﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>

#include "../Common/applet_Common.h"

NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

//-----------------------------------------------------------------------------

nn::applet::LibraryAppletHandle PrepareAloneLibraryAppletImpl(const void* data, size_t size)
{
    nn::applet::LibraryAppletHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLibraryApplet(&handle, nn::applet::AppletId_LibraryAppletWeb, nn::applet::LibraryAppletMode_AllForeground));
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, size));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, data, size));
        nn::applet::PushToInChannel(handle, storageHandle);
    }
    return handle;
}

void FinishAloneLibraryAppletImpl(nn::applet::LibraryAppletHandle handle, char expected) NN_NOEXCEPT
{
    nn::applet::JoinLibraryApplet(handle);
    static char g_Data[1024 * 4] = {};
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::applet::TryPopFromOutChannel(&storageHandle, handle));
        NN_ABORT_UNLESS(nn::applet::GetStorageSize(storageHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, g_Data, sizeof(g_Data)));
        nn::applet::ReleaseStorage(storageHandle);
    }
    nn::applet::CloseLibraryApplet(handle);
    for (auto c : g_Data)
    {
        NN_ABORT_UNLESS_EQUAL(expected, c);
    }
}

nn::ae::LibraryAppletStartHookResult AloneLibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void *) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(appletId == nn::applet::AppletId_LibraryAppletWeb);
    NN_ABORT_UNLESS(libraryAppletMode == nn::applet::LibraryAppletMode_AllForeground);
    return nn::ae::LibraryAppletStartHookResult_WindProgram;
}

void LibraryAppletMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    NN_ABORT_UNLESS(info.appletId == nn::applet::AppletId_LibraryAppletWeb);
    NN_ABORT_UNLESS(info.libraryAppletMode == nn::applet::LibraryAppletMode_AllForeground);

    //-------------------------------------------------------------------------
    // ライブラリアプレットの初期化
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    // ライブラリアプレットに FG 遷移要求
    NN_LOG("LAA: Wait Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LAA", &event, nn::ae::Message_ChangeIntoForeground);
    nn::ae::AcquireForegroundRights();

    if (!info.isUnwound)
    {
        NN_LOG("LAA: Not unwound.\n");
        // 入力データのチェック
        static char g_Data[1024 * 4] = {};
        {
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
            NN_ABORT_UNLESS(nn::applet::GetStorageSize(storageHandle));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(storageHandle, 0, g_Data, sizeof(g_Data)));
            nn::ae::UnpopToInChannel(storageHandle);
        }
        if (g_Data[0] == 0xAB)
        {
            NN_LOG("LAA1: launched...");
            // 1 段目の起動
            // 0xAB を確認
            for (auto c : g_Data)
            {
                NN_ABORT_UNLESS_EQUAL(0xAB, c);
            }
            NN_LOG("confirmed 0xAB\n");
            // 2 段目を 0xCD で起動
            NN_LOG("LAA1: start LAA2(0xCD) with winding.\n");
            std::memset(g_Data, 0xCD, sizeof(g_Data));
            auto handle = PrepareAloneLibraryAppletImpl(g_Data, sizeof(g_Data));
            nn::applet::StartLibraryApplet(handle);
            NN_ABORT("unexpected");
        }
        else if (g_Data[0] == 0xCD)
        {
            NN_LOG("LAA2: launched... ");
            // 2 段目の起動
            // 0xCD を確認
            for (auto c : g_Data)
            {
                NN_ABORT_UNLESS_EQUAL(0xCD, c);
            }
            NN_LOG("confirmed 0xCD\n");
            // 0xEF を返す
            NN_LOG("LAA2: return 0xEF.\n");
            std::memset(g_Data, 0xEF, sizeof(g_Data));
            {
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_Data)));
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, g_Data, sizeof(g_Data)));
                nn::ae::PushToOutChannel(storageHandle);
            }
            nn::ae::ExitLibraryApplet();
        }
        else
        {
            NN_ABORT("unexpected");
        }
    }
    else
    {
        NN_LOG("LAA1: unwound.\n");
        // 一段目の unwind 起動
        static char g_Data[1024 * 4] = {};
        FinishAloneLibraryAppletImpl(info.previousLibraryAppletHandle, static_cast<char>(0xEF));
        // 0x01 を返す
        NN_LOG("LAA2: return 0x01.\n");
        std::memset(g_Data, 0x01, sizeof(g_Data));
        {
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, sizeof(g_Data)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(storageHandle, 0, g_Data, sizeof(g_Data)));
            nn::ae::PushToOutChannel(storageHandle);
        }
        nn::ae::ExitLibraryApplet();
    }

    //-------------------------------------------------------------------------
    // ライブラリアプレットの終了
    NN_LOG("LAA: Invoke nn::ae::ExitLibraryApplet().\n");
    nn::ae::ExitLibraryApplet();
}

nn::applet::LibraryAppletHandle CreateSelfLibraryApplet() NN_NOEXCEPT
{
    static char g_Data[1024 * 4];
    std::memset(g_Data, 0xAB, sizeof(g_Data));
    return PrepareAloneLibraryAppletImpl(g_Data, sizeof(g_Data));
}

extern "C" void nnMain()
{
    NN_LOG("LAA: Invoked nn::ae::InvokeLibraryAppletMain().\n");

    nn::ae::SetLibraryAppletStartHook(AloneLibraryAppletStartHook);
    nn::ae::SetCreateSelfLibraryAppletForDevelop(CreateSelfLibraryApplet);
    nn::ae::InvokeLibraryAppletMain(LibraryAppletMain);
}

