﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>

#include "../Common/applet_Common.h"

//-----------------------------------------------------------------------------
//  Audio 制御
//
void ControlMainAppletAudio()
{
    // アプリのボリューム操作
    auto maExpectedVolume = nn::ae::GetMainAppletExpectedMasterVolume();
    auto laExpectedVolume = nn::ae::GetLibraryAppletExpectedMasterVolume();

    NN_LOG("LA3: Current Application's expected volume   = %1.2f\n", maExpectedVolume);
    NN_LOG("LA3: Current LibraryApplet's expected volume = %1.2f\n", laExpectedVolume);
    NN_ABORT_UNLESS(maExpectedVolume == 0.25f);
    NN_ABORT_UNLESS(laExpectedVolume == 1.0f );

    nn::ae::ChangeMainAppletMasterVolume( 0.47f, nn::TimeSpan::FromMilliSeconds(13) );
    NN_LOG("LA3: Invoked nn::ae::ChangeMainAppletMasterVolume()\n");
}


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}


//-----------------------------------------------------------------------------
//  メイン関数
//
void LibraryAppletAuthMain(const nn::ae::LibraryAppletSelfInfo&) NN_NOEXCEPT
{
    NN_LOG("LA3: Launched an LibraryApplet3.\n");

    //-------------------------------------------------------------------------
    // ライブラリアプレットの初期化
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    bool mainIsApplication;

    {
        // 小さいサイズの入力データのチェック
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
        int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
        NN_LOG("LA3: Pop from InChannel[0] -> size = %d\n", inStorageSize);
        const auto ExpectedSize = nn::ae::StartupParamSizeMax;
        NN_ABORT_UNLESS(inStorageSize == ExpectedSize);
        static char buffer[ExpectedSize];
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, buffer, ExpectedSize));
        ReleaseStorage(storageHandle);
        mainIsApplication = buffer[0] == 'A';
        CheckMemory(buffer, sizeof(buffer), mainIsApplication ? 'A' : 'a');
    }

    auto mainAppletIdentityInfo = nn::ae::GetMainAppletIdentityInfo();
    auto callerAppletIdentityInfo = nn::ae::GetCallerAppletIdentityInfo();
    if (mainIsApplication)
    {
        NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_Application, mainAppletIdentityInfo.appletId);
        // TODO: floating 起動の場合、現在 applicationId を取ることができず 0 が返る
        if (mainAppletIdentityInfo.applicationId.value != 0)
        {
            NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, mainAppletIdentityInfo.applicationId.value);
        }

        // アプリ系から呼ばれる場合は Winding-LA1 から呼ばれる
        NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_LibraryAppletAuth, callerAppletIdentityInfo.appletId);
    }
    else
    {
        NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_SystemAppletMenu, mainAppletIdentityInfo.appletId);
        NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_SystemAppletMenu, callerAppletIdentityInfo.appletId);
    }

    // ライブラリアプレットに FG 遷移要求
    NN_LOG("LA3: Wait Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LA3", &event, nn::ae::Message_ChangeIntoForeground);
    nn::ae::AcquireForegroundRights();

    // アプリの Audio 制御
    NN_LOG("LA3: Control main applet's volume.\n");
    ControlMainAppletAudio();

    //-------------------------------------------------------------------------
    // ライブラリアプレットの終了
    NN_LOG("LA3: Invoke nn::ae::ExitLibraryApplet().\n");
    {
        // 小さいデータの出力
        static char g_ReturnValueBuffer[nn::applet::ReturnValueSizeMax];
        FillMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'B');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_ReturnValueBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
        nn::ae::PushToOutChannel(storageHandle);
    }
}

extern "C" void nnMain()
{
    NN_LOG("LA3: Invoke nn::ae::InvokeLibraryAppletMain().\n");
    nn::ae::InvokeLibraryAppletMain(LibraryAppletAuthMain);
}

