﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

#include "../Common/applet_Common.h"

//-----------------------------------------------------------------------------

namespace {

const size_t OptionalStartupParamSize = 2 * 1024 * 1024;
const size_t OptionalReturnValueSize  = 2 * 1024 * 1024;
const size_t TransferStorageBufferSize = 1 * 1024 * 1024;
NN_ALIGNAS(4096) char g_OptionalStartupParamBuffer[ OptionalStartupParamSize ];
NN_ALIGNAS(4096) char g_OptionalReturnValueBuffer[ OptionalReturnValueSize ];

}

//-----------------------------------------------------------------------------
//  Audio 制御
//
void ControlMainAppletAudio()
{
    // アプリのボリューム操作
    auto maExpectedVolume = nn::ae::GetMainAppletExpectedMasterVolume();
    auto laExpectedVolume = nn::ae::GetLibraryAppletExpectedMasterVolume();

    NN_LOG("LA1: Current Application's expected volume   = %1.2f\n", maExpectedVolume);
    NN_LOG("LA1: Current LibraryApplet's expected volume = %1.2f\n", laExpectedVolume);
    NN_ABORT_UNLESS(maExpectedVolume == 0.25f);
    NN_ABORT_UNLESS(laExpectedVolume == 1.0f );

    nn::ae::ChangeMainAppletMasterVolume( 0.49f, nn::TimeSpan::FromMilliSeconds(11) );
    NN_LOG("LA1: Invoked nn::ae::ChangeMainAppletMasterVolume()\n");
}


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

//-----------------------------------------------------------------------------
//  LA 起動フック
//
char g_HookWindProgramMarker;
nn::ae::LibraryAppletStartHookResult La1LibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void* userParameter) NN_NOEXCEPT
{
    if (userParameter == nullptr)
    {
        return nn::ae::LibraryAppletStartHookResult_Normal;
    }
    else if (userParameter == &g_HookWindProgramMarker)
    {
        NN_LOG("LA1: Prepare for winding program");
        const size_t scratchPadSize = 4 * 1024;
        static char g_ContextBuffer[scratchPadSize];
        NN_LOG("LA1: Save Context ( %d KB )\n", scratchPadSize / 1024);
        // 小さいデータの入力
        FillMemory(g_ContextBuffer, sizeof(g_ContextBuffer), 'C');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_ContextBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_ContextBuffer, sizeof(g_ContextBuffer)));
        nn::ae::PushToContextStack(storageHandle);
        return nn::ae::LibraryAppletStartHookResult_WindProgram;
    }
    else
    {
        NN_ABORT("unexpected");
    }
}

//-----------------------------------------------------------------------------
//  メイン関数
//
void LibraryAppletAuthMain(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    auto id = nn::applet::GetAppletResourceUserId();
    NN_LOG("LA1: ARUID = 0x%016llx\n", id.lower);
    NN_ABORT_UNLESS(id.lower != 0);

    // 現在のシーケンスではアプリからのみ起動
    auto mainAppletIdentityInfo = nn::ae::GetMainAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_Application, mainAppletIdentityInfo.appletId);

    auto callerAppletIdentityInfo = nn::ae::GetCallerAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_Application, callerAppletIdentityInfo.appletId);

    // TODO: floating 起動の場合、現在 applicationId を取ることができず 0 が返る
    if (mainAppletIdentityInfo.applicationId.value != 0)
    {
        NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, mainAppletIdentityInfo.applicationId.value);
    }

    //-------------------------------------------------------------------------
    // ライブラリアプレットの初期化
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    //-------------------------------------------------------------------------
    // スクラッチパッド対応

    if (info.isUnwound)
    {
        auto la3 = info.previousLibraryAppletHandle;
        NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_LibraryAppletController, GetLibraryAppletId(la3));
        NN_ABORT_UNLESS_EQUAL(nn::applet::LibraryAppletMode_AllForeground, GetLibraryAppletMode(la3));

        // スクラッチパッド起動
        NN_LOG("LA1: Unwinding LibraryApplet1\n");

        // LA3 の終了確認
        NN_LOG("LA1: Join LA3\n");
        JoinLibraryApplet(la3);
        NN_ABORT_UNLESS(GetLibraryAppletExitReason(la3) == nn::applet::LibraryAppletExitReason_Normal);

        // LA1 に対して FG 遷移要求
        NN_LOG("LA1: Wait nn::applet::Message_ChangeIntoForeground\n");
        CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoForeground);
        nn::ae::AcquireForegroundRights();

        {
            // コンテキストスタックのチェック
            const size_t scratchPadSize = 4 * 1024;
            static char g_ContextBuffer[scratchPadSize];
            NN_LOG("LA1: Check LA3's context stack\n");
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS(nn::ae::TryPopFromContextStack(&storageHandle));
            NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_ContextBuffer));
            NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_ContextBuffer, sizeof(g_ContextBuffer)));
            ReleaseStorage(storageHandle);
            CheckMemory(g_ContextBuffer, sizeof(g_ContextBuffer), 'C');
        }

        // la3 のクローズ
        CloseLibraryApplet(la3);

        {
            // 小さいデータの出力
            static char g_ReturnValueBuffer[nn::applet::ReturnValueSizeMax];
            FillMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'D');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_ReturnValueBuffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
            nn::ae::PushToOutChannel(storageHandle);
        }
        {
            // 大きいサイズの入力データのチェック
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
            int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
            NN_LOG("LA1: Pop from InChannel[2] -> size = %d\n", inStorageSize);
            const auto ExpectedSize = sizeof(g_OptionalReturnValueBuffer);
            NN_ABORT_UNLESS(inStorageSize == ExpectedSize);
            FillMemory(g_OptionalReturnValueBuffer, sizeof(g_OptionalReturnValueBuffer), '2');
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_OptionalReturnValueBuffer, ExpectedSize));
            nn::ae::PushToOutChannel(storageHandle);
        }
        // あえてここでは ExitLibraryApplet() を呼ばない。
        // InvokeLibraryAppletMain() を経由して終了した場合に、
        // 終了コードが正しく LA 呼出元に返るかどうかをテストするため。
        return;
    }

    NN_LOG("LA1: Launched an LibraryApplet1 (Non-ScratchPad)\n");

    {
        // 小さいサイズの入力データのチェック
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
        int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
        NN_LOG("LA1: Pop from InChannel[0] -> size = %d\n", inStorageSize);
        const auto ExpectedSize = nn::ae::StartupParamSizeMax;
        NN_ABORT_UNLESS(inStorageSize == ExpectedSize);
        static char buffer[ExpectedSize];
        {
            // 非転送ストレージでの Map に失敗する
            void* address;
            size_t size;
            NN_ABORT_UNLESS(nn::ae::MapTransferStorage(&address, &size, storageHandle) <= nn::applet::ResultNotTransferStorage());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, buffer, ExpectedSize));
        ReleaseStorage(storageHandle);
        if (buffer[0] == 'Z')
        {
            NN_LOG("LA1: abort (expected for test)\n");
            NN_ABORT("for test");
        }
        CheckMemory(buffer, sizeof(buffer), 'A');
    }
    {
        // 大きいサイズの入力データのチェック
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
        int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
        NN_LOG("LA1: Pop from InChannel[1] -> size = %d\n", inStorageSize);
        const auto ExpectedSize = sizeof(g_OptionalStartupParamBuffer);
        NN_ABORT_UNLESS(inStorageSize == ExpectedSize);
        {
            // 非転送ストレージでの Map に失敗する
            void* address;
            size_t size;
            NN_ABORT_UNLESS(nn::ae::MapTransferStorage(&address, &size, storageHandle) <= nn::applet::ResultNotTransferStorage());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_OptionalStartupParamBuffer, ExpectedSize));
        ReleaseStorage(storageHandle);
        CheckMemory(g_OptionalStartupParamBuffer, sizeof(g_OptionalStartupParamBuffer), '0');
    }
    {
        // 転送ストレージの入力データのチェック
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));

        // 入力データをマップして内容をチェック
        void* address;
        size_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::MapTransferStorage(&address, &size, storageHandle));
        NN_LOG("LA1: Mapped TransferStorage of InChannel[2] -> address = %p\n", address);
        NN_LOG("LA1: Mapped TransferStorage of InChannel[2] -> size    = %p\n", size);
        CheckMemory(reinterpret_cast<char*>(address), size, '2');

        // ReadFromStorage(), WriteToStorage() は失敗する
        const auto ExpectedSize = TransferStorageBufferSize;
        NN_ABORT_UNLESS(size == ExpectedSize);
        NN_ABORT_UNLESS(nn::applet::ReadFromStorage(storageHandle, 0, g_OptionalStartupParamBuffer, ExpectedSize) <= nn::applet::ResultNotMemoryStorage());
        NN_ABORT_UNLESS(nn::applet::WriteToStorage(storageHandle, 0, g_OptionalStartupParamBuffer, ExpectedSize) <= nn::applet::ResultNotMemoryStorage());

        // マップ後のマップに失敗、アンマップに成功
        NN_ABORT_UNLESS(nn::ae::MapTransferStorage(&address, &size, storageHandle) <= nn::applet::ResultAlreadyMappedTransferStorage());
        nn::ae::UnmapTransferStorage(storageHandle);

        // リリース
        ReleaseStorage(storageHandle);
    }
    {
        // 終了用バッファのチェックと unpop
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&storageHandle));
        int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
        NN_LOG("LA1: Pop from InChannel[2] -> size = %d\n", inStorageSize);
        const auto ExpectedSize = sizeof(g_OptionalReturnValueBuffer);
        NN_ABORT_UNLESS(inStorageSize == ExpectedSize);
        NN_LOG("LA1: Unpop to InChannel[2] -> size = %d\n", inStorageSize);
        nn::ae::UnpopToInChannel(storageHandle);
    }

    // ライブラリアプレットに FG 遷移要求
    NN_LOG("LA1: Wait Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoForeground);
    nn::ae::AcquireForegroundRights();

    // アプリの Audio 制御
    NN_LOG("LA1: Control application's volume.\n");
    ControlMainAppletAudio();
    NN_LOG("LA1: WARNNING: TODO: implemet\n");


    //-------------------------------------------------------------------------
    // LA が FG 状態の場合に、SA からの FG 遷移要求を正しくハンドリングできるか
    //-------------------------------------------------------------------------
    // SA に対して HOME ボタン短押しを発行
    NN_LOG("LA1: Press Home Button Short.\n");
    nn::ae::NotifyShortPressingHomeButtonForDebug();

    // SA からの BG 遷移要求を待機
    NN_LOG("LA1: Wait for Message_ChangeIntoBackground from SystemAppletMenu.\n");
    CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoBackground);
    NN_LOG("LA1: Invoke nn::ae::ReleaseForegroundRights().\n");
    nn::ae::ReleaseForegroundRights();

    // SA からの FG 遷移要求を待機
    NN_LOG("LA1: Wait for Message_ChangeIntoForeground from SystemAppletMenu.\n");
    CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoForeground);
    NN_LOG("LA1: Invoke nn::ae::AcquireForegroundRights().\n");
    nn::ae::AcquireForegroundRights();

    //-------------------------------------------------------------------------
    // FG 状態の LA1 から更に別の LA3 を起動
    //-------------------------------------------------------------------------

    {
        NN_LOG("LA1: Invoke nn::applet::CreateLibraryApplet( LA3, Foreground )\n");

        // ライブラリアプレットの作成
        nn::applet::LibraryAppletHandle la3;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLibraryApplet(&la3, nn::applet::AppletId_LibraryAppletController, nn::applet::LibraryAppletMode_AllForeground));

        {
            // 小さいデータの入力
            static char g_StartupParamBuffer[ nn::applet::StartupParamSizeMax ];
            FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'A');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
            PushToInChannel(la3, storageHandle);
        }

        NN_LOG("LA1: Invoke nn::applet::StartLibraryApplet(LA3)\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(la3));

        // LA1 に対して BG 遷移要求
        NN_LOG("LA1: Wait nn::applet::Message_ChangeIntoBackground(LA3)\n");
        CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoBackground);
        nn::ae::ReleaseForegroundRights();


        // LA3 の終了待機
        NN_LOG("LA1: Wait for LA3 exit\n");
        WaitSystemEvent(GetLibraryAppletExitEvent(la3));

        // LA3 の終了確認
        NN_LOG("LA1: Join LA3\n");
        JoinLibraryApplet(la3);
        NN_ABORT_UNLESS(GetLibraryAppletExitReason(la3) == nn::applet::LibraryAppletExitReason_Normal);

        {
            // 小さいデータの出力確認
            static char g_ReturnValueBuffer[ nn::applet::ReturnValueSizeMax ];
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS(TryPopFromOutChannel(&storageHandle, la3));
            NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_ReturnValueBuffer));
            NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer)));
            CheckMemory(g_ReturnValueBuffer, sizeof(g_ReturnValueBuffer), 'B');
        }

        // ハンドルのクローズ
        NN_LOG("LA1: Invoke nn::applet::CloseLibraryApplet()\n");
        nn::applet::CloseLibraryApplet(la3);
    }

    // LA1 に対して FG 遷移要求
    NN_LOG("LA1: Wait nn::applet::Message_ChangeIntoForeground()\n");
    CheckAndProcessMessage("LA1", &event, nn::ae::Message_ChangeIntoForeground);
    nn::ae::AcquireForegroundRights();

    //-------------------------------------------------------------------------
    // FG 状態の LA1 から更に別の LA3 を起動（スクラッチパッド起動）
    //-------------------------------------------------------------------------

    {
        NN_LOG("LA1: Invoke nn::applet::CreateLibraryApplet( LA3, Foreground ) with ScratchPad\n");

        // ライブラリアプレットの作成
        nn::applet::LibraryAppletHandle la3;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateLibraryApplet(&la3, nn::applet::AppletId_LibraryAppletController, nn::applet::LibraryAppletMode_AllForeground));

        {
            // 小さいデータの入力
            static char g_StartupParamBuffer[ nn::applet::StartupParamSizeMax ];
            FillMemory(g_StartupParamBuffer, sizeof(g_StartupParamBuffer), 'A');
            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_StartupParamBuffer)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_StartupParamBuffer, sizeof(g_StartupParamBuffer)));
            PushToInChannel(la3, storageHandle);
        }

        NN_LOG("LA1: Invoke nn::applet::StartLibraryApplet(LA3)\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::StartLibraryApplet(la3, &g_HookWindProgramMarker));

        NN_ABORT("unexpected");
    }
}   // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    NN_LOG("LA1: Invoke nn::ae::InvokeLibraryAppletMain().\n");

    nn::ae::SetLibraryAppletStartHook(La1LibraryAppletStartHook);
    nn::ae::InvokeLibraryAppletMain(LibraryAppletAuthMain);
}

