﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "../Common/Common.h"
#include "Graphics.h"

#include <nn/fs/fs_SdCardForDebug.h>

// このマクロが有効だと QueueFinish() を呼ぶ従来のフレームワークになります。
#define QUEUE_FINISH_ENABLED

namespace Graphics {

namespace {

// クリアカラー
Rgba g_ClearColor;

// タイトル
std::string g_Title;

// partial かどうか
bool g_IsPartialForeground = false;

// 線形テクスチャを有効にするか
bool g_IsEnabledLinearTextures = false;
const char* g_pCaptureBufferName1 = "";
const char* g_pCaptureBufferName2 = "";

// 線形テクスチャの数
const size_t g_LinearTextureNum = 2;

// VSync イベント
nn::os::Event g_VsyncEvent( nn::os::EventClearMode_AutoClear );

// メッセージ
const char *g_pMessage = NULL;

// 描画するオブジェクト数
int g_ObjectCount = 0;

// スリープ時間 g_SleepCount * 200us
int g_SleepCount = 0;

// バッファ数 (2 以上を指定してください)
const int g_BufferCount = 2;

// Home Button 押し待ちモード
bool g_IsWaitPressHomeButton = false;

// フレーム番号
unsigned int g_FrameCount = 0;
unsigned int g_SavedFrameCount = 0;

// FS に使用するバッファ
void* g_MountRomCacheBuffer = NULL;
size_t g_MountRomCacheBufferSize = 0;

// Audio が再生中かどうか
bool g_IsAudioEnabled = false;

//-----------------------------------------------------------------------------
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

// グラフィックスフレームワークを初期化
nns::gfx::GraphicsFramework g_GraphicsFramework;
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
static int g_BufferIndex = 0;
void InitializeGraphicsFramework()
{
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 16 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 20 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, 0 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1 * 1024 * 1024);
    fwInfo.SetBufferCount(g_BufferCount);
    fwInfo.SetSwapChainBufferCount(g_BufferCount);
    g_GraphicsFramework.Initialize(fwInfo);
}

// グラフィックスフレームワークを破棄
void FinalizeGraphicsFramework()
{
    g_GraphicsFramework.Finalize();
}

// 線形テクスチャを初期化
nn::gfx::Texture g_LinearTexture[g_LinearTextureNum];
ptrdiff_t g_LinearTextureOffset[g_LinearTextureNum];
ptrdiff_t g_TextureMemoryPoolOffset[g_LinearTextureNum];
void InitializeLinearTextures()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    info.SetWidth( g_GraphicsFramework.GetDisplayWidth() );
    info.SetHeight( g_GraphicsFramework.GetDisplayHeight() );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    info.SetMipCount( 1 );
    info.SetTileMode( nn::gfx::TileMode_Linear );

    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_LinearTextureOffset[i] = 0;

        size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info );
        size_t size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info );
        size_t pitch = nn::gfx::Texture::GetRowPitch(g_GraphicsFramework.GetDevice(), info );
        g_TextureMemoryPoolOffset[i] = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, size, alignment);
        nn::gfx::MemoryPool* memoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
        uint8_t* pixels = nn::util::BytePtr(memoryPool->Map(), g_TextureMemoryPoolOffset[i]).Get< uint8_t >();
        for ( int y = 0; y < info.GetHeight(); ++y )
        {
            uint8_t* row = pixels + y * pitch;
            for ( int x = 0; x < info.GetWidth(); ++x )
            {
                uint8_t* pixel = row + x * 4;
                pixel[ 0 ] = static_cast< uint8_t >( x );
                pixel[ 1 ] = static_cast< uint8_t >( y );
                pixel[ 2 ] = 0x00;
                pixel[ 3 ] = 0xFF;
            }
        }
        g_LinearTexture[i].Initialize(g_GraphicsFramework.GetDevice(), info, memoryPool, g_TextureMemoryPoolOffset[i], size );
        g_LinearTextureOffset[i] = g_TextureMemoryPoolOffset[i];
    }
};

// 線形テクスチャを破棄
void FinalizeLinearTextures()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_GraphicsFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_TextureMemoryPoolOffset[i]);
        g_LinearTexture[i].Finalize(g_GraphicsFramework.GetDevice());
    }
}

// 線形テクスチャビューを初期化
nn::gfx::TextureView g_LinearTextureView[g_LinearTextureNum];
void InitializeLinearTextureViews()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetImageDimension( nn::gfx::ImageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        info.SetTexturePtr( &g_LinearTexture[i] );
        info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
            nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha );
        g_LinearTextureView[i].Initialize(g_GraphicsFramework.GetDevice(), info );
    }
}

// 線形テクスチャビューを破棄
void FinalizeLinearTextureViews()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_LinearTextureView[i].Finalize(g_GraphicsFramework.GetDevice());
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    g_Sampler.Initialize(g_GraphicsFramework.GetDevice(), info);
}

// サンプラを破棄
void FinalizeSampler()
{
    g_Sampler.Finalize(g_GraphicsFramework.GetDevice());
}

//  メモリ確保関数 for fs
void* AllocateForFs( size_t size )
{
    return malloc( size );
}

//  メモリ開放関数 for fs
void DeallocateForFs( void* ptr, size_t )
{
    free( ptr );
}

// FS の初期化
void InitializeFs()
{
    nn::Result result;

    result = nn::fs::QueryMountRomCacheSize( &g_MountRomCacheBufferSize );
    NN_ABORT_UNLESS( result.IsSuccess() );
    g_MountRomCacheBuffer = AllocateForFs( g_MountRomCacheBufferSize );
    NN_ABORT_UNLESS_NOT_NULL( g_MountRomCacheBuffer );

    result = nn::fs::MountRom( "Contents", g_MountRomCacheBuffer, g_MountRomCacheBufferSize );
    NN_ABORT_UNLESS( result.IsSuccess() );
}

// FS の終了
void FinalizeFs()
{
    nn::fs::Unmount( "Contents" );

    DeallocateForFs( g_MountRomCacheBuffer, g_MountRomCacheBufferSize );
    g_MountRomCacheBuffer = NULL;
    g_MountRomCacheBufferSize = 0;
}

// リソースの読み込み
char g_ResourceBuffer[ 4 * 1024 * 1024 ] NN_ALIGNAS(4096); // TORIAEZU : 静的確保
void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );

    void* pBuffer = g_ResourceBuffer;
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

// テクスチャリソースファイルの初期化
nn::gfx::ResTextureFile* g_pResTextureFile;
int g_IndexPressHomeButton = -1;
void InitializeResTextureFile()
{
    void *pTextureResource = ReadResource( "Contents:/Check.bntx" );
    g_pResTextureFile = nn::gfx::ResTextureFile::ResCast( pTextureResource );
    g_pResTextureFile->Initialize( g_GraphicsFramework.GetDevice() );
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Initialize( g_GraphicsFramework.GetDevice() );
    }
    g_IndexPressHomeButton = g_pResTextureFile->GetTextureDic()->FindIndex("press_home_button");
    NN_ASSERT(g_IndexPressHomeButton != -1);
}

// テクスチャリソースファイルの破棄
void FinalizeResTextureFile()
{
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
         idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Finalize( g_GraphicsFramework.GetDevice() );
    }
    g_pResTextureFile->Finalize( g_GraphicsFramework.GetDevice() );
}

// プリミティブレンダラの初期化
nn::mem::StandardAllocator  g_PrimitiveRendererAllocator;
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(g_GraphicsFramework.DefaultAllocateFunction, NULL);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetDrawCallCountMax(1024 * 4);
    info.SetViewFunctionCallCountMax(1024 * 4);
    info.SetMultiBufferQuantity(g_BufferCount);

    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(g_GraphicsFramework.GetDevice(), info);
    g_pPrimitiveRenderer->SetScreenWidth(g_GraphicsFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

// プリミティブレンダラの破棄
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), g_GraphicsFramework.DefaultFreeFunction, NULL);
}

// パフォーマンス計測クラスの初期化
nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
void*  g_MeterMemory;
ptrdiff_t g_MeterMemoryPoolOffset;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(4);
#ifdef QUEUE_FINISH_ENABLED
    info.SetGpuBufferCount(2);
#else
    info.SetGpuBufferCount(3);
#endif
    info.SetCpuSectionCountMax(8);
    info.SetGpuSectionCountMax(500);

    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_MeterMemory = g_GraphicsFramework.AllocateMemory(memorySize, memoryAlignment);

    // メモリプールの確保
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
    g_MeterMemoryPoolOffset = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info));
    nn::gfx::MemoryPool* meterMemoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);
    NN_PERF_INITIALIZE_METER(g_GraphicsFramework.GetDevice(), info,
        g_MeterMemory, memorySize,
        meterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
    NN_ASSERT(g_MeterMemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");
}

// パフォーマンス計測クラスの破棄
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(g_GraphicsFramework.GetDevice());
    g_GraphicsFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_MeterMemoryPoolOffset);
    g_GraphicsFramework.FreeMemory(g_MeterMemory);
}

// コマンド作成
void MakeCommand( unsigned int frame )
{
    nn::gfx::CommandBuffer* pCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex);
    g_GraphicsFramework.BeginFrame(g_BufferIndex);
    {
        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
        NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "All");

        nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

        pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
        pCommandBuffer->ClearDepthStencil(g_GraphicsFramework.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
        pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
        pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

        g_pPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::Vector3f translate;
        nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);

        // Blend
        pCommandBuffer->SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

        // ビューとプロジェクションを設定
        int count = g_IsWaitPressHomeButton ? g_SavedFrameCount : frame;
        float radius = 20.f;
        float x      = radius * sin( count / 500.f );
        float z      = radius * cos( count / 500.f );
                nn::util::Vector3fType camPos = { x, 10.f, z };
        nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);

        const float fovy = nn::util::FloatPi / 3.0f;
        const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);

        g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);


        // Depth Enable
        g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);

        // 軸を描画
        float interval = -10.f;
        nn::util::Vector3fType begin;
        nn::util::Vector3fType end;
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        for (int i = 0; i < 21; i++)
        {
            nn::util::VectorSet(&begin, -10.f, 0.f, interval);
            nn::util::VectorSet(&end, 10.f, 0.f, interval);
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            g_pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            nn::util::VectorSet(&begin, interval, 0.f, -10.f);
            nn::util::VectorSet(&end, interval, 0.f, 10.f);
            g_pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            interval += 1.0f;
        }

        // オブジェクトを描画
        nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType size = { 30.f, 30.f, 30.f };
        for (int i = 0; i < g_ObjectCount; ++i)
        {
            if (i % 2 == 0)
            {
                NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Black());
            }
            else
            {
                NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::White());
            }
            NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "test");
            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
            pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
            nn::util::VectorSet(&translate, 0, 0, 0);
            nn::util::MatrixSetAxisW(&modelMatrix, translate);
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
            g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
            g_pPrimitiveRenderer->SetLineWidth(14.f);
            g_pPrimitiveRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
            NN_PERF_END_MEASURE_GPU(pCommandBuffer);
        }

        // Depth Disable
        g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest );

        // タイトルを描画
        g_Writer.object.SetTextColor( nn::util::Color4u8::White() );
        g_Writer.object.SetScale( 3, 3 );
        g_Writer.object.SetCursor( 30, 150 );
        g_Writer.object.Print( g_Title.c_str() );

        // メッセージ描画
        if (g_pMessage)
        {
            g_Writer.object.SetCursor(30, 210);
            g_Writer.object.Print("%s", g_pMessage);
        }

        g_Writer.object.SetScale(1.5f, 1.5f);

        // フレームカウントを描画
        g_Writer.object.SetCursor( 30, 280 );
        g_Writer.object.Print( "Frame : %d", frame );

        // ビューをリセット
        g_pPrimitiveRenderer->SetDefaultParameters();

        // 2D Rect 描画
        if ( g_IsEnabledLinearTextures )
        {
            // 線形テクスチャのデスクリプタを取得
            nn::gfx::DescriptorSlot rectDescriptor[g_LinearTextureNum];
            nn::gfx::DescriptorSlot samplerDescriptor;
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&rectDescriptor[0], g_TextureDescriptorBaseIndex);
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&rectDescriptor[1], g_TextureDescriptorBaseIndex + 1);
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->GetDescriptorSlot(&samplerDescriptor, g_SamplerDescriptorBaseIndex);

            g_Writer.object.SetScale( 2, 2 );

            g_Writer.object.SetCursor( 760, 5 );
            g_Writer.object.Print( g_pCaptureBufferName1 );
            g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer,
                760, 53, 480, 270, rectDescriptor[0], samplerDescriptor );

            g_Writer.object.SetCursor( 760, 335 );
            g_Writer.object.Print( g_pCaptureBufferName2 );
            g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer,
                760, 383, 480, 270, rectDescriptor[1], samplerDescriptor );
        }

        if ( g_IsWaitPressHomeButton )
        {
            // テクスチャリソースファイルのデスクリプタを取得
            nn::gfx::DescriptorSlot rectDescriptor;
            nn::gfx::DescriptorSlot samplerDescriptor;
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&rectDescriptor, g_TextureDescriptorBaseIndex + 2);
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->GetDescriptorSlot(&samplerDescriptor, g_SamplerDescriptorBaseIndex);

            // Push Press Button 画像描画
            g_pPrimitiveRenderer->Draw2DRect( pCommandBuffer,
                440, 210, 400, 300, rectDescriptor, samplerDescriptor );
        }

        // 処理負荷メータを描画
        g_Writer.object.SetScale(1, 1);
        nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
        g_MeterDrawer.SetVisibleNoResult(false);
        nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, g_GraphicsFramework.GetDisplayHeight() - g_MeterDrawer.GetHeight(pFrameMeter) - 10);
        g_MeterDrawer.SetDebugFontTextWriter(&g_Writer.object);
        g_MeterDrawer.SetPosition(pos);
        g_MeterDrawer.SetScaleMax(15);
        g_MeterDrawer.SetWidth(g_GraphicsFramework.GetDisplayWidth() - 64.f);
        g_MeterDrawer.Draw(pCommandBuffer, g_pPrimitiveRenderer, pFrameMeter);

        float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
        g_Writer.object.SetScale(1.5f, 1.5f);
        g_Writer.object.SetCursor(30, 320);
        g_Writer.object.Print("FPS : %4.1f", fps);

        // オブジェクト数を描画
        g_Writer.object.SetCursor(30, 400);
        g_Writer.object.Print("Object : %d", g_ObjectCount);

        // GPU 負荷を描画
        nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
        nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
        g_Writer.object.SetCursor(30, 360);
        g_Writer.object.Print("GPU : %lldus", mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds());

        // Audio ステータス
        g_Writer.object.SetCursor(30, 440);
        g_Writer.object.Print("Audio : %s", g_IsAudioEnabled ? "Enabled" : "Disabled");

        // まとめて文字を書き出す
        g_Writer.object.Draw(pCommandBuffer);

        // カラーバッファからスキャンバッファへコピー
        {
            nn::gfx::TextureCopyRegion destRegion;
            destRegion.SetDefault();
            destRegion.SetWidth(480);
            destRegion.SetHeight(270);
            destRegion.SetOffsetU(260);
            destRegion.SetOffsetV(68);
            nn::gfx::TextureCopyRegion srcRegion;
            srcRegion.SetDefault();
            srcRegion.SetWidth(g_GraphicsFramework.GetDisplayWidth());
            srcRegion.SetHeight(g_GraphicsFramework.GetDisplayHeight());

            if (g_IsPartialForeground)
            {
                nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetSwapChain()->AcquireNextScanBufferView();
                pCommandBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.0f, 0.0f, NULL);
                pCommandBuffer->BlitImage(g_GraphicsFramework.GetSwapChain()->AcquireNextScanBuffer(), destRegion, g_GraphicsFramework.GetColorBuffer(), srcRegion, 0);
            }
            else
            {
                pCommandBuffer->BlitImage(g_GraphicsFramework.GetSwapChain()->AcquireNextScanBuffer(), srcRegion, g_GraphicsFramework.GetColorBuffer(), srcRegion, 0);
            }
        }

        NN_PERF_END_MEASURE_GPU(pCommandBuffer);
    }
    pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_QueryBuffer );
    pCommandBuffer->End();
} // NOLINT(impl/function_size)

} // namespace

// グラフィックスの初期処理
void InitializeGraphics( Rgba clearColor, std::string title )
{
    g_ClearColor = clearColor;
    g_Title = title;

    InitializeFs();

    const size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);

    // グラフィックスフレームワークの初期化
    InitializeGraphicsFramework();

    // テクスチャ初期化
    InitializeLinearTextures();
    InitializeLinearTextureViews();
    InitializeResTextureFile();
    g_TextureDescriptorBaseIndex = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 3);
    g_GraphicsFramework.SetTextureViewToDescriptorPool(g_TextureDescriptorBaseIndex, &g_LinearTextureView[0]);
    g_GraphicsFramework.SetTextureViewToDescriptorPool(g_TextureDescriptorBaseIndex + 1, &g_LinearTextureView[1]);
    g_GraphicsFramework.SetTextureViewToDescriptorPool(g_TextureDescriptorBaseIndex + 2,
        g_pResTextureFile->GetResTexture( g_IndexPressHomeButton )->GetTextureView() );

    //サンプラ初期化
    InitializeSampler();
    g_SamplerDescriptorBaseIndex = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    g_GraphicsFramework.SetSamplerToDescriptorPool(g_SamplerDescriptorBaseIndex, &g_Sampler);

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    g_GraphicsFramework.InitializeDebugFontTextWriter(&g_Writer, 256);

    // 負荷計測の初期化
    InitializeLoadMeter();
}

// グラフィックスの終了処理
void FinalizeGraphics()
{
    // 負荷計測の破棄
    FinalizeLoadMeter();

    // デバッグフォントの破棄
    g_GraphicsFramework.FinalizeDebugFontTextWriter(&g_Writer);

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    // サンプラ破棄
    g_GraphicsFramework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, g_SamplerDescriptorBaseIndex);
    FinalizeSampler();

    // テクスチャ破棄
    g_GraphicsFramework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, g_TextureDescriptorBaseIndex);
    FinalizeLinearTextures();
    FinalizeLinearTextureViews();
    FinalizeResTextureFile();

    // グラフィックスフレームワークの破棄
    FinalizeGraphicsFramework();

    FinalizeFs();
}

void SetPartialForeground(bool isPartialForeground)
{
    g_IsPartialForeground = isPartialForeground;
}

nn::vi::Layer* GetViLayer()
{
    return g_GraphicsFramework.GetLayer();
}

void EnableLinearTextures(const char* name1, const char* name2)
{
    g_IsEnabledLinearTextures = true;
    g_pCaptureBufferName1 = name1;
    g_pCaptureBufferName2 = name2;
}

void DisableLinearTextures()
{
    g_IsEnabledLinearTextures = false;
}

void SetMessage( const char* pMessage )
{
    g_pMessage = pMessage;
}

char* GetLinearTextureBuffer( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return NULL;
    }
    return nn::util::BytePtr(g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer)->Map(), g_LinearTextureOffset[index] ).Get< char >();
}

size_t GetLinearTextureSize( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayWidth() * g_GraphicsFramework.GetDisplayHeight() * 4;
}

size_t GetLinearTextureWidth( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayWidth();
}

size_t GetLinearTextureHeight( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayHeight();
}

int GetObjectCount()
{
    return g_ObjectCount;
}

void SetObjectCount(int objectCount)
{
    g_ObjectCount = std::min(objectCount, 400);
    g_ObjectCount = std::max(g_ObjectCount, 0);
}

int GetSleepCount()
{
    return g_SleepCount;
}

void SetSleepCount(int sleepCount)
{
    g_SleepCount = std::max(sleepCount, 0);
}

void SetAudioStatus(bool isEnable)
{
    g_IsAudioEnabled = isEnable;
}

void GraphicsRenderer()
{
    // フレーム全体の計測
    NN_PERF_BEGIN_FRAME();
    {
        nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

        NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
        NN_PERF_BEGIN_MEASURE();
        {
            // バッファの更新
            g_BufferIndex = g_GraphicsFramework.GetSwapChain()->AcquireNextScanBufferIndex();
            g_pPrimitiveRenderer->Update(g_BufferIndex);

            // コマンド作成と実行
            MakeCommand( g_FrameCount++ );
            queue->ExecuteCommand(g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex), NULL);
            queue->Flush();
        }
        NN_PERF_END_MEASURE();

        // sleep to increase CPU load
        {
            NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
            NN_PERF_BEGIN_MEASURE();
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_SleepCount * 200));
            NN_PERF_END_MEASURE();
        }

        // ディスプレイに描写するコマンドを提出
        queue->Present(g_GraphicsFramework.GetSwapChain(), 1);

#ifdef QUEUE_FINISH_ENABLED
        queue->Sync();
#endif
        g_VsyncEvent.Signal();
    }
    NN_PERF_END_FRAME();
}

void WaitVsync()
{
    g_VsyncEvent.Wait();
}

void SetWaitPressHomeButtonMode() NN_NOEXCEPT
{
    g_IsWaitPressHomeButton = true;
    g_SavedFrameCount = g_FrameCount;
}

} // Graphics
