﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/fs.h>
#include <nn/hid.h>
#include <nn/applet/applet.h>
#include <nn/util/util_Optional.h>
#include <nn/ns/ns_Result.h>

#include <nn/vi.private.h>

#include "../Common/Common.h"
#include "Graphics.h"
#include "../Common/ConvNpad.h"

//-----------------------------------------------------------------------------

namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// AE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// DebugPad 監視スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;

nn::os::SystemEventType g_MessageSystemEvent;

uint64_t g_GpuTimeSliceBoostInMicroSeconds;

// スレッド制御
ThreadControl g_ThreadControl;

class OverlayStateControl
{
public:
    void BeginUserInteraction() NN_NOEXCEPT
    {
        m_IsUserInteractionMode = true;
        nn::ae::SetTransparentVolumeRate(0.3f);
        nn::ae::BeginOverlayUserInteraction();
        NN_LOG("OA: Invoked nn::ae::BeginOverlayUserInteraction()\n");
        NN_LOG("OA: --> 'ZL' button enables APD setting time.\n");
        NN_LOG("OA:     'ZR' button disables APD setting time.\n");
    }

    void EndUserInteraction() NN_NOEXCEPT
    {
        m_IsUserInteractionMode = false;
        nn::ae::SetTransparentVolumeRate(1.0f);
        nn::ae::EndOverlayUserInteraction();
        NN_LOG("OA: Invoked nn::ae::EndOverlayUserInteraction()\n");
    }

    bool IsUserInteractionMode() NN_NOEXCEPT
    {
        return m_IsUserInteractionMode;
    }

private:
    bool    m_IsUserInteractionMode{false};
};

OverlayStateControl g_OverlayStateControl;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.0f, 0.0f, 0.0f, 0.0f };
    Graphics::InitializeGraphics( clearColor, "OverlayApplet" );

    // アプレット通知メッセージ用イベントの初期化
    nn::ae::InitializeNotificationMessageEvent( &g_MessageSystemEvent );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();

    // スリープの事前ハンドリングを行う
    nn::ae::BeginSleepHandling();
}

void Finalize()
{
    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

// VR モードのカーテン表示管理
nn::os::LightEvent g_IsVrModeCurtainRequired(nn::os::EventClearMode_ManualClear);

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void MessageThreadFunction( void *arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage( &g_MessageSystemEvent );
        switch (message)
        {
        // スリープ準備
        case nn::ae::Message_RequestToPrepareSleep:
            NN_LOG("OA: Received Message_RequestToPrepareSleep\n");
            g_ThreadControl.SetInFocusState(false);
            nn::ae::AllowToEnterSleepAndWait();
            break;

        // スリープ終了
        case nn::ae::Message_FinishedSleepSequence:
            NN_LOG("OA: Received Message_FinishedSleepSequence\n");
            g_ThreadControl.SetInFocusState(true);
            break;

        // 終了要求（通常、OA には来ない）
        case nn::ae::Message_Exit:
            NN_LOG("OA: Received Message_Exit\n");
            g_ThreadControl.RequestExit();
            return;

        // HOME ボタン短押し
        case nn::ae::Message_DetectShortPressingHomeButton:
            NN_LOG("OA: Received Message_DetectShortPressingHomeButton\n");
            if (g_OverlayStateControl.IsUserInteractionMode())
            {
                g_OverlayStateControl.EndUserInteraction();
            }
            else
            {
                NN_LOG("OA: Unexpected !!\n");
            }
            break;

        // HOME ボタン長押し
        case nn::ae::Message_DetectLongPressingHomeButton:
            NN_LOG("OA: Received Message_DetectLongPressingHomeButton\n");
            NN_LOG("OA: Current AppId = %016llx\n", nn::ae::GetApplicationIdForLogo().value);
            if (g_OverlayStateControl.IsUserInteractionMode())
            {
                g_OverlayStateControl.EndUserInteraction();
            }
            else
            {
                g_OverlayStateControl.BeginUserInteraction();
            }
            break;

        // POWER ボタン中押し
        case nn::ae::Message_DetectMiddlePressingPowerButton:
            NN_LOG("OA: Received Message_DetectMiddlePressingPowerButton\n");
            if (g_OverlayStateControl.IsUserInteractionMode())
            {
                g_OverlayStateControl.EndUserInteraction();
            }
            else
            {
                g_OverlayStateControl.BeginUserInteraction();
            }
            break;

        case nn::ae::Message_ControllerFirmwareUpdateSectionChanged:
            NN_LOG("OA: Received Message_ControllerFirmwareUpdateSectionChanged\n");
            NN_LOG("OA: ae::IsInControllerFirmwareUpdateSection() returns %s.\n", nn::ae::IsInControllerFirmwareUpdateSection() ? "true" : "false");
            break;

        case nn::ae::Message_OperationModeChanged:
            NN_LOG("OA: Received Message_OperationModeChanged\n");
            NN_LOG("OA: Currently Operation Mode is %s Mode\n",
            nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        case nn::ae::Message_VrModeChanged:
            NN_LOG("OA: Received Message_VrModeChanged\n");
            {
                auto isVrMode = nn::ae::IsVrMode();
                if (!isVrMode)
                {
                    g_IsVrModeCurtainRequired.Clear();
                }
                NN_LOG("OA: ae::IsVrMode() returns %s.\n", isVrMode ? "true" : "false");
            }
            break;

        case nn::ae::Message_VrModeCurtainRequired:
            g_IsVrModeCurtainRequired.Signal();
            nn::ae::BeginScreenShotCaptureForbiddenSection();
            NN_LOG("OA: ===============================================\n");
            NN_LOG("OA: Received Message_VrModeCurtainRequired\n");
            NN_LOG("OA: ===============================================\n");
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );
            if (nn::ae::IsVrMode())
            {
                NN_LOG("OA: ===============================================\n");
                NN_LOG("OA: Invoke applet::EndVrModeInternal()\n");
                NN_LOG("OA: ===============================================\n");
                nn::applet::EndVrModeInternal();
            }
            nn::ae::EndScreenShotCaptureForbiddenSection();
            break;

        case nn::ae::Message_ShowApplicationLogo:
            NN_LOG("OA: Received Message_ShowApplicationLogo for 0x%016llx\n", nn::ae::GetApplicationIdForLogo().value);
            break;

        case nn::ae::Message_HideApplicationLogo:
            NN_LOG("OA: Received Message_HideApplicationLogo\n");
            break;

        case nn::ae::Message_ForceHideApplicationLogo:
            NN_LOG("OA: Received Message_ForceHideApplicationLogo\n");
            break;

        default:
            NN_LOG("OA: Received unexpected message= 0x%08x\n", message);
            break;
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  グラフィックスのレンダリングスレッド
//

// OA の表示のオンオフ
bool g_IsSuspendGraphicsThread = false;
bool g_IsPrepareSuspend = false;

nn::os::Event g_SuspendGraphicsThread(nn::os::EventClearMode_AutoClear);
void SuspendGraphicsThread()
{
    g_IsPrepareSuspend = true;
}

void ResumeGraphicsThread()
{
    g_SuspendGraphicsThread.Signal();
    g_IsSuspendGraphicsThread = false;
    g_IsPrepareSuspend = false;
}

void WaitForResumeGraphicsThread()
{
    if (g_IsSuspendGraphicsThread && g_IsPrepareSuspend)
    {
        g_SuspendGraphicsThread.Wait();
    }
}

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer(g_IsPrepareSuspend);
        if (g_IsPrepareSuspend)
        {
            g_IsSuspendGraphicsThread = true;
        }
    }
}


//-----------------------------------------------------------------------------
//  DebugPad 入力状態を監視してログに出力するスレッド
//
void DebugPadThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState      debugPadState;
    nn::hid::DebugPadButtonSet  pressed;
    nn::hid::DebugPadButtonSet  trigger;

    char buf[32];

    for (;;)
    {
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(16666) );
        nnt::applet::hid::GetConvNpadState( &debugPadState );

        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // DebugPad の R で負荷を増やす
        if ((pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() + 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() + 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() + 1);
            }
        }
        // DebugPad の L で負荷を減らす
        if ((pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() - 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() - 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() - 1);
            }
        }
        // ZR + 上下で GPU ブースト
        if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            nn::util::optional<uint64_t> gpuTimeSliceInMicroSeconds;
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                gpuTimeSliceInMicroSeconds = g_GpuTimeSliceBoostInMicroSeconds + 10;
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                if (g_GpuTimeSliceBoostInMicroSeconds > 0)
                {
                    gpuTimeSliceInMicroSeconds = g_GpuTimeSliceBoostInMicroSeconds - 10;
                }
            }
            if (gpuTimeSliceInMicroSeconds)
            {
                nn::ae::SetGpuTimeSliceBoost(nn::TimeSpan::FromMicroSeconds(*gpuTimeSliceInMicroSeconds));
                g_GpuTimeSliceBoostInMicroSeconds = gpuTimeSliceInMicroSeconds.value();
                NN_LOG("OA: SetGpuTimeSliceBoost(%d us)\n", static_cast<int>(g_GpuTimeSliceBoostInMicroSeconds));
            }
        }
        // ZL 同時押し判定
        if ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn())
        {
            // ZL + X でアプリケーションロゴ破損としてアプリ強制終了
            if ((trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn())
            {
                nn::ae::TerminateApplicationAndSetReason( nn::ns::ResultApplicationLogoCorrupted() );
            }
        }
        // A ボタンで全画面クリアのオンオフを切り替える
        if ((trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn())
        {
            Graphics::SetClearEnabled(!Graphics::GetClearEnabled());
        }

        // B ボタンで OA 表示のオンオフを切り替える
        if ((trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn())
        {
            static bool isEnabled = true;

            if (isEnabled)
            {
                // グラフィックススレッド中断
                SuspendGraphicsThread();
                isEnabled = false;
            }
            else
            {
                // グラフィックススレッド再開
                ResumeGraphicsThread();
                isEnabled = true;
            }
        }

        if (!trigger.IsAnyOn())
        {
            continue;
        }

        char* dst = buf;
        *dst++ = (trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn() ? 'A' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn() ? 'B' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn() ? 'X' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn() ? 'Y' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::L::Mask).IsAnyOn() ? 'L' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::R::Mask).IsAnyOn() ? 'R' : '_';

        if ((trigger & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn())
        {
            *dst++ = 'Z';
            *dst++ = 'L';
            nn::ae::EnableAutoSleepTimeAndDimmingTime();
        }
        else
        {
            *dst++ = '_';
            *dst++ = '_';
        }

        if ((trigger & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            *dst++ = 'Z';
            *dst++ = 'R';
            nn::ae::DisableAutoSleepTimeAndDimmingTime();
        }
        else
        {
            *dst++ = '_';
            *dst++ = '_';
        }

        *dst++ = (trigger & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() ? 'U' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn() ? 'D' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn() ? 'L' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn() ? 'R' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Start::Mask).IsAnyOn() ? '+' : '_';
        *dst++ = (trigger & nn::hid::DebugPadButton::Select::Mask).IsAnyOn() ? '-' : '_';
        *dst   = '\0';

        NN_LOG("OA: DebugPad [%s]\n", buf);
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[38 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

//-----------------------------------------------------------------------------
//  メイン関数です。
//
void OverlayAppletMain(nn::ae::OverlayAppletParameters* param)
{
    NN_LOG("OA: Operating Core is #%d.\n", nn::os::GetCurrentCoreNumber());
    {
        NN_LOG("OA: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("OA: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    NN_UNUSED(param);

    // 便宜上、InFocus 扱いにしておく
    g_ThreadControl.SetInFocusState(true);

    // 初期化処理
    Initialize();

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, MessageThreadFunction, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1 ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec) );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );

    // 終了処理
    Finalize();
}

extern "C" void nnMain()
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, OverlayAppletMain);
}
