﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_KeyboardLayout.h>
#include <nn/hid.h>
#include <nn/fs.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "../Common/Common.h"
#include "../Common/Graphics.h"
#include "../Common/AudioOut.h"
#include "../Common/ConvNpad.h"


namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// AE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_AudioThread;

// アプレットパラメータ
char g_ParamBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// メッセージバッファ
char g_MessageBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// WLAN 優先モードの切替
bool g_IsWlanPriorityModeEnabled = false;

// Partial Foreground かどうか
bool g_IsPartialForeground = false;

// Audio が有効かどうか
bool g_IsAudioEnabled = false;

nn::os::SystemEventType g_MessageSystemEvent;

enum LibraryAppletState
{
    LibraryAppletState_ForeGround,
    LibraryAppletState_BackGround,
};

LibraryAppletState g_State = LibraryAppletState_ForeGround;

// スレッド制御
ThreadControl g_ThreadControl;

// キャプチャバッファの Map 用 Mutex
nn::os::Mutex g_MutexForMapLastForeground{false};
nn::os::Mutex g_MutexForMapCallerApplet{false};

void UpdateCallerAppletCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapCallerApplet);

    bool isScreenShotEnabled;
    void* mappedAddress = nullptr;
    for (;;)
    {
        auto result = nn::ae::MapCallerAppletCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("LA2: MapCallerAppletCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("LA2: MapCallerAppletCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 0 );
        auto dstSize = Graphics::GetLinearTextureSize( 0 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapCallerAppletCaptureBuffer();
    }
}

void UpdateLastForegroundCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapLastForeground);

    bool isScreenShotEnabled;
    void* mappedAddress = nullptr;
    for (;;)
    {
        auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("LA2: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("LA2: MapLastForegroundCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 1 );
        auto dstSize = Graphics::GetLinearTextureSize( 1 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapLastForegroundCaptureBuffer();
    }
}

void EliminateName( char *pStr, const char *pName )
{
    char *p = std::strstr( pStr, pName );

    if ( p != NULL )
    {
        *p = '\0';
    }
}

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void MessageThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage( &g_MessageSystemEvent );
        switch ( message )
        {
        case nn::ae::Message_ChangeIntoForeground:
            NN_LOG("LA2: Received Message_ChangeIntoForeground\n");
            nn::ae::AcquireForegroundRights();

            // キャプチャバッファの更新
            if ( g_State == LibraryAppletState_BackGround )
            {
                UpdateCallerAppletCaptureBuffer();
                UpdateLastForegroundCaptureBuffer();
            }

            g_State = LibraryAppletState_ForeGround;
            NN_LOG("LA2: Done AcquireForegroundRights()\n");
            g_ThreadControl.SetInFocusState(true);
            break;

        case nn::ae::Message_ChangeIntoBackground:
            NN_LOG("LA2: Received Message_ChangeIntoBackground\n");
            g_State = LibraryAppletState_BackGround;
            nn::ae::ReleaseForegroundRights();
            NN_LOG("LA2: Done ReleaseForegroundRights()\n");
            g_ThreadControl.SetInFocusState(false);
            break;

        case nn::ae::Message_Exit:
            NN_LOG("LA2: Received Message_Exit\n");
            g_ThreadControl.RequestExit();
            nn::ae::DisableHandlingForExitRequest();
            return;

        case nn::ae::Message_OperationModeChanged:
            NN_LOG("LA2: Received Message_OperationModeChanged\n");
            NN_LOG("LA2: Currently Operation Mode is %s Mode\n",
            nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        default:
            NN_LOG("LA2: Received unknown message= 0x%08x\n", message);
            break;
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed;
    nn::hid::DebugPadButtonSet trigger;

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::WaitVsync();

        // ボタン情報の更新
        nnt::applet::hid::GetConvNpadState( &debugPadState );
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // ZR と ZL を同時に押しながら～の操作
        if (((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn()) &&
            ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn()))
        {
            // ZR + ZL + 下 で WLAN 優先モードをトグル
            if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsWlanPriorityModeEnabled = !g_IsWlanPriorityModeEnabled;
                NN_LOG("LA2: Invoke applet::SetWirelessPriorityModeForLibraryApplet(%s)\n", g_IsWlanPriorityModeEnabled ? "WLAN" : "Default");
                nn::ae::SetWirelessPriorityModeForLibraryApplet(g_IsWlanPriorityModeEnabled ? nn::ae::WirelessPriorityMode_OptimizedForWlan : nn::ae::WirelessPriorityMode_Default);
            }

            continue;
        }

        // ZR を押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {

            // ZR + 右 で BGM ON/OFF
            if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                if ( g_IsAudioEnabled )
                {
                    NN_LOG( "LA2: Disable Audio\n" );
                    g_IsAudioEnabled = false;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                }
                else
                {
                    NN_LOG( "LA2: Enable Audio\n" );
                    g_IsAudioEnabled = true;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                }
            }

            continue;
        }

        // DebugPad の B が押された
        if ( (trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn() )
        {
            NN_LOG("LA2: Invoke nn::ae::ExitLibraryApplet().\n");

            nn::ae::EnterEntranceProhibitionSection();
            nn::ae::EnterInterruptSceneProhibitionSection();
            const char name[] = "LA2";
            std::strcpy( g_ParamBuffer, g_MessageBuffer );
            EliminateName( g_ParamBuffer, name );

            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &storageHandle, sizeof(g_ParamBuffer) ) );
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
            nn::ae::PushToOutChannel( storageHandle );
            nn::ae::LeaveInterruptSceneProhibitionSection();
            nn::ae::LeaveEntranceProhibitionSection();

            nn::ae::ExitLibraryApplet();
        }

        // DebugPad の R で負荷を増やす
        if ((pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() + 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() + 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() + 1);
            }
        }
        // DebugPad の L で負荷を減らす
        if ((pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() - 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() - 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() - 1);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer();
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    const int PlayCount = 14;
    int count = 0;

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        nnt::applet::audioout::WaitAudio();

        if (g_IsAudioEnabled && ++count == PlayCount)
        {
            // PlayCount ごとに SE 再生
            nnt::applet::audioout::PlaySe( nnt::applet::audioout::Se_B );
            count = 0;
        }

        nnt::applet::audioout::UpdateAudio();
        Graphics::WaitVsync();
    }
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.3f, 0.3f, 0.1f, 1.0f };
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredExecution;
    Graphics::InitializeGraphics( clearColor, "LibraryApplet2", frameworkMode, g_IsPartialForeground );
    Graphics::EnableLinearTextures("CallerAppletCaptureBuffer",
                                   "LastForegroundCaptureBuffer");

    // アプレット通知メッセージ用イベントの初期化
    nn::ae::InitializeNotificationMessageEvent( &g_MessageSystemEvent );

    // パラメータの取得
    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS( nn::ae::TryPopFromInChannel( &handle ), "入力データ数が足りない" );
    NN_ABORT_UNLESS_EQUAL( sizeof(g_ParamBuffer), nn::applet::GetStorageSize(handle) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(handle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
    nn::applet::ReleaseStorage(handle);

    std::strcpy( g_MessageBuffer, g_ParamBuffer );
    std::strcat( g_MessageBuffer, "LA2 " );
    Graphics::SetMessage( g_MessageBuffer );

    // Audio の初期化
    nnt::applet::audioout::InitializeAudio();
    nn::ae::ChangeMainAppletMasterVolume( 0.5, nn::TimeSpan::FromSeconds(1) );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void Finalize()
{
    // Audio 関連の終了処理
    nnt::applet::audioout::FinalizeAudio();

    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

nn::applet::LibraryAppletHandle PrepareLibraryAppletImpl(nn::applet::AppletId id, const char* inData, size_t inDataSize) NN_NOEXCEPT
{
    // 作成
    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id, nn::applet::LibraryAppletMode_AllForeground) );

    // 入力データの push
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
        nn::applet::PushToInChannel( appletHandle, storageHandle );
    }
    return appletHandle;
}

nn::applet::LibraryAppletHandle CreateLibraryAppletForDevelop() NN_NOEXCEPT
{
    return PrepareLibraryAppletImpl( nn::applet::AppletId_LibraryAppletCabinet, g_ParamBuffer, sizeof(g_ParamBuffer) );
}

void LibraryAppletMain( const nn::ae::LibraryAppletSelfInfo& info )
{
    NN_LOG("LA2: Operating Core is #%d.\n", nn::os::GetCurrentCoreNumber());
    {
        NN_LOG("LA2: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("LA2: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    NN_ABORT_UNLESS( info.appletId == nn::applet::AppletId_LibraryAppletCabinet );
    if ( info.libraryAppletMode == nn::applet::LibraryAppletMode_PartialForeground )
    {
        g_IsPartialForeground = true;
    }

    nn::ae::EnableHandlingForExitRequest();

    // 呼出元情報の取得
    {
        auto info = nn::ae::GetMainAppletIdentityInfo();
        NN_LOG("LA2: main appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA2: main applicationId = 0x%016llx\n", info.applicationId);
    }
    bool isCallerApplication = false;
    {
        auto info = nn::ae::GetCallerAppletIdentityInfo();
        NN_LOG("LA2: caller appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA2: caller applicationId = 0x%016llx\n", info.applicationId);
        isCallerApplication = (nn::applet::AppletId_Application == info.appletId);
    }
    {
        auto info = nn::ae::GetNextReturnDestinationAppletIdentityInfo();
        NN_LOG("LA2: returnDest appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA2: returnDest applicationId = 0x%016llx\n", info.applicationId);
    }
    {
        nn::Bit32 layout;
        auto ret = nn::applet::GetDesirableKeyboardLayout(&layout);
        NN_LOG("LA2: applet::GetDesirableKeyboardLayout() %s", ret ? "true" : "false\n");
        if (ret)
        {
            NN_LOG(" layout=0x%08x\n", layout);
        }
    }
    {
        nn::settings::LanguageCode lang;
        auto result = nn::ae::GetMainAppletApplicationDesiredLanguage(&lang);
        NN_LOG("LA2: ae::GetMainAppletApplicationDesiredLanguage()=0x%08x  lang=\x22%s\x22\n", result, result.IsSuccess() ? lang.string : "");
    }

    // 初期化処理
    Initialize();
    nn::ae::SetScreenShotAppletIdentityInfo( nn::ae::GetMainAppletIdentityInfo() );

    // 呼出元がアプリなら LastApp の画像を CallerApplet にコピーする
    if (isCallerApplication)
    {
        nn::ae::CopyBetweenCaptureBuffers(nn::ae::CaptureBufferIndex_CallerApplet, nn::ae::CaptureBufferIndex_LastApplication);
    }
    UpdateCallerAppletCaptureBuffer();
    UpdateLastForegroundCaptureBuffer();

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, MessageThreadFunction, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1 ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec ) );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_AudioThread );

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_AudioThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_AudioThread );

    // 終了処理
    Finalize();

    NN_LOG( "LA2: end.\n" );
}

// PartialForeground での起動のほうがメモリを多く消費するため(+4MB)、サイズ変更したときは PartialForeground の動作確認も行ってください。
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[56 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

extern "C" void nnMain()
{
    nn::ae::SetCreateSelfLibraryAppletForDevelop( CreateLibraryAppletForDevelop );
    nn::ae::InvokeLibraryAppletMain( LibraryAppletMain );
}
