﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/util/util_ScopeExit.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/ec/ec_ShopApiForSystem.h>
#include <nn/hid.h>
#include <nn/fs.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_KeyboardLayout.h>
#include <nn/fatal/fatal_Api.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "../Common/applet_Configs.h"
#include "../Common/Common.h"
#include "../Common/Graphics.h"
#include "../Common/AudioOut.h"
#include "../Common/ConvNpad.h"


namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// AE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// アプレットイベント処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_LibraryApplet2HandleThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Foreground キャプチャスレッド
NN_OS_ALIGNAS_THREAD_STACK char g_CaptureThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_LibraryApplet2HandleThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_CaptureThread;
nn::os::ThreadType  g_AudioThread;

// ライブラリアプレットハンドル
nn::applet::LibraryAppletHandle g_LibraryAppletHandle2;

// アプレットパラメータ
char g_ParamBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// メッセージバッファ
char g_MessageBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// Program Winding するかどうか
bool g_IsProgramWinding = true;

// 終了時ハンドリング方法
void (*g_ExitRequestHandler)() = nullptr;

// 終了時ハンドリング待機時間
int g_ExitWaitTimeInSeconds = 0;

// ホームボタン禁止
bool g_ProhibitingHomeButton;

// Partial Foreground かどうか
bool g_IsPartialForeground = false;

// Audio が有効かどうか
bool g_IsAudioEnabled = false;

// 終了要求受理時に LA 起動を行うかどうか
bool g_IsLaunchLAWhenExitRequestEnabled = false;

// VR モード二画面対応かどうか
bool g_IsVrModeDualScreen = false;

// LastForeground の定期自動更新
bool g_IsLastForegroundRepetableCapture = false;

// コントローラ F/W 更新区間
bool g_IsInControllerUpdateSection = false;

// WLAN 優先モードの切替
bool g_IsWlanPriorityModeEnabled = false;

nn::os::SystemEventType g_MessageSystemEvent;

enum LibraryAppletState
{
    LibraryAppletState_ForeGround,
    LibraryAppletState_BackGround,
};

LibraryAppletState g_State = LibraryAppletState_ForeGround;

// スレッド制御
ThreadControl g_ThreadControl;

// キャプチャバッファの Map 用 Mutex
nn::os::Mutex g_MutexForMapLastForeground{false};
nn::os::Mutex g_MutexForMapCallerApplet{false};

void UpdateCallerAppletCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapCallerApplet);
    nn::ae::EnableHandlingForExitRequest();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ae::DisableHandlingForExitRequest();
    };

    bool isScreenShotEnabled;
    void* mappedAddress = nullptr;
    for (;;)
    {
        auto result = nn::ae::MapCallerAppletCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("LA1: MapCallerAppletCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("LA1: MapCallerAppletCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 0 );
        auto dstSize = Graphics::GetLinearTextureSize( 0 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapCallerAppletCaptureBuffer();
    }
}

void UpdateLastForegroundCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapLastForeground);
    nn::ae::EnableHandlingForExitRequest();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ae::DisableHandlingForExitRequest();
    };

    bool isScreenShotEnabled;
    void* mappedAddress = nullptr;
    for (;;)
    {
        auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("LA1: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("LA1: MapLastForegroundCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 1 );
        auto dstSize = Graphics::GetLinearTextureSize( 1 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapLastForegroundCaptureBuffer();
    }
}

//-----------------------------------------------------------------------------
// LA の作成
//
nn::applet::LibraryAppletHandle PrepareLibraryAppletImpl( nn::applet::AppletId id, const char* inData, size_t inDataSize, bool isAllForeground ) NN_NOEXCEPT
{
    // 作成
    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id, isAllForeground ? nn::applet::LibraryAppletMode_AllForeground : nn::applet::LibraryAppletMode_PartialForeground ) );

    // 入力データの push
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
        nn::applet::PushToInChannel( appletHandle, storageHandle );
    }
    return appletHandle;
}

nn::applet::LibraryAppletHandle CreateLibraryAppletForDevelop() NN_NOEXCEPT
{
    return PrepareLibraryAppletImpl( nn::applet::AppletId_LibraryAppletAuth, g_ParamBuffer, sizeof(g_ParamBuffer), true );
}

bool LaunchLibraryApplet2(bool isAllForeground)
{
    // LA1 呼出元の画像をキャプチャしておく
    nn::applet::TakeScreenShotOfCallerApplet();

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle2 = PrepareLibraryAppletImpl( nn::applet::AppletId_LibraryAppletCabinet, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle2 == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("LA1: Request exit failed: Cannot create LA2 handle\n");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( g_LibraryAppletHandle2, NULL ) );
    return true;
}

bool JumpLibraryApplet2(bool isAllForeground)
{
    // LA1 呼出元の画像をキャプチャしておく
    nn::applet::TakeScreenShotOfCallerApplet();

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle2 = PrepareLibraryAppletImpl( nn::applet::AppletId_LibraryAppletCabinet, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle2 == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("LA1: Request exit failed: Cannot create LA2 handle\n");
        return false;
    }
    nn::applet::JumpLibraryApplet( g_LibraryAppletHandle2 );
    // Not reached
    return true;
}

void LaunchEcShopApplication()
{
    // LA1 呼出元の画像をキャプチャしておく
    nn::applet::TakeScreenShotOfCallerApplet();

    // AppletIApp の ApplicationId を渡すので無効ページになるはず
    nn::ec::ShowShopApplicationInformationForApplet(appletIntegrationApplicationId);
}

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void HandleAeMessage(void *arg)
{
    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage( &g_MessageSystemEvent );
        switch ( message )
        {
        case nn::ae::Message_ChangeIntoForeground:
            NN_LOG("LA1: Received Message_ChangeIntoForeground\n");
            nn::ae::AcquireForegroundRights();

            // キャプチャバッファの更新
            if ( g_State == LibraryAppletState_BackGround )
            {
                UpdateCallerAppletCaptureBuffer();
                UpdateLastForegroundCaptureBuffer();
            }

            g_State = LibraryAppletState_ForeGround;
            NN_LOG("LA1: Done AcquireForegroundRights()\n");
            g_ThreadControl.SetInFocusState(true);
            if (g_IsPartialForeground)
            {
                nn::ae::InvalidateTransitionLayerForDevelop();
            }
            break;

        case nn::ae::Message_ChangeIntoBackground:
            NN_LOG("LA1: Received Message_ChangeIntoBackground\n");
            g_State = LibraryAppletState_BackGround;
            nn::ae::ReleaseForegroundRights();
            NN_LOG("LA1: Done ReleaseForegroundRights()\n");
            g_ThreadControl.SetInFocusState(false);
            break;

        case nn::ae::Message_RequestToPrepareSleep:
            NN_LOG("LA1: Received Message_RequestToPrepareSleep\n");
            NN_LOG("LA1: Invoke nn::ae::AllowToEnterSleepAndWait()\n");
            nn::ae::AllowToEnterSleepAndWait();
            break;

        case nn::ae::Message_Exit:
            NN_LOG("LA1: Received Message_Exit\n");
            {
                // LastForeground キャプチャバッファが Unmap されるのを待機
                std::lock_guard<nn::os::Mutex> lock(g_MutexForMapLastForeground);
            }
            {
                // CallerApplet キャプチャバッファが Unmap されるのを待機
                std::lock_guard<nn::os::Mutex> lock(g_MutexForMapCallerApplet);
            }
            if (g_ExitRequestHandler)
            {
                g_ExitRequestHandler();
            }
            if (g_IsLaunchLAWhenExitRequestEnabled)
            {
                // 意図的に LA を起動
                LaunchLibraryApplet2(true);
            }
            g_ThreadControl.RequestExit();
            return;;

        case nn::ae::Message_OperationModeChanged:
            NN_LOG("LA1: Received Message_OperationModeChanged\n");
            NN_LOG("LA1: Currently Operation Mode is %s Mode\n",
                nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        case nn::ae::Message_VrModeChanged:
            NN_LOG("LA1: Received Message_VrModeChanged\n");
            NN_LOG("LA1: ae::IsVrMode() returns %s.\n", nn::ae::IsVrMode() ? "true" : "false");
            break;

        default:
            NN_LOG("LA1: Received unknown message= 0x%08x\n", message);
            break;
        }
    }
}

//-----------------------------------------------------------------------------

void EliminateName( char *pStr, const char *pName )
{
    char *p = std::strstr( pStr, pName );

    if ( p != NULL )
    {
        *p = '\0';
    }
}

void HandlePopFromOutChannelEvent()
{
    NN_LOG("LA1: Received PopFromOutChannelEvent\n" );
    nn::applet::StorageHandle storageHandle;
    NN_ABORT_UNLESS( TryPopFromOutChannel( &storageHandle, g_LibraryAppletHandle2 ) );
    NN_ABORT_UNLESS( GetStorageSize(storageHandle) == sizeof(g_ParamBuffer) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
    ReleaseStorage( storageHandle );

    std::strcpy( g_MessageBuffer, g_ParamBuffer );
}

const char* GetLibraryAppletExitReasonString(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    switch (reason)
    {
    case nn::applet::LibraryAppletExitReason_Normal: return "LibraryAppletExitReason_Normal";
    case nn::applet::LibraryAppletExitReason_Canceled : return "LibraryAppletExitReason_Canceled";
    case nn::applet::LibraryAppletExitReason_Abnormal: return "LibraryAppletExitReason_Abnormal";
    case nn::applet::LibraryAppletExitReason_Unexpected: return "LibraryAppletExitReason_Unexpected";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void WaitLibraryApplet2Exited()
{
    nn::applet::JoinLibraryApplet( g_LibraryAppletHandle2 );
    auto exitReason = nn::applet::GetLibraryAppletExitReason(g_LibraryAppletHandle2);
    NN_LOG("LA1: JoinLibraryApplet2: %s\n", GetLibraryAppletExitReasonString(exitReason));
    if (exitReason == nn::applet::LibraryAppletExitReason_Normal)
    {
        HandlePopFromOutChannelEvent();
    }
    nn::applet::CloseLibraryApplet( g_LibraryAppletHandle2 );
    g_LibraryAppletHandle2 = nn::applet::InvalidLibraryAppletHandle;
}

//-----------------------------------------------------------------------------
// LA 制御用のワーカースレッド
//
enum WorkerCommand
{
    WorkerCommand_None,
    WorkerCommand_LaunchLA,
    WorkerCommand_LaunchLAWinding,
    WorkerCommand_LaunchLAWindingWithoutUnwinding,
    WorkerCommand_LaunchLAWithPartialFG,
    WorkerCommand_LaunchEcShopAppWinding,
};

const int CommandBufferCount = 8;
uintptr_t g_CommandBuffer2[CommandBufferCount];
nn::os::MessageQueue g_CommandQueue2(g_CommandBuffer2, CommandBufferCount);

void LibraryApplet2HandleThread( void *arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    bool isLibraryAppletLaunched = false;

    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        uintptr_t cmd;
        g_CommandQueue2.Receive(&cmd);
        switch (cmd)
        {
            // LA2 の通常起動
            case WorkerCommand_LaunchLA:
            {
                NN_LOG("LA1: Invoke nn::applet::StartLibraryApplet() (LA2)\n");
                g_IsProgramWinding = false;
                isLibraryAppletLaunched = LaunchLibraryApplet2(true);
                break;
            }
            // LA2 の Winding 起動
            case WorkerCommand_LaunchLAWinding:
            {
                NN_LOG("LA1: Invoke nn::applet::StartLibraryApplet() (LA2, Winding)\n");
                g_IsProgramWinding = true;
                isLibraryAppletLaunched = LaunchLibraryApplet2(true);
                break;
            }
            // LA2 の Winding 起動（ただし LA1 の Unwind なし）
            case WorkerCommand_LaunchLAWindingWithoutUnwinding:
            {
                NN_LOG("LA1: Invoke nn::applet::JumpLibraryApplet() (LA2, Winding)\n");
                g_IsProgramWinding = true;
                isLibraryAppletLaunched = JumpLibraryApplet2(true);
                break;
            }
            // LA2 の Partial-FG 起動
            case WorkerCommand_LaunchLAWithPartialFG:
            {
                NN_LOG("LA1: Invoke nn::applet::StartLibraryApplet() (LA2, Partial FG)\n");
                g_IsProgramWinding = false;
                isLibraryAppletLaunched = LaunchLibraryApplet2(false);
                break;
            }
            case WorkerCommand_LaunchEcShopAppWinding:
            {
                NN_LOG("LA1: Invoke nn::ec::ShowShopApplicationInformationForApplet() (ecShop, Winding)\n");
                g_IsProgramWinding = true;
                LaunchEcShopApplication();
                break;
            }
            default:
            {
                NN_LOG("LA1: Unknown internal command: 0x%08x\n", cmd);
                break;
            }
        }

        // LA2 の終了を待機
        if (isLibraryAppletLaunched)
        {
            WaitLibraryApplet2Exited();
            isLibraryAppletLaunched = false;
        }
    }
}

//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed = nn::hid::DebugPadButtonSet();
    nn::hid::DebugPadButtonSet trigger = nn::hid::DebugPadButtonSet();

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::WaitVsync();

        // ボタン情報の更新
        nnt::applet::hid::GetConvNpadState( &debugPadState );
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // Partial Foreground な LA が起動していたら何もしない
        if ( g_LibraryAppletHandle2 != nn::applet::InvalidLibraryAppletHandle )
        {
            continue;
        }

        // START(+) ボタン
        if ((trigger & nn::hid::DebugPadButton::Start::Mask).IsAnyOn())
        {
            nn::diag::RegisterAbortObserver(nn::ae::GetAbortObserverHolderToUnmapCaptureBuffer());
            bool isScreenShotEnabled;
            void* mappedAddress;
            for (;;)
            {
                auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
                if (result.IsSuccess())
                {
                    if (!isScreenShotEnabled)
                    {
                        NN_LOG("LA1: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
                    }
                    break;
                }
                else if (result <= nn::ae::ResultCaptureBufferBusy())
                {
                    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            NN_ABORT("expected abort!!!");
        }

        // SELECT(-) ボタン
        if ((trigger & nn::hid::DebugPadButton::Select::Mask).IsAnyOn())
        {
            bool isScreenShotEnabled;
            void* mappedAddress;
            for (;;)
            {
                auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
                if (result.IsSuccess())
                {
                    if (!isScreenShotEnabled)
                    {
                        NN_LOG("LA1: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
                    }
                    break;
                }
                else if (result <= nn::ae::ResultCaptureBufferBusy())
                {
                    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            NN_ABORT("expected abort!!!");
        }

        // ZR と ZL を同時に押しながら～の操作
        if (((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn()) &&
            ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn()))
        {
            // ZR + ZL + 下 で WLAN 優先モードをトグル
            if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsWlanPriorityModeEnabled = !g_IsWlanPriorityModeEnabled;
                NN_LOG("LA1: Invoke applet::SetWirelessPriorityModeForLibraryApplet(%s)\n", g_IsWlanPriorityModeEnabled ? "WLAN" : "Default");
                nn::ae::SetWirelessPriorityModeForLibraryApplet(g_IsWlanPriorityModeEnabled ? nn::ae::WirelessPriorityMode_OptimizedForWlan : nn::ae::WirelessPriorityMode_Default);
            }
            continue;
        }

        // ZR を同時に押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            // ZR + A,B,X,Y で終了時ハンドリングの設定を変更
            if ((trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn())
            {
                NN_LOG("LA1: Set exit handler: normally exit.\n");
                if (!g_ExitRequestHandler)
                {
                    nn::ae::EnableHandlingForExitRequest();
                }
                g_ExitRequestHandler = []()
                {
                    NN_LOG("LA1: Waiting exit. in %d sec...\n", g_ExitWaitTimeInSeconds);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(g_ExitWaitTimeInSeconds));
                    NN_LOG("LA1: exit normally\n");
                    nn::ae::ExitLibraryApplet();
                };
            }
            else if ((trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn())
            {
                NN_LOG("LA1: Set exit handler: abort.\n");
                if (!g_ExitRequestHandler)
                {
                    nn::ae::EnableHandlingForExitRequest();
                }
                g_ExitRequestHandler = []()
                {
                    NN_LOG("LA1: Waiting exit. in %d sec...\n", g_ExitWaitTimeInSeconds);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(g_ExitWaitTimeInSeconds));
                    NN_LOG("LA1: abort (expected)\n");
                    NN_ABORT("");
                };
            }
            else if ((trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn())
            {
                NN_LOG("LA1: Set exit handler: cancel.\n");
                if (!g_ExitRequestHandler)
                {
                    nn::ae::EnableHandlingForExitRequest();
                }
                g_ExitRequestHandler = []()
                {
                    NN_LOG("LA1: Waiting exit. in %d sec...\n", g_ExitWaitTimeInSeconds);
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(g_ExitWaitTimeInSeconds));
                    NN_LOG("LA1: canceled\n");
                    nn::ae::DisableHandlingForExitRequest();
                };
            }
            else if ((trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn())
            {
                NN_LOG("LA1: Unset exit handler.\n");
                if (g_ExitRequestHandler)
                {
                    nn::ae::DisableHandlingForExitRequest();
                    g_ExitRequestHandler = nullptr;
                }
            }
            else if ((trigger & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
            {
                ++g_ExitWaitTimeInSeconds;
                NN_LOG("LA1: Set exit wait time: %d sec.\n", g_ExitWaitTimeInSeconds);
            }
            else if ((trigger & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
            {
                if (g_ExitWaitTimeInSeconds > 0)
                {
                    --g_ExitWaitTimeInSeconds;
                    NN_LOG("LA1: Set exit wait time: %d sec.\n", g_ExitWaitTimeInSeconds);
                }
            }
            // ZR + 左 で VR モード二画面モードの on/off
            else if ((trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                g_IsVrModeDualScreen = !g_IsVrModeDualScreen;
                NN_LOG("LA1: VR-mode double screen is %s.\n", g_IsVrModeDualScreen ? "true" : "false");
                nn::ae::SetVrModeDualScreenSupported(g_IsVrModeDualScreen);
            }
            // ZR + 右 で SE ON/OFF
            else if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                if ( g_IsAudioEnabled )
                {
                    NN_LOG( "LA1: Disable Audio\n" );
                    g_IsAudioEnabled = false;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                }
                else
                {
                    NN_LOG( "LA1: Enable Audio\n" );
                    g_IsAudioEnabled = true;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                }
            }
            continue;
        }

        // ZL を同時に押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn())
        {
            // ZL + A で HOME ボタン禁止セクションに入る
            if ((trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn())
            {
                if (!g_ProhibitingHomeButton)
                {
                    NN_LOG("LA1: EnterHomeButtonProhibitionSection\n");
                    nn::ae::EnterHomeButtonProhibitionSection();
                    g_ProhibitingHomeButton = true;
                }
            }
            // ZL + B で HOME ボタン禁止セクションから出る
            else if ((trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn())
            {
                if (g_ProhibitingHomeButton)
                {
                    NN_LOG("LA1: LeaveHomeButtonProhibitionSection\n");
                    nn::ae::LeaveHomeButtonProhibitionSection();
                    g_ProhibitingHomeButton = false;
                }
            }
            // ZL + 下 で終了要求受理後の LA 起動の ON/OFF
            else if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsLaunchLAWhenExitRequestEnabled = !g_IsLaunchLAWhenExitRequestEnabled;
                NN_LOG("LA1: %s launching LA when exit requested.\n", g_IsLaunchLAWhenExitRequestEnabled ? "Enable" : "Disable");
            }
            // ZL + 上 で LastForeground を 500ms 毎に自動キャプチャ更新
            else if ((trigger & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                g_IsLastForegroundRepetableCapture = !g_IsLastForegroundRepetableCapture;
                NN_LOG("LA1: %s capture to LastForeground capture buffer repeatedly.\n", g_IsLastForegroundRepetableCapture ? "Enable" : "Disable");
            }
            // ZL + 左 で コントローラ F/W 更新区間の on/off
            else if ((trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                g_IsInControllerUpdateSection = !g_IsInControllerUpdateSection;
                NN_LOG("LA1: Controller F/W update section: %s\n", g_IsInControllerUpdateSection ? "true" : "false");
                if (g_IsInControllerUpdateSection)
                {
                    nn::ae::BeginControllerFirmwareUpdateSection();
                }
                else
                {
                    nn::ae::EndControllerFirmwareUpdateSection();
                }
            }
            // ZL + 右 でスリープハンドリングを切替（SIGLO-49257）
            else if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                // 本当はスレッド優先度を -2 に設定すると再現させやすいが、
                // desc の変更等が結構コスト高なので以下の実装で再現させる。
                auto coreNum = nn::os::GetCurrentCoreNumber();
                nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 2, 0x1 << 2);
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );
                NN_LOG("LA1: Invoke nn::ae::EndSleepHandling() and nn::ae::BeginSleepHandling()\n");
                nn::ae::EndSleepHandling();
                // ここの End～Begin の間にログを挟むと再現しにくい
                nn::ae::BeginSleepHandling();
                // SIGLO-49257 修正前だと上記 nn::ae::BeginSleepHandling() から
                // 返ってこない。
                nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), coreNum, 0x1 << coreNum);
            }
            continue;
        }

        // DebugPad の Y が押された
        if ( (trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn() &&
             (g_LibraryAppletHandle2 == nn::applet::InvalidLibraryAppletHandle) )
        {
            // 上 + Y で LA2 を Partial-FG 起動
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                g_CommandQueue2.Send(WorkerCommand_LaunchLAWithPartialFG);
            }
            // 下 + Y で LA2 を通常起動（LA1 の Winding/Unwinding あり）
            else if ( (pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn() )
            {
                g_CommandQueue2.Send(WorkerCommand_LaunchLAWinding);
            }
            // 右 + Y で LA2 を起動（LA1 の Winding あり、Unwinding なし）
            else if ( (pressed & nn::hid::DebugPadButton::Right::Mask).IsAnyOn() )
            {
                g_CommandQueue2.Send(WorkerCommand_LaunchLAWindingWithoutUnwinding);
            }
            // 左 + Y でアプリ情報ショップページを起動（Winding/Unwinding あり）
            else if ( (pressed & nn::hid::DebugPadButton::Left::Mask).IsAnyOn() )
            {
                g_CommandQueue2.Send(WorkerCommand_LaunchEcShopAppWinding);
            }
            // LA2 の通常起動（Winding なし）
            else
            {
                g_CommandQueue2.Send(WorkerCommand_LaunchLA);
            }
            continue;
        }
        // DebugPad の A が押された
        else if ( (trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn() )
        {
            // 上 + A で App の起動要求を出す
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                char buf[nn::applet::StartupParamSizeMax] = {};
                std::strcpy(buf, "LA1 jump to ");
                nn::ae::RequestToLaunchApplication(appletIntegrationApplicationId, buf, sizeof(buf));
            }
        }
        // DebugPad の B が押された
        else if ( (trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn() )
        {
            NN_LOG("LA1: Invoke nn::ae::ExitLibraryApplet().\n");

            nn::ae::EnterEntranceProhibitionSection();
            nn::ae::EnterInterruptSceneProhibitionSection();
            const char name[] = "LA1";
            std::strcpy( g_ParamBuffer, g_MessageBuffer );
            EliminateName( g_ParamBuffer, name );

            nn::applet::StorageHandle storageHandle;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &storageHandle, sizeof(g_ParamBuffer) ) );
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
            nn::ae::PushToOutChannel( storageHandle );
            nn::ae::LeaveInterruptSceneProhibitionSection();
            nn::ae::LeaveEntranceProhibitionSection();

            nn::ae::ExitLibraryApplet();
        }
        // DebugPad の R で負荷を増やす
        else if ((pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() + 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() + 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() + 1);
            }
        }
        // DebugPad の L で負荷を減らす
        else if ((pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() - 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() - 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() - 1);
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer();
    }
}

void CaptureThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        // 500ms 経過するまで寝る
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 500 ) );

        // LastForegroundCaptureBuffer の更新
        if (g_IsLastForegroundRepetableCapture)
        {
            nn::applet::TakeScreenShotOfOwnLayerIntoLastForegroundBuffer(true);
            UpdateLastForegroundCaptureBuffer();
        }
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    const int PlayCount = 12;
    int count = 0;

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        nnt::applet::audioout::WaitAudio();

        if (g_IsAudioEnabled && ++count == PlayCount)
        {
            // PlayCount ごとに SE 再生
            nnt::applet::audioout::PlaySe( nnt::applet::audioout::Se_D );
            count = 0;
        }

        nnt::applet::audioout::UpdateAudio();
        Graphics::WaitVsync();
    }
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize( const nn::ae::LibraryAppletSelfInfo& info )
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.1f, 0.3f, 0.1f, 1.0f };
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredExecution;
    Graphics::InitializeGraphics( clearColor, "LibraryApplet1", frameworkMode, g_IsPartialForeground );
    Graphics::EnableLinearTextures("CallerAppletCaptureBuffer",
                                   "LastForegroundCaptureBuffer");

    // アプレット通知メッセージ用イベントの初期化
    nn::ae::InitializeNotificationMessageEvent( &g_MessageSystemEvent );

    // パラメータの取得
    if ( info.isUnwound )
    {
        nn::applet::JoinLibraryApplet(info.previousLibraryAppletHandle);
        nn::applet::CloseLibraryApplet(info.previousLibraryAppletHandle);
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS( nn::ae::TryPopFromContextStack( &storageHandle ) );
        NN_ABORT_UNLESS_EQUAL( sizeof(g_ParamBuffer), nn::applet::GetStorageSize( storageHandle ) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        nn::applet::ReleaseStorage( storageHandle );
        std::strcpy( g_MessageBuffer, g_ParamBuffer );
    }
    else
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS( nn::ae::TryPopFromInChannel( &storageHandle ) );
        NN_ABORT_UNLESS_EQUAL( sizeof(g_ParamBuffer), nn::applet::GetStorageSize( storageHandle ) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        nn::applet::ReleaseStorage( storageHandle );
        std::strcpy( g_MessageBuffer, g_ParamBuffer );
        std::strcat( g_MessageBuffer, "LA1 " );
    }
    Graphics::SetMessage( g_MessageBuffer );

    // Audio の初期化
    nnt::applet::audioout::InitializeAudio();
    auto volume = nn::ae::GetMainAppletExpectedMasterVolume();
    nn::ae::ChangeMainAppletMasterVolume( volume, nn::TimeSpan::FromSeconds(1) );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void Finalize()
{
    // Audio 関連の終了処理
    nnt::applet::audioout::FinalizeAudio();

    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

void SaveCallContext() NN_NOEXCEPT
{
    nn::applet::StorageHandle storageHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, sizeof(g_ParamBuffer) ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, &g_ParamBuffer, sizeof(g_ParamBuffer) ) );
    nn::ae::PushToContextStack(storageHandle);
}

nn::ae::LibraryAppletStartHookResult LibraryAppletStartHook(nn::applet::AppletId appletId, nn::applet::LibraryAppletMode libraryAppletMode, void *) NN_NOEXCEPT
{
    if ( !g_IsProgramWinding )
    {
        return nn::ae::LibraryAppletStartHookResult_Normal;
    }
    SaveCallContext();
    return nn::ae::LibraryAppletStartHookResult_WindProgram;
}

void LibraryAppletMain( const nn::ae::LibraryAppletSelfInfo& info )
{
    NN_LOG("LA1: Operating Core is #%d.\n", nn::os::GetCurrentCoreNumber());
    {
        NN_LOG("LA1: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("LA1: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    NN_ABORT_UNLESS( info.appletId == nn::applet::AppletId_LibraryAppletAuth );
    if ( info.libraryAppletMode == nn::applet::LibraryAppletMode_PartialForeground )
    {
        g_IsPartialForeground = true;
    }

    {
        auto info = nn::ae::GetMainAppletIdentityInfo();
        NN_LOG("LA1: main appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA1: main applicationId = 0x%016llx\n", info.applicationId);
    }
    bool isCallerApplication = false;
    {
        auto info = nn::ae::GetCallerAppletIdentityInfo();
        NN_LOG("LA1: caller appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA1: caller applicationId = 0x%016llx\n", info.applicationId);
        isCallerApplication = (nn::applet::AppletId_Application == info.appletId);
    }
    {
        auto info = nn::ae::GetNextReturnDestinationAppletIdentityInfo();
        NN_LOG("LA1: returnDest appletId      = 0x%016llx\n", info.appletId);
        NN_LOG("LA1: returnDest applicationId = 0x%016llx\n", info.applicationId);
    }
    {
        nn::Bit32 layout;
        auto ret = nn::applet::GetDesirableKeyboardLayout(&layout);
        NN_LOG("LA1: applet::GetDesirableKeyboardLayout() %s", ret ? "true" : "false\n");
        if (ret)
        {
            NN_LOG(" layout=0x%08x\n", layout);
        }
    }
    {
        nn::settings::LanguageCode lang;
        auto result = nn::ae::GetMainAppletApplicationDesiredLanguage(&lang);
        NN_LOG("LA1: ae::GetMainAppletApplicationDesiredLanguage()=0x%08x  lang=\x22%s\x22\n", result, result.IsSuccess() ? lang.string : "");
    }

    // 初期化処理
    Initialize( info );
    nn::ae::BeginSleepHandling();

    // 呼出元がアプリなら LastApp の画像を CallerApplet にコピーする
    if (isCallerApplication)
    {
        nn::ae::CopyBetweenCaptureBuffers(nn::ae::CaptureBufferIndex_CallerApplet, nn::ae::CaptureBufferIndex_LastApplication);
    }
    UpdateCallerAppletCaptureBuffer();
    UpdateLastForegroundCaptureBuffer();

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, HandleAeMessage, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1 ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_LibraryApplet2HandleThread, LibraryApplet2HandleThread, NULL, g_LibraryApplet2HandleThreadStack, ThreadStackSize, AppletPriorityForLimit10msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_CaptureThread, CaptureThreadFunction, NULL, g_CaptureThreadStack, ThreadStackSize, AppletPriorityForOver10msec + 1 ) );

    // アプリのコアが使用可能かどうか
    NN_LOG("LA1: CanUseApplicationCore() = %s\n", nn::ae::CanUseApplicationCore() ? "true" : "false");

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_LibraryApplet2HandleThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_CaptureThread );
    nn::os::StartThread( &g_AudioThread );

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_LibraryApplet2HandleThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_CaptureThread );
    nn::os::WaitThread( &g_AudioThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_LibraryApplet2HandleThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_CaptureThread );
    nn::os::DestroyThread( &g_AudioThread );

    // 終了処理
    Finalize();

    NN_LOG( "LA1: end.\n" );
}

// 本来 80 MiB で足りるが、ぎりぎりまでメモリをとるために 160MiB 使用する SIGLO-36704
// PartialForeground での起動のほうがメモリを多く消費するため(+4MB)、サイズ変更したときは PartialForeground の動作確認も行ってください。
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[56 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

extern "C" void nnMain()
{
    // VR モード 2 画面対応
    g_IsVrModeDualScreen = true;
    nn::ae::SetVrModeDualScreenSupported(g_IsVrModeDualScreen);

    nn::fatal::SetFatalPolicy(nn::fatal::FatalPolicy_OnlyErrorReport);
    nn::ae::SetLibraryAppletStartHook( &LibraryAppletStartHook );
    nn::ae::SetCreateSelfLibraryAppletForDevelop( CreateLibraryAppletForDevelop );
    nn::ae::InvokeLibraryAppletMain( LibraryAppletMain );
}
