﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "GraphicsFrameworkEx.h"

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nvn/nvn_FuncPtrInline.h>

namespace Graphics
{
    void GraphicsFrameworkEx::Initialize(
        const nns::gfx::GraphicsFramework::FrameworkInfo& info
    ) NN_NOEXCEPT
    {
        GraphicsFramework::Initialize(info);
    }

    void GraphicsFrameworkEx::Initialize(
        const nns::gfx::GraphicsFramework::FrameworkInfo& info,
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        nn::FreeFunctionWithUserData pFreeFunction,
        void* pAllocateFunctionUserData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        if ((pAllocateFunction != nullptr) && (pFreeFunction != nullptr)) {
            m_AllocateFunction = pAllocateFunction;
            m_FreeFunction = pFreeFunction;
            m_pAllocateFunctionUserData = pAllocateFunctionUserData;
        }
        else {
            m_AllocateFunction = DefaultAllocateFunction;
            m_FreeFunction = DefaultFreeFunction;
            m_pAllocateFunctionUserData = nullptr;
        }

        m_DisplayWidth = info.GetDisplayWidth();
        m_DisplayHeight = info.GetDisplayHeight();
        m_BufferCount = info.GetBufferCount();
        m_SwapChainBufferCount = info.GetSwapChainBufferCount();
        m_NextScanBufferIndex = 0;

        if (m_DisplayWidth <= 0)
        {
            m_DisplayWidth = DefaultDisplayWidth;
        }
        if (m_DisplayHeight <= 0)
        {
            m_DisplayHeight = DefaultDisplayHeight;
        }

        bool isSharedBufferSupported = nn::vi::fbshare::SharedLayerWindowGfx::IsSupported();
        NN_LOG("IsSharedBufferSupported: %s\n", isSharedBufferSupported ? "yes" : "no");

        // VI
        if(!isSharedBufferSupported)
        {
            // VI 初期化
            nn::vi::Initialize();

            // ディスプレイ, レイヤー
            nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
            NN_SDK_ASSERT(result.IsSuccess());
            NN_UNUSED(result);

            result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
            NN_SDK_ASSERT(result.IsSuccess());

            result = nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
            NN_SDK_ASSERT(result.IsSuccess());
        }
        else
        {
            nn::vi::Initialize();
            m_pDisplay = nullptr;
            m_pLayer = nullptr;
        }

        // GFX
        {
            // GFX 初期化
            nn::gfx::Initialize();

            // Device
            {
                nn::gfx::Device::InfoType deviceInfo;
                deviceInfo.SetDefault();
                deviceInfo.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
                deviceInfo.SetDebugMode(info.GetDebugMode());
                m_Device.Initialize(deviceInfo);
            }

            // デプスモード、ウインドウ座標 ( 0,0 )位置モード の設定
            nn::gfx::util::ScalableViewport::OriginMode originMode = info.GetPhysicalWindowCoordinate().GetOriginMode();

#if NN_GFX_IS_TARGET_NVN
            // 深度モード
            if (info.GetDepthMode() == DepthMode_Near_Is_Zero)
            {
                nvnDeviceSetDepthMode(m_Device.ToData()->pNvnDevice, NVN_DEPTH_MODE_NEAR_IS_ZERO);
            }
            else if (info.GetDepthMode() == DepthMode_Near_Is_Minus_W)
            {
                nvnDeviceSetDepthMode(m_Device.ToData()->pNvnDevice, NVN_DEPTH_MODE_NEAR_IS_MINUS_W);
            }

            // ウィンドウ原点モード
            if (originMode == nn::gfx::util::ScalableViewport::OriginMode_LowerLeft)
            {
                nvnDeviceSetWindowOriginMode(m_Device.ToData()->pNvnDevice, NVN_WINDOW_ORIGIN_MODE_LOWER_LEFT);
            }
            else if (originMode == nn::gfx::util::ScalableViewport::OriginMode_UpperLeft)
            {
                nvnDeviceSetWindowOriginMode(m_Device.ToData()->pNvnDevice, NVN_WINDOW_ORIGIN_MODE_UPPER_LEFT);
            }
#endif

            // Queue
            {
                nn::gfx::Queue::InfoType queueInfo;
                queueInfo.SetDefault();
                queueInfo.SetCapability(nn::gfx::QueueCapability_Graphics
                    | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
                m_Queue.Initialize(&m_Device, queueInfo);
            }

#if NN_GFX_IS_TARGET_GL
            GLenum windowOriginMode = GL_LOWER_LEFT;
            GLenum depthMode = GL_NEGATIVE_ONE_TO_ONE;

            if (info.GetDepthMode() == DepthMode_Near_Is_Zero)
            {
                depthMode = GL_ZERO_TO_ONE;
            }

            if (originMode == nn::gfx::util::ScalableViewport::OriginMode_UpperLeft)
            {
                windowOriginMode = GL_UPPER_LEFT;
            }

#ifdef NN_BUILD_TARGET_PLATFORM_OS_WIN
            BOOL result = TRUE;
            HDC hDc = m_Queue.ToData()->hDc;
            HGLRC hGlRc = m_Queue.ToData()->renderingContext.hGlRc;
            ::wglMakeCurrent(hDc, hGlRc);
            glClipControl(windowOriginMode, depthMode);
            auto error = glGetError();
            NN_ASSERT(error == GL_NO_ERROR);
            result = ::wglMakeCurrent(NULL, NULL);
            NN_ASSERT(result == TRUE);
#else // NN_BUILD_TARGET_PLATFORM_OS_NN
            nn::gfx::Queue::DataType& data = m_Queue.ToData();
            EGLBoolean result = eglMakeCurrent(data.hDisplay, data.hSurface, data.hSurface, data.renderingContext.hGlRc);
            NN_ASSERT(result);
            glClipControl(windowOriginMode, depthMode);
            GLenum error = glGetError();
            NN_ASSERT(error == GL_NO_ERROR);
#endif // NN_BUILD_TARGET_PLATFORM_OS_WIN
#endif

            // DisplayFence
            {
                nn::gfx::Fence::InfoType fenceInfo;
                fenceInfo.SetDefault();

                int count = m_BufferCount;
                m_pDisplayFence = reinterpret_cast<nn::gfx::Fence*>(m_AllocateFunction(sizeof(nn::gfx::Fence) * count, NN_ALIGNOF(nn::gfx::Fence), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pDisplayFence);
                for (int i = 0; i < count; i++) {
                    new(&m_pDisplayFence[i])nn::gfx::Fence;

                    m_pDisplayFence[i].Initialize(&m_Device, fenceInfo);
                }
            }
            // GpuFence
            {
                nn::gfx::Fence::InfoType fenceInfo;
                fenceInfo.SetDefault();

                int count = m_BufferCount;
                m_pGpuFence = reinterpret_cast<nn::gfx::Fence*>(m_AllocateFunction(sizeof(nn::gfx::Fence) * count, NN_ALIGNOF(nn::gfx::Fence), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pGpuFence);
                for (int i = 0; i < count; i++) {
                    new(&m_pGpuFence[i])nn::gfx::Fence;

                    m_pGpuFence[i].Initialize(&m_Device, fenceInfo);
                }
            }

            // DisplaySemaphore
            {
                nn::gfx::Semaphore::InfoType semaphoreInfo;
                semaphoreInfo.SetDefault();

                int count = m_BufferCount;
                m_pDisplaySemaphore = reinterpret_cast<nn::gfx::Semaphore*>(m_AllocateFunction(sizeof(nn::gfx::Semaphore) * count, NN_ALIGNOF(nn::gfx::Semaphore), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pDisplaySemaphore);
                for (int i = 0; i < count; i++) {
                    new(&m_pDisplaySemaphore[i])nn::gfx::Semaphore;

                    m_pDisplaySemaphore[i].Initialize(&m_Device, semaphoreInfo);
                }
            }

            // MemoryPool
            {
                nn::gfx::MemoryPool::InfoType memorypoolInfo;
                int memoryPoolProperty[MemoryPoolType_End] = {
                    /* CommandBuffer */     nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached,
                    /* RenderTarget */      nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible,
                    /* ConstantBuffer */    nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
                    /* Shader */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode,
                    /* Data */              nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached,
                    /* Others */            nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible | nn::gfx::MemoryPoolProperty_ShaderCode,
                };

                for (int i = 0; i < MemoryPoolType_End; i++)
                {
                    memorypoolInfo.SetDefault();
                    memorypoolInfo.SetMemoryPoolProperty(memoryPoolProperty[i]);

                    size_t size = info.GetMemoryPoolSize(static_cast<MemoryPoolType>(i));
                    if (size == 0) {
                        m_MemoryPool[i].size = 0;
                        continue;
                    }
                    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, memorypoolInfo);
                    void* pAlignedMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
                    NN_SDK_ASSERT_NOT_NULL(pAlignedMemory);

                    memorypoolInfo.SetPoolMemory(pAlignedMemory, nn::util::align_down(size,
                        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, memorypoolInfo)));
                    m_MemoryPool[i].object.Initialize(&m_Device, memorypoolInfo);

                    m_MemoryPool[i].allocator.Initialize(
                        MemoryPoolAllocatorMalloc,
                        reinterpret_cast<void*>(this),
                        MemoryPoolAllocatorFree,
                        reinterpret_cast<void*>(this),
                        &m_MemoryPool[i].object,
                        0,
                        size,
                        nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
                        true
                    );
                    NN_SDK_ASSERT(m_MemoryPool[i].allocator.IsInitialized());

                    m_MemoryPool[i].pMemory = pAlignedMemory;
                    m_MemoryPool[i].size = size;
                }
            }

            // DescriptorPool
            {
                int descriptorBaseIndex[nn::gfx::DescriptorPoolType_End];
                descriptorBaseIndex[nn::gfx::DescriptorPoolType_BufferView] = 0;
                descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView] = 0;
                descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler] = 0;
#if NN_GFX_IS_TARGET_NVN
                nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
                nvnDeviceGetInteger(deviceData.pNvnDevice,
                    NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_TextureView]);
                nvnDeviceGetInteger(deviceData.pNvnDevice,
                    NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &descriptorBaseIndex[nn::gfx::DescriptorPoolType_Sampler]);
#endif
                nn::gfx::DescriptorPool::InfoType descriptorpoolInfo;

                for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
                {
                    nn::gfx::DescriptorPoolType descriptorpoolType = static_cast<nn::gfx::DescriptorPoolType>(i);
                    m_DescriptorPoolSlotCount[i] = info.GetDescriptorPoolSlotCount(descriptorpoolType);
                    if (m_DescriptorPoolSlotCount[i] > 0) {
                        NN_SDK_ASSERT_GREATER(m_DescriptorPoolSlotCount[i], descriptorBaseIndex[i]);

                        descriptorpoolInfo.SetDefault();
                        descriptorpoolInfo.SetDescriptorPoolType(descriptorpoolType);
                        descriptorpoolInfo.SetSlotCount(m_DescriptorPoolSlotCount[i]);

                        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, descriptorpoolInfo);
                        MemoryPoolType memorypoolType = MemoryPoolType_ConstantBuffer;
                        size_t alignment = nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, descriptorpoolInfo);
                        ptrdiff_t offset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                        NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);

                        m_DescriptorPool[i].Initialize(&m_Device, descriptorpoolInfo, &m_MemoryPool[memorypoolType].object, offset, size);

                        m_DescriptorPoolAllocator[i].Initialize(
                            DescriptorPoolAllocatorMalloc,
                            reinterpret_cast<void*>(this),
                            DescriptorPoolAllocatorFree,
                            reinterpret_cast<void*>(this),
                            &m_DescriptorPool[i],
                            descriptorBaseIndex[i],
                            m_DescriptorPoolSlotCount[i] - descriptorBaseIndex[i],
                            true
                        );
                        NN_SDK_ASSERT(m_DescriptorPoolAllocator[i].IsInitialized());
                    }
                }
            }

            // RootCommandBuffer
            {
                int count = m_BufferCount;
                m_pCommandBuffer = reinterpret_cast<GraphicsFramework::CommandBuffer*>(m_AllocateFunction(sizeof(GraphicsFramework::CommandBuffer) * count, NN_ALIGNOF(GraphicsFramework::CommandBuffer), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pCommandBuffer);
                for (int i = 0; i < count; i++) {
                    new(&m_pCommandBuffer[i])GraphicsFramework::CommandBuffer;
                    m_pCommandBuffer[i].commandMemorySize = info.GetRootCommandBufferCommandMemorySize();
                    m_pCommandBuffer[i].controlMemorySize = info.GetRootCommandBufferControlMemorySize();
                }
                m_ShaderScratchMemorySize = 0;

                // CommandBuffer
                for (int i = 0; i < count; i++) {
                    {
                        nn::gfx::CommandBuffer::InfoType commandbufferInfo;
                        commandbufferInfo.SetDefault();
                        commandbufferInfo.SetQueueCapability(nn::gfx::QueueCapability_Graphics
                            | nn::gfx::QueueCapability_Compute | nn::gfx::QueueCapability_Copy);
                        commandbufferInfo.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);

                        m_pCommandBuffer[i].object.Initialize(&m_Device, commandbufferInfo);
                    }

                    // CommandMemory
                    {
                        MemoryPoolType memorypoolType = MemoryPoolType_CommandBuffer;
                        size_t alignment = nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device);
                        size_t size = m_pCommandBuffer[i].commandMemorySize;
                        m_pCommandBuffer[i].commandMemoryOffset = m_MemoryPool[memorypoolType].allocator.Allocate(size, alignment);
                        NN_SDK_ASSERT(m_pCommandBuffer[i].commandMemoryOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    }

                    // ControlMemory
                    {
                        size_t alignment = nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device);
                        size_t size = m_pCommandBuffer[i].controlMemorySize;
                        m_pCommandBuffer[i].pControlMemory = m_AllocateFunction(size, alignment, m_pAllocateFunctionUserData);
                        NN_SDK_ASSERT_NOT_NULL(m_pCommandBuffer[i].pControlMemory);
                    }
                }
            }

            // ShaderScratchMemory
            {
                int count = m_BufferCount;
                m_pShaderScratchMemory = reinterpret_cast<ShaderScratchMemoryInfo*>(m_AllocateFunction(sizeof(ShaderScratchMemoryInfo) * count, NN_ALIGNOF(ShaderScratchMemoryInfo), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pShaderScratchMemory);
                for (int i = 0; i < count; i++) {
                    new(&m_pShaderScratchMemory[i])ShaderScratchMemoryInfo;
                    m_pShaderScratchMemory[i].offset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
                    m_pShaderScratchMemory[i].size = 0;
                }
            }

            if(!isSharedBufferSupported)
            {
                // SwapChain
                nn::gfx::SwapChain::InfoType swapchainInfo;
                swapchainInfo.SetDefault();
                swapchainInfo.SetLayer(m_pLayer);
                swapchainInfo.SetWidth(m_DisplayWidth);
                swapchainInfo.SetHeight(m_DisplayHeight);
                swapchainInfo.SetFormat(info.GetSwapChainFormat());
                swapchainInfo.SetBufferCount(m_SwapChainBufferCount);
                {
                    MemoryPoolType type = MemoryPoolType_RenderTarget;
                    size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, swapchainInfo);
                    size_t alignment = nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, swapchainInfo);
                    ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                    NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                    m_SwapChain.Initialize(&m_Device, swapchainInfo, &m_MemoryPool[type].object, offset, size);
                }

                // ScanBuffer
                int count = m_SwapChainBufferCount;
                m_pScanBufferViews = reinterpret_cast<nn::gfx::ColorTargetView**>(m_AllocateFunction(sizeof(nn::gfx::ColorTargetView*) * count, NN_ALIGNOF(nn::gfx::ColorTargetView*), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pScanBufferViews);
                m_pScanBuffers = reinterpret_cast<nn::gfx::Texture**>(m_AllocateFunction(sizeof(nn::gfx::Texture*) * count, NN_ALIGNOF(nn::gfx::Texture*), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pScanBuffers);

                m_SwapChain.GetScanBufferViews(m_pScanBufferViews, count);
                m_SwapChain.GetScanBuffers(m_pScanBuffers, count);
            }
            else
            {
                // 共有バッファウィンドウ
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_SharedWindow.Initialize(&m_Device, info.GetBufferCount(), info.GetColorBufferFormat()));
                // ScanBuffer（値は AcquireTexture ごとに設定する）
                int count = m_SwapChainBufferCount;
                m_pScanBufferViews = reinterpret_cast<nn::gfx::ColorTargetView**>(m_AllocateFunction(sizeof(nn::gfx::ColorTargetView*) * count, NN_ALIGNOF(nn::gfx::ColorTargetView*), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pScanBufferViews);
                m_pScanBuffers = reinterpret_cast<nn::gfx::Texture**>(m_AllocateFunction(sizeof(nn::gfx::Texture*) * count, NN_ALIGNOF(nn::gfx::Texture*), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pScanBuffers);
            }

            if (info.IsColorBufferEnabled())
            {
                // ColorBuffer
                nn::gfx::Texture::InfoType textureInfo;
                textureInfo.SetDefault();
                textureInfo.SetWidth(m_DisplayWidth);
                textureInfo.SetHeight(m_DisplayHeight);
                textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
                textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
                textureInfo.SetImageFormat(info.GetColorBufferFormat());
                textureInfo.SetMipCount(1);
                textureInfo.SetDepth(1);

                MemoryPoolType type = MemoryPoolType_RenderTarget;
                size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, textureInfo);
                size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, textureInfo);

                int count = 1; //m_BufferCount;
                m_pColorBuffer = reinterpret_cast<nn::gfx::Texture*>(m_AllocateFunction(sizeof(nn::gfx::Texture) * count, NN_ALIGNOF(nn::gfx::Texture), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pColorBuffer);
                for (int i = 0; i < count; i++) {
                    new(&m_pColorBuffer[i])nn::gfx::Texture;
                    {
                        ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                        NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                        m_pColorBuffer[i].Initialize(&m_Device, textureInfo, &m_MemoryPool[type].object, offset, size);
                    }
                }

                // ColorTargetView
                nn::gfx::ColorTargetView::InfoType colorTargetViewInfo;
                colorTargetViewInfo.SetDefault();
                colorTargetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
                colorTargetViewInfo.SetImageFormat(info.GetColorBufferFormat());

                m_pColorTargetView = reinterpret_cast<nn::gfx::ColorTargetView*>(m_AllocateFunction(sizeof(nn::gfx::ColorTargetView) * count, NN_ALIGNOF(nn::gfx::ColorTargetView), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pColorTargetView);
                for (int i = 0; i < count; i++) {
                    new(&m_pColorTargetView[i])nn::gfx::ColorTargetView;
                    colorTargetViewInfo.SetTexturePtr(&m_pColorBuffer[i]);
                    m_pColorTargetView[i].Initialize(&m_Device, colorTargetViewInfo);
                }
            }
            else
            {
                m_pColorBuffer = nullptr;
                m_pColorTargetView = nullptr;
            }

            if (info.IsDepthStencilBufferEnabled())
            {
                // DepthStencilBuffer
                nn::gfx::Texture::InfoType textureInfo;
                textureInfo.SetDefault();
                textureInfo.SetWidth(m_DisplayWidth);
                textureInfo.SetHeight(m_DisplayHeight);
                textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
                textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
                textureInfo.SetImageFormat(info.GetDepthStencilBufferFormat());
                textureInfo.SetMipCount(1);
                textureInfo.SetDepth(1);

                MemoryPoolType type = MemoryPoolType_RenderTarget;
                size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, textureInfo);
                size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, textureInfo);

                int count = 1; //m_BufferCount;
                m_pDepthStencilBuffer = reinterpret_cast<nn::gfx::Texture*>(m_AllocateFunction(sizeof(nn::gfx::Texture) * count, NN_ALIGNOF(nn::gfx::Texture), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pDepthStencilBuffer);
                for (int i = 0; i < count; i++) {
                    new(&m_pDepthStencilBuffer[i])nn::gfx::Texture;
                    {
                        ptrdiff_t offset = m_MemoryPool[type].allocator.Allocate(size, alignment);
                        NN_SDK_ASSERT(offset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset);
                        m_pDepthStencilBuffer[i].Initialize(&m_Device, textureInfo, &m_MemoryPool[type].object, offset, size);
                    }
                }

                // DepthStencilView
                nn::gfx::DepthStencilView::InfoType depthStencilViewInfo;
                depthStencilViewInfo.SetDefault();
                depthStencilViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);

                m_pDepthStencilView = reinterpret_cast<nn::gfx::DepthStencilView*>(m_AllocateFunction(sizeof(nn::gfx::DepthStencilView) * count, NN_ALIGNOF(nn::gfx::DepthStencilView), m_pAllocateFunctionUserData));
                NN_SDK_ASSERT_NOT_NULL(m_pDepthStencilView);
                for (int i = 0; i < count; i++) {
                    new(&m_pDepthStencilView[i])nn::gfx::DepthStencilView;
                    depthStencilViewInfo.SetTexturePtr(&m_pDepthStencilBuffer[i]);
                    m_pDepthStencilView[i].Initialize(&m_Device, depthStencilViewInfo);
                }
            }
            else
            {
                m_pDepthStencilBuffer = nullptr;
                m_pDepthStencilView = nullptr;
            }
        }

        m_IsInitialized = true;

        // その他組み込み GFX オブジェクト
        {
            // ViewportScissorState
            {
                nn::gfx::ViewportStateInfo viewportInfo;
                viewportInfo.SetDefault();
                viewportInfo.SetWidth(static_cast< float >(m_DisplayWidth));
                viewportInfo.SetHeight(static_cast< float >(m_DisplayHeight));

                nn::gfx::ScissorStateInfo scissorInfo;
                scissorInfo.SetDefault();
                scissorInfo.SetWidth(m_DisplayWidth);
                scissorInfo.SetHeight(m_DisplayHeight);

                nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;
                viewportScissorInfo.SetDefault();
                viewportScissorInfo.SetScissorEnabled(true);
                viewportScissorInfo.SetViewportStateInfoArray(&viewportInfo, 1);
                viewportScissorInfo.SetScissorStateInfoArray(&scissorInfo, 1);

                NN_SDK_ASSERT(nn::gfx::ViewportScissorState::GetRequiredMemorySize(viewportScissorInfo) == 0);
                m_ViewportScissorState.Initialize(GetDevice(), viewportScissorInfo);
            }

            // VertexState
            {
                nn::gfx::VertexState::InfoType vertexInfo;

                // VertexStateType_Float3_Float2
                vertexInfo.SetDefault();
                ptrdiff_t stride = sizeof(float) * 5;
                nn::gfx::VertexAttributeStateInfo attribs[2];
                {
                    attribs[0].SetDefault();
                    //attribs[0].SetNamePtr("i_Position");
                    attribs[0].SetBufferIndex(0);
                    attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_32_Float);
                    attribs[0].SetOffset(0);
                    attribs[0].SetShaderSlot(0);
                }
                {
                    attribs[1].SetDefault();
                    //attribs[1].SetNamePtr("i_TexCoord");
                    attribs[1].SetBufferIndex(0);
                    attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
                    attribs[1].SetOffset(sizeof(float) * 3);
                    attribs[1].SetShaderSlot(1);
                }
                nn::gfx::VertexBufferStateInfo buffer;
                {
                    buffer.SetDefault();
                    buffer.SetStride(stride);
                }
                vertexInfo.SetVertexAttributeStateInfoArray(attribs, 2);
                vertexInfo.SetVertexBufferStateInfoArray(&buffer, 1);

                InitializeVertexState(&m_VertexState[VertexStateType_Float3_Float2], vertexInfo);
            }

            // RasterizerState
            {
                nn::gfx::RasterizerState::InfoType rasterizerInfo;

                // RasterizerStateType_FillSolid_CullNone
                rasterizerInfo.SetDefault();
                rasterizerInfo.SetCullMode(nn::gfx::CullMode_None);
                rasterizerInfo.SetScissorEnabled(true);
                rasterizerInfo.SetDepthClipEnabled(false);
                m_RasterizerState[RasterizerStateType_FillSolid_CullNone].Initialize(GetDevice(), rasterizerInfo);
            }

            // BlendState
            {
                nn::gfx::BlendState::InfoType blendInfo;
                nn::gfx::BlendTargetStateInfo targetInfo;

                // BlendStateType_Disabled
                blendInfo.SetDefault();
                targetInfo.SetDefault();
                targetInfo.SetBlendEnabled(false);
                blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
                InitializeBlendState(&m_BlendState[BlendStateType_Disabled], blendInfo);

                // BlendStateType_Alpha
                blendInfo.SetDefault();
                targetInfo.SetDefault();
                targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue);
                targetInfo.SetBlendEnabled(true);
                targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
                targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
                targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
                blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
                InitializeBlendState(&m_BlendState[BlendStateType_Alpha], blendInfo);
            }

            // DepthStencilState
            {
                nn::gfx::DepthStencilState::InfoType depthStencilInfo;

                // DepthStencilStateType_Disabled
                depthStencilInfo.SetDefault();
                depthStencilInfo.SetDepthTestEnabled(false);
                depthStencilInfo.SetDepthWriteEnabled(false);
                m_DepthStencilState[DepthStencilStateType_Disabled].Initialize(GetDevice(), depthStencilInfo);
            }

            // Sampler
            {
                nn::gfx::Sampler::InfoType samplerInfo;

                // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat
                samplerInfo.SetDefault();
                samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
                samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
                samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
                samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Repeat);
                m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat].Initialize(GetDevice(), samplerInfo);

                // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror
                samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
                samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
                samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
                m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror].Initialize(GetDevice(), samplerInfo);

                // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp
                samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
                samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
                samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
                m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp].Initialize(GetDevice(), samplerInfo);

                // SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border
                samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
                samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
                samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToBorder);
                m_Sampler[SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border].Initialize(GetDevice(), samplerInfo);
            }
        }

        // フレーム処理用
        m_FrameworkMode = FrameworkMode_DeferredSubmission;
        m_pCalculateCallback = nullptr;
        m_pCalculateCallbackUserData = nullptr;
        m_pMakeCommandCallback = nullptr;
        m_pMakeCommandCallbackUserData = nullptr;
        m_PresentInterval = 1;
        m_WaitSyncTimeout = nn::TimeSpan::FromSeconds(1);
        m_CurrentFrameIndex = 0;
        m_PrevFrameworkMode = m_FrameworkMode;
    }// NOLINT(impl/function_size)

    void GraphicsFrameworkEx::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        bool isSharedBufferSupported = nn::vi::fbshare::SharedLayerWindowGfx::IsSupported();

        // GFX 終了
        {
            // m_MemoryPoolAllocator, m_DescriptorPoolAllocator は Finalize() するので
            // 確保していた offset, index の個々の Free() は不要

            for (int i = 0; i < SamplerType_End; i++) {
                m_Sampler[i].Finalize(GetDevice());
            }
            for (int i = 0; i < DepthStencilStateType_End; i++) {
                m_DepthStencilState[i].Finalize(GetDevice());
            }
            for (int i = 0; i < BlendStateType_End; i++) {
                FinalizeBlendState(&m_BlendState[i]);
            }
            for (int i = 0; i < RasterizerStateType_End; i++) {
                m_RasterizerState[i].Finalize(GetDevice());
            }
            for (int i = 0; i < VertexStateType_End; i++) {
                FinalizeVertexState(&m_VertexState[i]);
            }
            m_ViewportScissorState.Finalize(GetDevice());

            if(isSharedBufferSupported)
            {
                m_FreeFunction(m_pScanBufferViews, m_pAllocateFunctionUserData);
                m_FreeFunction(m_pScanBuffers, m_pAllocateFunctionUserData);
                m_SharedWindow.Finalize();
            }
            else
            {
                m_FreeFunction(m_pScanBufferViews, m_pAllocateFunctionUserData);
                m_FreeFunction(m_pScanBuffers, m_pAllocateFunctionUserData);
                m_SwapChain.Finalize(&m_Device);
            }

            if (m_pDepthStencilBuffer != nullptr) {
                for (int i = 0; i < 1/*m_BufferCount*/; i++) {
                    m_pDepthStencilView[i].Finalize(&m_Device);
                    m_pDepthStencilBuffer[i].Finalize(&m_Device);
                }
                m_FreeFunction(m_pDepthStencilView, m_pAllocateFunctionUserData);
                m_FreeFunction(m_pDepthStencilBuffer, m_pAllocateFunctionUserData);
            }
            if (m_pColorBuffer != nullptr) {
                for (int i = 0; i < 1/*m_BufferCount*/; i++) {
                    m_pColorTargetView[i].Finalize(&m_Device);
                    m_pColorBuffer[i].Finalize(&m_Device);
                }
                m_FreeFunction(m_pColorTargetView, m_pAllocateFunctionUserData);
                m_FreeFunction(m_pColorBuffer, m_pAllocateFunctionUserData);
            }
            for (int i = 0; i < m_BufferCount; i++) {
                m_pCommandBuffer[i].object.Finalize(&m_Device);
                m_FreeFunction(m_pCommandBuffer[i].pControlMemory, m_pAllocateFunctionUserData);
            }
            m_FreeFunction(m_pCommandBuffer, m_pAllocateFunctionUserData);
            m_FreeFunction(m_pShaderScratchMemory, m_pAllocateFunctionUserData);

            for (int i = 0; i < nn::gfx::DescriptorPoolType_End; i++)
            {
                if (m_DescriptorPoolSlotCount[i] > 0) {
                    m_DescriptorPool[i].Finalize(&m_Device);
                    m_DescriptorPoolAllocator[i].Finalize();
                }
                m_DescriptorPoolSlotCount[i] = 0;
            }

            for (int i = 0; i < MemoryPoolType_End; i++)
            {
                if (m_MemoryPool[i].size == 0) {
                    continue;
                }
                m_MemoryPool[i].allocator.Finalize();
                m_MemoryPool[i].object.Finalize(&m_Device);
                m_MemoryPool[i].size = 0;
                m_FreeFunction(m_MemoryPool[i].pMemory, m_pAllocateFunctionUserData);
            }

            for (int i = 0; i < m_BufferCount; i++) {
                m_pDisplayFence[i].Finalize(&m_Device);
            }
            m_FreeFunction(m_pDisplayFence, m_pAllocateFunctionUserData);

            for (int i = 0; i < m_BufferCount; i++) {
                m_pGpuFence[i].Finalize(&m_Device);
            }
            m_FreeFunction(m_pGpuFence, m_pAllocateFunctionUserData);

            for (int i = 0; i < m_BufferCount; i++) {
                m_pDisplaySemaphore[i].Finalize(&m_Device);
            }
            m_FreeFunction(m_pDisplaySemaphore, m_pAllocateFunctionUserData);

            m_Queue.Finalize(&m_Device);
            m_Device.Finalize();
            nn::gfx::Finalize();
        }

        // VI 終了
        if(!isSharedBufferSupported)
        {
            nn::vi::DestroyLayer(m_pLayer);
            nn::vi::CloseDisplay(m_pDisplay);
            nn::vi::Finalize();
        }
        else
        {
            nn::vi::Finalize();
        }

        m_AllocateFunction = nullptr;
        m_FreeFunction = nullptr;
        m_pAllocateFunctionUserData = nullptr;

        m_IsInitialized = false;
    }// NOLINT(impl/function_size)

    void GraphicsFrameworkEx::AcquireTextureEx(int bufferIndex) NN_NOEXCEPT
    {
        if(!nn::vi::fbshare::SharedLayerWindowGfx::IsSupported())
        {
            return GraphicsFramework::AcquireTexture(bufferIndex);
        }

        NN_SDK_ASSERT(!m_SharedWindow.IsTextureAcquired());

        int index = -1;
        nn::gfx::Texture* pTexture = nullptr;
        // チェック用に長いタイムアウトを設定する（ am より長い）
        auto preAcqResult= m_SharedWindow.PreAcquireTexture(nn::TimeSpan::FromMilliSeconds(3000));
        if(preAcqResult.IsFailure())
        {
            NN_LOG("[I][graphic] PreAcquireTexture timed-out\n");
        }

        // 動作確認用に AcquireTexture と AcquirePreAcquiredTexture を交互に呼び出す。
        // ・ 1 フレームごとに交互に呼び出す。
        // ・ただし PreAcquireTexture() が失敗していると AcquireTexture() が無限待ちになる場合があるので
        //   失敗した場合は必ず AcquirePreAcquiredTexture を使う。
        static int64_t s_Count = 0;
        if(s_Count % 2 == 0 && preAcqResult.IsSuccess())
        {
            m_SharedWindow.AcquireTexture(&m_pDisplaySemaphore[bufferIndex], &m_pDisplayFence[bufferIndex], &index, &pTexture);
        }
        else
        {
            m_SharedWindow.AcquirePreAcquiredTexture(&m_pDisplaySemaphore[bufferIndex], &m_pDisplayFence[bufferIndex], &index, &pTexture);
        }
        s_Count++;

        nn::gfx::ColorTargetView* pView = m_SharedWindow.GetTextureColorTargetView(pTexture);
        m_pScanBuffers[index] = pTexture;
        m_pScanBufferViews[index] = pView;

        m_NextScanBufferIndex = index;
    }

    void GraphicsFrameworkEx::QueuePresentTextureEx(int presentInterval) NN_NOEXCEPT
    {
        if(!nn::vi::fbshare::SharedLayerWindowGfx::IsSupported())
        {
            return GraphicsFramework::QueuePresentTexture(presentInterval);
        }

        if(!m_SharedWindow.IsTextureAcquired())
        {
            NN_LOG("[I][graphic]skipped QueuePresentTextureEx. no buffer is acquired\n");
            return;
        }

        m_SharedWindow.SetSwapInterval(presentInterval);
        m_SharedWindow.PresentTexture(GetQueue(), m_NextScanBufferIndex);
    }

}

