﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <vector>

#include <nn/init.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/hid.h>
#include <nn/fs.h>
#include <nn/applet/applet.h>
#include <nn/applet/applet_KeyboardLayout.h>
#include <nn/apm/apm.h>
#include <nn/rid.h>
#include <nn/album.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/oe/oe_HomeButtonControl.h>
#include <nn/oe/oe_DebugApis.h>
#include <nn/oe/oe_ApplicationSelf.private.h>
#include <nn/oe/oe_MediaPlaybackSectionApi.private.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>
#include <nn/oe/oe_GamePlayRecordingApiForDebug.h>
#include <nn/oe/oe_GamePlayRecordingApiForDebug.private.h>
#include <nn/oe/oe_SpecificToOceanApis.private.h>
#include <nn/oe/oe_CaptureButtonControl.private.h>
#include <nn/oe/oe_LcdBacklightControlApi.private.h>
#include <nn/oe/oe_TvPowerStateMatchingControl.private.h>
#include <nn/oe/oe_SubProgramJump.private.h>
#include <nn/oe/oe_RidSpecificApis.private.h>
#include <nn/oe/oe_InStoreDemoApis.h>
#include <nn/pl/pl_VrModeApi.private.h>
#include <nn/applet/applet_LibraryAppletSelf.h>
#include <nn/applet/applet_InternalFunctions.h>
#include <nn/account.h>
#include <nn/account/account_StateRetention.h>

#include "../Common/Common.h"
#include "../Common/Graphics.h"
#include "../Common/AudioRenderer.h"
#include "../Common/ConvNpad.h"
#include "../Common/applet_Configs.h"

#include <nn/svc/svc_Base.h>

namespace nn { namespace oe {
    void SetFocusHandlingModeForDebug(bool, bool, bool, bool) NN_NOEXCEPT;
    bool IsGamePlayRecordingSupported() NN_NOEXCEPT;
}}


namespace {

// CAPTURE ボタン自前ハンドリング時に保存する画像データバッファ
NN_ALIGNAS(4096) nn::Bit32 g_ScreenShotBuffer[1280 * 720];

// アプリの権利表記用フレームバッファ
NN_ALIGNAS(4096) char g_CopyrightFrameBuffer[nn::oe::CopyrightFrameBufferSize];

// アプリの権利表記用画像バッファ
static const int CopyrightImageWidth = 640;
static const int CopyrightImageHeight = 360;
static const int CopyrightImagePositionX = 640;
static const int CopyrightImagePositionY = 360;
NN_ALIGNAS(4096) nn::Bit32 g_CopyrightImage[CopyrightImageWidth * CopyrightImageHeight];

// アプリの権利表記用画像の合成有無
bool g_IsCopyrightCompositionEnabled = false;

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// OE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

// ライブラリアプレット終了対応スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_LibraryApplet1HandleThreadStack[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_LibraryApplet2HandleThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_AudioThread;
nn::os::ThreadType  g_LibraryApplet1HandleThread;
nn::os::ThreadType  g_LibraryApplet2HandleThread;

// ライブラリアプレットハンドル
nn::applet::LibraryAppletHandle g_LibraryAppletHandle1;
nn::applet::LibraryAppletHandle g_LibraryAppletHandle2;

// アプレットパラメータ
char g_ParamBuffer[ nn::applet::StartupParamSizeMax ];

// メッセージバッファ
char g_MessageBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

enum AudioState
{
    AudioState_Disabled,
    AudioState_Enabled,
    AudioState_Stress,
};

AudioState g_AudioState = AudioState_Disabled;

// Audio が有効かどうか
bool g_IsAudioEnabled = false;

// VR モードかどうか
bool g_IsVrMode = false;

// LCD バックライトが OFF かどうか
bool g_IsLcdBacklightOff = false;

// HOME ブロック機能が有効かどうか
bool g_IsHomeBlockEnabled = false;

// スクリーンショット画像の回転方向
int g_AlbumImageOrientation = 0;

// CAPTURE ボタン短押しのハンドリングが有効かどうか
bool g_IsShortedCaptureButtonHandling = false;

// キャプチャボタン短押しによる撮影が有効かどうか
bool g_IsScreenShotCaptureEnabled = true;

// 常時動画撮影機能が有効かどうか
bool g_IsGamePlayRecordingEnabled = false;

// 終了要求受理時に LA1 起動を行うかどうか
bool g_IsLaunchLAWhenExitRequestEnabled = false;

// OutOfFocus 時のサスペンド有効無効設定
int g_OutOfFocusSuspendCount = 0;
auto g_FocusHandlingMode = nn::oe::FocusHandlingMode_Notify;

// WLAN 優先モードの切替
bool g_IsWlanPriorityModeEnabled = false;

// CEC テレビ電源連動モードの無効化
bool g_IsCecTvPowerStateMatchingDisabled = false;

// スレッド制御
ThreadControl g_ApplicationThreadControl;

std::string GetSubProgramName(int programIndex)
{
    switch (programIndex)
    {
        case 0: return "main";
        case 1: return "sub1";
        default: return "unknown";
    }
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // OE ライブラリの初期化
    nn::oe::Initialize();
    {
        NN_LOG("APP: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("APP: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.1f, 0.1f, 0.3f, 1.0f };
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredSubmission;
    auto name = "Application:" + GetSubProgramName(NNT_APPLETIAPP_PROGRAM_INDEX);
    Graphics::InitializeGraphics( clearColor, name, frameworkMode, false );

    // パラメータの取得 (AppletISA から起動されていれば取得できる)
    size_t realParameterSize;
    std::strcpy( g_MessageBuffer, "" );
    while ( nn::oe::TryPopLaunchParameter( &realParameterSize, g_ParamBuffer, sizeof(g_ParamBuffer) ) )
    {
        NN_LOG("APP: Launch parameter size = 0x%zx\n", realParameterSize);
        if (realParameterSize <= sizeof(g_MessageBuffer))
        {
            std::strcpy( g_MessageBuffer, g_ParamBuffer );
        }
    }
    std::strcat( g_MessageBuffer, "APP " );
    Graphics::SetMessage( g_MessageBuffer );

    // Audio の初期化
    nnt::applet::audiorenderer::InitializeAudio();
    nnt::applet::audiorenderer::LoadBgm( nnt::applet::audiorenderer::Bgm_B );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void OpenAccount()
{
    auto previousProgramIndex = nn::oe::GetPreviousProgramIndex();
    NN_LOG("APP: PreviousProgramIndex = %d\n", previousProgramIndex);

    nn::account::Initialize();
    if (previousProgramIndex < 0)
    {
        nn::account::UserHandle userHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenPreselectedUser(&userHandle));
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, userHandle));
        nn::account::Nickname nickname;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname, uid));
        NN_LOG("APP: first user: %s\n", nickname.name);
    }
    else
    {
        const int UserHandleCount = 8;
        nn::account::UserHandle userHandles[UserHandleCount];
        int n;
        nn::account::PopOpenUsers(&n, userHandles, 1);
        NN_LOG("APP: PopOpenUsers -> %d users\n", n);
        for (auto i = 0; i < n; ++i)
        {
            const auto& userHandle = userHandles[i];
            nn::account::Uid uid;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, userHandle));
            nn::account::Nickname nickname;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname, uid));
            NN_LOG("APP: - %s\n", nickname.name);
        }


        NN_LOG("APP: PreviousProgramIndex = %d\n", previousProgramIndex);
    }
}

void Finalize()
{
    // Audio 関連の終了処理
    nnt::applet::audiorenderer::FinalizeAudio();

    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

//-----------------------------------------------------------------------------
// LA の作成
//
nn::applet::LibraryAppletHandle PrepareLibraryApplet( nn::applet::AppletId id, const char* inData, size_t inDataSize, bool isAllForeground ) NN_NOEXCEPT
{
    auto mode = isAllForeground ? nn::applet::LibraryAppletMode_AllForeground : nn::applet::LibraryAppletMode_PartialForeground;

    // 作成
    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id, mode ) );
    NN_ABORT_UNLESS_EQUAL(mode, nn::applet::GetLibraryAppletMode(appletHandle));

    // 入力データの push
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
        nn::applet::PushToInChannel( appletHandle, storageHandle );
    }

    // LA 起動時のアプリサスペンドの有効化
    if (g_OutOfFocusSuspendCount > 0)
    {
        nn::applet::SetOutOfFocusApplicationSuspendingEnabled(appletHandle, true);
    }
    return appletHandle;
}

bool LaunchLibraryApplet1(bool isAllForeground)
{
    // Ocean の APP->LA 起動時は、CallerApplet にキャプチャ画像を残さないため、
    // AppletIApp でも残さないでおき、LA1 および LA2 にて、LastApplication の
    // キャプチャ画像を CallerApplet にもコピーするようにする。

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle1 = PrepareLibraryApplet( nn::applet::AppletId_LibraryAppletAuth, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle1 == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("APP: Request exit failed: Cannot create LA1 handle\n");
        return false;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( g_LibraryAppletHandle1, NULL ) );
    return true;
}

bool LaunchLibraryApplet2(bool isAllForeground, bool useImmediately)
{
    // Ocean の APP->LA 起動時は、CallerApplet にキャプチャ画像を残さないため、
    // AppletIApp でも残さないでおき、LA1 および LA2 にて、LastApplication の
    // キャプチャ画像を CallerApplet にもコピーするようにする。

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle2 = PrepareLibraryApplet( nn::applet::AppletId_LibraryAppletCabinet, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle2 == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("APP: Request exit failed: Cannot create LA1 handle\n");
        return false;
    }
    if (useImmediately)
    {
    retry:
        auto result = nn::applet::StartLibraryAppletImmediately( g_LibraryAppletHandle2, NULL );
        if (!result.IsSuccess())
        {
            if (result <= nn::applet::ResultCannotStartImmediately())
            {
                NN_LOG("#################################\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                goto retry;
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( g_LibraryAppletHandle2, NULL ) );
    }
    return true;
}

const char* GetLibraryAppletExitReasonString(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    switch (reason)
    {
        case nn::applet::LibraryAppletExitReason_Normal: return "LibraryAppletExitReason_Normal";
        case nn::applet::LibraryAppletExitReason_Canceled : return "LibraryAppletExitReason_Canceled";
        case nn::applet::LibraryAppletExitReason_Abnormal: return "LibraryAppletExitReason_Abnormal";
        case nn::applet::LibraryAppletExitReason_Unexpected: return "LibraryAppletExitReason_Unexpected";
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void WaitLibraryApplet1Exited()
{
    nn::applet::JoinLibraryApplet( g_LibraryAppletHandle1 );
    auto exitReason = nn::applet::GetLibraryAppletExitReason(g_LibraryAppletHandle1);
    NN_LOG("APP: JoinLibraryApplet1: %s\n", GetLibraryAppletExitReasonString(exitReason));

    if (exitReason == nn::applet::LibraryAppletExitReason_Normal)
    {
        NN_LOG("APP: Received PopFromOutChannelEvent\n" );
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS( TryPopFromOutChannel( &storageHandle, g_LibraryAppletHandle1 ) );
        NN_ABORT_UNLESS( GetStorageSize(storageHandle) == sizeof(g_ParamBuffer) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        ReleaseStorage( storageHandle );

        std::strcpy( g_MessageBuffer, g_ParamBuffer );
    }
    nn::applet::CloseLibraryApplet( g_LibraryAppletHandle1 );
    g_LibraryAppletHandle1 = nn::applet::InvalidLibraryAppletHandle;
}

void WaitLibraryApplet2Exited()
{
    nn::applet::JoinLibraryApplet( g_LibraryAppletHandle2 );
    NN_LOG("APP: JoinLibraryApplet2: %s\n", GetLibraryAppletExitReasonString(nn::applet::GetLibraryAppletExitReason(g_LibraryAppletHandle2)));

    nn::applet::CloseLibraryApplet( g_LibraryAppletHandle2 );
    g_LibraryAppletHandle2 = nn::applet::InvalidLibraryAppletHandle;
}

void MakeScreenShotPicture(void* buffer, size_t bufferSize)
{
    auto dst = static_cast<nn::Bit32*>(buffer);
    for (auto i=0u; i<bufferSize / sizeof(nn::Bit32); ++i)
    {
        *dst++ = i | 0xff000000;
    }
}

} // namespace

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void MessageThreadFunction(void *arg) NN_NOEXCEPT
{
    for (;;)
    {
        auto message = nn::oe::PopNotificationMessage();
        switch (message)
        {
        case nn::oe::MessageFocusStateChanged:
            {
                auto focusState = nn::oe::GetCurrentFocusState();
                switch (focusState)
                {
                    case nn::oe::FocusState_InFocus:
                    {
                        NN_LOG("APP: Received MessageFocusStateChanged: current=InFocus\n");
                        g_ApplicationThreadControl.SetInFocusState(true);
                        break;
                    }
                    case nn::oe::FocusState_OutOfFocus:
                    {
                        NN_LOG("APP: Received MessageFocusStateChanged: current=OutOfFocus\n");
                        g_ApplicationThreadControl.SetInFocusState(false);
                        break;
                    }
                    case nn::oe::FocusState_Background:
                    {
                        NN_LOG("APP: Received MessageFocusStateChanged: current=Background\n");
                        g_ApplicationThreadControl.SetInFocusState(false);
                        break;
                    }
                    default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;

        case nn::oe::MessageResume:
            NN_LOG("APP: Received MessageResume\n");
            break;

        case nn::oe::MessageRequestToEndVrMode:
            NN_LOG("APP: Received MessageRequestToEndVrMode\n");
            g_IsVrMode = nn::pl::IsVrMode();
            NN_LOG("APP: Current VR Mode is %s\n", g_IsVrMode ? "true" : "false");
            break;

        case nn::oe::MessageLcdBacklightSwitchedOn:
            NN_LOG("APP: Received MessageLcdBacklightSwitchedOn\n");
            g_IsLcdBacklightOff = false;
            break;

        case nn::oe::MessageCaptureButtonPressedShortly:
            NN_LOG("APP: Received Message(90) which means capture button pressed shortly\n");
            MakeScreenShotPicture(g_ScreenShotBuffer, sizeof(g_ScreenShotBuffer));
            NN_LOG("APP: Invoke nn::album::SaveScreenshot()\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotBuffer, sizeof(g_ScreenShotBuffer), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );
            break;

        case nn::oe::MessageOperationModeChanged:
            NN_LOG("APP: Received MessageOperationModeChanged\n");
            NN_LOG("APP: Currently Operation Mode is %s Mode\n", nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        case nn::oe::MessagePerformanceModeChanged:
            NN_LOG("APP: Received MessagePerformanceModeChanged\n");
            NN_LOG("APP: Currently Performance Mode is %s Mode\n", nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");
            break;

        case nn::oe::MessageExitRequest:
            NN_LOG("APP: Received MessageExitRequest\n");
            g_ApplicationThreadControl.RequestExit();

            if (g_IsLaunchLAWhenExitRequestEnabled)
            {
                // 意図的に LA1 を起動する
                LaunchLibraryApplet1(true);
            }

            // 終了リクエストのハンドリングを終了
            nn::oe::LeaveExitRequestHandlingSection();
            return;

        default:
            NN_LOG("APP: Received unknown message= 0x%08x\n", message);
            break;
        }
    }
}

//-----------------------------------------------------------------------------
// LA 制御用のワーカースレッド
//
enum WorkerCommand
{
    WorkerCommand_None,
    WorkerCommand_LaunchLA,
    WorkerCommand_LaunchLAWithPartialFG,
    WorkerCommand_LaunchLAImmediately,
    WorkerCommand_Wait3Seconds,
    WorkerCommand_SendRequestExitAfter5Seconds,
};

const int CommandBufferCount = 8;
uintptr_t g_CommandBuffer1[CommandBufferCount];
nn::os::MessageQueue g_CommandQueue1(g_CommandBuffer1, CommandBufferCount);
uintptr_t g_CommandBuffer2[CommandBufferCount];
nn::os::MessageQueue g_CommandQueue2(g_CommandBuffer2, CommandBufferCount);

void LibraryApplet1HandleThread( void *arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    bool isLibraryAppletLaunched = false;

    while (g_ApplicationThreadControl.WaitForInFocusOrExitRequested())
    {
        uintptr_t cmd;
        g_CommandQueue1.Receive(&cmd);
        switch (cmd)
        {
            // LA1 の通常起動
            case WorkerCommand_LaunchLA:
            {
                static nn::Bit32 layout = 0x000F0001;
                layout = (layout << 1) | (layout >> 31);
                NN_LOG("APP: Invoke nn::applet::SetDesirableKeyboardLayout(0x%08x)\n", layout);
                nn::applet::SetDesirableKeyboardLayout(layout);
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet() (LA1)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(true);
                break;
            }
            // LA1 の Partial-FG 起動
            case WorkerCommand_LaunchLAWithPartialFG:
            {
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet() (LA1, Partial FG)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(false);
                break;
            }
            // LA1 起動後に、5 秒待機して LA1 に終了を要求
            case WorkerCommand_SendRequestExitAfter5Seconds:
            {
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet(LA1)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(true);

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
                NN_LOG("APP: Invoke nn::applet::RequestExitLibraryApplet(LA1)\n");
                nn::applet::RequestExitLibraryApplet( g_LibraryAppletHandle1 );
                break;
            }
            default:
            {
                NN_LOG("APP: Unknown internal command: 0x%08x\n", cmd);
                break;
            }
        }

        // LA1 の終了を待機
        if (isLibraryAppletLaunched)
        {
            WaitLibraryApplet1Exited();
            isLibraryAppletLaunched = false;
        }
    }
}

void LibraryApplet2HandleThread( void *arg ) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    bool isLibraryAppletLaunched = false;

    while (g_ApplicationThreadControl.WaitForInFocusOrExitRequested())
    {
        uintptr_t cmd;
        g_CommandQueue2.Receive(&cmd);
        switch (cmd)
        {
            // LA2 の通常起動
            case WorkerCommand_LaunchLA:
            {
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet() (LA2)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet2(true, false);
                break;
            }
            // LA2 の Partial-FG 起動
            case WorkerCommand_LaunchLAWithPartialFG:
            {
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet() (LA1, Partial FG)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet2(false, false);
                break;
            }
            // LA2 の通常起動（StartImmediately を使う）
            case WorkerCommand_LaunchLAImmediately:
            {
                NN_LOG("APP: Invoke nn::applet::CreateLibraryApplet() (LA2)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet2(true, true);
                break;
            }
            // LA2 起動前の 3 秒待機
            case WorkerCommand_Wait3Seconds:
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
                break;
            }
            default:
            {
                NN_LOG("APP: Unknown internal command: 0x%08x\n", cmd);
                break;
            }
        }

        // LA2 の終了を待機
        if (isLibraryAppletLaunched)
        {
            WaitLibraryApplet2Exited();
            isLibraryAppletLaunched = false;
        }
    }
}

//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed;
    nn::hid::DebugPadButtonSet trigger;

    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::WaitVsync();
        nnt::applet::hid::GetConvNpadState( &debugPadState );

        // ボタン情報の更新
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // Partial Foreground な LA1 が起動していたら何もしない
        if ( g_LibraryAppletHandle1 != nn::applet::InvalidLibraryAppletHandle )
        {
            continue;
        }

        // DebugPad の ZL/ZR/L/R ボタンを 4 つ同時押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // ZL/ZR/L/R 同時押し ＋ SELECT(-) 押しでシャットダウンを要求
            if ((trigger & nn::hid::DebugPadButton::Select::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::RequestToShutdown()\n");
                nn::oe::RequestToShutdown();
            }

            // ZL/ZR/L/R 同時押し ＋ START(+) 押しで再起動を要求
            if ((trigger & nn::hid::DebugPadButton::Start::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::RequestToReboot()\n");
                nn::oe::RequestToReboot();
            }
            continue;
        }

        // DebugPad の ZR と ZL を同時に押しながら～の操作
        if (((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn()) &&
            ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn()))
        {
            // ZR + ZL + 右 で画像の回転方向を変更（静止画・動画の両方に影響）
            if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                g_AlbumImageOrientation = (g_AlbumImageOrientation + 1) & 0x3;
                NN_LOG("APP: Invoke oe::SetAlbumImageOrientation(%d)\n", g_AlbumImageOrientation);
                nn::oe::SetAlbumImageOrientation(static_cast<nn::album::ImageOrientation>(g_AlbumImageOrientation));
            }
            // ZR + ZL + 左 で権利表記画像の合成の ON/OFF
            else if ((trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                g_IsCopyrightCompositionEnabled = !g_IsCopyrightCompositionEnabled;
                NN_LOG("APP: Invoke oe::SetCopyrightVisibility(%s)\n", g_IsCopyrightCompositionEnabled ? "true" : "false");
                nn::oe::SetCopyrightVisibility(g_IsCopyrightCompositionEnabled);
            }

            // ZR + ZL + 下 で WLAN 優先モードをトグル
            else if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsWlanPriorityModeEnabled = !g_IsWlanPriorityModeEnabled;
                NN_LOG("APP: Invoke applet::SetWirelessPriorityMode(%s)\n", g_IsWlanPriorityModeEnabled ? "WLAN" : "Bluetooth");
                nn::applet::SetWirelessPriorityMode(g_IsWlanPriorityModeEnabled ? nn::applet::WirelessPriorityMode_OptimizedForWlan : nn::applet::WirelessPriorityMode_OptimizedForBluetooth);
            }

            // ZR + ZL + 上 で CEC テレビ電源連動モードの無効化
            else if ((trigger & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                g_IsCecTvPowerStateMatchingDisabled = !g_IsCecTvPowerStateMatchingDisabled;
                NN_LOG("APP: Invoke oe::SetTvPowerStateMatchingMode(%s)\n", g_IsCecTvPowerStateMatchingDisabled ? "Disabled" : "Default");
                nn::oe::SetTvPowerStateMatchingMode(g_IsCecTvPowerStateMatchingDisabled ? nn::oe::TvPowerStateMatchingMode_Disabled : nn::oe::TvPowerStateMatchingMode_Default);
            }
            continue;
        }

        // X でアプリ自身による終了
        if ( (trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn() )
        {
            // 上 + X で std::quick_exit() による自発終了
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke std::quick_exit(0)\n");
                std::quick_exit(0);
            }
            // 下 + X で nn::oe::ExitApplication() による自発終了
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::ExitApplication()\n");
                nn::oe::ExitApplication();
            }
            // 左 + X で nn::oe::ExitApplicationAndGoBackToInStoreDemoMenu() による自発終了
            else if ((pressed & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::ExitApplicationFromInStoreDemo()\n");
                nn::oe::EnterExitRequestHandlingSection();
                // 上記 EnterExitRequestHandlingSection() の効果は無効化される
                nn::oe::ExitApplicationAndGoBackToInStoreDemoMenu();
            }
            // 右 + X で nn::oe::ExitApplication() による自発終了 ＋ 動画保存
            else if ((pressed & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::RequestToSaveRecordingForDebug() and wait 5 sec.\n");
                nn::oe::RequestToSaveRecordingForDebug();
                NN_LOG("APP: Invoke nn::oe::ExitApplication()\n");
                nn::oe::ExitApplication();
            }
            // 左 + X で 動画保存待機後 nn::oe::ExitApplication() による自発終了
            else if ((pressed & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::oe::SaveGamePlayMovieForDebug()\n");
                auto result = nn::oe::SaveGamePlayRecordingForDebug(0);
                if (!result.IsSuccess())
                {
                    NN_LOG("APP: SaveGamePlayMovieForDebug(): error %08x\n", result.GetInnerValueForDebug());
                }
                nn::oe::ExitApplication();
            }
        }

        // DebugPad の Y が押された
        if ( (trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn() &&
             (g_LibraryAppletHandle1 == nn::applet::InvalidLibraryAppletHandle) )
        {
            // まず LA1 を起動
            bool isAllForeground = true;
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                // UP + Y なら Partial Foreground 起動
                g_CommandQueue1.Send(WorkerCommand_LaunchLAWithPartialFG);
                isAllForeground = false;
            }
            else
            {
                if ( (pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn() )
                {
                    // DOWN + Y なら、5 秒後に自動的に終了
                    g_CommandQueue1.Send(WorkerCommand_SendRequestExitAfter5Seconds);
                }
                else
                {
                    // LA1 の通常起動
                    g_CommandQueue1.Send(WorkerCommand_LaunchLA);
                }
            }

            // ZL + Y なら LA2 も起動
            // 上記 LA1 起動と同時に実行可能（SIGLO-43604 検証用）
            if ( (pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn() )
            {
                // LA1 が非同期の場合 3 秒待機する(3 秒後に LA1 は終了する)
                if (!isAllForeground)
                {
                    g_CommandQueue2.Send(WorkerCommand_Wait3Seconds);
                }
                // さらに ZR も押している場合には StartImmediately を使用する
                if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
                {
                    g_CommandQueue2.Send(WorkerCommand_LaunchLAImmediately);
                }
                else
                {
                    // LA2 の通常起動
                    g_CommandQueue2.Send(WorkerCommand_LaunchLA);
                }
            }
        }

        // DebugPad の R で負荷を増やす
        if ((pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() + 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() + 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() + 1);
            }
        }
        // DebugPad の L で負荷を減らす
        if ((pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() - 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() - 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() - 1);
            }
        }

        // DebugPad の ZR を押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            // ZR + "+" (Start) で BGM の on/off
            if ((trigger & nn::hid::DebugPadButton::Start::Mask).IsAnyOn())
            {
                if( g_AudioState == AudioState_Stress )
                {
                    NN_LOG( "APP: Disable Audio\n" );
                    g_IsAudioEnabled = false;
                    g_AudioState = AudioState_Disabled;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                    nnt::applet::audiorenderer::PauseBgm();
                    nnt::applet::audiorenderer::PauseSine();
                }
                else if( g_AudioState == AudioState_Disabled )
                {
                    NN_LOG( "APP: Enable Audio\n" );
                    g_IsAudioEnabled = true;
                    g_AudioState = AudioState_Enabled;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                    nnt::applet::audiorenderer::PlayBgm();
                }
                else
                {
                    NN_LOG( "APP: Add Audio\n" );
                    g_IsAudioEnabled = true;
                    g_AudioState = AudioState_Stress;
                    Graphics::SetAudioStatus( g_IsAudioEnabled );
                    nnt::applet::audiorenderer::PlaySine();
                }
            }
            // ZR + B で LCD バックライトの ON/OFF
            else if ((trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn())
            {
                g_IsLcdBacklightOff = !g_IsLcdBacklightOff;
                NN_LOG("APP: LCD-Backlight is %s.\n", g_IsLcdBacklightOff ? "OFF" : "ON");
                if (g_IsLcdBacklightOff)
                {
                    nn::oe::SwitchLcdBacklightOff();
                }
                else
                {
                    nn::oe::SwitchLcdBacklightOn();
                }
            }
            // ZR + 左 で VR モードの on/off
            else if ((trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                g_IsVrMode = !g_IsVrMode;
                NN_LOG("APP: VR-mode is %s.\n", g_IsVrMode ? "true" : "false");
                if (g_IsVrMode)
                {
                    nn::pl::BeginVrMode();
                }
                else
                {
                    nn::pl::EndVrMode();
                }
            }
            // ZR + 上 で キャプチャボタン短押しハンドリングの on/off
            else if ((trigger & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                g_IsShortedCaptureButtonHandling = !g_IsShortedCaptureButtonHandling;
                if (g_IsShortedCaptureButtonHandling)
                {
                    NN_LOG("APP: Invoke oe::BeginHandlingCaptureButtonShortPressed()\n");
                    nn::oe::BeginHandlingCaptureButtonShortPressed();
                }
                else
                {
                    NN_LOG("APP: Invoke oe::EndHandlingCaptureButtonShortPressed()\n");
                    nn::oe::EndHandlingCaptureButtonShortPressed();
                }
            }
            // ZR + 下 で 常時動画撮影 の on/off
            else if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsGamePlayRecordingEnabled = !g_IsGamePlayRecordingEnabled;
                if (g_IsGamePlayRecordingEnabled)
                {
                    NN_LOG("APP: Invoke oe::EnableRecording()\n");
                    nn::oe::EnableRecording();
                }
                else
                {
                    NN_LOG("APP: Invoke oe::DisableRecording()\n");
                    nn::oe::DisableRecording();
                }
            }
            // ZR + 右 で キャプチャ撮影 の on/off
            else if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                g_IsScreenShotCaptureEnabled = !g_IsScreenShotCaptureEnabled;
                if (g_IsScreenShotCaptureEnabled)
                {
                    NN_LOG("APP: Invoke oe::EnableScreenShot()\n");
                    nn::oe::EnableScreenShot();
                }
                else
                {
                    NN_LOG("APP: Invoke oe::DisableScreenShot()\n");
                    nn::oe::DisableScreenShot();
                }
            }

            // ZR + X で試遊台向けのアプリ終了プレイタイマー発動
            // 自アプリをアプリ終了タイマー付きで起動要求
            // 起動パラメータとして、0byte のストレージを積む
            if ( (trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn() )
            {
                NN_LOG("APP: Invoke nn::applet::RequestToLaunchApplicationForQuest(30s, 10s)\n");
                nn::applet::ApplicationAttributeForQuest attribute = {};
                attribute.playableTime      = 30;   // 秒
                attribute.idleDetectionTime = 10;   // 秒
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storageHandle, 0));
                nn::applet::RequestToLaunchApplicationForQuest(appletIntegrationApplicationId, storageHandle, &attribute);
            }

            // ZR + A で自プログラムの再起動を SA に要求
            // 起動パラメータは積まない
            if ( (trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn() )
            {
                NN_LOG("APP: Invoke nn::oe::RequestToRelaunchApplication()\n");
                nn::oe::RequestToRelaunchApplication();
            }
        }

        // DebugPad の ZL ボタンを押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn())
        {
            // ZL + 上 で HOME ブロックの ON/OFF
            if ((trigger & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                g_IsHomeBlockEnabled = !g_IsHomeBlockEnabled;
                if (g_IsHomeBlockEnabled)
                {
                    NN_LOG("APP: nn::oe::BeginBlockingHomeButton()\n");
                    nn::oe::BeginBlockingHomeButton();
                }
                else
                {
                    NN_LOG("APP: nn::oe::EndBlockingHomeButton()\n");
                    nn::oe::EndBlockingHomeButton();
                }
            }
            // ZL + 下 で終了要求受理後の LA1 起動の ON/OFF
            else if ((trigger & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                g_IsLaunchLAWhenExitRequestEnabled = !g_IsLaunchLAWhenExitRequestEnabled;
                NN_LOG("APP: %s launching LA1 when exit requested.\n", g_IsLaunchLAWhenExitRequestEnabled ? "Enable" : "Disable");
            }
            // ZL + 左 でフォーカスハンドリングモードの変更
            else if ((trigger & nn::hid::DebugPadButton::Left::Mask).IsAnyOn())
            {
                switch (g_FocusHandlingMode)
                {
                    case nn::oe::FocusHandlingMode_Suspend:
                    {
                        g_FocusHandlingMode = nn::oe::FocusHandlingMode_Notify;
                        NN_LOG("APP: SetFocusHandlingMode( Notify )\n");
                        break;
                    }
                    case nn::oe::FocusHandlingMode_Notify:
                    {
                        g_FocusHandlingMode = nn::oe::FocusHandlingMode_SuspendAndNotify;
                        NN_LOG("APP: SetFocusHandlingMode( SuspendAndNotify )\n");
                        break;
                    }
                    case nn::oe::FocusHandlingMode_SuspendAndNotify:
                    {
                        g_FocusHandlingMode = nn::oe::FocusHandlingMode_InFocusOnly;
                        NN_LOG("APP: SetFocusHandlingMode( InFocusOnly )\n");
                        break;
                    }
                    case nn::oe::FocusHandlingMode_InFocusOnly:
                    {
                        g_FocusHandlingMode = nn::oe::FocusHandlingMode_Suspend;
                        NN_LOG("APP: SetFocusHandlingMode( Suspend )\n");
                        break;
                    }
                    default: NN_UNEXPECTED_DEFAULT;
                }
                nn::oe::SetFocusHandlingMode(g_FocusHandlingMode);
            }
            // ZL + 右 で OutOfFocus 時のサスペンドカウントを 0<->1 でトグル
            else if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                g_OutOfFocusSuspendCount = 1 - g_OutOfFocusSuspendCount;
                NN_LOG( "APP: OutOfFocus suspend count = %d\n", g_OutOfFocusSuspendCount);
            }
            // ZL + X でプログラムの再起動
            else if ((trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn())
            {
                NN_LOG( "APP: JumpToSubProgram(self)\n");
                uint8_t programIndex = []
                {
                    switch (NN_STATIC_CONDITION(NNT_APPLETIAPP_PROGRAM_INDEX))
                    {
                        case 0: return 1;
                        case 1: return 0;
                        default: return 0;
                    }
                }();
                auto prefix = GetSubProgramName(NNT_APPLETIAPP_PROGRAM_INDEX) + " > ";
                std::memset(g_ParamBuffer, 0, sizeof(g_ParamBuffer));
                std::strcpy(g_ParamBuffer, prefix.c_str());
                nn::account::PushOpenUsers();
                nn::oe::ExecuteProgram(programIndex, g_ParamBuffer, 4096);
                //nn::oe::JumpToSubProgramForDevelopment({0x010000000000b1c1}, g_ParamBuffer, sizeof(g_ParamBuffer));
                //nn::oe::RestartProgram(g_ParamBuffer, sizeof(g_ParamBuffer));
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer();
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    while ( g_ApplicationThreadControl.WaitForInFocusOrExitRequested() )
    {
        nnt::applet::audiorenderer::WaitAudio();
        nnt::applet::audiorenderer::UpdateAudio();
        Graphics::WaitVsync();
    }
}

const char* GetStorageTypeString(nn::oe::StorageTypeForDebug type)
{
    switch (type)
    {
    case nn::oe::StorageTypeForDebug_None:
            return "None";

    case nn::oe::StorageTypeForDebug_Host:
            return "PC-Host";

    case nn::oe::StorageTypeForDebug_GameCard:
            return "GameCard";

    case nn::oe::StorageTypeForDebug_BuiltInStorage:
            return "NAND";

    case nn::oe::StorageTypeForDebug_SdCard:
            return "SD-Card";

    default:
            return "Unknown";
    }
}

void DisplayMemoryInfo()
{
    nn::os::MemoryInfo info;
    nn::os::QueryMemoryInfo( &info );

    NN_LOG("Total Available Memory    : 0x%p\n", info.totalAvailableMemorySize);
    NN_LOG("Total Used Memory         : 0x%p\n", info.totalUsedMemorySize);
    NN_LOG("Total MemoryHeap size     : 0x%p\n", info.totalMemoryHeapSize);
    NN_LOG("Program Size              : 0x%p\n", info.programSize);
    NN_LOG("Allocated MemoryHeap size : 0x%p\n", info.allocatedMemoryHeapSize);
    NN_LOG("Total Threads Stack size  : 0x%p\n", info.totalThreadStackSize);
    NN_LOG("Current thread count      : %d\n",   info.threadCount);
}

// アプリの権利表記画像を読み込む
size_t ReadCopyrightImage(void* buffer, size_t bufferSize)
{
    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::OpenFile(&handle, "asset:/Images/ApplicationCopyright.dat", nn::fs::OpenMode_Read) );

    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::GetFileSize(&size, handle) );
    NN_ABORT_UNLESS( size <= bufferSize );

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::ReadFile(handle, 0, buffer, size) );
    nn::fs::CloseFile(handle);

    return size;
}

// 権利表記画像を（再）設定する
void SetCopyrightImage()
{
#if 0
    // nn::oe::SetCopyrightImage() に sMMU マップしたメモリを渡せることの確認用コード
    // nn::oe::SetCopyrightImage() の機能のみを見る分には不要
    nn::dd::DeviceAddressSpaceType device;
    const nn::dd::DeviceVirtualAddress deviceAddress = 0x40000000;
    {
        const uint64_t DeviceAddressSpaceSize  = 4ull * 1024ull * 1024ull * 1024ull;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::dd::CreateDeviceAddressSpace(&device, DeviceAddressSpaceSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::dd::MapDeviceAddressSpaceNotAligned( &device, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(g_CopyrightImage), sizeof(g_CopyrightImage), deviceAddress, nn::dd::MemoryPermission_ReadWrite) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::dd::AttachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata) );

        nn::svc::MemoryInfo info;
        nn::svc::PageInfo pageInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::svc::QueryMemory(&info, &pageInfo, reinterpret_cast<uint64_t>(g_CopyrightImage)));
        NN_LOG("APP: svc::QueryMemoryInfo() address=0x%p attribute=0x%08x\n", g_CopyrightImage, info.attribute);
        NN_ABORT_UNLESS((info.attribute & nn::svc::MemoryAttribute_DeviceShared) != 0);
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::dd::UnmapDeviceAddressSpace(&device, nn::dd::GetCurrentProcessHandle(), reinterpret_cast<uint64_t>(g_CopyrightImage), sizeof(g_CopyrightImage), deviceAddress);
        nn::dd::DetachDeviceAddressSpace(&device, nn::dd::DeviceName_Sata);
        nn::dd::DestroyDeviceAddressSpace(&device);
    };
#endif

    // DeviceShared となったフレームバッファを転送元として渡す
    nn::oe::SetCopyrightImage(g_CopyrightImage, sizeof(g_CopyrightImage), CopyrightImagePositionX, CopyrightImagePositionY, CopyrightImageWidth, CopyrightImageHeight, nn::oe::WindowOriginMode_LowerLeft);
    g_IsCopyrightCompositionEnabled = true;

}

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[256 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );

    DisplayMemoryInfo();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( 0x400000 ) );
}

extern "C" void nnMain()
{
    NN_LOG( "APP: start.\n" );

    // すでに InFocus 状態である
    g_ApplicationThreadControl.SetInFocusState(true);

    // 初期化処理
    Initialize();
    nn::album::Initialize();

    OpenAccount();

    // 権利表記の準備
    {
        nn::oe::InitializeCopyrightFrameBuffer(g_CopyrightFrameBuffer, sizeof(g_CopyrightFrameBuffer));
        auto size = ReadCopyrightImage(g_CopyrightImage, sizeof(g_CopyrightImage));
        NN_ABORT_UNLESS_EQUAL(size, static_cast<size_t>(4 * CopyrightImageWidth * CopyrightImageHeight));

        // 設定
        SetCopyrightImage();
    }

    // メモリ使用状況
    DisplayMemoryInfo();

    {
        auto lang = nn::oe::GetDesiredLanguage();
        NN_LOG("APP: oe::GetDesiredLanguage() lang=\x22%s\x22\n", lang.string);
    }

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, MessageThreadFunction, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_LibraryApplet1HandleThread, LibraryApplet1HandleThread, NULL, g_LibraryApplet1HandleThreadStack, ThreadStackSize, AppletPriorityForLimit10msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_LibraryApplet2HandleThread, LibraryApplet2HandleThread, NULL, g_LibraryApplet2HandleThreadStack, ThreadStackSize, AppletPriorityForLimit10msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec) );

    // 終了リスクエストをハンドリングする宣言
    nn::oe::EnterExitRequestHandlingSection();

    // 常時動画撮影が有効なファームウェアか？
    g_IsGamePlayRecordingEnabled = nn::oe::IsGamePlayRecordingSupported();
    NN_LOG("APP: IsGamePlayRecordingSupported() = %s\n", g_IsGamePlayRecordingEnabled ? "true" : "false");

    // 動作モードと性能モードの変更通知を受け取る
    nn::oe::SetOperationModeChangedNotificationEnabled(true);
    nn::oe::SetPerformanceModeChangedNotificationEnabled(true);

    // 状態遷移ハンドリングモードの指定
    nn::oe::SetFocusHandlingMode( g_FocusHandlingMode );
    //nn::oe::SetFocusHandlingModeForDebug(false, true, false, false);
    nn::oe::SetResumeNotificationEnabled(true);

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_AudioThread );
    nn::os::StartThread( &g_LibraryApplet1HandleThread );
    nn::os::StartThread( &g_LibraryApplet2HandleThread );

    // 起動デバイスの取得
    nn::oe::LaunchStorageInfoForDebug storageInfo;
    nn::oe::GetLaunchStorageInfoForDebug(&storageInfo);
    NN_LOG("APP: LaunchStorage = %s\n", GetStorageTypeString(storageInfo.launchStorage));
    NN_LOG("APP: patchStorage  = %s\n", GetStorageTypeString(storageInfo.patchStorage));

    // 動作モードと性能モードの取得
    NN_LOG("APP: Currently Operation Mode is %s Mode\n", nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
    NN_LOG("APP: Currently Performance Mode is %s Mode\n", nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");

    // API が呼べることだけ確認しておく
    nn::applet::OverrideAutoSleepTimeAndDimmingTime(60, 60, 10, 10);
    nn::oe::BeginMediaPlaybackSection();
    nn::oe::BeginMediaPlaybackSection();
    nn::applet::SetMediaPlaybackState(true);
    nn::applet::SetMediaPlaybackState(false);
    nn::oe::EndMediaPlaybackSection();
    nn::oe::EndMediaPlaybackSection();
    nn::applet::OverrideAutoSleepTimeAndDimmingTime(0, 0, 0, 0);

#if 1
    nn::oe::DisableScreenShot();
    nn::oe::EnableScreenShot();
    g_IsScreenShotCaptureEnabled = true;
#else
    for (;;)
    {
        nn::oe::DisableScreenShot();
        NN_LOG("APP: Invoked nn::oe::DisableScreenShot()\n");
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

        nn::oe::EnableScreenShot();
        NN_LOG("APP: Invoked nn::oe::EnableScreenShot()\n");
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );
    }
#endif

    // ロゴ消去を 5 秒遅延
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
    nn::oe::FinishStartupLogo();

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_AudioThread );
    nn::os::WaitThread( &g_LibraryApplet1HandleThread );
    nn::os::WaitThread( &g_LibraryApplet2HandleThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_AudioThread );
    nn::os::DestroyThread( &g_LibraryApplet1HandleThread );
    nn::os::DestroyThread( &g_LibraryApplet2HandleThread );

    // 終了処理
    Finalize();

    NN_LOG( "APP: end.\n" );

    return;
}
