﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nnt/nsutil/nsutil_InstallUtils.h>
#include <nnt/ncmutil/ncmutil_InstallUtils.h>

#include <nn/aoc.h>
#include <nn/aoc/aoc_PrivateApi.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/nifm.h>

#include <nn/aoc/detail/aoc_Service.h>
#include <nn/aocsrv/aocsrv_AddOnContentHipcServer.h>
#include <nn/ns/ns_ApiForDfc.h>

// noclean でシステムセーブデータが消えない間のワークアラウンド
#include <nn/fs/fs_SaveDataManagement.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/os/os_Thread.h>
#endif

#include <nn/es/es_Api.h>
#include <nn/es/es_InitializationApi.h>

namespace {
    class AocListingTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::InitializeDependenciesForDfc();
#endif
            m_NspPathForNand = std::string(nnt::GetHostArgv()[2]);
            NN_LOG("nspForNandPath %s\n", m_NspPathForNand.c_str());
            m_NspPathForSd = std::string(nnt::GetHostArgv()[3]);
            NN_LOG("nspForSdPath %s\n", m_NspPathForSd.c_str());

            nn::ncm::Initialize();
            nn::es::Initialize();
            nn::ns::Initialize();
            nn::ns::InvalidateAllApplicationControlCache();
            {
                const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
                nn::ns::ApplicationRecord recordList[128];
                auto recordCount = nn::ns::ListApplicationRecord(recordList, sizeof(recordList) / sizeof(recordList[0]), 0);
                for (int i = 0; i < recordCount; i++)
                {
                    NNT_EXPECT_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletelyForDebug(recordList[i].id, flags));
                }
            }

            NNT_EXPECT_RESULT_SUCCESS(InstallAllTestApplications());
            DeleteTestApplicationTicket();
        }

        virtual void TearDown()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::FinalizeDependenciesForDfc();
#endif

            nn::ns::Finalize();
            nn::ncm::Finalize();
            nn::es::Finalize();
        }

        const char* GetNspForNandPath()
        {
            return m_NspPathForNand.c_str();
        }

        const char* GetNspForSdPath()
        {
            return m_NspPathForSd.c_str();
        }

        void DeleteTestApplicationTicket()
        {
            nn::es::RightsIdIncludingKeyId rightsId = {
                { 0x00, 0x05, 0x00, 0x0c, 0x10, 0x00, 0x10, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05 },
            };

            nn::es::DeleteTicket(&rightsId, 1);
        }

        nn::Result InstallAllTestApplications()
        {
            std::vector<nn::ncm::ApplicationContentMetaKey> installed;
            NN_RESULT_DO(m_ApplicationInstaller.InstallAll(&installed, GetNspForNandPath(), nn::ncm::StorageId::BuildInUser));
            NN_RESULT_DO(m_ApplicationInstaller.InstallAll(&installed, GetNspForSdPath(), nn::ncm::StorageId::SdCard));
            NN_RESULT_SUCCESS;
        }

        static void SetUpTestCase()
        {
            NN_ASSERT(nnt::GetHostArgc() > 2);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }


    private:
        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;
        nnt::ncmutil::SubmissionPackageFileInstaller m_SubmissionPackageInstaller;
        std::string m_NspPathForNand;
        std::string m_NspPathForSd;
    };


    const int MaxListupCount = 2048;
    const int AocTopIndex = 1;
    struct TestParameter {
        nn::ApplicationId baseId;
        int offset;
        int count;
        int expectCount;
        int expectListedCount;
        nn::aoc::AddOnContentIndex expectIndices[MaxListupCount];
        TestParameter(
            nn::ApplicationId baseId,
            int offset,
            int count,
            int expectCount,
            int expectListedCount) : baseId(baseId), offset(offset), count(count), expectCount(expectCount), expectListedCount(expectListedCount)
        {
            for (int i = 0; i < expectCount; ++i) {
                expectIndices[i] = offset + i + AocTopIndex;
            }
        }
    };

    nn::aoc::AddOnContentIndex g_ListBuffer[MaxListupCount];

    void TestListing(const TestParameter& param)
    {
        NN_LOG("[TestParameter]\n");
        NN_LOG("BaseId            : %016llx\n", param.baseId);
        NN_LOG("Count             : %0d\n", param.count);
        NN_LOG("Offset            : %0d\n", param.offset);
        NN_LOG("ExpectCount       : %0d\n", param.expectCount);
        NN_LOG("ExpectListedCount : %0d\n", param.expectListedCount);

        int count = nn::aoc::CountAddOnContent(param.baseId);
        EXPECT_EQ(param.expectCount, count);

        int listedCount = nn::aoc::ListAddOnContent(g_ListBuffer, param.offset, param.count, param.baseId);
        EXPECT_EQ(param.expectListedCount, listedCount);
        EXPECT_TRUE(std::memcmp(g_ListBuffer, param.expectIndices, sizeof(nn::aoc::AddOnContentIndex) * param.expectListedCount) == 0);

    }
}

TestParameter minimum(
    { 0x0005000c10000000ull },
    0,
    3,
    3,
    3);

TestParameter minimumOffset(
    { 0x0005000c10000000ull },
    1,
    3,
    3,
    2);

TestParameter minimumOffset2(
    { 0x0005000c10000000ull },
    1,
    1,
    3,
    1);

TestParameter overCount(
    { 0x0005000c10000000ull },
    0,
    4,
    3,
    3);
TestParameter overCountOffset(
    { 0x0005000c10000000ull },
    1,
    4,
    3,
    2);

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
TEST_F(AocListingTest, MinimumListing)
{
    TestListing(minimum);
    TestListing(minimumOffset);
    TestListing(minimumOffset2);
    TestListing(overCount);
    TestListing(overCountOffset);
}
#endif

// Windows 環境で HIPC の動作を見るケース
// NX 環境では不要
#if defined( NN_BUILD_CONFIG_OS_WIN )

namespace {
const size_t ServerStackSize = 1024 * 16;
NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ServerStack[ServerStackSize];
nn::os::ThreadType g_ServerThread;

void IpcDispatcher(void*) NN_NOEXCEPT
{
    nn::ncm::Initialize();
    nn::ns::Initialize();

    nn::aocsrv::LoopAuto();

    nn::ns::Finalize();
    nn::ncm::Finalize();
}

void InitializeService() NN_NOEXCEPT
{
    nn::aocsrv::CreateServer();
    nn::aocsrv::StartServer();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &g_ServerThread,
        IpcDispatcher,
        nullptr,
        g_ServerStack,
        ServerStackSize,
        NN_SYSTEM_THREAD_PRIORITY(aocsrv, IpcDispatcher)));
    nn::os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(aocsrv, IpcDispatcher));
    nn::os::StartThread(&g_ServerThread);
}

void FinalizeService() NN_NOEXCEPT
{
    nn::aocsrv::RequestStopServer();
    nn::os::WaitThread(&g_ServerThread);
    nn::os::DestroyThread(&g_ServerThread);
    nn::aocsrv::DestroyServer();
}
} // ~namespace <anonymous>

TEST_F(AocListingTest, MinimumListingByServer)
{
    InitializeService();

    nn::aoc::detail::InitializeForHipc();

    TestListing(minimum);
    TestListing(minimumOffset);

    nn::aoc::detail::Finalize();

    // INFO: Win版で Aoc サーバの終了処理がうまくいかないのでとりあえずパス
    FinalizeService();
}
#endif
