﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/gtest/gtest.h>
#include <nnd/bh1730fvc/bh1730fvc.h>

#include "testBh1730_Common.h"

namespace nnt { namespace bh1730fvc {

// 成功する遷移パターン
TEST(TransitionSuccessTest, State)
{
    using namespace nnd::bh1730fvc;

    Initialize();

    MeasurementMode mode;

    // Standby
    GetMeasurementMode(&mode);
    EXPECT_EQ(MeasurementMode::Standby, mode);

    // Standby -> Standby
    auto result = SetMeasurementMode(MeasurementMode::Standby);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Standby, mode);

    // Standby -> Continuous
    result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Continuous, mode);

    // Continuous -> Standby
    result = SetMeasurementMode(MeasurementMode::Standby);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Standby, mode);

    // Standby -> OneShot -> Standby (自動遷移)
    result = SetMeasurementMode(MeasurementMode::OneShot);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::OneShot, mode);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(DefaultWaitMilliSec));
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Standby, mode);

    // OneShot -> Standby の手動遷移のためにサイクル数を長めにセット
    SetIntegralCycle(255);

    // Standby -> OneShot
    result = SetMeasurementMode(MeasurementMode::OneShot);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::OneShot, mode);

    // OneShot -> Standby
    result = SetMeasurementMode(MeasurementMode::Standby);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Standby, mode);

    // Standby -> Manual
    result = SetMeasurementMode(MeasurementMode::Manual);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Manual, mode);

    // Manual -> Standby
    result = SetMeasurementMode(MeasurementMode::Standby);
    EXPECT_TRUE(result.IsSuccess());
    result = GetMeasurementMode(&mode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_EQ(MeasurementMode::Standby, mode);

    Finalize();
}

// 失敗する遷移パターン
TEST(TransitionErrorTest, State)
{
    using namespace nnd::bh1730fvc;

    Initialize();

    // Standby -> Continuous
    SetMeasurementMode(MeasurementMode::Continuous);

    // TEST: Continuous -> OneShot
    auto result = SetMeasurementMode(MeasurementMode::OneShot);
    EXPECT_TRUE(result.IsFailure());

    // TEST: Continuous -> Manual
    result = SetMeasurementMode(MeasurementMode::Manual);
    EXPECT_TRUE(result.IsFailure());

    // Continuous -> Standby
    SetMeasurementMode(MeasurementMode::Standby);

    // OneShot で自動的に Standby に遷移することを防ぐためにサイクル数を長めにセット
    SetIntegralCycle(255);

    // OneShot
    SetMeasurementMode(MeasurementMode::OneShot);

    // TEST: OneShot -> Continuous
    result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsFailure());

    // TEST: OneShot -> Manual
    result = SetMeasurementMode(MeasurementMode::Manual);
    EXPECT_TRUE(result.IsFailure());

    // OneShot -> Standby
    SetMeasurementMode(MeasurementMode::Standby);

    // Standby -> Manual
    SetMeasurementMode(MeasurementMode::Manual);

    // TEST: Manual -> Continuous
    result = SetMeasurementMode(MeasurementMode::Continuous);
    EXPECT_TRUE(result.IsFailure());

    // TEST: Manual -> OneShot
    result = SetMeasurementMode(MeasurementMode::OneShot);
    EXPECT_TRUE(result.IsFailure());

    // Continuous -> Standby
    SetMeasurementMode(MeasurementMode::Standby);

    Finalize();
}

}}
