﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album.h>
#include <nn/album/album_AlbumFileEntry.private.h>
#include <nn/album/album_AlbumFileAccess.private.h>
#include <nn/album/album_AlbumFileAccessForDebug.private.h>

#include <cstring>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/time.h>
#include <nn/capsrv/capsrv_ApplicationAlbumFileEntry.h>

#include <nnt.h>

namespace {

void PrintAlbumFileEntry(const nn::album::AlbumFileEntry* p, int index)
{
    // 作成日時の取得
    auto dateTime = nn::album::GetAlbumFileCreatedTime(p[index]);
    auto pp = reinterpret_cast<const nn::capsrv::ApplicationAlbumFileEntry*>(p + index);
    auto ar = reinterpret_cast<const uint32_t*>(p + index);
    NN_LOG("[%4d] %4d/%02d/%02d %02d:%02d:%02d(%02d) [0x%08x][0x%08x][0x%08x][0x%08x]\n", index, dateTime.year, dateTime.month, dateTime.day, dateTime.hour, dateTime.minute, dateTime.second, pp->dateTime.id, ar[0], ar[1], ar[2], ar[3]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[4], ar[5], ar[6], ar[7]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[8], ar[9], ar[10], ar[11]);
}

void DumpAlbumFileEntry(const nn::album::AlbumFileEntry* p, int index)
{
    auto ar = reinterpret_cast<const uint32_t*>(p + index);
    NN_LOG("[%4d] ----/--/-- --:--:--(--) [0x%08x][0x%08x][0x%08x][0x%08x]\n", index, ar[0], ar[1], ar[2], ar[3]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[4], ar[5], ar[6], ar[7]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[8], ar[9], ar[10], ar[11]);
}

NN_ALIGNAS(4096) uint8_t g_ScreenShotImage[ nn::album::AlbumScreenShotImageDataSize + 1 ];
NN_ALIGNAS(4096) uint8_t g_ScreenShotWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadImage ];

NN_ALIGNAS(4096) uint8_t g_ThumbnailImage[ nn::album::AlbumThumbnailImageDataSize + 1 ];
NN_ALIGNAS(4096) uint8_t g_ThumbnailWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadThumbnailImage ];

// ExtraMovie のファイルを列挙し、ファイル数と最新の entry を返す
int GetNewestScreenshotFile(nn::album::AlbumFileEntry* pOut, int* pOutIndex, bool isLogPrint)
{
    // 現在のファイルリストを取得
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(10000);

    int fileCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetAlbumScreenshotFileList(&fileCount, entryList.data(), entryList.size()));
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount);
    if (fileCount == 0)
    {
        return 0;
    }

    int newestIndex = 0;
    nn::time::CalendarTime newestTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&newestTime, nullptr, nn::time::InputPosixTimeMin));
    for (int index=0; index<fileCount; ++index)
    {
        if (isLogPrint)
        {
            PrintAlbumFileEntry(entryList.data(), index);
        }
        auto dateTime = nn::album::GetAlbumFileCreatedTime(entryList[index]);
        if (dateTime > newestTime)
        {
            newestTime  = dateTime;
            newestIndex = index;
        }
    }

    *pOut = entryList[newestIndex];
    *pOutIndex = newestIndex;
    return fileCount;
}

void CreateScreenShotFile()
{
    auto p = reinterpret_cast<uint32_t*>(g_ScreenShotImage);
    for (int y=0; y<720; ++y)
    {
        for (int x=0; x<1280; ++x)
        {
            uint32_t r = 255 * y / 720;
            uint32_t g = 255 * x / 1280;
            p[y * 1280 + x] = 0xff000000 | (g << 8) | (r);
        }
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );
}

}   // namespace


// Scoop の静止画リスト読込み
TEST(AlbumFileAccessPrivate, ScoopScreenshotApi)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    // album の初期化
    nn::album::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::album::Finalize(); };

    // 現在のファイルリストを取得
    nn::album::AlbumFileEntry entry = {};
    int newestIndex = 0;
    int fileCount1 = GetNewestScreenshotFile(&entry, &newestIndex, true);
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount1);
    NN_UNUSED(entry);
    NN_UNUSED(newestIndex);

    // 静止画ファイルを作成
    CreateScreenShotFile();

    // 再度ファイルリストを取得（１ファイル増えている）
    // 最も新しいもののインデックスも取得
    int fileCount2 = GetNewestScreenshotFile(&entry, &newestIndex, true);
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount2);
    ASSERT_EQ(fileCount1 + 1, fileCount2);

    // 最新ファイルの静止画を読み込む（画像バッファ大きめ）
    {
        const auto imageBufferSize = nn::album::AlbumScreenShotImageDataSize + 1;
        NN_STATIC_ASSERT(sizeof(g_ScreenShotImage) >= imageBufferSize);
        std::memset(g_ScreenShotImage, 0xff, imageBufferSize);

        int width = 0;
        int height = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumImage(&width, &height, g_ScreenShotImage, imageBufferSize, g_ScreenShotWorkBuffer, sizeof(g_ScreenShotWorkBuffer), entry) );
        ASSERT_EQ(1280, width);
        ASSERT_EQ(720, height);
        ASSERT_EQ(0, g_ScreenShotImage[imageBufferSize - 1]);
    }

    // 最新ファイルの静止画を読み込む（成功）
    int width = 0;
    int height = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumImage(&width, &height, g_ScreenShotImage, nn::album::AlbumScreenShotImageDataSize, g_ScreenShotWorkBuffer, sizeof(g_ScreenShotWorkBuffer), entry) );
    ASSERT_EQ(1280, width);
    ASSERT_EQ(720, height);

    // 最新ファイルのサムネイル画像を読み込む（画像バッファ大きめ）
    {
        const auto imageBufferSize = nn::album::AlbumThumbnailImageDataSize + 1;
        NN_STATIC_ASSERT(sizeof(g_ThumbnailImage) >= imageBufferSize);
        std::memset(g_ThumbnailImage, 0xff, imageBufferSize);

        int thumbnailWidth  = 0;
        int thumbnailHeight = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumThumbnailImage(&thumbnailWidth, &thumbnailHeight, g_ThumbnailImage, imageBufferSize, g_ThumbnailWorkBuffer, sizeof(g_ThumbnailWorkBuffer), entry) );
        ASSERT_EQ(320, thumbnailWidth);
        ASSERT_EQ(180, thumbnailHeight);
        ASSERT_EQ(0, g_ThumbnailImage[imageBufferSize - 1]);
    }

    // 最新ファイルのサムネイル画像を読み込む（成功）
    int thumbnailWidth  = 0;
    int thumbnailHeight = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumThumbnailImage(&thumbnailWidth, &thumbnailHeight, g_ThumbnailImage, nn::album::AlbumThumbnailImageDataSize, g_ThumbnailWorkBuffer, sizeof(g_ThumbnailWorkBuffer), entry) );
    ASSERT_EQ(320, thumbnailWidth);
    ASSERT_EQ(180, thumbnailHeight);

    // 静止画の上にサムネイルを重ねる
    for (int y=0; y<thumbnailHeight; ++y)
    {
        std::memcpy(g_ScreenShotImage + y * width * sizeof(uint32_t), g_ThumbnailImage + y * thumbnailWidth * sizeof(uint32_t), thumbnailWidth * sizeof(uint32_t));
    }

    // 合成した画像を書き出す
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );

    // 再度ファイルリストを取得（さらに１ファイル増えている）
    int fileCount3 = GetNewestScreenshotFile(&entry, &newestIndex, true);
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount3);
    ASSERT_EQ(fileCount2 + 1, fileCount3);

    // スクショ画像を削除
    NN_LOG("nn::album::DeleteAlbumFileForDebug()\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteAlbumFileForDebug(entry));

    // 再度ファイルリストを取得
    int fileCount4 = GetNewestScreenshotFile(&entry, &newestIndex, true);
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount3);
    ASSERT_EQ(fileCount2, fileCount4);

    // スクショ画像を削除
    NN_LOG("nn::album::DeleteAlbumFileForDebug()\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteAlbumFileForDebug(entry));

}   // NOLINT(impl/function_size)

// Scoop の静止画リスト読込み（オーバーランチェック）
TEST(AlbumFileAccessPrivate, CheckOverrun)
{
    // album の初期化
    nn::album::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::album::Finalize(); };

    // テスト前のファイル数をカウントしておく
    nn::album::AlbumFileEntry entry = {};
    int newestIndex = 0;
    int count1 = GetNewestScreenshotFile(&entry, &newestIndex, false);

    // 現在のファイルリストを取得（とりあえず 100 個）
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(100);

    // 先頭から２つ分のエントリを 0x55 で埋める
    std::memset(entryList.data(), 0x55, sizeof(nn::album::AlbumFileEntry) * 2);

    // スクショ画像を１つ作成
    NN_LOG("nn::album::SaveScreenshot() for dummy\n");
    std::memset(g_ScreenShotImage, 0, sizeof(g_ScreenShotImage));
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );

    // 長さ 1 でリストを取得
    int fileCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetAlbumScreenshotFileList(&fileCount, entryList.data(), 1));
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount);

    // 必ず１つは存在する
    ASSERT_EQ(1, fileCount);

    // 先頭 2 エントリ分のダンプ
    PrintAlbumFileEntry(entryList.data(), 0);
    DumpAlbumFileEntry(entryList.data(), 1);

    // 先頭から２つ目のエントリが 0x55 で埋まっているかをチェック
    auto p = reinterpret_cast<uint8_t*>(entryList.data() + 1);
    for (int i=0; i<sizeof(nn::album::AlbumFileEntry); ++i)
    {
        ASSERT_EQ(0x55, p[i]);
    }

    // スクショ画像を削除
    NN_LOG("nn::album::DeleteAlbumFileForDebug()\n");
    int count2 = GetNewestScreenshotFile(&entry, &newestIndex, false);
    ASSERT_EQ(count1 + 1, count2);
    ASSERT_TRUE(count2 >= 1);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteAlbumFileForDebug(entry));

    int count3 = GetNewestScreenshotFile(&entry, &newestIndex, false);
    ASSERT_EQ(count1, count3);
}

// Scoop の動画リスト読込み
TEST(AlbumFileAccessPrivate, ScoopMovieApi)
{
    // album の初期化
    nn::album::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::album::Finalize(); };

    // 現在のファイルリストを取得（とりあえず 1000 個）
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(1000);
    int fileCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetAlbumMovieFileList(&fileCount, entryList.data(), entryList.size()));
    NN_LOG("nn::album::GetAlbumMovieFileList()       count=%d\n", fileCount);
    for (int i=0; i<fileCount; ++i)
    {
        PrintAlbumFileEntry(entryList.data(), i);
    }

    {
        uint64_t tmp;
        asm volatile (" mrs %0, CTR_EL0" : "=r"(tmp) ::);
        NN_LOG("CTR_EL0=0x%08x\n", static_cast<uint32_t>(tmp));
    }
}

