﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAlbum_SceneUtility.h"

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nvn/nvn_FuncPtrInline.h>

namespace nnt{ namespace album{ namespace scene{

    void ClearTexture(NVNcommandBuffer* pCommandBuffer, NVNtexture* pTexture, const nn::util::Color4f& color) NN_NOEXCEPT
    {
        NVNcopyRegion clearRegion;
        clearRegion.xoffset = 0;
        clearRegion.yoffset = 0;
        clearRegion.zoffset = 0;
        clearRegion.width  = nvnTextureGetWidth(pTexture);
        clearRegion.height = nvnTextureGetHeight(pTexture);
        clearRegion.depth  = 1;
        float colorArray[4] = {};
        nn::util::VectorStore(colorArray, color);
        nvnCommandBufferClearTexture(pCommandBuffer, pTexture, nullptr, &clearRegion, colorArray, NVN_CLEAR_COLOR_MASK_RGBA);
    }

    //-----------------------------------------------------

    void CopyTextureParameter::SetPreset() NN_NOEXCEPT
    {
        const int IntMax = std::numeric_limits<int>::max();
        this->dstX = 0;
        this->dstY = 0;
        this->dstWidth = IntMax;
        this->dstHeight = IntMax;
        this->pDstView = nullptr;
        this->srcX = 0;
        this->srcY = 0;
        this->srcWidth = IntMax;
        this->srcHeight = IntMax;
        this->pSrcView = nullptr;
    }

    void CopyTexture(NVNcommandBuffer* pCommandBuffer, NVNtexture* pDstTexture, NVNtexture* pSrcTexture, const CopyTextureParameter& param) NN_NOEXCEPT
    {
        NVNcopyRegion srcRegion;
        srcRegion.xoffset = std::max(0, param.srcX);
        srcRegion.yoffset = std::max(0, param.srcY);
        srcRegion.zoffset = 0;
        srcRegion.width   = std::max(0, std::min(param.srcWidth, nvnTextureGetWidth(pSrcTexture)));
        srcRegion.height  = std::max(0, std::min(param.srcHeight, nvnTextureGetHeight(pSrcTexture)));
        srcRegion.depth   = 1;
        NVNcopyRegion dstRegion;
        dstRegion.xoffset = std::max(0, param.dstX);
        dstRegion.yoffset = std::max(0, param.dstY);
        dstRegion.zoffset = 0;
        dstRegion.width   = std::max(0, std::min(param.dstWidth,  nvnTextureGetWidth(pDstTexture)));
        dstRegion.height  = std::max(0, std::min(param.dstHeight, nvnTextureGetHeight(pDstTexture)));
        dstRegion.depth   = 1;
        int flag = NVN_COPY_FLAGS_NONE;
        nvnCommandBufferCopyTextureToTexture(pCommandBuffer, pSrcTexture, param.pSrcView, &srcRegion, pDstTexture, param.pDstView, &dstRegion, flag);
    }

    //-----------------------------------------------------

    CopyBufferToTextureParameter CopyBufferToTextureParameter::GetPreset() NN_NOEXCEPT
    {
        CopyBufferToTextureParameter param;
        param.SetPreset();
        return param;
    }

    void CopyBufferToTextureParameter::SetPreset() NN_NOEXCEPT
    {
        const int IntMax = std::numeric_limits<int>::max();
        this->dstX = 0;
        this->dstY = 0;
        this->dstWidth = IntMax;
        this->dstHeight = IntMax;
        this->pDstView = nullptr;
    }

    void CopyBufferToTexture(NVNcommandBuffer* pCommandBuffer, NVNtexture* pDstTexture, NVNbuffer* pSrcBuffer, ptrdiff_t srcOffset, size_t srcSize, const CopyBufferToTextureParameter& param) NN_NOEXCEPT
    {
        NN_UNUSED(srcSize);
        NVNbufferAddress srcAddress = nvnBufferGetAddress(pSrcBuffer) + srcOffset;
        NVNcopyRegion dstRegion;
        dstRegion.xoffset = std::max(0, param.dstX);
        dstRegion.yoffset = std::max(0, param.dstY);
        dstRegion.zoffset = 0;
        dstRegion.width   = std::max(0, std::min(param.dstWidth,  nvnTextureGetWidth(pDstTexture)));
        dstRegion.height  = std::max(0, std::min(param.dstHeight, nvnTextureGetHeight(pDstTexture)));
        dstRegion.depth   = 1;
        int flag = NVN_COPY_FLAGS_NONE;
        nvnCommandBufferCopyBufferToTexture(pCommandBuffer, srcAddress, pDstTexture, param.pDstView, &dstRegion, flag);
    }

    //-----------------------------------------------------

    void RotationBoxParameter::SetPreset() NN_NOEXCEPT
    {
        this->boxSize           = 50;
        this->boxColor          = nn::util::Color4f(1, 1, 1, 1);
        this->trajectoryCenterX = 1280 / 2;
        this->trajectoryCenterY = 720 / 2;
        this->trajectoryRadius  = 300;
        this->angularVelocity   = 2 * M_PI / 5;
        this->initialAngle      = 0;
    }

    void DrawRotationBox(NVNcommandBuffer* pCommandBuffer, NVNtexture* pTexture, NVNtextureView* pTexView, const RotationBoxParameter& param, float time) NN_NOEXCEPT
    {
        float angle = param.angularVelocity * time + param.initialAngle;
        float x = param.trajectoryRadius * std::cos(angle) + param.trajectoryCenterX;
        float y = param.trajectoryRadius * std::sin(angle) + param.trajectoryCenterY;
        float edge = param.boxSize;
        NVNcopyRegion clearRegion;
        clearRegion.xoffset = x - edge * 0.5f;
        clearRegion.yoffset = y - edge * 0.5f;
        clearRegion.zoffset = 0;
        clearRegion.width  = edge;
        clearRegion.height = edge;
        clearRegion.depth  = 1;
        float color[4] = { param.boxColor.GetR(), param.boxColor.GetG(), param.boxColor.GetB(), param.boxColor.GetA() };
        nvnCommandBufferClearTexture(pCommandBuffer, pTexture, pTexView, &clearRegion, color, NVN_CLEAR_COLOR_MASK_RGBA);
    }

    //-----------------------------------------------------

    void DrawRotationBoxScene(NVNcommandBuffer* pCommandBuffer, NVNtexture* pTexture, NVNtextureView* pTexView, float time) NN_NOEXCEPT
    {
        RotationBoxParameter boxParam;
        boxParam.SetPreset();
        // clear
        {
            float angle = boxParam.angularVelocity * time + boxParam.initialAngle;
            float r = 0.5f * std::cos(angle               ) + 0.5f;
            float g = 0.5f * std::cos(angle + 3.14f       ) + 0.5f;
            float b = 0.5f * std::cos(angle + 3.14f * 0.5f) + 0.5f;
            ClearTexture(pCommandBuffer, pTexture, nn::util::Color4f(r, g, b, 1));
        }
        DrawRotationBox(pCommandBuffer, pTexture, pTexView, boxParam, time);
    }

}}}
