﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album.h>
#include <nn/album/album_AlbumFileAccess.private.h>
#include <nn/album/album_ExtraMediaFileAccess.private.h>
#include <nn/album/album_MovieMaker.h>
#include <nn/capsrv/capsrv_ApplicationAlbumFileEntry.h>

#include <mutex>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/vi.h>

#include <nv/nv_MemoryManagement.h>
#include "testAlbum_NvnUtility.h"
#include "testAlbum_SceneUtility.h"

#include <nnt.h>

namespace {

    // オーディオバッファ
    const int AudioSampleRate = 48000;
    const int AudioChannelCount = 2;
    const int AudioBufferLength = AudioChannelCount * AudioSampleRate / 30;
    NN_ALIGNAS(4096) int16_t g_AudioBuffer[AudioBufferLength];

    NN_ALIGNAS(4096) nn::Bit8 g_DonateMemory[16 * 1024 * 1024];
    NN_ALIGNAS(4096) nn::Bit8 g_MovieWorkMemory[128 * 1024 * 1024];
    NN_ALIGNAS(4096) nn::Bit8 g_MovieReadBuffer[128 * 1024 * 1024];
    NN_ALIGNAS(4096) nn::Bit32 g_MovieThumbnailBuffer[1280 * 720];

    const size_t ControlMemorySize     = 4 * 4096;
    const size_t CommandPoolMemorySize = 10 * 4096;
    const size_t FramePoolMemorySize   = 8 * 1024 * 1024;

    void SetupAlbumUserData(char* pOutUserData, size_t userDataSize)
    {
        NN_ASSERT(userDataSize <= nn::album::AlbumUserDataSizeMax);
        for (int i=0; i<userDataSize; ++i)
        {
            pOutUserData[i] = 0x80 + (i & 0x7f);
        }
    }

    bool VerifyAlbumUserData(const char* pUserData, size_t userDataSize)
    {
        NN_ASSERT(userDataSize <= nn::album::AlbumUserDataSizeMax);

        NN_LOG("VerifyAlbumUserData() userData={%02x,%02x,%02x,%02x, ... ,%02x,%02x,%02x,%02x}\n", pUserData[0], pUserData[1], pUserData[2], pUserData[3], pUserData[userDataSize - 4], pUserData[userDataSize - 3], pUserData[userDataSize - 2], pUserData[userDataSize - 1]);
        for (int i=0; i<userDataSize; ++i)
        {
            if (pUserData[i] != 0x80 + (i & 0x7f))
            {
                return false;
            }
        }
        return true;
    }

}

void PrintAlbumFileEntry(const nn::album::AlbumFileEntry* p, int index)
{
    // 作成日時の取得
    auto dateTime = nn::album::GetAlbumFileCreatedTime(p[index]);
    auto pp = reinterpret_cast<const nn::capsrv::ApplicationAlbumFileEntry*>(p + index);
    auto ar = reinterpret_cast<const uint32_t*>(p + index);
    NN_LOG("[%4d] %4d/%02d/%02d %02d:%02d:%02d(%02d) [0x%08x][0x%08x][0x%08x][0x%08x]\n", index, dateTime.year, dateTime.month, dateTime.day, dateTime.hour, dateTime.minute, dateTime.second, pp->dateTime.id, ar[0], ar[1], ar[2], ar[3]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[4], ar[5], ar[6], ar[7]);
    NN_LOG("                               [0x%08x][0x%08x][0x%08x][0x%08x]\n", ar[8], ar[9], ar[10], ar[11]);
}

// ExtraMovie のファイルを列挙し、ファイル数と最新の entry を返す
int GetNewestExtraMovieFile(nn::album::AlbumFileEntry* pOut, int* pOutIndex, bool isLogPrint)
{
    // 現在のファイルリストを取得
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(10000);

    int fileCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetExtraMovieFileList(&fileCount, entryList.data(), entryList.size()));
    NN_LOG("nn::album::GetExtraMovieFileList()       count=%d\n", fileCount);
    if (fileCount == 0)
    {
        return 0;
    }

    // 降順ソートされていることもチェックする
    nn::time::CalendarTime prevTime = { 2100, 1, 1, 0, 0, 0};

    int newestIndex = 0;
    nn::time::CalendarTime newestTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&newestTime, nullptr, nn::time::InputPosixTimeMin));
    for (int index=0; index<fileCount; ++index)
    {
        if (isLogPrint)
        {
            PrintAlbumFileEntry(entryList.data(), index);
        }
        auto dateTime = nn::album::GetAlbumFileCreatedTime(entryList[index]);
        if (dateTime > newestTime)
        {
            newestTime  = dateTime;
            newestIndex = index;
        }

        // 降順ソートされていることをチェック
        NN_ABORT_UNLESS(dateTime <= prevTime);
        prevTime = dateTime;
    }

    *pOut = entryList[newestIndex];
    *pOutIndex = newestIndex;

    // 降順ソートされている場合は必ず先頭に最新日付のものがくる
    NN_ABORT_UNLESS(newestIndex == 0);

    return fileCount;
}

nn::os::Mutex g_Mutex(false);
bool          g_IsGraphicsInitialized = false;

void InitializeEnvironment()
{
    {
        std::lock_guard<nn::os::Mutex> lock(g_Mutex);
        if (g_IsGraphicsInitialized)
        {
            return;
        }
        g_IsGraphicsInitialized = true;
    }

    // nv 関連の初期化
    nv::InitializeGraphics(g_DonateMemory, sizeof(g_DonateMemory));

    nv::SetGraphicsAllocator(
        [] (size_t size, size_t alignment, void*) -> void*
        {
            return std::aligned_alloc(alignment, size);
        },
        [] (void* p, void*)
        {
            std::free(p);
        },
        [] (void* p, size_t size, void*)
        {
            return std::realloc(p, size);
        },
        nullptr
    );

    // 各種ライブラリの初期化
    nn::album::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
}

NN_ALIGNAS(4096) uint8_t g_ScreenShotImage[ nn::album::AlbumScreenShotImageDataSize ];
NN_ALIGNAS(4096) uint8_t g_ScreenShotWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadImage ];

NN_ALIGNAS(4096) uint8_t g_ThumbnailImage[ nn::album::AlbumThumbnailImageDataSize ];
NN_ALIGNAS(4096) uint8_t g_ThumbnailWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadThumbnailImage ];

void InitializeMovieMakerForTest()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::InitializeMovieMaker(g_MovieWorkMemory, sizeof(g_MovieWorkMemory)));
    nn::vi::Initialize();
}

void FinalizeMovieMakerForTest()
{
    nn::vi::Finalize();
    nn::album::FinalizeMovieMaker();
}

const int ExtraMovieFrameCountOne     = 1;
const int ExtraMovieFrameCountNormal  = 30 * 5;
const int ExtraMovieFrameCountTooLong = 10000000; // 2GiB を超えるようにする

nn::Result CreateExtraMovie(const char* pUserData, size_t userDataSize, int frameLimit)
{
    // 録画用レイヤを作成
    nn::vi::NativeWindowHandle hVideoNativeWindow;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetMovieMakerNativeWindow(&hVideoNativeWindow));

    // nvn 初期化
    auto pControlMemory     = aligned_alloc(4096, ControlMemorySize);
    auto pCommandPoolMemory = aligned_alloc(4096, CommandPoolMemorySize);
    auto pFramePoolMemory   = aligned_alloc(4096, FramePoolMemorySize);
    NN_ABORT_UNLESS_NOT_NULL(pControlMemory);
    NN_ABORT_UNLESS_NOT_NULL(pCommandPoolMemory);
    NN_ABORT_UNLESS_NOT_NULL(pFramePoolMemory);
    NN_UTIL_SCOPE_EXIT{
        free(pControlMemory);
        free(pCommandPoolMemory);
        free(pFramePoolMemory);
    };

    NVNdevice device;
    ::nnt::album::NvnUtility::InitializeNvnAndDevice(device);
    NN_UTIL_SCOPE_EXIT{ nvnDeviceFinalize(&device); };

    NVNmemoryPool commandPool;
    ::nnt::album::NvnUtility::InitializeCommandMemoryPool(commandPool, device, pCommandPoolMemory, CommandPoolMemorySize);
    NN_UTIL_SCOPE_EXIT{ nvnMemoryPoolFinalize(&commandPool); };

    NVNqueue queue;
    ::nnt::album::NvnUtility::InitializeQueue(queue, device);
    NN_UTIL_SCOPE_EXIT{ nvnQueueFinalize(&queue); };

    NVNcommandBuffer commandBuffer;
    ::nnt::album::NvnUtility::InitializeCommandBuffer(commandBuffer, device);
    NN_UTIL_SCOPE_EXIT { nvnCommandBufferFinalize(&commandBuffer); };

    NVNmemoryPool framePool;
    ::nnt::album::NvnUtility::InitializeTextureMemoryPool(framePool, device, pFramePoolMemory, FramePoolMemorySize);
    NN_UTIL_SCOPE_EXIT { nvnMemoryPoolFinalize(&framePool); };

    NVNtexture frameTexs[2];
    ::nnt::album::NvnUtility::InitializeFrameBufferTexture(frameTexs, 2, 1280, 720, device, framePool, 0, FramePoolMemorySize);
    NN_UTIL_SCOPE_EXIT
    {
        for (auto& e : frameTexs)
        {
            nvnTextureFinalize(&e);
        }
    };

    NVNwindow videoWindow;
    NVNtexture* videoWindowTextureList[2];
    for(int i = 0; i < 2; i++)
    {
        videoWindowTextureList[i] = &frameTexs[i];
    };
    ::nnt::album::NvnUtility::InitializeWindow(videoWindow, videoWindowTextureList, 2, hVideoNativeWindow, device);
    NN_UTIL_SCOPE_EXIT { nvnWindowFinalize(&videoWindow); };

    NVNtextureView texView;
    ::nnt::album::NvnUtility::InitializeTextureViewForFrameBuffer(texView);

    // 録画用関数定義
    auto frameFunction = [&](int frame) -> nn::Result
    {
        NVNsync acqSync;
        nvnSyncInitialize(&acqSync, &device);
        NN_UTIL_SCOPE_EXIT{ nvnSyncFinalize(&acqSync); };

        int texIdx = -1;
        NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&videoWindow, &acqSync, &texIdx), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);
        if (frameLimit == ExtraMovieFrameCountNormal)
        {
            NN_LOG("frame-no:%d (tex=%d)\n", frame, texIdx);
        }
        else if (frameLimit == ExtraMovieFrameCountTooLong)
        {
            if (frame % (30 * 30) == 0)
            {
                NN_LOG("frame-no:%d (tex=%d) [%d(sec)]\n", frame, texIdx, frame / 30);
            }
        }

        nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(CommandPoolMemorySize));
        nvnCommandBufferAddControlMemory(&commandBuffer, pControlMemory, ControlMemorySize);
        nvnCommandBufferBeginRecording(&commandBuffer);

        // wait display
        nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

        // ここで描画コマンドを積んでもよい
        auto pTargetTexture = videoWindowTextureList[texIdx];
        {
            nnt::album::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

            // フレーム数を描画
            {
                char buf[256];
                nn::util::SNPrintf(buf, sizeof(buf), "F%08d", frame + 1);
                nnt::album::scene::DrawStringParameter param = {};
                param.color = nn::util::Color4f(1, 1, 1);
                param.size  = 32;
                param.posX  = 0;
                param.posY  = 0;
                nnt::album::scene::DrawString(&commandBuffer, pTargetTexture, &texView, buf, param);
            }
        }

        auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
        nvnQueueSubmitCommands(&queue, 1, &hCommand);
        nvnQueuePresentTexture(&queue, &videoWindow, texIdx);
        nvnQueueFinish(&queue);

        {
            auto result = nn::album::CheckMovieMakerError();
            if (!result.IsSuccess())
            {
                NN_LOG("nn::album::CheckMovieMakerError() result=0x%08x\n", result.GetInnerValueForDebug());
                nn::album::AbortMovieMaker();
                NN_RESULT_DO(result);
            }
        }

        // オーディオ（無音）
        nn::album::EncodeMovieMakerAudioSample(g_AudioBuffer, sizeof(g_AudioBuffer));
        NN_RESULT_SUCCESS;
    };

    // MovieMaker の開始
    nn::album::MovieMakerMovieParameter mmParam = nn::album::MovieMakerMovieParameter::GetDefaultValue();
    mmParam.SetVideoFrameRate(30);
    {
        auto result = nn::album::PrecheckToStartMovieMaker(20ull * 1024 * 1024);
        if (!result.IsSuccess())
        {
            NN_LOG("nn::album::PrecheckToStartMovieMaker() result=0x%08x\n", result.GetInnerValueForDebug());
            NN_RESULT_DO(result);
        }
    }
    {
        auto result = nn::album::StartMovieMaker(mmParam);
        if (!result.IsSuccess())
        {
            NN_LOG("nn::album::StartMovieMaker() result=0x%08x\n", result.GetInnerValueForDebug());
            NN_RESULT_DO(result);
        }
    }

    EXPECT_TRUE(nn::album::IsMovieMakerInitialized());
    EXPECT_TRUE(nn::album::IsMovieMakerRunning());

    // 5 秒間の黒画面の動画を作成
    for (int frame=0; frame<frameLimit; frame++)
    {
        NN_RESULT_DO( frameFunction(frame) );
    }

    // 後処理
    {
        if (!pUserData)
        {
            nn::album::FinishMovieMaker();
        }
        else
        {
            for (int y=0; y<720; ++y)
            {
                for (int x=0; x<1280; ++x)
                {
                    uint32_t r = (x / 75) * 15;
                    uint32_t g = (y / 42) * 15;
                    g_MovieThumbnailBuffer[y * 1280 + x] = 0xff000000 | (g << 8) | r;
                }
            }
            nn::album::FinishMovieMaker(pUserData, userDataSize, g_MovieThumbnailBuffer, sizeof(g_MovieThumbnailBuffer), nn::album::ImageSize_1280x720);
        }
        EXPECT_TRUE(nn::album::IsMovieMakerInitialized());
        EXPECT_FALSE(nn::album::IsMovieMakerRunning());
    };

    NN_RESULT_SUCCESS;

}   // NOLINT(impl/function_size)

void DeleteAllExtraMovieFiles()
{
    // 現在のファイルリストを取得して全削除
    int fileCount1 = 0;
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(20000);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetExtraMovieFileList(&fileCount1, entryList.data(), entryList.size()));
    if (fileCount1 == 0)
    {
        NN_LOG("DeleteAllExtraMovieFiles(): Extra movie files not found.\n");
        return;
    }

    NN_LOG("DeleteAllExtraMovieFiles(): Begin: file count = %d\n", fileCount1);
    for (int i=0; i<fileCount1; ++i)
    {
        NN_LOG(".");
        // PrintAlbumFileEntry(entryList.data(), i);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteExtraMovieFile(entryList[i]));
    }
    NN_LOG("\n");

    // 現在のファイルリストを再取得して残ファイルがないことを確認
    int fileCount2 = -1;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetExtraMovieFileList(&fileCount2, entryList.data(), entryList.size()));
    ASSERT_EQ(0, fileCount2);
    entryList.clear();
    NN_LOG("DeleteAllExtraMovieFiles(): End   file count = %d\n", fileCount2);
}

//----------------------------------------------------------------------------

// 動画ファイルのストリーム読込み
void DoTestReadExtraMovie()
{
    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

    // 動画ファイルの作成
    {
        char userData[nn::album::AlbumUserDataSizeMax] = {};
        SetupAlbumUserData( userData, sizeof(userData) );
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateExtraMovie(userData, sizeof(userData), ExtraMovieFrameCountNormal));
    }

    // 現在のファイルリストと最新 entry を取得
    nn::album::AlbumFileEntry targetEntry = {};
    int targetIndex;
    auto fileCount = GetNewestExtraMovieFile(&targetEntry, &targetIndex, true);
    NN_LOG("nn::album::GetExtraMovieFileList()       count=%d\n", fileCount);
    NN_ABORT_UNLESS(fileCount > 0);

    // 日付が最も新しいファイルを読む
    // 読込みストリームのオープン
    nn::album::MovieStreamHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::OpenAlbumMovieStream(&handle, targetEntry) );
    NN_LOG("nn::album::OpenAlbumMovieStream() handle=0x%016llx\n", handle.value);

    // 読込みストリームオープン中の画像読込み API の動作確認
    {
        // 最新ファイルの静止画を読み込む
        int width = 0;
        int height = 0;
        char userData[nn::album::AlbumUserDataSizeMax] = {};
        size_t userDataSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumImage(&width, &height, &userDataSize, &userData, sizeof(userData), g_ScreenShotImage, sizeof(g_ScreenShotImage), g_ScreenShotWorkBuffer, sizeof(g_ScreenShotWorkBuffer), targetEntry) );
        ASSERT_EQ(1280, width);
        ASSERT_EQ(720, height);
        ASSERT_EQ(sizeof(userData), userDataSize);
        ASSERT_TRUE( VerifyAlbumUserData(userData, sizeof(userData)) );

        // 読み込んだ静止画をスクショ画像として出力
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );
    }
    {
        // 最新ファイルのサムネイル画像を読み込む
        int thumbnailWidth  = 0;
        int thumbnailHeight = 0;
        char userData[nn::album::AlbumUserDataSizeMax] = {};
        size_t userDataSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumThumbnailImage(&thumbnailWidth, &thumbnailHeight, &userDataSize, &userData, sizeof(userData), g_ThumbnailImage, sizeof(g_ThumbnailImage), g_ThumbnailWorkBuffer, sizeof(g_ThumbnailWorkBuffer), targetEntry) );
        ASSERT_EQ(320, thumbnailWidth);
        ASSERT_EQ(180, thumbnailHeight);
        ASSERT_EQ(sizeof(userData), userDataSize);
        ASSERT_TRUE( VerifyAlbumUserData(userData, sizeof(userData)) );

        // 読み込んだサムネイル画像を x3 してスクショ画像として出力
        std::memset(g_ScreenShotImage, 0, sizeof(g_ScreenShotImage));
        for (int y=0; y<thumbnailHeight; ++y)
        {
            auto* pSrc = reinterpret_cast<uint32_t*>(g_ThumbnailImage) + y * 320;
            auto* pDst = reinterpret_cast<uint32_t*>(g_ScreenShotImage) + (90 + y * 3) * 1280 + 160;
            for (int x=0; x<thumbnailWidth; ++x)
            {
                auto c = pSrc[x];
                pDst[x * 3 + 0] = c;
                pDst[x * 3 + 1] = c;
                pDst[x * 3 + 2] = c;
            }
            std::memcpy(pDst + 1280 * 4, pDst, 320 * 3 * 4);
            std::memcpy(pDst + 2560 * 4, pDst, 320 * 3 * 4);
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );
    }

    // 読込みストリームの動画データの読込み
    NN_STATIC_ASSERT(sizeof(g_MovieReadBuffer) % nn::album::MovieFileDataUnitSize == 0);
    {
        uint64_t movieSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::GetAlbumMovieStreamSize(&movieSize, handle) );
        NN_LOG("nn::album::GetAlbumMovieStreamSize() movieSize=0x%016llx\n", movieSize);
        NN_ABORT_UNLESS(movieSize > 0);

        uint64_t alreadReadSize = 0;
        uint64_t offset = 0;
        while (alreadReadSize < movieSize)
        {
            size_t readMovieSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::ReadAlbumMovieStream(&readMovieSize, g_MovieReadBuffer, sizeof(g_MovieReadBuffer), handle, offset) );
            NN_LOG("nn::album::ReadAlbumMovieStream() offset=0x%016llx readSize=0x%016llx\n", offset, readMovieSize, offset);
            offset += readMovieSize;
            alreadReadSize += readMovieSize;
        }
    }

    // 読込みストリームをクローズ
    nn::album::CloseAlbumMovieStream(handle);
    NN_LOG("nn::album::CloseAlbumMovieStream()\n");

    // 作成した動画ファイルを削除する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteExtraMovieFile(targetEntry));
    NN_LOG("nn::album::DeleteExtraMovieFile()        index=%d\n", targetIndex);

}   // NOLINT(impl/function_size)

//----------------------------------------------------------------------------

// 動画ファイルのファイルリスト取得
void DoTestGetExtraMovieFileList()
{
    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

    // 現在のファイルリストを取得
    nn::album::AlbumFileEntry targetEntry = {};
    int targetIndex;
    auto fileCount1 = GetNewestExtraMovieFile(&targetEntry, &targetIndex, true);
    NN_LOG("nn::album::GetExtraMovieFileList()       count=%d\n", fileCount1);
    NN_UNUSED(targetEntry);
    NN_UNUSED(targetIndex);

    // 動画ファイルの作成
    {
        char userData[nn::album::AlbumUserDataSizeMax] = {};
        SetupAlbumUserData( userData, sizeof(userData) );
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateExtraMovie(userData, sizeof(userData), ExtraMovieFrameCountNormal));
    }

    // 現在のファイルリストと最新 entry を取得
    int fileCount2 = GetNewestExtraMovieFile(&targetEntry, &targetIndex, true);
    NN_LOG("nn::album::GetExtraMovieFileList()       count=%d\n", fileCount2);
    ASSERT_EQ(fileCount1 + 1, fileCount2);

    // 最新ファイルの静止画を読み込む
    int width = 0;
    int height = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumImage(&width, &height, g_ScreenShotImage, sizeof(g_ScreenShotImage), g_ScreenShotWorkBuffer, sizeof(g_ScreenShotWorkBuffer), targetEntry) );
    ASSERT_EQ(1280, width);
    ASSERT_EQ(720, height);

    // 最新ファイルのサムネイル画像を読み込む
    int thumbnailWidth  = 0;
    int thumbnailHeight = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::LoadAlbumThumbnailImage(&thumbnailWidth, &thumbnailHeight, g_ThumbnailImage, sizeof(g_ThumbnailImage), g_ThumbnailWorkBuffer, sizeof(g_ThumbnailWorkBuffer), targetEntry) );
    ASSERT_EQ(320, thumbnailWidth);
    ASSERT_EQ(180, thumbnailHeight);

    // 静止画の上にサムネイルを重ねる
    for (int y=0; y<thumbnailHeight; ++y)
    {
        std::memcpy(g_ScreenShotImage + y * width * 4, g_ThumbnailImage + y * thumbnailWidth * 4, thumbnailWidth * 4);
    }

    // 合成した画像を書き出す
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::album::SaveScreenshot(g_ScreenShotImage, sizeof(g_ScreenShotImage), nn::album::ImageSize_1280x720, nn::album::AlbumReportOption_ReportAlways) );

    // 作成した動画ファイルを削除する
    NN_LOG("nn::album::DeleteExtraMovieFile()        index=%d\n", targetIndex);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::DeleteExtraMovieFile(targetEntry));

    // 最後にもう一度ファイルリストを取得（１ファイル減っている）
    int fileCount3 = GetNewestExtraMovieFile(&targetEntry, &targetIndex, true);
    NN_LOG("nn::album::GetAlbumScreenshotFileList()       count=%d\n", fileCount3);
    ASSERT_EQ(fileCount2 - 1, fileCount3);

}   // NOLINT(impl/function_size)

//----------------------------------------------------------------------------

void DoTestCheckExtraMovieFileCountLimit()
{
    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

    // Extra 動画を 1000 個作成
    const int extraMovieCreateCount = 1000;
    for (int i=0; i<extraMovieCreateCount; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateExtraMovie(nullptr, 0, ExtraMovieFrameCountOne));
        NN_LOG("created extra movie file: %d\n", i + 1);
    }

    // Extra 動画の数をカウント
    {
        nn::album::AlbumFileEntry targetEntry = {};
        int targetIndex;
        auto count = GetNewestExtraMovieFile(&targetEntry, &targetIndex, false);
        ASSERT_EQ(extraMovieCreateCount, count);
    }

    // さらにもう１つ Extra 動画を作成しようとしてみる
    auto result = CreateExtraMovie(nullptr, 0, ExtraMovieFrameCountOne);
    if (!(result <= nn::album::ResultAlbumFileCountLimit()))
    {
        NN_LOG("CreateExtraMovie(): result=0x%08x\n", result.GetInnerValueForDebug());
        NN_ABORT();
    }

    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

}   // NOLINT(impl/function_size)

//----------------------------------------------------------------------------

void DoTestExtraMovieSizeLimit()
{
    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

    // 2GiB を超える動画を作成してみる
    auto result = CreateExtraMovie(nullptr, 0, ExtraMovieFrameCountTooLong);
    if (!(result <= nn::album::ResultAlbumFileSizeLimit()))
    {
        NN_LOG("CreateExtraMovie(): result=0x%08x\n", result.GetInnerValueForDebug());
        NN_ABORT();
    }

    // Extra 動画を全削除
    DeleteAllExtraMovieFiles();

}   // NOLINT(impl/function_size)

//----------------------------------------------------------------------------

// SIGLO-82864: いきなり動画を作成する
// album::PrecheckToStartMovieMaker() がアボートする不具合の確認用
TEST(AlbumExtraAccess, CreateMovieMaker)
{
    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    InitializeEnvironment();
    InitializeMovieMakerForTest();

    // 動画ファイルの作成
    {
        char userData[nn::album::AlbumUserDataSizeMax] = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateExtraMovie(userData, sizeof(userData), ExtraMovieFrameCountNormal));
    }

    FinalizeMovieMakerForTest();

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());
}

// 動画ファイルのファイルリスト取得
TEST(AlbumExtraAccess, GetExtraMovieFileList)
{
    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    InitializeEnvironment();
    InitializeMovieMakerForTest();
    DoTestGetExtraMovieFileList();
    FinalizeMovieMakerForTest();

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());
}

// 動画ファイルのストリーム読込み
TEST(AlbumExtraAccess, ReadExtraMovie)
{
    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    InitializeEnvironment();
    InitializeMovieMakerForTest();
    DoTestReadExtraMovie();
    FinalizeMovieMakerForTest();

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());
}

#if 0
// 2GiB の長時間動画作成には時間がかかるので一旦無効にしておく。
// ここを有効にしてビルドすればテストを実行できます。

// 動画ファイル作成時の SizeLimit のテスト
TEST(AlbumExtraAccess, ReadExtraMovieSizeLimit)
{
    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    InitializeEnvironment();
    InitializeMovieMakerForTest();
    DoTestExtraMovieSizeLimit();
    FinalizeMovieMakerForTest();

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());
}
#endif

#if 0
// 1000 回の MovieMaker 実行は時間がかかるので一旦無効にしておく。
// ここを有効にしてビルドすればテストを実行できます。

// アプリ独自の長時間動画ファイルのファイル数上限チェック
TEST(AlbumExtraAccess, CheckExtraMovieFileCountLimit)
{
    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    InitializeEnvironment();
    InitializeMovieMakerForTest();
    DoTestCheckExtraMovieFileCountLimit();
    FinalizeMovieMakerForTest();

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());
}
#endif
