﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/ahid/hdr/hdr.h>
#include <nn/ahid/ahid.h>
#include "../../../../../Programs/Eris/Sources/Libraries/ahid/hdr/libnn_hdrServer/ahid_Hdr.h"
#include "../../../../../Programs/Eris/Sources/Libraries/cdhid/libnn_cdhidWin32Keyboard/cdhidWin32Keyboard_Main.h"

//////////////////////////////////////////////////////////////////////////////
static void printString(uint8_t *p)
{
    int len = *p++;

    if (*p++ != nn::ahid::hdr::AhidDescriptorTypeString)
    {
        NN_SDK_LOG("invalid string\n");
        return;
    }

    len -= 2;

    while (len)
    {
        NN_SDK_LOG("%c", *p++);
        p++;
        len -= 2;
    }

    NN_SDK_LOG("\n");
}



//////////////////////////////////////////////////////////////////////////////
static void testInitialize(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters* pDeviceParameters)
{
    nn::Result result;

    NN_SDK_LOG("calling Initialize\n");
    result = pAhid->Initialize(pDeviceParameters);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static nn::ahid::CodeBookHeader *pCodeBookHeader;
static nn::ahid::Item           *pItemLControl;
static nn::ahid::Item           *pItemLShift;
static nn::ahid::Item           *pItemRControl;
static nn::ahid::Item           *pItemRShift;
static nn::ahid::Item           *pItemNumLock;
static nn::ahid::Item           *pItemCapsLock;
static nn::ahid::Item           *pItemScrollLock;
static nn::ahid::Item           *pItemKana;
static nn::ahid::Item           *pItemScanCode[6];


static void testCodeBook(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling GetCodeBookHeader\n");
    result = pAhid->GetCodeBookHeader(&pCodeBookHeader);
    NN_SDK_LOG("result %d\n", result);

    //                  store item          reportId    usagePage   usageMinMax index
    pAhid->GetInputItem(&pItemLControl,     0,          0x07,       0xe0, 0xe0, 0);
    pAhid->GetInputItem(&pItemLShift,       0,          0x07,       0xe1, 0xe1, 0);
    pAhid->GetInputItem(&pItemRControl,     0,          0x07,       0xe4, 0xe4, 0);
    pAhid->GetInputItem(&pItemRShift,       0,          0x07,       0xe5, 0xe5, 0);
    pAhid->GetInputItem(&pItemNumLock,      0,          0x08,       0x01, 0x01, 0);
    pAhid->GetInputItem(&pItemCapsLock,     0,          0x08,       0x02, 0x02, 0);
    pAhid->GetInputItem(&pItemScrollLock,   0,          0x08,       0x03, 0x03, 0);
    pAhid->GetInputItem(&pItemKana,         0,          0x08,       0x05, 0x05, 0);
    pAhid->GetInputItem(&pItemScanCode[0],  0,          0x07,       0x00, 0x65, 0);
    pAhid->GetInputItem(&pItemScanCode[1],  0,          0x07,       0x00, 0x65, 1);
    pAhid->GetInputItem(&pItemScanCode[2],  0,          0x07,       0x00, 0x65, 2);
    pAhid->GetInputItem(&pItemScanCode[3],  0,          0x07,       0x00, 0x65, 3);
    pAhid->GetInputItem(&pItemScanCode[4],  0,          0x07,       0x00, 0x65, 4);
    pAhid->GetInputItem(&pItemScanCode[5],  0,          0x07,       0x00, 0x65, 5);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetString(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling GetString\n");
    result = pAhid->GetString(data, 64, 1);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetIdle\n");
    result = pAhid->SetIdle(0, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetIdle(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t idle;

    NN_SDK_LOG("calling GetIdle\n");
    result = pAhid->GetIdle(&idle, 0);
    NN_SDK_LOG("result %d idle %d\n", result, idle);
}


//////////////////////////////////////////////////////////////////////////////
static void testSetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling SetProtocol\n");
    result = pAhid->SetProtocol(0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetProtocol(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t protocol;

    NN_SDK_LOG("calling GetProtocol\n");
    result = pAhid->GetProtocol(&protocol);
    NN_SDK_LOG("result %d protocol %d\n", result, protocol);
}


//////////////////////////////////////////////////////////////////////////////
static void testGetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling GetReport\n");
    result = pAhid->GetReport(data, 64, 0x01, 0);
    NN_SDK_LOG("result %d\n", result);

    if (result.IsSuccess())
    {
        NN_SDK_LOG("Report:\n");

        for (int i = 0; i < 63; i++)
        {
            NN_SDK_LOG("%02x ", data[i]);
        }

        NN_SDK_LOG("\n");
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testSetReport(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[64];

    NN_SDK_LOG("calling SetReport\n");
    result = pAhid->SetReport(data, 64, 0x02, 0);
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void testRead(nn::ahid::Ahid *pAhid)
{
    int         iteration = 0;
    uint8_t     data[64];
    uint32_t    bytesRead;
    nn::Result  result;

    NN_SDK_LOG("type on keyboard, program will run for 100 inputs\n");

    while (iteration++ < 100)
    {
        result = pAhid->Read(&bytesRead, data, pCodeBookHeader->inputSize);

        if (result.IsSuccess())
        {
            if (bytesRead)
            {
                NN_SDK_LOG("%4d read data: ", iteration);

                for (uint32_t i = 0; i < bytesRead; i++)
                {
                    NN_SDK_LOG("%02x ", data[i]);
                }

                int32_t lControl, lShift, rControl, rShift,
                        numLockLed, capsLockLed, scrollLockLed, kanaLed,
                        scanCode[6];

                //            value             input   item
                pAhid->Decode(&lControl,        data,   pItemLControl);
                pAhid->Decode(&lShift,          data,   pItemLShift);
                pAhid->Decode(&rControl,        data,   pItemRControl);
                pAhid->Decode(&rShift,          data,   pItemRShift);
                pAhid->Decode(&numLockLed,      data,   pItemNumLock);
                pAhid->Decode(&capsLockLed,     data,   pItemCapsLock);
                pAhid->Decode(&scrollLockLed,   data,   pItemScrollLock);
                pAhid->Decode(&kanaLed,         data,   pItemKana);
                pAhid->Decode(&scanCode[0],     data,   pItemScanCode[0]);
                pAhid->Decode(&scanCode[1],     data,   pItemScanCode[1]);
                pAhid->Decode(&scanCode[2],     data,   pItemScanCode[2]);
                pAhid->Decode(&scanCode[3],     data,   pItemScanCode[3]);
                pAhid->Decode(&scanCode[4],     data,   pItemScanCode[4]);
                pAhid->Decode(&scanCode[5],     data,   pItemScanCode[5]);

                if (lControl) {NN_SDK_LOG("L Control ");}
                if (lShift) {NN_SDK_LOG("L Shift ");}
                if (rControl) {NN_SDK_LOG("R Control ");}
                if (rShift) {NN_SDK_LOG("R Shift ");}

                if (numLockLed) {NN_SDK_LOG("Num Lock ");}
                if (capsLockLed) {NN_SDK_LOG("Caps Lock ");}
                if (scrollLockLed) {NN_SDK_LOG("Scroll Lock ");}
                if (kanaLed) {NN_SDK_LOG("Kana ");}

                NN_SDK_LOG("scan code: ");

                if (scanCode[0]) {NN_SDK_LOG("%02x ", scanCode[0]);}
                if (scanCode[1]) {NN_SDK_LOG("%02x ", scanCode[1]);}
                if (scanCode[2]) {NN_SDK_LOG("%02x ", scanCode[2]);}
                if (scanCode[3]) {NN_SDK_LOG("%02x ", scanCode[3]);}
                if (scanCode[4]) {NN_SDK_LOG("%02x ", scanCode[4]);}
                if (scanCode[5]) {NN_SDK_LOG("%02x ", scanCode[5]);}

                NN_SDK_LOG("\n");
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
static void testWrite(nn::ahid::Ahid *pAhid)
{
    nn::Result result;
    uint8_t data[4];
    uint32_t bytesWritten;

    NN_SDK_LOG("calling Write\n");
    result = pAhid->Write(&bytesWritten, data, 4);
    NN_SDK_LOG("result %d bytesWritten %d\n", result, bytesWritten);
}


//////////////////////////////////////////////////////////////////////////////
static void testFinalize(nn::ahid::Ahid *pAhid)
{
    nn::Result result;

    NN_SDK_LOG("calling Finalize\n");
    result = pAhid->Finalize();
    NN_SDK_LOG("result %d\n", result);
}


//////////////////////////////////////////////////////////////////////////////
static void clientFunction()
{
    bool run = true;

    nn::ahid::hdr::Hdr  hdr;
    nn::ahid::Ahid      ahid; // use one client per device

    NN_SDK_LOG("->%s\n", __FUNCTION__);

    hdr.Initialize();

    nn::ahid::hdr::AttachFilter attachFilter;

    memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

    // Client queries HDR for attached devices
    while (run)
    {
        size_t outEntries, inEntries;

        nn::Result result = hdr.GetDeviceEntries(&outEntries);

        if (result.IsSuccess() && outEntries)
        {
            nn::ahid::hdr::DeviceHandle deviceHandle[nn::ahid::hdr::AhidDevicesCountMax];

            inEntries = nn::ahid::hdr::AhidDevicesCountMax;

            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            attachFilter.attachedAfterTicks = nn::os::GetSystemTick().GetInt64Value();

            if (result.IsSuccess() && outEntries)
            {
                for (size_t j = 0; j < outEntries; j++)
                {
                    nn::ahid::hdr::DeviceParameters deviceParameters;

                    result = hdr.GetDeviceParameters(&deviceParameters, deviceHandle[j]);

                    if (result.IsSuccess())
                    {
                        //if (strcmp((const char*)deviceParameters.servicePath, "kbd") == 0)

                        // look for usagePage: Generic Desktop usageId: Mouse
                        if ((deviceParameters.usagePage == 0x01) && (deviceParameters.usageId == 0x06))
                        {
                            NN_SDK_LOG("DeviceHandle:       %08x\n",    deviceHandle[j]);
                            NN_SDK_LOG("    busId:          %d\n",      deviceParameters.busId);
                            NN_SDK_LOG("    servicePath:    %s\n",      deviceParameters.servicePath);
                            NN_SDK_LOG("    vendorId:       %08x\n",    deviceParameters.vendorId);
                            NN_SDK_LOG("    productId:      %08x\n",    deviceParameters.productId);
                            NN_SDK_LOG("    versionNumber:  %08x\n",    deviceParameters.versionNumber);
                            NN_SDK_LOG("    manufacturer:   ");
                            printString(deviceParameters.manufacturer);
                            NN_SDK_LOG("    product:        ");
                            printString(deviceParameters.product);
                            NN_SDK_LOG("    serialNumber:   ");
                            printString(deviceParameters.serialNumber);

                            testInitialize(&ahid, &deviceParameters);
                            testCodeBook(&ahid);
                            testGetString(&ahid);
                            testSetProtocol(&ahid);
                            testGetProtocol(&ahid);
                            testSetIdle(&ahid);
                            testGetIdle(&ahid);
                            testSetReport(&ahid);
                            testGetReport(&ahid);
                            testRead(&ahid);
                            testWrite(&ahid);
                            testFinalize(&ahid);

                            run = false;
                        }
                    }
                }
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    hdr.Finalize();

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
}


//////////////////////////////////////////////////////////////////////////////
extern "C" void nnMain()
{
    NN_SDK_LOG("HdrTest built %s %s\n", __DATE__, __TIME__);

    nn::ahid::hdr::HdrInitialize();                 // HDR server
    nn::cdhid::win32::Win32KeyboardInitialize();    // Win32 AHID Keyboard driver

    clientFunction();

    nn::cdhid::win32::Win32KeyboardFinalize();
    nn::ahid::hdr::HdrFinalize();

    NN_SDK_LOG("End of HdrTest\n");
}
