﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/ahid/hdr/hdr.h>
#include <nn/ahid/ahid.h>

//////////////////////////////////////////////////////////////////////////////
/*

    HID "Usage Page" and "Usage" are defined by the USB org. The documentation
    can be found at http://www.usb.org/developers/hidpage/Hut1_12v2.pdf.

    The following symbols are created for items this program looks for.
 */

enum UsagePageId
{
    GenericDesktop  = 0x0001,
    KeyboardKeypad  = 0x0007,
    LED             = 0x0008,
    Button          = 0x0009,
};

enum GenericDesktopPageUsageId
{
    Pointer         = 0x0001,
    Mouse           = 0x0002,
    Joystick        = 0x0004,
    Gamepad         = 0x0005,
    Keyboard        = 0x0006,

    X               = 0x0030,
    Y               = 0x0031,
    Z               = 0x0032,
    Rz              = 0x0035,
    Slider          = 0x0036,
    Wheel           = 0x0038,
    HatSwitch       = 0x0039,
};

enum KeyboardUsagePageId
{
    NoEvent         = 0x0000,
    Equals          = 0x0086,
    LeftControl     = 0x00e0,
    LeftShift       = 0x00e1,
    LeftAlt         = 0x00e2,
    LeftGui         = 0x00e3,
    RightControl    = 0x00e4,
    RightShift      = 0x00e5,
    RightAlt        = 0x00e6,
    RightGui        = 0x00e7,
};


enum LedUsagePageId
{
    NumLock         = 0x0001,
    CapsLock        = 0x0002,
    ScrollLock      = 0x0003,
};


enum ButtonUsagePageId
{
    Button1         = 0x0001,
    Button2         = 0x0002,
    Button3         = 0x0003,
    Button4         = 0x0004,
    Button5         = 0x0005,
    Button6         = 0x0006,
    Button7         = 0x0007,
    Button8         = 0x0008,
    Button9         = 0x0009,
    Button10        = 0x000a,
    Button11        = 0x000b,
    Button12        = 0x000c,
};


//////////////////////////////////////////////////////////////////////////////
static void printString(uint8_t *p)
{
    int len = *p++;

    if (*p++ != nn::ahid::hdr::AhidDescriptorTypeString)
    {
        NN_SDK_LOG("invalid string\n");
        return;
    }

    len -= 2;

    while (len)
    {
        NN_SDK_LOG("%c", *p++);
        p++;
        len -= 2;
    }

    NN_SDK_LOG("\n");
}


//////////////////////////////////////////////////////////////////////////////
static void setProtocolToReport(nn::ahid::Ahid *pAhid)
{
    for (int i = 0; i < 10; i++)
    {
        nn::Result result = pAhid->SetProtocol(1);

        NN_SDK_LOG("SetProtocol result module %d description %d\n", result.GetModule(), result.GetDescription());

        if (result.IsSuccess())
        {
            uint8_t protocol;

            result = pAhid->GetProtocol(&protocol);

            NN_SDK_LOG("GetProtocol result module %d descriptopn %d protocol %d\n", result.GetModule(), result.GetDescription(), protocol);

            if (protocol == 1)
            {
                return;
            }
        }
    }

    NN_SDK_LOG("SetProtocol failed after 10 retries\n");
}


//////////////////////////////////////////////////////////////////////////////
static void setIdleToZero(nn::ahid::Ahid *pAhid)
{
    for (int i = 0; i < 10; i++)
    {
        nn::Result result = pAhid->SetIdle(0, 0);

        NN_SDK_LOG("SetIdle result module %d description %d\n", result.GetModule(), result.GetDescription());

        if (result.IsSuccess())
        {
            uint8_t idle;

            result = pAhid->GetIdle(&idle, 0);

            NN_SDK_LOG("GetIdle result module %d descriptopn %d idle %d\n", result.GetModule(), result.GetDescription(), idle);

            if (idle == 0)
            {
                return;
            }
        }
    }

    NN_SDK_LOG("SetIdle failed after 10 retries\n");
}


//////////////////////////////////////////////////////////////////////////////
static void readJoystickDevice(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters *pDeviceParameters)
{
    NN_SDK_LOG("->%s\n", __FUNCTION__);

    nn::Result result;

    // Ahid::Initialize() will attempt to own the device associated with the device parameters.
    // The device may have been acquired by another user, in this case it will fail.
    result = pAhid->Initialize(pDeviceParameters);

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Ahid::Initialize() returned module %d description %d\n", result.GetModule(), result.GetDescription());
    }

    if (result.IsSuccess())
    {
        nn::ahid::CodeBookHeader *pCodeBookHeader;
        nn::ahid::Item           *pItemButton[12];
        nn::ahid::Item           *pItemX;
        nn::ahid::Item           *pItemY;
        nn::ahid::Item           *pItemZ;
        nn::ahid::Item           *pItemRZ;
        nn::ahid::Item           *pHatSwitch;
        nn::ahid::Item           *pSlider;

        // Get state change event
        nn::os::SystemEvent stateChangeEvent;
        result = pAhid->CreateStateChangeEvent(stateChangeEvent.GetBase(), nn::os::EventClearMode_AutoClear);

        // Set protocol to report
        setProtocolToReport(pAhid);

        // Set idle to 0, HID devices input a frame of data per idle (ms), when set to 0 it only delivers a frame of data when there is input.
        setIdleToZero(pAhid);

        // Get the CodeBook header. This is needed to know read, write, feature buffer size.
        result = pAhid->GetCodeBookHeader(&pCodeBookHeader);

        if (result.IsSuccess())
        {
            bool read = true;

            /*
                Here we try to get input itmes for controls this application cares about.

                - Not all joysticks will have these items (controls).
                - If the item does not exist in the device the call will fail and the pItem will be set to nullptr.
                - The program can check the return code here or just try to decode the pItem, it does not matter.
                - Beware some devices report items for controls that do not exist on the hardware.

                - If the control does not have an usage range then the usage min and usage max should be set to same.
                - Index is supported where these is an array of controls with the same usage id.
             */

            // Find report ID for the gamepad / joystic (some mice have more than one report ID for non-existing functions)
            uint8_t reportId = 0;
            uint8_t reportCount;

            result = pAhid->GetReportCount(&reportCount);

            if (result.IsSuccess()) // GetReportCount always returns success
            {
                for (int i = 0; i < reportCount; i++)
                {
                    uint8_t usagePage, usage;
                    result = pAhid->GetReportIdUsage(&reportId, &usagePage, &usage, i);

                    if (result.IsSuccess())
                    {
                        NN_SDK_LOG("reportId %02x usageTable %02x usage %02x\n", reportId, usagePage, usage);

                        if (usagePage == UsagePageId::GenericDesktop)
                        {
                            // Accept pointer or mouse
                            if ((usage == GenericDesktopPageUsageId::Joystick) || (usage == GenericDesktopPageUsageId::Gamepad))
                            {
                                break;
                            }
                        }
                    }

                    // incase we don't find it then at least we'd try reportId 0
                    reportId = 0;
                }
            }

            //                      store item          report id   usage page                      usage min                               usage max                               index
            pAhid->GetInputItem(    &pItemButton[0],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button1,             ButtonUsagePageId::Button1,             0);
            pAhid->GetInputItem(    &pItemButton[1],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button2,             ButtonUsagePageId::Button2,             0);
            pAhid->GetInputItem(    &pItemButton[2],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button3,             ButtonUsagePageId::Button3,             0);
            pAhid->GetInputItem(    &pItemButton[3],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button4,             ButtonUsagePageId::Button4,             0);
            pAhid->GetInputItem(    &pItemButton[4],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button5,             ButtonUsagePageId::Button5,             0);
            pAhid->GetInputItem(    &pItemButton[5],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button6,             ButtonUsagePageId::Button6,             0);
            pAhid->GetInputItem(    &pItemButton[6],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button7,             ButtonUsagePageId::Button7,             0);
            pAhid->GetInputItem(    &pItemButton[7],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button8,             ButtonUsagePageId::Button8,             0);
            pAhid->GetInputItem(    &pItemButton[8],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button9,             ButtonUsagePageId::Button9,             0);
            pAhid->GetInputItem(    &pItemButton[9],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button10,            ButtonUsagePageId::Button10,            0);
            pAhid->GetInputItem(    &pItemButton[10],   reportId,   UsagePageId::Button,            ButtonUsagePageId::Button11,            ButtonUsagePageId::Button11,            0);
            pAhid->GetInputItem(    &pItemButton[11],   reportId,   UsagePageId::Button,            ButtonUsagePageId::Button12,            ButtonUsagePageId::Button12,            0);
            pAhid->GetInputItem(    &pItemX,            reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::X,           GenericDesktopPageUsageId::X,           0);
            pAhid->GetInputItem(    &pItemY,            reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Y,           GenericDesktopPageUsageId::Y,           0);
            pAhid->GetInputItem(    &pItemZ,            reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Z,           GenericDesktopPageUsageId::Z,           0);
            pAhid->GetInputItem(    &pItemRZ,           reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Rz,          GenericDesktopPageUsageId::Rz,          0);
            pAhid->GetInputItem(    &pSlider,           reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Slider,      GenericDesktopPageUsageId::Slider,      0);
            pAhid->GetInputItem(    &pHatSwitch,        reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::HatSwitch,   GenericDesktopPageUsageId::HatSwitch,   0);

            // If we find a hat switch we can store some runtime units here for special handling
            int32_t hatSwitchMinimum, hatSwitchMaximum;
            uint32_t hatSwitchUnit, hatSwitchExponent;

            if (pHatSwitch)
            {
                // GetUnit returns min, max, unit and exponent for the item. We will use this to deal with hat switch.
                if (pAhid->GetUnit(&hatSwitchMinimum, &hatSwitchMaximum, &hatSwitchUnit, &hatSwitchExponent, pHatSwitch).IsSuccess())
                {
                    NN_SDK_LOG("hat switch unit of measure %08x exponent %08x\n", hatSwitchUnit, hatSwitchExponent);
                }
            }

            // read loop, this will exit when the joystick is detached or there is an error
            while (read)
            {
                // We need a read buffer of pCodeBookHeader->inputSize to perform the read
                uint8_t data[pCodeBookHeader->inputSize];
                uint32_t bytesRead;

                // Read returns when the hardware delivers a "frame" of data, for HID devices the frequency is per SetIdle()
                result = pAhid->Read(&bytesRead, data, pCodeBookHeader->inputSize);

                if (result.IsSuccess())
                {
                    if (bytesRead)
                    {
                        // We will try to decode for 12 buttons, it doens't matter if the joystick doesn't have that many.
                        int32_t button[12], x, y, z, rz, hatSwitch, slider;

                        //            value         input   item
                        pAhid->Decode(&button[0],   data,   pItemButton[0]);
                        pAhid->Decode(&button[1],   data,   pItemButton[1]);
                        pAhid->Decode(&button[2],   data,   pItemButton[2]);
                        pAhid->Decode(&button[3],   data,   pItemButton[3]);
                        pAhid->Decode(&button[4],   data,   pItemButton[4]);
                        pAhid->Decode(&button[5],   data,   pItemButton[5]);
                        pAhid->Decode(&button[6],   data,   pItemButton[6]);
                        pAhid->Decode(&button[7],   data,   pItemButton[7]);
                        pAhid->Decode(&button[8],   data,   pItemButton[8]);
                        pAhid->Decode(&button[9],   data,   pItemButton[9]);
                        pAhid->Decode(&button[10],  data,   pItemButton[10]);
                        pAhid->Decode(&button[11],  data,   pItemButton[11]);
                        pAhid->Decode(&x,           data,   pItemX);
                        pAhid->Decode(&y,           data,   pItemY);
                        pAhid->Decode(&z,           data,   pItemZ);
                        pAhid->Decode(&rz,          data,   pItemRZ);
                        pAhid->Decode(&slider,      data,   pSlider);

                        NN_SDK_LOG("button: %d%d%d%d%d%d%d%d%d%d%d%d x: %d y: %d z: %d rz: %d slider: %d",
                            button[0], button[1], button[2], button[3], button[4], button[5], button[6], button[7], button[8], button[9], button[10], button[11], x, y, z, rz, slider
                            );

                        // We can also selectively decode what we have items for, in this case the hatSwitch needs some special attention.
                        // This same logic can be applied to all the items but for common items there is no need to.
                        if (pHatSwitch)
                        {
                            pAhid->Decode(&hatSwitch, data, pHatSwitch);

                            // If value is in range then it is valid... this is true on some joysticks.
                            // On others it may give valid number when the switch is not engaged :0
                            // Application has to decide how it wants to deal with these items.
                            if ((hatSwitch >= hatSwitchMinimum) && (hatSwitch <= hatSwitchMaximum))
                            {
                                NN_SDK_LOG(" hatSwitch: %d", hatSwitch);
                            }
                        }

                        NN_SDK_LOG("\n");
                    }
                }
                else
                {
                    read = false;

                    // Wait for detach event
                    stateChangeEvent.Wait();
                    pAhid->DestroyStateChangeEvent(stateChangeEvent.GetBase());
                }
            }
        }

        // Release the ownership of the device
        pAhid->Finalize();
    }

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
} // NOLINT(impl/function_size)


//////////////////////////////////////////////////////////////////////////////
static void readMouseDevice(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters *pDeviceParameters)
{
    NN_SDK_LOG("->%s\n", __FUNCTION__);

    nn::Result result;

    // Ahid::Initialize() will attempt to own the device associated with the device parameters.
    // The device may have been acquired by another user, in this case it will fail.
    result = pAhid->Initialize(pDeviceParameters);

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Ahid::Initialize() returned module %d description %d\n", result.GetModule(), result.GetDescription());
    }
    if (result.IsSuccess())
    {
        nn::ahid::CodeBookHeader *pCodeBookHeader;
        nn::ahid::Item           *pItemButton[3];
        nn::ahid::Item           *pItemX;
        nn::ahid::Item           *pItemY;
        nn::ahid::Item           *pItemWheel;

        // Get state change event
        nn::os::SystemEvent stateChangeEvent;
        result = pAhid->CreateStateChangeEvent(stateChangeEvent.GetBase(), nn::os::EventClearMode_AutoClear);

        // Set protocol to report
        setProtocolToReport(pAhid);

        // Set idle to 0, HID devices input a frame of data per idle (ms), when set to 0 it only delivers a frame of data when there is input.
        setIdleToZero(pAhid);

        // Get the CodeBook header. This is needed to know read, write, feature buffer size.
        result = pAhid->GetCodeBookHeader(&pCodeBookHeader);

        if (result.IsSuccess())
        {
            bool read = true;

            /*
                Here we try to get input itmes for controls this application cares about.

                - Not all mice will have these items (controls).
                - If the item does not exist in the device the call will fail and the pItem will be set to nullptr.
                - The program can check the return code here or just try to decode the pItem, it does not matter.
                - Beware some devices report items for controls that do not exist on the hardware.

                - If the control does not have an usage range then the usage min and usage max should be set to same.
                - Index is supported where these is an array of controls with the same usage id.

                - Mice can also be set to "boot protocol" if one wants to just cast the input data to known usage.
                - In this case there would be no need to use the CodeBook, GetInputItem, Decdode.
                - Just read data and deal with known data usage.
             */

            // Find report ID for the mouse (some mice have more than one report ID for non-existing functions)
            uint8_t reportId = 0;
            uint8_t reportCount;

            result = pAhid->GetReportCount(&reportCount);

            if (result.IsSuccess()) // GetReportCount always returns success
            {
                for (int i = 0; i < reportCount; i++)
                {
                    uint8_t usagePage, usage;
                    result = pAhid->GetReportIdUsage(&reportId, &usagePage, &usage, i);

                    if (result.IsSuccess())
                    {
                        NN_SDK_LOG("reportId %02x usageTable %02x usage %02x\n", reportId, usagePage, usage);

                        if (usagePage == UsagePageId::GenericDesktop)
                        {
                            // Accept pointer or mouse
                            if ((usage == GenericDesktopPageUsageId::Pointer) || (usage == GenericDesktopPageUsageId::Mouse))
                            {
                                break;
                            }
                        }
                    }

                    // incase we don't find it then at least we'd try reportId 0
                    reportId = 0;
                }
            }

            //                      store item          report id   usage page                      usage min                               usage max                               index
            pAhid->GetInputItem(    &pItemButton[0],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button1,             ButtonUsagePageId::Button1,             0);
            pAhid->GetInputItem(    &pItemButton[1],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button2,             ButtonUsagePageId::Button2,             0);
            pAhid->GetInputItem(    &pItemButton[2],    reportId,   UsagePageId::Button,            ButtonUsagePageId::Button3,             ButtonUsagePageId::Button3,             0);
            pAhid->GetInputItem(    &pItemX,            reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::X,           GenericDesktopPageUsageId::X,           0);
            pAhid->GetInputItem(    &pItemY,            reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Y,           GenericDesktopPageUsageId::Y,           0);
            pAhid->GetInputItem(    &pItemWheel,        reportId,   UsagePageId::GenericDesktop,    GenericDesktopPageUsageId::Wheel,       GenericDesktopPageUsageId::Wheel,       0);

            // read loop, this will exit when the mouse is detached or there is an error
            while (read)
            {
                // We need a read buffer of pCodeBookHeader->inputSize to perform the read
                uint8_t data[pCodeBookHeader->inputSize];
                uint32_t bytesRead;

                // Read returns when the hardware delivers a "frame" of data, for HID devices the frequency is per SetIdle()
                result = pAhid->Read(&bytesRead, data, pCodeBookHeader->inputSize);

                if (result.IsSuccess())
                {
                    if (bytesRead)
                    {
                        // We will try to decode for 3 buttons, it doens't matter if the mouse doesn't have that many.
                        int32_t button[3], x, y, wheel;

                        //            value         input   item
                        pAhid->Decode(&button[0],   data,   pItemButton[0]);
                        pAhid->Decode(&button[1],   data,   pItemButton[1]);
                        pAhid->Decode(&button[2],   data,   pItemButton[2]);
                        pAhid->Decode(&x,           data,   pItemX);
                        pAhid->Decode(&y,           data,   pItemY);
                        pAhid->Decode(&wheel,       data,   pItemWheel);

                        NN_SDK_LOG(
                                "buttons: %d %d %d x: %d y: %d wheel: %d\n",
                                button[0],
                                button[1],
                                button[2],
                                x,
                                y,
                                wheel
                                );
                    }
                }
                else
                {
                    read = false;

                    // Wait for detach event
                    stateChangeEvent.Wait();
                    pAhid->DestroyStateChangeEvent(stateChangeEvent.GetBase());
                }
            }
        }

        // Release the ownership of the device
        pAhid->Finalize();
    }

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
}


//////////////////////////////////////////////////////////////////////////////
static void readKeyboardDevice(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters *pDeviceParameters)
{
    NN_SDK_LOG("->%s\n", __FUNCTION__);

    nn::Result result;

    // Ahid::Initialize() will attempt to own the device associated with the device parameters.
    // The device may have been acquired by another user, in this case it will fail.
    result = pAhid->Initialize(pDeviceParameters);

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Ahid::Initialize() returned module %d description %d\n", result.GetModule(), result.GetDescription());
    }

    if (result.IsSuccess())
    {
        nn::ahid::CodeBookHeader *pCodeBookHeader;
        nn::ahid::Item           *pItemLControl;
        nn::ahid::Item           *pItemLShift;
        nn::ahid::Item           *pItemLAlt;
        nn::ahid::Item           *pItemLGui;
        nn::ahid::Item           *pItemRControl;
        nn::ahid::Item           *pItemRShift;
        nn::ahid::Item           *pItemRAlt;
        nn::ahid::Item           *pItemRGui;
        nn::ahid::Item           *pItemScanCode[6];
        nn::ahid::Item           *pItemNumLockLed;
        nn::ahid::Item           *pItemCapsLockLed;
        nn::ahid::Item           *pItemScrollLockLed;

        // Get state change event
        nn::os::SystemEvent stateChangeEvent;
        pAhid->CreateStateChangeEvent(stateChangeEvent.GetBase(), nn::os::EventClearMode_AutoClear);

        // Set protocol to report
        setProtocolToReport(pAhid);

        // Set idle to 0, HID devices input a frame of data per idle (ms), when set to 0 it only delivers a frame of data when there is input.
        setIdleToZero(pAhid);

        // Get the CodeBook header. This is needed to know read, write, feature buffer size.
        result = pAhid->GetCodeBookHeader(&pCodeBookHeader);

        if (result.IsSuccess())
        {
            bool read = true;

            /*
                Here we try to get input itmes for controls this application cares about.

                - Not all keyboards will have these items (controls).
                - If the item does not exist in the device the call will fail and the pItem will be set to nullptr.
                - The program can check the return code here or just try to decode the pItem, it does not matter.
                - Beware some devices report items for controls that do not exist on the hardware.

                - If the control does not have an usage range then the usage min and usage max should be set to same.
                - Index is supported where these is an array of controls with the same usage id.

                - Keyboards can also be set to "boot protocol" if one wants to just cast the input data to known usage.
                - In this case there would be no need to use the CodeBook, GetInputItem, Decdode.
                - Just read data and deal with known data usage.
             */

            //                      store item              reportId    usagePage                       usage Min                           usage Max                           index
            pAhid->GetInputItem(    &pItemLControl,         0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::LeftControl,   KeyboardUsagePageId::LeftControl,   0);
            pAhid->GetInputItem(    &pItemLShift,           0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::LeftShift,     KeyboardUsagePageId::LeftShift,     0);
            pAhid->GetInputItem(    &pItemLAlt,             0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::LeftAlt,       KeyboardUsagePageId::LeftAlt,       0);
            pAhid->GetInputItem(    &pItemLGui,             0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::LeftGui,       KeyboardUsagePageId::LeftGui,       0);
            pAhid->GetInputItem(    &pItemRControl,         0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::RightControl,  KeyboardUsagePageId::RightControl,  0);
            pAhid->GetInputItem(    &pItemRShift,           0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::RightShift,    KeyboardUsagePageId::RightShift,    0);
            pAhid->GetInputItem(    &pItemRAlt,             0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::RightAlt,      KeyboardUsagePageId::RightAlt,      0);
            pAhid->GetInputItem(    &pItemRGui,             0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::RightGui,      KeyboardUsagePageId::RightGui,      0);
            pAhid->GetInputItem(    &pItemScanCode[0],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        0);
            pAhid->GetInputItem(    &pItemScanCode[1],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        1);
            pAhid->GetInputItem(    &pItemScanCode[2],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        2);
            pAhid->GetInputItem(    &pItemScanCode[3],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        3);
            pAhid->GetInputItem(    &pItemScanCode[4],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        4);
            pAhid->GetInputItem(    &pItemScanCode[5],      0,          UsagePageId::KeyboardKeypad,    KeyboardUsagePageId::NoEvent,       KeyboardUsagePageId::Equals,        5);

            // keyboard drivers have to set LEDs using SetReport
            pAhid->GetOutputItem(   &pItemNumLockLed,       0,          UsagePageId::LED,               LedUsagePageId::NumLock,            LedUsagePageId::NumLock,            0);
            pAhid->GetOutputItem(   &pItemCapsLockLed,      0,          UsagePageId::LED,               LedUsagePageId::CapsLock,           LedUsagePageId::CapsLock,           0);
            pAhid->GetOutputItem(   &pItemScrollLockLed,    0,          UsagePageId::LED,               LedUsagePageId::ScrollLock,         LedUsagePageId::ScrollLock,         0);

            // Set LEDs to desired initial state (all off)
            uint8_t outputReport[pCodeBookHeader->outputSize];

            uint32_t numLockLedState    = 0;
            uint32_t capsLockLedState   = 0;
            uint32_t scrollLockLedState = 0;

            int32_t numLockLed          = 0;
            int32_t capsLockLed         = 0;
            int32_t scrollLockLed       = 0;

            memset(outputReport, 0, pCodeBookHeader->outputSize);

            pAhid->Encode(&numLockLed,      outputReport, pItemNumLockLed);
            pAhid->Encode(&capsLockLed,     outputReport, pItemCapsLockLed);
            pAhid->Encode(&scrollLockLed,   outputReport, pItemScrollLockLed);

            // 0x02 is "output report", see HID spec
            pAhid->SetReport(outputReport, sizeof(outputReport), 0x02, 0);

            bool setLeds = false;

            // read loop, this will exit when the keyboard is detached or there is an error
            while (read)
            {
                // We need a read buffer of pCodeBookHeader->inputSize to perform the read
                uint8_t data[pCodeBookHeader->inputSize];
                uint32_t bytesRead;

                // Read returns when the hardware delivers a "frame" of data, for HID devices the frequency is per SetIdle()
                result = pAhid->Read(&bytesRead, data, pCodeBookHeader->inputSize);

                if (result.IsSuccess())
                {
                    if (bytesRead)
                    {
                        NN_SDK_LOG("Raw input data: ");

                        for (uint32_t i = 0; i < bytesRead; i++)
                        {
                            NN_SDK_LOG("%02x ", data[i]);
                        }

                        NN_SDK_LOG("\n");

                        // We will try to decode for keyboard buttons.
                        int32_t lControl, lShift, lAlt, lGui, rControl, rShift, rAlt, rGui,
                                scanCode[6];

                        //            value             input   item
                        pAhid->Decode(&lControl,        data,   pItemLControl);
                        pAhid->Decode(&lShift,          data,   pItemLShift);
                        pAhid->Decode(&lAlt,            data,   pItemLAlt);
                        pAhid->Decode(&lGui,            data,   pItemLGui);
                        pAhid->Decode(&rControl,        data,   pItemRControl);
                        pAhid->Decode(&rShift,          data,   pItemRShift);
                        pAhid->Decode(&rAlt,            data,   pItemRAlt);
                        pAhid->Decode(&rGui,            data,   pItemRGui);
                        pAhid->Decode(&scanCode[0],     data,   pItemScanCode[0]);
                        pAhid->Decode(&scanCode[1],     data,   pItemScanCode[1]);
                        pAhid->Decode(&scanCode[2],     data,   pItemScanCode[2]);
                        pAhid->Decode(&scanCode[3],     data,   pItemScanCode[3]);
                        pAhid->Decode(&scanCode[4],     data,   pItemScanCode[4]);
                        pAhid->Decode(&scanCode[5],     data,   pItemScanCode[5]);

                        if (lControl) {NN_SDK_LOG("L Control ");}
                        if (lShift) {NN_SDK_LOG("L Shift ");}
                        if (lAlt) {NN_SDK_LOG("L Alt ");}
                        if (lGui) {NN_SDK_LOG("L Gui ");}
                        if (rControl) {NN_SDK_LOG("R Control ");}
                        if (rShift) {NN_SDK_LOG("R Shift ");}
                        if (rAlt) {NN_SDK_LOG("R Alt ");}
                        if (rGui) {NN_SDK_LOG("R Gui ");}

                        NN_SDK_LOG("scan code: ");

                        // process scan codes
                        for (int i = 0; i < 6; i++)
                        {
                            if (scanCode[i])
                            {
                                NN_SDK_LOG("%02x ", scanCode[i]);

                                if (scanCode[i] == 0x39)    // caps lock
                                {
                                    capsLockLedState++;
                                    setLeds = true;
                                }

                                if (scanCode[i] == 0x53)    // num lock
                                {
                                    numLockLedState++;
                                    setLeds = true;
                                }

                                if (scanCode[i] == 0x47)    // scroll lock
                                {
                                    scrollLockLedState++;
                                    setLeds = true;
                                }
                            }
                        }

                        NN_SDK_LOG("\n");

                        // set LEDs
                        if (setLeds)
                        {
                            capsLockLed     = capsLockLedState & 1;
                            numLockLed      = numLockLedState & 1;
                            scrollLockLed   = scrollLockLedState & 1;

                            memset(outputReport, 0, pCodeBookHeader->outputSize);

                            pAhid->Encode(&numLockLed, outputReport, pItemNumLockLed);
                            pAhid->Encode(&capsLockLed, outputReport, pItemCapsLockLed);
                            pAhid->Encode(&scrollLockLed, outputReport, pItemScrollLockLed);

                            // 0x02 is "output report", see HID spec
                            pAhid->SetReport(outputReport, sizeof(outputReport), 0x02, 0);

                            setLeds = false;
                        }
                    }
                }
                else
                {
                    read = false;

                    // Wait for detach event
                    stateChangeEvent.Wait();
                    pAhid->DestroyStateChangeEvent(stateChangeEvent.GetBase());
                }
            }
        }

        // Release the ownership of the device
        pAhid->Finalize();
    }

    NN_SDK_LOG("<-%s\n", __FUNCTION__);

} // NOLINT(impl/function_size)


//////////////////////////////////////////////////////////////////////////////
static void sendIntrOutput(nn::ahid::Ahid *pAhid, nn::ahid::hdr::DeviceParameters *pDeviceParameters)
{
    NN_SDK_LOG("->%s\n", __FUNCTION__);

    nn::Result result;

    // Ahid::Initialize() will attempt to own the device associated with the device parameters.
    // The device may have been acquired by another user, in this case it will fail.
    result = pAhid->Initialize(pDeviceParameters);

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Ahid::Initialize() returned module %d description %d\n", result.GetModule(), result.GetDescription());
    }

    if (result.IsSuccess())
    {
        bool write = true;
        uint16_t counter = 0;
        uint32_t bytesWritten;

        // Get state change event
        nn::os::SystemEvent stateChangeEvent;
        result = pAhid->CreateStateChangeEvent(stateChangeEvent.GetBase(), nn::os::EventClearMode_AutoClear);

        while (write)
        {
            NN_SDK_LOG("Calling pAhid->Write()\n");
            result = pAhid->Write(&bytesWritten, &counter, 2);
            NN_SDK_LOG("Returned from pAhid->Write() result %x\n", result);

            if (result.IsSuccess())
            {
                counter++;
            }
            else
            {
                NN_SDK_LOG("Result module %d description %d\n", result.GetModule(), result.GetDescription());

                write = false;

                // Wait for detach event
                stateChangeEvent.Wait();
                pAhid->DestroyStateChangeEvent(stateChangeEvent.GetBase());
            }
        }

        // Release the ownership of the device
        pAhid->Finalize();
    }

    NN_SDK_LOG("<-%s\n", __FUNCTION__);

} // NOLINT(impl/function_size)


//////////////////////////////////////////////////////////////////////////////
extern "C" void nnMain()
{
    nn::Result result = nn::ResultSuccess();
    nn::ahid::hdr::Hdr hdr;
    nn::ahid::hdr::AttachFilter attachFilter;
    nn::ahid::Ahid ahid;

    NN_SDK_LOG("->%s\n", __FUNCTION__);

    // Initialize HID Device Registry client library
    hdr.Initialize();

    // The call to SetDeviceFilterForHipc should return ResultFunctionNotSupported from HIPC client.
    nn::ahid::hdr::DeviceFilter deviceFilter = {0, 0, 0, 0};
    nn::Result result1 = hdr.SetDeviceFilterForHipc(&deviceFilter, 1);
    NN_SDK_LOG("SetDeviceFilterForHipc() returns module %d description %d\n", result1.GetModule(), result1.GetDescription());

    // Clear attach filter, to accept any devices
    memset(&attachFilter, 0, sizeof(nn::ahid::hdr::AttachFilter));

    // While no error, keep demonstrating connected HID device behavior
    while(result.IsSuccess())
    {
        size_t outEntries = 0;

        // While devices are not yet available
        while(result.IsSuccess())
        {
            // Get the number of connected devices
            if((result = hdr.GetDeviceEntries(&outEntries)).IsSuccess())
            {
                // stop looking if device is found
                if(outEntries != 0) break;

                // Nothing connected, wait
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        };

        // Device available?
        if (result.IsSuccess() && outEntries)
        {
            nn::ahid::hdr::DeviceHandle deviceHandle[nn::ahid::hdr::AhidDevicesCountMax];
            size_t inEntries = nn::ahid::hdr::AhidDevicesCountMax;

            // Get details for available devices
            result = hdr.GetDeviceList(&outEntries, inEntries, deviceHandle, &attachFilter);

            // If unable to retrieve details, it's possible that device has since disconnected
            if (!result.IsSuccess() || (outEntries == 0)) continue;

            // For all available devices
            for (size_t j = 0; j < outEntries; j++)
            {
                nn::ahid::hdr::DeviceParameters deviceParameters;
                result = hdr.GetDeviceParameters(&deviceParameters, deviceHandle[j]);
                if (result.IsSuccess())
                {
                    NN_SDK_LOG("HDR DeviceHandle:   %08x\n",    deviceHandle[j]);
                    NN_SDK_LOG("    deviceHandle:   %08x\n",    deviceParameters.deviceHandle);
                    NN_SDK_LOG("    servicePath:    %s\n",      deviceParameters.servicePath);
                    NN_SDK_LOG("    usagePage:      %04x\n",    deviceParameters.usagePage);
                    NN_SDK_LOG("    usageId:        %04x\n",    deviceParameters.usageId);
                    NN_SDK_LOG("    busId:          %d\n",      deviceParameters.busId);
                    NN_SDK_LOG("    vendorId:       %08x\n",    deviceParameters.vendorId);
                    NN_SDK_LOG("    productId:      %08x\n",    deviceParameters.productId);
                    NN_SDK_LOG("    versionNumber:  %08x\n",    deviceParameters.versionNumber);
                    NN_SDK_LOG("    manufacturer:   ");
                    printString(deviceParameters.manufacturer);
                    NN_SDK_LOG("    product:        ");
                    printString(deviceParameters.product);
                    NN_SDK_LOG("    serialNumber:   ");
                    printString(deviceParameters.serialNumber);

                    // Application looking for test intr out device
                    if ((deviceParameters.vendorId == 0x000004b4) && (deviceParameters.productId == 0x00006025))
                    {
                        NN_SDK_LOG("Found Intr out test device\n");
                        sendIntrOutput(&ahid, &deviceParameters);
                    }
                    else
                    {
                        // Application looking for a joystick
                        if ((deviceParameters.usagePage == UsagePageId::GenericDesktop) && (deviceParameters.usageId == GenericDesktopPageUsageId::Joystick))
                        {
                            readJoystickDevice(&ahid, &deviceParameters);
                        }

                        // Application looking for a gamepad
                        if ((deviceParameters.usagePage == UsagePageId::GenericDesktop) && (deviceParameters.usageId == GenericDesktopPageUsageId::Gamepad))
                        {
                            // gamepad has lots of typical things found in joysticks
                            readJoystickDevice(&ahid, &deviceParameters);
                        }

                        // Application looking for a mouse
                        if ((deviceParameters.usagePage == UsagePageId::GenericDesktop) && (deviceParameters.usageId == GenericDesktopPageUsageId::Mouse))
                        {
                            readMouseDevice(&ahid, &deviceParameters);
                        }

                        // Application looking for a keyboard
                        if ((deviceParameters.usagePage == UsagePageId::GenericDesktop) && (deviceParameters.usageId == GenericDesktopPageUsageId::Keyboard))
                        {
                            readKeyboardDevice(&ahid, &deviceParameters);
                        }

                        /*

                            Note:

                            - An application might also look for specific vendorId productId if it does not want generic device.
                            - In this case it may also know exactly how many bytes to read and cast the Read and Write data to know usage.
                            - CodeBook, Items, Decode and Encode does not need to be used in this case.
                            - Simply Read Write data.

                         */
                    }

                    NN_SDK_LOG("Device no longer available, looking for another...\n");
                }
            }
        }
    };

    hdr.Finalize();

    NN_SDK_LOG("<-%s\n", __FUNCTION__);
}
