﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace account {

template <typename ContextType>
class OperatorBase
{
private:
    nn::Result m_Result;

protected:
    OperatorBase() NN_NOEXCEPT
        : m_Result(nn::ResultSuccess())
    {
    }
    nn::Result SetResut(const nn::Result r) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Result);
        if (!r.IsSuccess())
        {
            m_Result = r;
            NN_RESULT_THROW(r);
        }
        NN_RESULT_SUCCESS;
    }
public:
    typedef ContextType Context;

    virtual nn::Result Create(Context* pOut, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT = 0;

    nn::Result GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }
};

class NaIdTokenOperator final
    : public OperatorBase<nn::account::NintendoAccountAuthorizationRequestContext>
{
private:
    const uint64_t m_ClientId;
    char m_Uri[128];
    const nn::account::NintendoAccountAuthorizationRequestParameters m_Params;

    void* m_Buffer;
    size_t m_BufferSize;

public:
    NaIdTokenOperator(
        uint64_t clientId, const char* uri, const nn::account::NintendoAccountAuthorizationRequestParameters& params,
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_ClientId(clientId)
        , m_Params(params)
        , m_Buffer(buffer)
        , m_BufferSize(bufferSize)
    {
        NN_SDK_ASSERT(strlen(uri) < sizeof(m_Uri));
        std::strncpy(m_Uri, uri, sizeof(m_Uri));
    }

    using OperatorBase<nn::account::NintendoAccountAuthorizationRequestContext>::Context;
    virtual nn::Result Create(Context* pOut, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT final NN_OVERRIDE
    {
        return SetResut(
            admin.CreateNintendoAccountAuthorizationRequest(
                pOut,
                m_ClientId, m_Uri, sizeof(m_Uri), m_Params, m_Buffer, m_BufferSize));
    }
};

class NaInfoCacheOperator final
    : public OperatorBase<nn::account::AsyncContext>
{
public:
    using OperatorBase<nn::account::AsyncContext>::Context;
    virtual nn::Result Create(Context* pOut, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT final NN_OVERRIDE
    {
        return SetResut(admin.RefreshCachedNintendoAccountInfoAsync(pOut));
    }
};

class NsaProfileOperator final
    : public OperatorBase<nn::account::AsyncContext>
{
public:
    using OperatorBase<nn::account::AsyncContext>::Context;
    virtual nn::Result Create(Context* pOut, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT final NN_OVERRIDE
    {
        return SetResut(admin.SynchronizeProfileAsync(pOut));
    }
};

}} // ~namespace nnt::account
