﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include "testAccount_Util.h"

#include <nn/util/util_ScopeExit.h>

bool TestOpenClose() NN_NOEXCEPT
{
    // 作成
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        // テスト用の明示的な終了
        nn::account::Finalize();
    };

    nn::account::Uid u;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
    nn::account::Finalize();

    // Open -> Close
    nn::account::Initialize();
    nn::account::UserHandle handle;
    auto r = nn::account::OpenUser(&handle, u);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(r);
    if (r.IsSuccess())
    {
        nn::account::CloseUser(handle);
    }
    nn::account::Finalize();

    // 削除
    nn::account::InitializeForAdministrator();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));

    return true;
}

bool TestRegistrationAsCommon(
    const int ExpectedCount = 0,
    const int ExpectedOpenCount = 0) NN_NOEXCEPT
{
    nn::account::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        // テスト用の明示的な終了
        nn::account::Finalize();
    };

    int count;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(ExpectedCount, count);

    int actualCount;
    nn::account::Uid users[nn::account::UserCountMax + 1];
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::ListAllUsers(&actualCount, users, sizeof(users) / sizeof(users[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(count, actualCount);

    // リストが仕様通りに構成されているか
    //  - 有効な要素数までの要素が true か
    //  - それ以降の要素が false か
    for (auto i = 0; i < static_cast<int>(sizeof(users) / sizeof(users[0])); ++ i)
    {
        if (i < actualCount)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(users[i]);
        }
        else
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(users[i]);
        }
    }
    // 列挙されたユーザーはすべて実在するか
    for (auto i = 0; i < actualCount; ++ i)
    {
        const auto& u = users[i];
        bool exist;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserExistence(&exist, u));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(exist);
    }
    // 最後に Open 状態にしたユーザーがユーザーリストに含まれるか
    nn::account::Uid lastOpened;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::GetLastOpenedUser(&lastOpened));
    if (lastOpened)
    {
        bool found = false;
        for (const auto& u: users)
        {
            found = (u && u == lastOpened);
            if (found) break;
        }
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(found);
    }

    int openCount;
    nn::account::Uid openUsers[nn::account::UserCountMax + 1];
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
        nn::account::ListOpenUsers(&openCount, openUsers, sizeof(openUsers) / sizeof(openUsers[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(ExpectedOpenCount, openCount);
    // リストが仕様通りに構成されているか
    //  - 有効な要素数までの要素が true か
    //  - それ以降の要素が false か
    for (auto i = 0; i < static_cast<int>(sizeof(openUsers) / sizeof(openUsers[0])); ++i)
    {
        if (i < openCount)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(openUsers[i]);
        }
        else
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(openUsers[i]);
        }
    }

    return true;
} // NOLINT(readability/fn_size)

bool TestRegistrationAsWatcher() NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        // テスト用の明示的な終了
        nn::account::Finalize();
    };

    // User 登録の通知の取得
    nn::account::Notifier notifiers[4];
    for (auto& n: notifiers)
    {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetUserRegistrationNotifier(&n));
        nn::os::SystemEvent e;
        n.GetSystemEvent(&e);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e.TryWait());

        // 手放す
        nn::account::Notifier n1(std::move(n));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e.TryWait());

        // これは落ちる
        // NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e.TryWait());
    }

    // User 状態変化の通知の取得
    for (auto i = 0u; i < sizeof(notifiers) / sizeof(notifiers[0]) - 1; ++ i)
    {
        auto& n = notifiers[i];

        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetUserRegistrationNotifier(&n));
        nn::os::SystemEvent e;
        n.GetSystemEvent(&e);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e.TryWait());

        // 付け替え
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            nn::account::GetUserStateChangeNotifier(&n));
        nn::os::SystemEvent e1;
        n.GetSystemEvent(&e1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e1.TryWait());

        // これは落ちる
        // NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(e.TryWait());
    }

    return true;
} // NOLINT(readability/fn_size)

bool TestRegistrationAsManager() NN_NOEXCEPT
{
    nn::account::InitializeForAdministrator();
    NN_UTIL_SCOPE_EXIT
    {
        // テスト用の明示的な終了
        nn::account::Finalize();
    };

    int count;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);

    nn::account::Uid u;
    auto declare = nn::account::EnableInterprogramOpenUserRetention({0x010000000000B121ull});

    // 1. 追加開始 → キャンセル
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(u);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CancelUserRegistration(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);

    // Notifier 取得
    nn::account::Notifier regNfr;
    nn::os::SystemEvent regEv;
    nn::account::Notifier statNfr;
    nn::os::SystemEvent statEv;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserRegistrationNotifier(&regNfr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(regNfr.GetSystemEvent(&regEv));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserStateChangeNotifier(&statNfr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(statNfr.GetSystemEvent(&statEv));

    // 2. 追加開始 → 確定
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(u);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(statEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(0, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(regEv.TryWait());
    regEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(statEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(1, count);

    // 3. 2 人目
    nn::account::Uid u1;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(regEv.TryWait());
    regEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(2, count);

    // 4. Open -> 状態検査
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateOpen(u1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(2, 1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateOpen(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(2, 2));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateClose(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(2, 1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateClose(u1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(2, 0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateOpen(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(2, 1));

    // 5. 並び替え
    nn::account::Uid users[2];
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::SetUserPosition(u1, 0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(2, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(u1, users[0]);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(u, users[1]);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::SetUserPosition(u1, 1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(2, count);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(u, users[0]);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_EQ(u1, users[1]);

    // 6. 削除
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultUnexpectedUserState,
        nn::account::DeleteUser(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(statEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(regEv.TryWait());
    regEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(statEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(1, 1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DebugSetUserStateClose(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(regEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(statEv.TryWait());
    statEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(1, 0));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(regEv.TryWait());
    regEv.Clear();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_NOT(statEv.TryWait());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestRegistrationAsCommon(0, 0));

    return true;
} // NOLINT(readability/fn_size)
