﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include <nnt/nntest.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
/*
    InitializeWith 用のリソース
*/
typedef std::aligned_storage<sizeof(nnt::account::DefaultTestResource), NN_ALIGNOF(nnt::account::DefaultTestResource)>::type Storage;
Storage g_AccountResourceStorage;
Storage g_AccountResourceStorage2;

bool TestCleanupAtExit()
{
    // サービスの初期化
    auto pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
    nn::account::InitializeWith(pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
    NN_UTIL_SCOPE_EXIT
    {
        pSrc->~ServiceResource();
        nn::account::Finalize();
    };

    // サービスの開始
    nn::account::Uid u;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::DeleteUser(u);
    };
    nn::account::Uid tmp;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&tmp));
    nn::account::Uid tmp2;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultRegistrationAlreadyBegins, nn::account::BeginUserRegistration(&tmp2));

    // サービスの終了
    //  - 作成中ユーザーは削除される
    nn::account::Finalize();
    pSrc->~ServiceResource();

    // サービスの再開
    // - 作成済みのユーザーが 1 人存在する
    // - 作成中のユーザーは存在しない
    pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
    nn::account::InitializeWith(pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
    int count;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 1);
    nn::account::Uid t;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, &t, 1));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 1);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(u == t);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&tmp));

    return true;
}

bool TestCleanupAtStart()
{
    {
        // サービスの初期化
        auto pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
        nn::account::InitializeWith(pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
        NN_UTIL_SCOPE_EXIT
        {
            pSrc->~ServiceResource();
            nn::account::Finalize();
        };

        // サービスの開始
        nn::account::Uid u;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistration(u));
        nn::account::Uid tmp;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&tmp));
        nn::account::Uid tmp2;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::account::ResultRegistrationAlreadyBegins, nn::account::BeginUserRegistration(&tmp2));

        // サービスの再開
        // - 作成済みのユーザーが 1 人存在する
        // - 作成中のユーザーは削除され、存在しなくなる
        auto pSrc2 = new(&g_AccountResourceStorage2) nnt::account::DefaultTestResource;
        NN_UTIL_SCOPE_EXIT
        {
            pSrc2->~ServiceResource();
        };
        nn::account::Finalize();
        nn::account::InitializeWith(pSrc2->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(pSrc->CreateServiceObjectForApplication()); // 前のサービスはまだ生きている
        int count;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetUserCount(&count));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 1);
        nn::account::Uid t;
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, &t, 1));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(count == 1);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(u == t);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&tmp));
    }

    // ユーザーの削除
    auto pSrc = new(&g_AccountResourceStorage) nnt::account::DefaultTestResource;
    nn::account::InitializeWith(pSrc->GetServicePointer<nn::account::IAccountServiceForAdministrator>());
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
        pSrc->~ServiceResource();
    };

    int lengthActual;
    nn::account::Uid users[nn::account::UserCountMax];
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&lengthActual, users, nn::account::UserCountMax));
    for (auto& u : users)
    {
        if (u)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::DeleteUser(u));
        }
    }
    return true;
}
} // ~namespace <anonymous>


TEST(AccountService, CleanUpPendings)
{
    ASSERT_TRUE(TestCleanupAtExit());
    ASSERT_TRUE(TestCleanupAtStart());
}
