﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiNetworkServiceLicense.h>

#include "testAccount_ServiceUtil.h"
#include "testAccount_Util.h"

#include "testAccount_LinkageApi.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_NAS_OP2_MEMBERSHIP

#if defined(NNT_ACCOUNT_ENABLE_NAS_OP2_MEMBERSHIP)

namespace
{
nn::Result WaitAsyncContext(nn::account::AsyncContext&& async) NN_NOEXCEPT
{
    nn::os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&e));
    e.Wait();
    return async.GetResult();
}

bool TestBasic(const nn::account::Uid& user, const nnt::account::NasLoginInfo& loginInfo) NN_NOEXCEPT
{
    nn::account::InitializeForSystemService();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::NetworkServiceAccountManager manager;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(
        &manager, user));

    nn::account::CachedNetworkServiceLicenseInfo cachedInfo;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetCachedNetworkServiceLicenseInfo(&cachedInfo));

    nn::account::AsyncContext async;
    bool matched;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNetworkServiceLicenseInfoAsyncIfTimeElapsed(
        &matched,
        &async, nn::TimeSpan::FromSeconds(0xFFFF)));
    EXPECT_FALSE(matched);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNetworkServiceLicenseInfoAsyncIfTimeElapsed(
        &matched,
        &async, nn::TimeSpan::FromSeconds(3)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(matched);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncContext(std::move(async)));

    // Invalidate -> Refresh
    nn::account::DebugInvalidateUserResourceCache(user);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        nn::account::ResultResourceCacheUnavailable,
        manager.GetCachedNetworkServiceLicenseInfo(&cachedInfo));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.RefreshCachedNetworkServiceLicenseInfoAsync(&async));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncContext(std::move(async)));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(manager.GetCachedNetworkServiceLicenseInfo(&cachedInfo));
    return true;
}

bool TestBasicForApplication(const nn::account::Uid& user)
{
    nn::account::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::Finalize();
    };

    nn::account::UserHandle userHandle;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, user));
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(userHandle);
    };

    nn::account::AsyncNetworkServiceLicenseKindContext async;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(nn::account::LoadNetworkServiceLicenseKindAsync(&async, userHandle));

    nn::os::SystemEvent systemEvent;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&systemEvent));
    systemEvent.Wait();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(async.GetResult());
    auto license = async.GetNetworkServiceLicenseKind();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(license == nn::account::NetworkServiceLicenseKind_None || license == nn::account::NetworkServiceLicenseKind_Common);
    return true;
}

} // ~namespace <anonymous>

TEST(AccountNintendoAccount, Op2Membership)
{
    NN_ABORT_UNLESS(nn::nifm::IsNetworkAvailable());

    NN_UTIL_SCOPE_EXIT
    {
        // 削除
        nnt::account::Cleanup();
    };

    // 作成
    auto naList = nnt::account::LoadNaList();
    std::unique_ptr<nn::account::Uid[]> users(new nn::account::Uid[naList.Count()]);
    nnt::account::CreateUsers(users.get(), naList.Count());

    {
        nn::account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            nn::account::Finalize();
        };

        for (auto i = 0; i < naList.Count(); ++ i)
        {
            nn::account::NetworkServiceAccountAdministrator admin;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(
                &admin, users[i]));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));

            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(LinkNintendoAccount(naList[i], admin));
        }
    }

    for (auto i = 0; i < naList.Count(); ++ i)
    {
        ASSERT_TRUE(TestBasic(users[i], naList[i]));
        ASSERT_TRUE(TestBasicForApplication(users[i]));
    }
}

#endif // NNT_ACCOUNT_ENABLE_NAS_OP2_MEMBERSHIP
