﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/ndas/account_NdasDriver.h>
#include <nn/account/ndas/account_ResultForNdas.h>

#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

#include <curl/curl.h>

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/dauth/dauth_Api.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/socket.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_NDAS_DRIVER

#if defined(NNT_ACCOUNT_ENABLE_NDAS_DRIVER)
namespace
{
bool IsDeviceAuthenticationCacheAvailable(uint64_t clientId) NN_NOEXCEPT
{
    auto t = nn::os::GetSystemTick();
    nn::TimeSpan exp;
    int length;
    std::unique_ptr<char[]> buffer(new char[2048]);
    auto r = nn::dauth::AcquireDeviceAuthenticationToken(&exp, &length, buffer.get(), 2048, clientId, false, nullptr);
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    return (nn::os::GetSystemTick() - t).ToTimeSpan() < nn::TimeSpan::FromSeconds(1);
}

void NdasDriverBasicDeviceAuthentication(
    CURL* curlHandle,
    const a::detail::AbstractLocalStorage& storage,
    const int TestCount, bool force) NN_NOEXCEPT
{
    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // NDAS Driver
    nn::account::ndas::NdasDriver ndasDriver(appAuthCache);

    // Client ID
    const uint64_t ClientIDs[] = {
        a::ndas::ClientIdForApplications,
        a::ndas::ClientIdForNintendoAccount,
    };

    // ----------------------------------------

    NN_SDK_ASSERT(curlHandle != nullptr);
    t::Buffer buffer(a::ndas::NdasDriver::RequiredBufferSize);

    // ----------------------------------------
    // デバイス認証トークン
    for (const auto ClientId: ClientIDs)
    {
        NN_LOG("[nnt::account] Device Authentication\n");
        for (auto i = 0; i < TestCount; ++ i)
        {
            auto begin = nn::os::GetSystemTick();
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasDriver.EnsureDeviceAuthenticationCache(
                ClientId, force, nullptr));
            NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
            EXPECT_TRUE(IsDeviceAuthenticationCacheAvailable(ClientId));
        }
    }
}

void NdasDriverBasicApplicationAuthentication(
    CURL* curlHandle,
    const a::detail::AbstractLocalStorage& storage,
    const int TestCount, bool force) NN_NOEXCEPT
{
    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // NDAS Driver
    nn::account::ndas::NdasDriver ndasDriver(appAuthCache);

    // アプリケーション
    const struct
    {
        a::detail::ApplicationInfo appInfo;
        a::ndas::ApplicationMeta meta;
    } tests[] = {
        {
            a::detail::ApplicationInfo::Get(0x010000000000B121ull, 0, a::detail::ApplicationMediaType::System),
            {a::nas::InvalidNasClientInfo}
        },
        {
            a::detail::ApplicationInfo::Get(0x010000000000B123ull, 0, a::detail::ApplicationMediaType::System),
            {{0x1234567890abcdefull, "https://www.pokemon.jp/"}}
        },
        {
            a::detail::ApplicationInfo::Get(0x0100000000002803ull, 0, a::detail::ApplicationMediaType::System),
            {{0x57d3dbaa12cb06a9ull, "nintendo://0100000000002803.app"}}
        }
    };

    // ----------------------------------------

    NN_SDK_ASSERT(curlHandle != nullptr);
    t::Buffer buffer(a::ndas::NdasDriver::RequiredBufferSize);

    // デバイス認証トークン
    NN_LOG("[nnt::account] Device Authentication\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasDriver.EnsureDeviceAuthenticationCache(
        a::ndas::ClientIdForApplications, force, nullptr));
    EXPECT_TRUE(IsDeviceAuthenticationCacheAvailable(a::ndas::ClientIdForApplications));

    // ----------------------------------------
    // アプリケーション認証トークン
    NN_LOG("[nnt::account] Application Authentication\n");
    for (auto test : tests)
    {
        EXPECT_FALSE(ndasDriver.IsApplicationAuthenticationCacheAvailable(test.appInfo));

        for (auto i = 0; i < TestCount; ++ i)
        {
            auto begin = nn::os::GetSystemTick();
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasDriver.EnsureApplicationAuthenticationCache(
                test.appInfo, force, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
            NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
            EXPECT_TRUE(ndasDriver.IsApplicationAuthenticationCacheAvailable(test.appInfo));

            a::ndas::ApplicationMeta meta;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasDriver.GetApplicationMetaCache(&meta, test.appInfo));
            if (test.meta.nasClientInfo)
            {
                EXPECT_TRUE(meta.nasClientInfo);
                EXPECT_EQ(test.meta.nasClientInfo.clientId, meta.nasClientInfo.clientId);
                EXPECT_EQ(0, std::strncmp(test.meta.nasClientInfo.redirectUri, meta.nasClientInfo.redirectUri, sizeof(test.meta.nasClientInfo.redirectUri)));
            }
            else
            {
                EXPECT_FALSE(meta.nasClientInfo);
            }
        }
    }
}

void NdasDriverFailure(
    CURL* curlHandle,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_SDK_ASSERT(curlHandle != nullptr);
    t::Buffer buffer(a::ndas::NdasDriver::RequiredBufferSize);

    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // NDAS Driver
    nn::account::ndas::NdasDriver ndasDriver(appAuthCache);
    // デバイス認証トークン
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasDriver.EnsureDeviceAuthenticationCache(
        a::ndas::ClientIdForApplications, true, nullptr));

    // アプリケーション認証トークン 取得失敗
    const struct
    {
        a::detail::ApplicationInfo appInfo;
        nn::Result result;
    } TestCases[] = {
        {
            // Unauthorized application (0x0000000000000001ull)
            a::detail::ApplicationInfo::Get(0x0000000000000001ull, 0, a::detail::ApplicationMediaType::System),
            a::ndas::ResultNdasStatusNo0106(),
        },
#if 0
        {
            // Device has been banned by application (0x010000000000B124ull)
            a::detail::ApplicationInfo::Get(0x010000000000B124ull, 0, a::detail::ApplicationMediaType::System),
            a::ndas::ResultNdasStatusNo0107(),
        },
#endif
        {
            // Service is not started (0x010000000000B1250ull)
            a::detail::ApplicationInfo::Get(0x010000000000B1250ull, 0, a::detail::ApplicationMediaType::System),
            a::ndas::ResultNdasStatusNo0108(),
        },
        {
            // Service closed (0x010000000000B126ull)
            a::detail::ApplicationInfo::Get(0x010000000000B126ull, 0, a::detail::ApplicationMediaType::System),
            a::ndas::ResultNdasStatusNo0109(),
        },
        {
            // Service is not available (010000000000B127)
            a::detail::ApplicationInfo::Get(0x010000000000B127ull, 0, a::detail::ApplicationMediaType::System),
            a::ndas::ResultNdasStatusNo0110(),
        },
        {
            // Application update is required (010000000000B128)
            a::detail::ApplicationInfo::Get(0x010000000000B128ull, 0, a::detail::ApplicationMediaType::System),
            a::ResultUnacceptableApplicationVersion(),
        },
    };
    const auto TestCount = sizeof(TestCases) / sizeof(TestCases[0]);

    NN_LOG("[nnt::account] Application Authentication\n");
    for (auto i = 0; i < TestCount; ++ i)
    {
        auto begin = nn::os::GetSystemTick();
        ASSERT_TRUE(
            TestCases[i].result.CanAccept(
                ndasDriver.EnsureApplicationAuthenticationCache(TestCases[i].appInfo, true, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr)));
        NN_LOG("  <- %lld msec\n", (nn::os::GetSystemTick() - begin).ToTimeSpan().GetMilliSeconds());
        EXPECT_FALSE(ndasDriver.IsApplicationAuthenticationCacheAvailable(TestCases[i].appInfo));
    }
}
} // ~namespace <anonymous>

TEST(AccountDriver, NdasDriverBasic)
{
    const int TestCount = 5;

    // ストレージ
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto handle = curl_easy_init();
    ASSERT_TRUE(handle != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(handle);
    };

    NdasDriverBasicDeviceAuthentication(handle, s, TestCount, false);
    NdasDriverBasicDeviceAuthentication(handle, s, TestCount, true);

    NdasDriverBasicApplicationAuthentication(handle, s, TestCount, false);
    NdasDriverBasicApplicationAuthentication(handle, s, TestCount, true);

    NdasDriverFailure(handle, s);
}

#endif

