﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasLoginDriver.h>
#include <nn/account/nas/account_NasUserDriver.h>
#include <nn/account/nas/account_NasOp2Driver.h>
#include <nn/account/nas/account_NasUserResourceCache.h>

#include <nn/account/baas/account_BaasLoginDriver.h>
#include <nn/account/baas/account_BaasRegistrationDriver.h>
#include <nn/account/baas/account_BaasUserDriver.h>
#include <nn/account/baas/account_BaasUserInfoHolder.h>
#include <nn/account/baas/account_ResultForBaas.h>

#include "nas/account_NasCredentialHolder.h"
#include "nas/account_NasLoginAdaptor.h"
#include "nas/account_NasResourceResolver.h"
#include "nas/account_NasUserAdaptor.h"

#include "detail/account_CacheUtil.h"
#include "detail/account_PathUtil.h"
#include "http/account_OAuthUtil.h"
#include <nn/account/account_RuntimeResource.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>

#include "testAccount_NaList.h"
#include "testAccount_NasProxy.h"
#include "testAccount_File.h"
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

#include <algorithm>
#include <curl/curl.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_NAS_DRIVER

#if defined(NNT_ACCOUNT_ENABLE_NAS_DRIVER)

namespace {

bool NasAuthorizationRequest(
    a::detail::Uuid* pOutCodeCacheId,
    const char* url, const char (&callbackUri)[256], const a::nas::State& state,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    t::Buffer callback;

    const bool UseNasProxy = true;
    if (NN_STATIC_CONDITION(UseNasProxy))
    {
        callback = t::GetAuthorizationViaNasProxy(t::LoadNaList()[0], url);
        a::detail::CacheUtil::StoreCacheFile(pOutCodeCacheId, callback.GetAddress(), strlen(callback.Get<char>()), storage);
    }

    a::nas::NasServiceAuthorizationAdaptor authzAdaptor(callbackUri, state, storage);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(authzAdaptor.Parse(callback.Get<char>(), callback.GetSize()));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(authzAdaptor.Adapt());
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(authzAdaptor.PullAuthorizationCode(pOutCodeCacheId));
    return true;
}

bool NasRefreshUserResourceCache(
    a::nas::NasUserResourceCache& resourceCache,
    const a::NintendoAccountId id, const a::nas::NasClientInfo& clientInfo,
    CURL* curlHandle, const t::Buffer& buffer,
    a::nas::NasLoginDriver& loginDriver, a::nas::NasUserDriver& userDriver, const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    // Token Refresh
    a::nas::NasAccessTokenCache accessTokenCache = {0, a::detail::InvalidUuid};
    NN_UTIL_SCOPE_EXIT
    {
        if (accessTokenCache.cacheId)
        {
            a::detail::CacheUtil::DeleteCacheFile(accessTokenCache.cacheId, storage);
        }
    };

    NN_LOG("[account] Refresh access token\n");
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(loginDriver.RefreshAccessToken(
        &accessTokenCache, id, clientInfo, a::nas::NasResourceResolver::GetScopeForAccountSystem(),
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // User resource refresh
    NN_LOG("[account] Refresh user resource\n");
    a::NintendoAccountId naId;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(userDriver.AcquireUserResourceCache(
        &naId, resourceCache, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    EXPECT_EQ(id, naId);
    return true;
}

bool CompareNasUserBase(const a::nas::NasUserBase& base, const a::nas::NasUserBase& base2) NN_NOEXCEPT
{
    NN_LOG("[account] User resource comparison\n");
    EXPECT_EQ(base.isChild, base2.isChild);
    EXPECT_EQ(0, std::strncmp(base.screenName, base2.screenName, sizeof(base.email)));
    EXPECT_EQ(0, std::strncmp(base.email, base2.email, sizeof(base.email)));
    EXPECT_EQ(0, std::strncmp(base.loginId, base2.loginId, sizeof(base.loginId)));
    EXPECT_EQ(0, std::strncmp(base.nickname, base2.nickname, sizeof(base.nickname)));
    EXPECT_EQ(0, std::strncmp(base.birthday, base2.birthday, sizeof(base.birthday)));
    EXPECT_EQ(0, std::strncmp(base.gender, base2.gender, sizeof(base.gender)));
    EXPECT_EQ(base.analyticsOptedIn, base2.analyticsOptedIn);
    EXPECT_EQ(0, std::strncmp(base.language, base2.language, sizeof(base.language)));
    EXPECT_EQ(0, std::strncmp(base.country, base2.country, sizeof(base.country)));
    EXPECT_EQ(0, std::strncmp(base.region, base2.region, sizeof(base.region)));
    EXPECT_EQ(0, std::strncmp(base.timezone, base2.timezone, sizeof(base.timezone)));
    EXPECT_EQ(base.isNnLinked, base2.isNnLinked);
    EXPECT_EQ(base.isTwitterLinked, base2.isTwitterLinked);
    EXPECT_EQ(base.isFacebookLinked, base2.isFacebookLinked);
    EXPECT_EQ(base.isGoogleLinked, base2.isGoogleLinked);
    return true;
}

struct AuthorizationRequestParameter
{
    a::nas::NasClientInfo clientInfo;
    a::http::CodeVerifier codeVerifier;
    a::nas::State state;
    char loginUrl[4097];
};

void AcquireUserResource(
    const AuthorizationRequestParameter& param,
    CURL* curlHandle, const t::Buffer& buffer, const t::Buffer& largeBuffer,
    a::nas::NasLoginDriver& loginDriver, a::nas::NasUserDriver& userDriver, a::nas::NasUserResourceCache& resourceCache,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    a::detail::Uuid codeCacheId = a::detail::InvalidUuid;
    ASSERT_TRUE(NasAuthorizationRequest(&codeCacheId, param.loginUrl, param.clientInfo.redirectUri, param.state, storage));

    // NAS Login
    NN_LOG("[account] Acquire primary tokens\n");
    a::nas::NasCredentialCache credentialCache = {a::detail::InvalidUuid, a::detail::InvalidUuid};
    a::nas::NasAccessTokenCache accessTokenCache = {0, a::detail::InvalidUuid};
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(loginDriver.AcquirePrimaryTokens(
        &credentialCache, &accessTokenCache, param.clientInfo, codeCacheId, param.codeVerifier,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // User リソースの取得
    NN_LOG("[account] Acquire user resource\n");
    a::NintendoAccountId naId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userDriver.AcquireUserResourceCache(
        &naId, resourceCache, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // トークンとユーザーリソースの永続化
    {
        auto lock = storage.AcquireWriterLock();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::nas::NasCredentialHolder::Store(
            naId, credentialCache, param.codeVerifier, storage, buffer.GetAddress(), buffer.GetSize()));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Perpetuate(naId));
    }

    // リソースキャッシュの内容の検討
    a::nas::NasUserBase base;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Load(&base, naId, buffer.GetAddress(), buffer.GetSize()));

    // リソースリフレッシュ操作
    ASSERT_TRUE(NasRefreshUserResourceCache(resourceCache, naId, param.clientInfo, curlHandle, buffer, loginDriver, userDriver, storage));

    // リソースキャッシュの内容の検討
    a::nas::NasUserBase base2;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Load(&base2, naId, buffer.GetAddress(), buffer.GetSize()));
    ASSERT_TRUE(CompareNasUserBase(base, base2));
}

void LinkToNetworkServiceAccount(
    const a::NetworkServiceAccountId& id, const a::baas::BaasCredential& credential,
    const AuthorizationRequestParameter& param,
    CURL* curlHandle, const t::Buffer& buffer, const t::Buffer& largeBuffer,
    a::baas::BaasLoginDriver& baasLoginDriver, a::baas::BaasUserDriver& baasUserDriver,
    a::nas::NasLoginDriver& loginDriver, a::nas::NasUserDriver& userDriver, a::nas::NasUserResourceCache& resourceCache,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    a::NetworkServiceAccountId loggedInAs = {};
    NN_UTIL_SCOPE_EXIT
    {
        if (loggedInAs.id != 0x00ull)
        {
            NN_LOG("[account] Unlink nintendo account\n");
            baasUserDriver.UnlinkFromNintendoAccount(loggedInAs, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr);
        }
    };

    // TODO すべてのアカウント BG タスクの停止？

    // ～ Authorization code の取得
    a::detail::Uuid codeCacheId = a::detail::InvalidUuid;
    ASSERT_TRUE(NasAuthorizationRequest(&codeCacheId, param.loginUrl, param.clientInfo.redirectUri, param.state, storage));

    // NAS Login
    NN_LOG("[account] Acquire primary tokens\n");
    a::nas::NasCredentialCache credentialCache = {a::detail::InvalidUuid, a::detail::InvalidUuid};
    a::nas::NasAccessTokenCache accessTokenCache = {0, a::detail::InvalidUuid};
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(loginDriver.AcquirePrimaryTokens(
        &credentialCache, &accessTokenCache, param.clientInfo, codeCacheId, param.codeVerifier,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // User リソースの取得
    NN_LOG("[account] Acquire user resource\n");
    a::NintendoAccountId naId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userDriver.AcquireUserResourceCache(
        &naId, resourceCache, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // サーバー上での紐づけ処理
    // POST federation
    a::baas::UserProfile profile;
    auto r = baasLoginDriver.ExecuteFederationLoginWithNintendoAccount(
        &loggedInAs, &profile, credential, naId, credentialCache.idTokenCacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr);
    if (r.IsSuccess())
    {
        NN_LOG("[account] Logged in as: %016llx\n", loggedInAs.id);
        NN_LOG("[account] Nickname    : %s\n", profile.base.nickname);
    }
    else
    {
        loggedInAs = id;
        ASSERT_TRUE(a::baas::ResultBaasStatus400LinkedUserNotFound::Includes(r));
        // POST Link
        NN_LOG("[account] This is first linkage\n");
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.LinkWithNintendoAccount(
            id, naId, credentialCache.idTokenCacheId,
            curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    }
}

void LinkUnlink(
    const a::NetworkServiceAccountId& id, const a::baas::BaasCredential& credential,
    const AuthorizationRequestParameter& param,
    CURL* curlHandle, const t::Buffer& buffer, const t::Buffer& largeBuffer,
    a::baas::BaasLoginDriver& baasLoginDriver, a::baas::BaasUserDriver& baasUserDriver,
    a::nas::NasLoginDriver& loginDriver, a::nas::NasUserDriver& userDriver, a::nas::NasUserResourceCache& resourceCache,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        NN_LOG("[account] Unlink nintendo account\n");
        baasUserDriver.UnlinkFromNintendoAccount(id, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr);
    };

    // TODO すべてのアカウント BG タスクの停止？

    // ～ Authorization code の取得
    a::detail::Uuid codeCacheId = a::detail::InvalidUuid;
    ASSERT_TRUE(NasAuthorizationRequest(&codeCacheId, param.loginUrl, param.clientInfo.redirectUri, param.state, storage));

    // NAS Login
    NN_LOG("[account] Acquire primary tokens\n");
    a::nas::NasCredentialCache credentialCache;
    a::nas::NasAccessTokenCache accessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(loginDriver.AcquirePrimaryTokens(
        &credentialCache, &accessTokenCache, param.clientInfo, codeCacheId, param.codeVerifier,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // User リソースの取得
    NN_LOG("[account] Acquire user resource\n");
    a::NintendoAccountId naId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userDriver.AcquireUserResourceCache(
        &naId, resourceCache, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // POST Link
    NN_LOG("[account] Link\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.LinkWithNintendoAccount(
        id, naId, credentialCache.idTokenCacheId,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    NN_LOG("[account] Link\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.LinkWithNintendoAccount(
        id, naId, credentialCache.idTokenCacheId,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    // POST Unlink
    NN_LOG("[account] Unlink\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.UnlinkFromNintendoAccount(
        id, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    // POST Link
    NN_LOG("[account] Link\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.LinkWithNintendoAccount(
        id, naId, credentialCache.idTokenCacheId,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    // POST Federation
    NN_LOG("[account] Federation\n");
    a::NetworkServiceAccountId loggedInAs;
    a::baas::UserProfile profile;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.ExecuteFederationLoginWithNintendoAccount(
        &loggedInAs, &profile, credential, naId, credentialCache.idTokenCacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    EXPECT_EQ(id, loggedInAs);
    // POST Unlink
    NN_LOG("[account] Unlink\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasUserDriver.UnlinkFromNintendoAccount(
        id, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    // POST Federation
    NN_LOG("[account] Federation\n");
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::baas::ResultBaasStatus400LinkedUserNotFound,
        baasLoginDriver.ExecuteFederationLoginWithNintendoAccount(
            &loggedInAs, &profile, credential, naId, credentialCache.idTokenCacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));
}

void Op2Membership(
    const AuthorizationRequestParameter& param,
    CURL* curlHandle, const t::Buffer& buffer, const t::Buffer& largeBuffer,
    a::nas::NasLoginDriver& loginDriver, a::nas::NasUserDriver& userDriver, a::nas::NasOp2Driver& op2Driver,
    a::nas::NasUserResourceCache& resourceCache,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    a::detail::Uuid codeCacheId = a::detail::InvalidUuid;
    ASSERT_TRUE(NasAuthorizationRequest(&codeCacheId, param.loginUrl, param.clientInfo.redirectUri, param.state, storage));

    // NAS Login
    NN_LOG("[account] Acquire primary tokens\n");
    a::nas::NasCredentialCache credentialCache = {a::detail::InvalidUuid, a::detail::InvalidUuid};
    a::nas::NasAccessTokenCache accessTokenCache = {0, a::detail::InvalidUuid};
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(loginDriver.AcquirePrimaryTokens(
        &credentialCache, &accessTokenCache, param.clientInfo, codeCacheId, param.codeVerifier,
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    // User リソースの取得
    NN_LOG("[account] Acquire user resource\n");
    a::NintendoAccountId naId;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userDriver.AcquireUserResourceCache(
        &naId, resourceCache, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    NN_LOG("[account] Acquire access token\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(loginDriver.RefreshAccessToken(
        &accessTokenCache, naId, param.clientInfo, a::nas::NasResourceResolver::GetScopeForOp2Membership(),
        curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    NN_LOG("[account] Acquire OP2 membership\n");
    a::NetworkServiceLicense license;
    nn::time::PosixTime expiration;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(op2Driver.GetMembership(
        &license, &expiration, naId, accessTokenCache.cacheId, curlHandle, buffer.GetAddress(), buffer.GetSize(), nullptr));

    switch (license)
    {
    case a::NetworkServiceLicense_Common:
        {
            auto calendar = nn::time::ToCalendarTimeInUtc(expiration);
            NN_LOG("[account] Acquire OP2 membership: Common (%d-%02d-%02d %02d:%02d:%02d)\n",
                calendar.year, calendar.month, calendar.day,
                calendar.hour, calendar.minute, calendar.second);
        }
        break;
    case a::NetworkServiceLicense_None:
        NN_LOG("[account] Acquire OP2 membership: None\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
} // ~namespace <anonymous>

TEST(AccountDriver, NasDriverBasic)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());

    // NDAS リソース
    a::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));
    a::ndas::NdasOperator ndasOperator(appAuthCache);

    // Nas リソース
    a::nas::NasUserResourceCache resourceCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Initialize(storage));
    a::nas::NasLoginDriver loginDriver(ndasOperator, storage);
    a::nas::NasUserDriver userDriver(storage);
    a::nas::NasOp2Driver op2Driver(storage);

    // 実行時リソース
    auto handle = curl_easy_init();
    ASSERT_TRUE(handle != nullptr);
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(handle);
    };
    t::Buffer buffer(a::RequiredBufferSize);
    t::Buffer largeBuffer(512 * 1024);
    a::ExecutionResource resource = {handle, {buffer.GetAddress(), buffer.GetSize()}};

    // NAS に送信するパラメータの生成
    AuthorizationRequestParameter param = {};
    param.clientInfo = a::nas::NasResourceResolver::GetClientIdForAccountSystem();

    std::memset(param.codeVerifier.data, 'A', sizeof(param.codeVerifier.data));
    a::http::GenerateStateString(param.state.data, sizeof(param.state.data));
    loginDriver.GetUrlForAuthorizationRequest(
        param.loginUrl, sizeof(param.loginUrl), param.clientInfo, param.codeVerifier, param.state, a::NintendoAccountAuthorizationPageTheme_Intro);
    NN_LOG("Login URL: %s\n", param.loginUrl);

    // -----------------------------------------------------------------------------------------
    // 単機能のテスト

    // NAS アクセス用のデバイス認証
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasOperator.AuthenticateServiceForNintendoAccount(resource, nullptr));

    // NAS アクセス単体のテスト
    AcquireUserResource(
        param, handle, buffer, largeBuffer,
        loginDriver, userDriver, resourceCache, storage);
    Op2Membership(
        param, handle, buffer, largeBuffer,
        loginDriver, userDriver, op2Driver, resourceCache, storage);


    // -----------------------------------------------------------------------------------------
    // BaaS との連携する機能のテスト

    // BaaS リソース
    a::baas::ClientAccessTokenCache clientAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(clientAccessTokenCache.Initialize(storage));
    a::baas::UserAccessTokenCache userAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userAccessTokenCache.Initialize(storage));
    a::baas::UserIdTokenCache userIdTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userIdTokenCache.Initialize(storage));
    a::baas::BaasRegistrationDriver baasRegDriver(userAccessTokenCache, userIdTokenCache, clientAccessTokenCache);
    a::baas::BaasLoginDriver baasLoginDriver(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, ndasOperator, storage);
    a::baas::BaasUserDriver baasUserDriver(userAccessTokenCache, storage);

    // BaaS アカウント
    NN_LOG("[account] Preparing BaaS account\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ndasOperator.AuthenticateServiceForApplications(resource, nullptr));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureClientAccessTokenCache(
        handle, buffer.GetAddress(), buffer.GetSize(), nullptr));
    a::NetworkServiceAccountId id;
    a::baas::BaasCredential credential;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasRegDriver.RegisterDeviceAccount(
        &id, &credential, handle, buffer.Get<char>(), buffer.GetSize(), nullptr));
    NN_UTIL_SCOPE_EXIT
    {
        NN_LOG("[account] Unregister device account\n");
    baasRegDriver.UnregisterDeviceAccount(
        id, credential, handle, buffer.Get<char>(), buffer.GetSize(), nullptr);
    };

    // BaaS アクセス用のデバイス認証
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasLoginDriver.EnsureUserAccessTokenCache(
        id, credential, nullptr, handle, buffer.Get<char>(), buffer.GetSize(), nullptr));

    // BaaS との連携テスト
    LinkToNetworkServiceAccount(
        id, credential, param, handle, buffer, largeBuffer,
        baasLoginDriver, baasUserDriver,
        loginDriver, userDriver, resourceCache, storage);
    LinkUnlink(
        id, credential, param, handle, buffer, largeBuffer,
        baasLoginDriver, baasUserDriver,
        loginDriver, userDriver, resourceCache, storage);
}

#endif // NNT_ACCOUNT_ENABLE_NAS_DRIVER
