﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserRegistry.h>

#include "testAccount_Module.h"

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_USER_REGISTRY_BASE

#if defined(NNT_ACCOUNT_ENABLE_USER_REGISTRY_BASE)

// 初期状態の検査
TEST(AccountUser, UserRegistry_InitialState)
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    a::Uid users[a::UserCountMax];

    auto count = pManager->GetUserCount();
    EXPECT_EQ(0, count);

    std::memset(users, 0xFF, sizeof(users));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(
        pManager->ListAllUsers(users, sizeof(users) / sizeof(users[0])));
    for (auto& u: users)
    {
        EXPECT_FALSE(u);
    }
}

// ユーザーの並び替え機能
TEST(AccountUser, UserRegistry_Relocation)
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    // 登録
    a::Uid users[a::UserCountMax];
    for (auto& u: users)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->BeginUserRegistration(&u));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->CompleteUserRegistration(u));
    }

    a::Uid before[a::UserCountMax];
    a::Uid after[a::UserCountMax];
    auto relocate = [&](int from, int to)-> nn::Result {
        NNT_ACCOUNT_RESULT_DO(
            pManager->ListAllUsers(before, a::UserCountMax));
        NNT_ACCOUNT_RESULT_DO(
            pManager->SetUserPosition(before[from], to));
        NNT_ACCOUNT_RESULT_DO(
            pManager->ListAllUsers(after, a::UserCountMax));
        NN_RESULT_SUCCESS;
    };
    auto validate = [&](const int* E, size_t length)-> bool {
        NN_ASSERT(length == a::UserCountMax);
        for (auto i = 0u; i < length; ++ i)
        {
            NNT_ACCOUNT_RETURN_FALSE_UNLESS(after[i] == before[E[i]]);
        }
        return true;
    };

    // - 先頭 -> 2 番目
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(0, 1));
    {
        // after[i] == before[Expected[i]] を期待する。
        const int Expected[] = {1, 0, 2, 3, 4, 5, 6, 7};
        static_assert(sizeof(Expected) / sizeof(Expected[0]) == a::UserCountMax, "Invalid test data");
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 先頭 -> 末尾のひとつ前
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(0, a::UserCountMax - 2));
    {
        const int Expected[] = {1, 2, 3, 4, 5, 6, 0, 7};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 先頭 -> 末尾
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(0, a::UserCountMax - 1));
    {
        const int Expected[] = {1, 2, 3, 4, 5, 6, 7, 0};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 2番目 -> 先頭
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(1, 0));
    {
        const int Expected[] = {1, 0, 2, 3, 4, 5, 6, 7};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 2番目 -> 末尾のひとつ前
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(1, a::UserCountMax - 2));
    {
        const int Expected[] = {0, 2, 3, 4, 5, 6, 1, 7};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 2番目 -> 末尾
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(1, a::UserCountMax - 1));
    {
        const int Expected[] = {0, 2, 3, 4, 5, 6, 7, 1};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾のひとつ前 -> 先頭
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 2, 0));
    {
        const int Expected[] = {6, 0, 1, 2, 3, 4, 5, 7};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾のひとつ前 -> 2番目
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 2, 1));
    {
        const int Expected[] = {0, 6, 1, 2, 3, 4, 5, 7};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾のひとつ前 -> 末尾
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 2, a::UserCountMax - 1));
    {
        const int Expected[] = {0, 1, 2, 3, 4, 5, 7, 6};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾 -> 先頭
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 1, 0));
    {
        const int Expected[] = {7, 0, 1, 2, 3, 4, 5, 6};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾 -> 2番目
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 1, 1));
    {
        const int Expected[] = {0, 7, 1, 2, 3, 4, 5, 6};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
    // - 末尾 -> 末尾のひとつ前
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(relocate(a::UserCountMax - 1, a::UserCountMax - 2));
    {
        const int Expected[] = {0, 1, 2, 3, 4, 5, 7, 6};
        ASSERT_TRUE(validate(Expected, a::UserCountMax));
    }
} // NOLINT(readability/fn_size)

// ユーザー追加, 状態変化の通知
TEST(AccountUser, UserRegistry_Notification)
{
    struct Notifier
    {
        nn::os::SystemEventType* e;
        Notifier() NN_NOEXCEPT
            : e(nullptr)
        {
        }

        bool TryWait() NN_NOEXCEPT
        {
            return nn::os::TryWaitSystemEvent(e);
        }
        void Clear() NN_NOEXCEPT
        {
            nn::os::ClearSystemEvent(e);
        }
        bool TryWaitAndClear() NN_NOEXCEPT
        {
            NN_UTIL_SCOPE_EXIT
            {
                Clear();
            };
            return TryWait();
        }
        NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
        {
            return e != nullptr;
        }
    } nfrs[a::UserCountMax];

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    const int TestCount = 1000;
    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    // Reg の Notifier を取得する
    auto initNotifier = [&](Notifier* pNfr)->bool {
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->GetUserRegistrationEvent(&pNfr->e));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(*pNfr);
        return true;
    };

    // Notifier を何も指していない状態にする
    auto resetNotifier = [&](Notifier* pNfr)->void {
        pManager->ReleaseSystemEvent(pNfr->e);
        pNfr->e = nullptr;
    };

    // Notifier を与え、各イベントで適切にシグナルされるかを評価する
    auto signalAndCheckAll = [&](int count)->bool {
        // すべての Notifier について、指定された条件で通知されているかを評価する。
        auto checkAll = [&](int *pOutCount, bool signaled, bool clear)->bool {
            int count = 0;
            NN_UTIL_SCOPE_EXIT
            {
                *pOutCount = count;
            };
            for (auto& n: nfrs)
            {
                if (n)
                {
                    NNT_ACCOUNT_RETURN_FALSE_UNLESS((clear? n.TryWaitAndClear(): n.TryWait()) == signaled);
                    count ++;
                }
            }
            return true;
        };

        int signaledCount = 0;
        a::Uid user;

        // ユーザーの登録開始では何もシグナルされない
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->BeginUserRegistration(&user));
        EXPECT_TRUE(checkAll(&signaledCount, false, true));
        EXPECT_EQ(count, signaledCount);

        // ユーザーの登録完了でシグナルされる
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->CompleteUserRegistration(user));
        EXPECT_TRUE(checkAll(&signaledCount, true, true));
        EXPECT_EQ(count, signaledCount);

        // ユーザーの削除でシグナルされる
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(pManager->DeleteUser(user));
        EXPECT_TRUE(checkAll(&signaledCount, true, false)); // あえてクリアしない
        EXPECT_EQ(count, signaledCount);
        return true;
    };

    // RegistrationNotifier x1
    for (auto i = 0u; i < TestCount; ++ i)
    {
        ASSERT_TRUE(initNotifier(&nfrs[0]));
        EXPECT_TRUE(signalAndCheckAll(1));
        resetNotifier(&nfrs[0]);
    }
    // RegistrationNotifier のみ Max
    for (auto i = 0u; i < TestCount; ++ i)
    {
        const int Count = a::detail::UserRegistryNotifierCountMax;
        for (int ct = 0; ct < Count; ++ ct)
        {
            ASSERT_TRUE(initNotifier(&nfrs[ct]));
        }
        EXPECT_TRUE(signalAndCheckAll(Count));
        for (int ct = 0; ct < Count; ++ ct)
        {
            resetNotifier(&nfrs[ct]);
        }
    }
} // NOLINT(readability/fn_size)

// API 事前条件違反
TEST(AccountUser, UserRegistry_BadInput)
{
    // TODO 各種異常系 (事前条件違反) あとでやる
}

#endif // NNT_ACCOUNT_ENABLE_USER_REGISTRY_BASE
