﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_SharedCounter.h>
#include <nn/account/user/account_UserRegistry.h>

#include "testAccount_Module.h"

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

namespace a = nn::account;
namespace t = nnt::account;

TEST(AccountSharedCounter, Basic)
{
    a::detail::Counter ct;
    ASSERT_EQ(0, ct.Get());
    {
        a::detail::SharedCounter sct(ct);
        ASSERT_TRUE(sct);
        ASSERT_EQ(1, ct.Get());

        a::detail::SharedCounter sct1(sct);
        ASSERT_TRUE(sct);
        ASSERT_TRUE(sct1);
        ASSERT_EQ(2, ct.Get());

        a::detail::SharedCounter sct2(std::move(sct1));
        ASSERT_TRUE(sct);
        ASSERT_FALSE(sct1);
        ASSERT_TRUE(sct2);
        ASSERT_EQ(2, ct.Get());

        sct1 = sct2;
        ASSERT_TRUE(sct);
        ASSERT_TRUE(sct1);
        ASSERT_TRUE(sct2);
        ASSERT_EQ(3, ct.Get());

        sct1 = std::move(sct2);
        ASSERT_TRUE(sct);
        ASSERT_TRUE(sct1);
        ASSERT_FALSE(sct2);
        ASSERT_EQ(2, ct.Get());

        sct1.Release();
        ASSERT_TRUE(sct);
        ASSERT_FALSE(sct1);
        ASSERT_EQ(1, ct.Get());

        a::detail::SharedCounter sct3 = ct;
        ASSERT_TRUE(sct);
        ASSERT_TRUE(sct3);
        ASSERT_EQ(2, ct.Get());
    }
    ASSERT_EQ(0, ct.Get());
}

TEST(AccountUser, UserRef_Basic)
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pRegistry = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pRegistry->Initialize(s));

    a::Uid user;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pRegistry->BeginUserRegistration(&user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pRegistry->CompleteUserRegistration(user));

    a::user::UserRef ref;
    ASSERT_FALSE(ref.IsValid());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pRegistry->GetUserRef(&ref, user));
    ASSERT_TRUE(ref.IsValid());
    ASSERT_EQ(2, ref.GetCount());

    // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));

    a::user::UserRef ref1 = ref;
    ASSERT_TRUE(ref.IsValid());
    ASSERT_TRUE(ref1.IsValid());
    ASSERT_EQ(3, ref.GetCount());
    // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));

    {
        a::user::UserRef ref2 = ref1;
        ASSERT_TRUE(ref.IsValid());
        ASSERT_TRUE(ref1.IsValid());
        ASSERT_TRUE(ref2.IsValid());
        ASSERT_EQ(4, ref.GetCount());
        // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));
    }
    ASSERT_TRUE(ref.IsValid());
    ASSERT_TRUE(ref1.IsValid());
    ASSERT_EQ(3, ref.GetCount());

    ref1.Release();
    ASSERT_TRUE(ref.IsValid());
    ASSERT_FALSE(ref1.IsValid());
    ASSERT_EQ(2, ref.GetCount());
    // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));

    ref1 = std::move(ref);
    ASSERT_FALSE(ref.IsValid());
    ASSERT_TRUE(ref1.IsValid());
    ASSERT_EQ(2, ref1.GetCount());
    // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));

    ref = std::move(ref1);
    ASSERT_TRUE(ref.IsValid());
    ASSERT_FALSE(ref1.IsValid());
    ASSERT_EQ(2, ref.GetCount());
    // NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultUnexpectedUserState, pRegistry->DeleteUser(user));

    ref.Release();
    ASSERT_FALSE(ref.IsValid());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pRegistry->DeleteUser(user));
}
