﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/profile/account_ProfileStorage.h>
#include "profile/account_SaveDataAccessor.h"

#include "testAccount_Module.h"

#include <nn/account/account_Types.h>

#include "detail/account_UuidUtil.h"
#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"

#include <cstdlib>
#include <type_traits>

#include <nn/nn_Log.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>

namespace a = nn::account;
namespace p = nn::account::profile;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_STORAGE_SINGLE
#define NNT_ACCOUNT_ENABLE_STORAGE_MANY
#define NNT_ACCOUNT_ENABLE_STORAGE_OUT_OF_ORDER
#define NNT_ACCOUNT_ENABLE_PROFILE_SAVEDATA
#define NNT_ACCOUNT_ENABLE_DIGEST

namespace
{
// const uint16_t EmptyNickname[a::NicknameLengthMax] = {0x0000};
const char EmptyUserData[a::UserDataBytesMax] = {0x00};

char g_Nicknames[a::UserCountMax][a::NicknameBytesMax];
char g_UserDatas[a::UserCountMax][a::UserDataBytesMax];
uint64_t g_TimeStamps[a::UserCountMax];
char g_Images[a::UserCountMax][128];

struct TestDataInitializer
{
    TestDataInitializer() NN_NOEXCEPT
    {
        for (auto i = 0; i < a::UserCountMax; ++ i)
        {
            for (auto j = 0; j < static_cast<int>(a::NicknameBytesMax); ++ j)
            {
                auto u8 = static_cast<uint8_t>(((i & 0xF) << 4) | (j & 0xF));
                g_Nicknames[i][j] = (j > i? '\0': (0x3B ^ u8)); // 最初は文字数少な目
            }
            for (auto j = 0; j < static_cast<int>(a::UserDataBytesMax); ++ j)
            {
                auto u8 = static_cast<uint8_t>(((i & 0xF) << 4) | (j & 0xF));
                g_UserDatas[i][j] = u8;
            }
            g_TimeStamps[i] = static_cast<uint64_t>(nn::os::GetSystemTick().GetInt64Value());
            std::memset(g_Images[i], i * 7, sizeof(g_Images[i]));
        }
    }
} g_TestDataInitializer;

// 指定されたユーザーのプロフィールが空であることの確認
typedef std::shared_ptr<a::profile::ProfileStorage> StorageType;
bool CheckEmpty(StorageType storage, const a::Uid& uid) NN_NOEXCEPT
{
    bool isSuccess = false;
    [&]()-> void {
        a::profile::ProfileBase base;
        a::profile::UserData userData;
        size_t size;

        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&base, &userData, uid));
        EXPECT_TRUE(base);
        EXPECT_EQ(uid, base.author);
        EXPECT_EQ(0ull, base.timeStamp);
        EXPECT_EQ(0u, a::profile::GetNicknameLength(base.nickname, sizeof(base.nickname)));
        EXPECT_EQ(0, memcmp(EmptyUserData, userData.data, sizeof(EmptyUserData)));
        NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, storage->GetImageSize(&size, uid));
        NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, storage->LoadImage(&size, nullptr, 0, uid));

        isSuccess = true;
    }();
    return isSuccess;
}

// 指定されたユーザーのプロフィールが存在しないことの確認
bool CheckDeleted(StorageType storage, const a::Uid& uid) NN_NOEXCEPT
{
    bool isSuccess = false;
    [&]()-> void {
        a::profile::ProfileBase base;
        a::profile::UserData userData;
        size_t size;

        NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, storage->GetProfile(&base, &userData, uid));
        NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, storage->GetImageSize(&size, uid));
        NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultUserNotExist, storage->LoadImage(&size, nullptr, 0, uid));

        isSuccess = true;
    }();
    return isSuccess;
}

// 指定されたユーザーのプロフィールが、指定されたインデクスのテストデータと一致するかの確認
bool CheckModified(StorageType storage, const a::Uid& uid, int testDataIndex) NN_NOEXCEPT
{
    bool isSuccess = false;
    [&]()-> void {
        a::profile::ProfileBase base;
        a::profile::UserData userData;
        size_t size;

        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetProfile(&base, &userData, uid));
        EXPECT_TRUE(base);
        EXPECT_EQ(uid, base.author);
        EXPECT_EQ(g_TimeStamps[testDataIndex], base.timeStamp);
        EXPECT_EQ(0, std::strncmp(g_Nicknames[testDataIndex], base.nickname, sizeof(g_Nicknames[testDataIndex])));
        EXPECT_EQ(0, memcmp(g_UserDatas[testDataIndex], userData.data, sizeof(g_UserDatas[testDataIndex])));
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->GetImageSize(&size, uid));
        EXPECT_EQ(sizeof(g_Images[testDataIndex]), size);

        size = 0;
        char p[256];
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->LoadImage(&size, p, sizeof(p), uid));
        EXPECT_EQ(sizeof(g_Images[testDataIndex]), size);
        EXPECT_EQ(0, std::memcmp(g_Images[testDataIndex], p, size));

        isSuccess = true;
    }();
    return isSuccess;
}

}

#if defined(NNT_ACCOUNT_ENABLE_STORAGE_SINGLE)
void TestStorageSingle(const a::detail::AbstractLocalStorage& s)
{
    auto u = a::detail::ConvertToUid(s.GenerateUuidWithContext());

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(nullptr, 0, s));

    a::profile::ProfileBase update;
    a::profile::UserData updateUserData;

    // Check newly added empty profile
    storage->Add(u);
    EXPECT_TRUE(CheckEmpty(storage, u));
    // Delete
    storage->Delete(u);
    EXPECT_TRUE(CheckDeleted(storage, u));
    // Add
    storage->Add(u);
    EXPECT_TRUE(CheckEmpty(storage, u));

    // Modify profile
    update.author = u;
    update.timeStamp = g_TimeStamps[0];
    std::strncpy(update.nickname, g_Nicknames[0], sizeof(update.nickname));
    std::memcpy(updateUserData.data, g_UserDatas[0], sizeof(updateUserData.data));
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Update(u, update, updateUserData, g_Images[0], sizeof(g_Images[0])));

    // Check modify
    EXPECT_TRUE(CheckModified(storage, u, 0));

    // Delete+Add+Reset
    storage->Delete(u);
    storage->Add(u);
    EXPECT_TRUE(CheckEmpty(storage, u));
}

TEST(AccountProfile, ProfileStorage_Single)
{
    a::detail::LocalStorage<a::detail::DefaultFileSystem, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    TestStorageSingle(s);
};

TEST(AccountProfile, ProfileStorage_SingleOnRamFs)
{
    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    TestStorageSingle(s);
};
#endif

#if defined(NNT_ACCOUNT_ENABLE_STORAGE_MANY)
TEST(AccountProfile, ProfileStorage_ManyOnRamFs)
{
    const int TestCount = 1000;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::Uid users[a::UserCountMax];
    for (auto& u: users)
    {
        u = a::detail::ConvertToUid(s.GenerateUuidWithContext());
    }

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(users, a::UserCountMax, s));

    for (auto i = 0; i < TestCount; ++ i)
    {
        for (int j = 0; j < a::UserCountMax; ++j)
        {
            const auto& u = users[j];
            a::profile::ProfileBase update;
            a::profile::UserData updateUserData;

            // Check newly added empty profile
            EXPECT_TRUE(CheckEmpty(storage, u));
            // Delete
            storage->Delete(u);
            EXPECT_TRUE(CheckDeleted(storage, u));
            // Add
            storage->Add(u);
            EXPECT_TRUE(CheckEmpty(storage, u));
            // Modify profile
            update.author = u;
            update.timeStamp = g_TimeStamps[j];
            std::strncpy(update.nickname, g_Nicknames[j], sizeof(update.nickname));
            std::memcpy(updateUserData.data, g_UserDatas[j], sizeof(updateUserData.data));
            NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Update(u, update, updateUserData, g_Images[j], sizeof(g_Images[j])));
            // Check modify
            EXPECT_TRUE(CheckModified(storage, u, j));
            // Delete+Add+Reset
            storage->Delete(u);
            storage->Add(u);
            EXPECT_TRUE(CheckEmpty(storage, u));
        }
    }
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_STORAGE_OUT_OF_ORDER)
TEST(AccountProfile, ProfileStorage_OutOfOrderOnRamFs)
{
    const int TestCount = 1000;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::Uid users[a::UserCountMax];
    for (auto& u: users)
    {
        u = a::detail::ConvertToUid(s.GenerateUuidWithContext());
    }

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(users, a::UserCountMax, s));

    for (auto i = 0; i < TestCount; ++ i)
    {
        for (auto j = 0; j < a::UserCountMax; ++ j)
        {
            // Delete a::UserCountMax - (j + 1)
            const auto Head = j;
            for (auto k = Head; k < a::UserCountMax - Head; ++ k)
            {
                auto& u = users[k];
                EXPECT_TRUE(CheckEmpty(storage, u));
                storage->Delete(u);
            }

            // Add a::UserCountMax - j
            for (auto k = Head; k < a::UserCountMax - Head; ++ k)
            {
                auto& u = users[k];
                EXPECT_TRUE(CheckDeleted(storage, u));
                storage->Add(u);
                EXPECT_TRUE(CheckEmpty(storage, u));
                // Modify
                a::profile::ProfileBase update;
                a::profile::UserData updateUserData;
                update.author = u;
                update.timeStamp = g_TimeStamps[k];
                std::strncpy(update.nickname, g_Nicknames[k], sizeof(update.nickname));
                std::memcpy(updateUserData.data, g_UserDatas[k], sizeof(updateUserData.data));
                NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Update(u, update, updateUserData, g_Images[k], sizeof(g_Images[k])));
            }

            // Delete a::UserCountMax - (j + 1)
            for (auto k = a::UserCountMax - Head - 1; k >= Head; -- k)
            {
                auto& u = users[k];
                EXPECT_TRUE(CheckModified(storage, u, k));
                storage->Delete(u);
            }

            // Add a::UserCountMax - j
            for (auto k = a::UserCountMax - Head - 1; k >= Head; -- k)
            {
                auto& u = users[k];
                EXPECT_TRUE(CheckDeleted(storage, u));
                storage->Add(u);
            }
        }
    }
}
#endif

#if defined(NNT_ACCOUNT_ENABLE_PROFILE_SAVEDATA)

TEST(AccountProfile, ProfileStorage_SaveDataAccessor)
{
    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    NN_STATIC_ASSERT(std::is_pod<a::profile::Profile>::value);
    a::profile::Profile profiles[a::UserCountMax];

    for (auto i = 0; i < sizeof(profiles) / sizeof(profiles[0]); ++ i)
    {
        auto& pf = profiles[i];
        pf.user = a::detail::ConvertToUid(s.GenerateUuidWithContext());

        char nickname[a::NicknameBytesMax];
        std::memset(nickname, 'A' + i, sizeof(nickname));
        a::profile::ProfileBase base = {
            a::detail::ConvertToUid(s.GenerateUuidWithContext()),
            static_cast<uint64_t>(nn::os::GetSystemTick().ToTimeSpan().GetSeconds()),
            {}
        };
        std::memcpy(base.nickname, nickname, sizeof(nickname));
        a::profile::UserData userData;
        std::memset(userData.data, 'z' - i, sizeof(userData.data));
        pf.base = base;
    }

    {
        t::Buffer buffer(1024);
        auto lock = s.AcquireWriterLock();
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(a::profile::SaveDataAccessor::Store(
            profiles, sizeof(profiles) / sizeof(profiles[0]), buffer.GetAddress(), buffer.GetSize(), s));
    }

    a::profile::Profile loaded[a::UserCountMax];
    {
        t::Buffer buffer(1024);
        NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(a::profile::SaveDataAccessor::Load(
            loaded, sizeof(loaded) / sizeof(loaded[0]), buffer.GetAddress(), buffer.GetSize(), s));

    }
    for (auto i = 0; i < sizeof(loaded) / sizeof(loaded[0]); ++ i)
    {
        const auto& pf = profiles[i];
        const auto& ld = loaded[i];
        EXPECT_EQ(pf.user, ld.user);
        EXPECT_EQ(pf.base.author, ld.base.author);
        EXPECT_EQ(pf.base.timeStamp, ld.base.timeStamp);
        EXPECT_EQ(0, std::memcmp(pf.base.nickname, ld.base.nickname, sizeof(pf.base.nickname)));
        EXPECT_EQ(0, std::memcmp(pf.userData.data, ld.userData.data, sizeof(pf.userData.data)));
        EXPECT_EQ(pf, ld);
    }
}

#endif // NNT_ACCOUNT_ENABLE_PROFILE_SAVEDATA


#if defined(NNT_ACCOUNT_ENABLE_DIGEST)
TEST(AccountProfile, ProfileStorage_DIGEST)
{
    const int TestCount = 100;

    a::detail::LocalStorage<t::RamFs, t::HostSaveData<>::Policy> s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::Uid users[a::UserCountMax];
    for (auto& u: users)
    {
        u = a::detail::ConvertToUid(s.GenerateUuidWithContext());
    }

    auto storage = t::CreateProfileStorage();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(storage->Initialize(users, a::UserCountMax, s));

    for (auto i = 0; i < a::UserCountMax; ++ i)
    {
        std::unique_ptr<a::ProfileDigest[]> history(new a::ProfileDigest[TestCount]);

        for (auto j = 0; j < TestCount; ++ j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(storage->GetDigest(&history[j], users[i]));
            EXPECT_EQ(history[j], history[j]);
            for (auto k = 0; k < j; ++ k)
            {
                EXPECT_NE(history[k], history[j]);
            }

            auto base = a::profile::DefaultProfileBase;
            base.author = users[i];
            base.timeStamp = static_cast<uint64_t>(nn::os::GetSystemTick().ToTimeSpan().GetSeconds());
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(storage->Update(users[i], base, a::profile::DefaultUserData, nullptr, 0));
        }
    }
}
#endif
