﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/json/account_JsonAdaptor.h>

#include "testAccount_JsonPrintAdaptor.h"

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

struct Expect;

template <int Depth, int Length, bool FailOnDepthExcess>
class AdaptorForTest
{
protected:
    typedef nn::account::json::JsonPath<Depth, Length, FailOnDepthExcess> JsonPathType;

    virtual const Expect* Get(int index) NN_NOEXCEPT = 0;
    virtual int GetExpectCount() const NN_NOEXCEPT = 0;

private:
    int m_Index;
    bool m_Success;

    const Expect* GetNext() NN_NOEXCEPT
    {
        if (!(m_Index < GetExpectCount()))
        {
            m_Success = false;
            return nullptr;
        }
        return Get(m_Index ++);
    }

public:
    AdaptorForTest() NN_NOEXCEPT
        : m_Index(0)
        , m_Success(true)
    {
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_Success && m_Index == GetExpectCount();
    }
    void Update(const JsonPathType& jsonPath, std::nullptr_t) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, uint64_t value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, double value) NN_NOEXCEPT;
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT;
};

struct Expect
{
    // Path
    const char* path;
    // Kind
    enum ValueKind
    {
        Null, Bool, Integer, Uint64, Numeric, String,
    } kind;
    union
    {
        bool b;
        int64_t i;
        uint64_t u64;
        struct
        {
            double min;
            double max;
        } n;
        char s[128];
    } u;
    Expect() NN_NOEXCEPT
        : path(""), kind(Null)
    {
    }
    Expect(const Expect& rhs) NN_NOEXCEPT
        : path(rhs.path), kind(rhs.kind)
    {
        switch (kind)
        {
        case Null:
            break;
        case Bool:
            u.b = rhs.u.b;
            break;
        case Integer:
            u.i = rhs.u.i;
            break;
        case Uint64:
            u.u64 = rhs.u.u64;
            break;
        case Numeric:
            u.n = rhs.u.n;
            break;
        case String:
            std::strncpy(u.s, rhs.u.s, sizeof(u.s));
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    Expect(const char* p, std::nullptr_t) NN_NOEXCEPT
        : path(p), kind(Null)
    {
    }
    Expect(const char* p, bool v) NN_NOEXCEPT
        : path(p), kind(Bool)
    {
        u.b = v;
    }
    Expect(const char* p, int v) NN_NOEXCEPT
        : path(p), kind(Integer)
    {
        u.i = v;
    }
    Expect(const char* p, uint64_t u64) NN_NOEXCEPT
        : path(p), kind(Uint64)
    {
        u.u64 = u64;
    }
    Expect(const char* p, double min, double max) NN_NOEXCEPT
        : path(p), kind(Numeric)
    {
        u.n.min = min;
        u.n.max = max;
    }
    Expect(const char* p, const char* s) NN_NOEXCEPT
        : path(p), kind(String)
    {
        kind = String;
        strcpy(u.s, s);
    }
    static const char* ValueKindToStr(ValueKind k) NN_NOEXCEPT
    {
        switch (k)
        {
        case Null:
            return "Null";
        case Bool:
            return "Bool";
        case Integer:
            return "Integer";
        case Uint64:
            return "Uint64";
        case Numeric:
            return "Numeric";
        case String:
            return "String";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

inline void PrintPathMismatch(const char* e, const char* a) NN_NOEXCEPT
{
    NN_LOG("Invalid Path\n");
    NN_LOG("  Expected: %s\n", e);
    NN_LOG("  Actual:   %s\n", a);
}
inline void PrintValueKindMismatch(const char* path, Expect::ValueKind e, Expect::ValueKind a) NN_NOEXCEPT
{
    NN_LOG("Invalid ValueKind at % s\n", path);
    NN_LOG("  Expected: %s\n", Expect::ValueKindToStr(e));
    NN_LOG("  Actual:   %s\n", Expect::ValueKindToStr(a));
}

template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, std::nullptr_t) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::Null))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::Null);
        success = false;
    }
    m_Success = (m_Success && success);
}
template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, bool value) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::Bool))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::Bool);
        success = false;
    }
    if (!(pE->u.b == value))
    {
        NN_LOG("Invalid Value at %s\n", path);
        NN_LOG("  Expected: %s\n", pE->u.b? "true": "false");
        NN_LOG("  Actual:   %s\n", value? "true": "false");
        success = false;
    }
    m_Success = (m_Success && success);
}
template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::Integer))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::Integer);
        success = false;
    }
    if (!(pE->u.i == value))
    {
        NN_LOG("Invalid Value at %s\n", path);
        NN_LOG("  Expected: %lld\n", pE->u.i);
        NN_LOG("  Actual:   %lld\n", value);
        success = false;
    }
    m_Success = (m_Success && success);
}
template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, uint64_t value) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::Uint64))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::Uint64);
        success = false;
    }
    if (!(pE->u.u64 == value))
    {
        NN_LOG("Invalid Value at %s\n", path);
        NN_LOG("  Expected: %llu\n", pE->u.u64);
        NN_LOG("  Actual:   %llu\n", value);
        success = false;
    }
    m_Success = (m_Success && success);
}
template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, double value) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::Numeric))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::Numeric);
        success = false;
    }
    if (!(pE->u.n.min <= value && value <= pE->u.n.max))
    {
        NN_LOG("Invalid Value at %s\n", path);
        NN_LOG("  Expected: [%.3lf, %.3lf]\n", pE->u.n.min, pE->u.n.max);
        NN_LOG("  Actual:   %lf\n", value);
        success = false;
    }
    m_Success = (m_Success && success);
}
template <int Depth, int Length, bool FailOnDepthExcess>
inline void AdaptorForTest<Depth, Length, FailOnDepthExcess>::Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    size_t length;
    auto path = jsonPath.ToString(&length);

    auto pE = GetNext();
    if (!pE) {NN_LOG("%s: skipped\n", path); return;}

    bool success = true;
    if (!(strlen(path) == length && strncmp(pE->path, path, length + 1) == 0))
    {
        PrintPathMismatch(pE->path, path);
        success = false;
    }
    if (!(pE->kind == Expect::String))
    {
        PrintValueKindMismatch(path, pE->kind, Expect::String);
        success = false;
    }
    if (!(static_cast<int>(strlen(value)) == valueLength && strncmp(pE->u.s, value, valueLength + 1) == 0))
    {
        NN_LOG("Invalid Value at %s\n", path);
        NN_LOG("  Expected: \"%s\"(%d)\n", pE->u.s, strlen(pE->u.s));
        NN_LOG("  Actual:   \"%s\"(%d)\n", value, valueLength);
        success = false;
    }
    m_Success = (m_Success && success);
}
