﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/account/account_IAccountService.sfdl.h>
#include <nn/account/account_ServiceNames.h>

#include <type_traits>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>
#include <nn/sf/sf_IServiceObject.h>

#include <nnt.h>
#include <nnt/nntest.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

namespace
{
struct InitializationManager
{
    uint32_t _counter;
    nn::os::MutexType _mutex;

    InitializationManager() NN_NOEXCEPT
        : _counter(0)
    {
        nn::os::InitializeMutex(&_mutex, false, 0);
    }

    NN_EXPLICIT_OPERATOR bool() NN_NOEXCEPT
    {
        std::lock_guard<InitializationManager> lock(*this);
        return _counter > 0;
    }
    void Initialize(std::function<void()> initializer) NN_NOEXCEPT
    {
        std::lock_guard<decltype(*this)> lock(*this);
        if (_counter == 0)
        {
            initializer();
        }
        ++ _counter;
    }
    void Finalize(std::function<void()> finalizer) NN_NOEXCEPT
    {
        std::lock_guard<decltype(*this)> lock(*this);
        if (_counter == 1)
        {
            finalizer();
        }
        -- _counter;
    }
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&_mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&_mutex);
    }
};

struct ObjectHolderAllocatorTag {};
static const size_t SizeOfHeapToAcquireObject = 1024 * 2;
typedef nn::sf::ExpHeapStaticAllocator<SizeOfHeapToAcquireObject, ObjectHolderAllocatorTag> AllocatorType;

struct AllocatorInitializer
{
    AllocatorInitializer() NN_NOEXCEPT
    {
        AllocatorType::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_AllocatorInitialzier;

class ObjectHolder
{
private:
    nn::account::IAccountServiceBase* m_pBase;

    nn::sf::HipcSimpleClientSessionManager m_HipcDomain;

    template <typename Interface>
    nn::Result AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT
    {
        nn::sf::SharedPointer<Interface> ptr;
        NN_RESULT_DO((m_HipcDomain.InitializeByName<Interface, AllocatorType::Policy>(&ptr, serviceName)));

        *ppOutService = ptr.Detach();
        NN_ABORT_UNLESS(*ppOutService != nullptr, "[nn::account] Initialization failure while connecting.\n");
        NN_RESULT_SUCCESS;
    }

public:
    ObjectHolder() NN_NOEXCEPT
        : m_pBase(nullptr)
    {
    }
    ~ObjectHolder() NN_NOEXCEPT
    {
        Release();
    }

    nn::Result AcquireForApplication() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_pBase);
        nn::account::IAccountServiceForApplication* pForApplication;
        NN_RESULT_DO(AcquireImpl(&pForApplication, nn::account::ServiceNameForApplication));
        m_pBase = pForApplication;
        NN_RESULT_SUCCESS;
    }
    nn::Result AcquireForSystemService() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_pBase);
        nn::account::IAccountServiceForSystemService* pForSystemService;
        NN_RESULT_DO(AcquireImpl(&pForSystemService, nn::account::ServiceNameForSystemService));
        m_pBase = pForSystemService;
        NN_RESULT_SUCCESS;
    }
    nn::Result AcquireForAdministrator() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_pBase);
        nn::account::IAccountServiceForAdministrator* pForAdministrator;
        NN_RESULT_DO(AcquireImpl(&pForAdministrator, nn::account::ServiceNameForAdministrator));
        m_pBase= pForAdministrator;
        NN_RESULT_SUCCESS;
    }

    void Release() NN_NOEXCEPT
    {
        if (m_pBase != nullptr)
        {
            nn::sf::ReleaseSharedObject(m_pBase);
            m_pBase = nullptr;
            m_HipcDomain.Finalize();
        }
    }

    nn::account::IAccountServiceBase& GetBase() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_pBase);
        return *m_pBase;
    }
};

class InitializationTest
{
private:
    InitializationManager m_InitMgr;
    ObjectHolder m_Obj;
    int m_Count;

    void InitializeForApplicationImpl() NN_NOEXCEPT
    {
        return m_InitMgr.Initialize([&]()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Obj.AcquireForApplication());
        });
    }
    void InitializeForSystemServiceImpl() NN_NOEXCEPT
    {
        return m_InitMgr.Initialize([&]()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Obj.AcquireForSystemService());
        });
    }
    void InitializeForAdministratorImpl() NN_NOEXCEPT
    {
        return m_InitMgr.Initialize([&]()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_Obj.AcquireForAdministrator());
        });
    }
    void FinalizeImpl() NN_NOEXCEPT
    {
        m_InitMgr.Finalize([&] { m_Obj.Release(); });
    }

public:
    enum Mode
    {
        Mode_Application,
        Mode_Administrator,
        Mode_SystemService
    };
    InitializationTest(int count, Mode mode) NN_NOEXCEPT
        : m_Count(count)
    {
        NN_SDK_REQUIRES(count > 0);


        for (int i = 0; i < m_Count; ++ i)
        {
            switch (mode)
            {
            case Mode_Application:
                InitializeForApplicationImpl();
                break;
            case Mode_Administrator:
                InitializeForAdministratorImpl();
                break;
            case Mode_SystemService:
                InitializeForSystemServiceImpl();
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
    ~InitializationTest() NN_NOEXCEPT
    {
        for (int i = 0; i < m_Count; ++ i)
        {
            FinalizeImpl();
        }
    }
};
} // ~namespace anonymous

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    testing::InitGoogleTest(&argc, argv);
    auto& listeners = testing::UnitTest::GetInstance()->listeners();
    auto* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    auto result = RUN_ALL_TESTS();
    nnt::Exit(result);
}

TEST(AccountInit, Initialization)
{
    const int ApplicationCount = 2;
    const int AdministratorCount = 4;
    const int SystemServiceCount = 8;

    InitializationTest* pTestForApplication[ApplicationCount] = {};
    InitializationTest* pTestForAdministrator[AdministratorCount] = {};
    InitializationTest* pTestForSystemService[SystemServiceCount] = {};

    for (auto& pt : pTestForApplication)
    {
        pt = new InitializationTest(8, InitializationTest::Mode_Application);
    }
    for (auto& pt : pTestForAdministrator)
    {
        pt = new InitializationTest(8, InitializationTest::Mode_Administrator);
    }
    for (auto& pt : pTestForSystemService)
    {
        pt = new InitializationTest(8, InitializationTest::Mode_SystemService);
    }

    for (auto& pt : pTestForApplication)
    {
        delete pt;
    }
    for (auto& pt : pTestForAdministrator)
    {
        delete pt;
    }
    for (auto& pt : pTestForSystemService)
    {
        delete pt;
    }
}
