﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nas/account_NasProblemDetails.h"
#include "nas/account_NasLoginAdaptor.h"

#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/account/account_ResultPrivate.h>

#include "testAccount_Mounter.h"
#include "testAccount_RapidJsonInputStream.h"
#include "testAccount_Util.h"

#include <nn/result/result_HandlingUtility.h>

namespace a = nn::account;
namespace t = nnt::account;

namespace nasAuthorization {
class AuthorizationErrorAdaptor final
    : public a::nas::NasAuthorizationErrorAdaptor
{
private:
    static const char RedirectUri[a::detail::NasCallbackUriLengthMax + 1];
    a::nas::State m_State;

protected:
    virtual void UpdateImplImpl(const char*, size_t, const char*, size_t) NN_NOEXCEPT NN_OVERRIDE
    {
    }
    virtual nn::Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
public:
    explicit AuthorizationErrorAdaptor(const char* state) NN_NOEXCEPT
        : a::nas::NasAuthorizationErrorAdaptor(RedirectUri, m_State)
    {
        std::strncpy(m_State.data, state, sizeof(m_State.data));
    }
};
const char AuthorizationErrorAdaptor::RedirectUri[a::detail::NasCallbackUriLengthMax + 1] = "";

template <typename ResultType, size_t DocumentSize>
bool TestCase(const char (&document)[DocumentSize], const char* state) NN_NOEXCEPT
{
    AuthorizationErrorAdaptor adaptor(state);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(adaptor.Parse(document, sizeof(document)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(ResultType, adaptor.Adapt());
    return true;
}

template <size_t DocumentSize>
bool TestSuccessCase(const char (&document)[DocumentSize], const char* state) NN_NOEXCEPT
{
    AuthorizationErrorAdaptor adaptor(state);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(adaptor.Parse(document, sizeof(document)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Adapt());
    return true;
}

bool RunTest() NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusUnauthorizedClient>(
        "?state=xyz&error=unauthorized_client",
        "xyz"));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusUnauthorizedClient>(
        "#state=xyz&error=unauthorized_client&error_detail=banned",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDenied>(
        "?state=xyz&error=access_denied",
        "xyz"));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDenied>(
        "#state=xyz&error=access_denied&error_detail=user_terms_agreement_required1",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDenied>(
        "?state=xyz&error=access_denied&error_detail=user_insufficient_age",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid>(
        "?state=xyz&error=access_denied&error_detail=id_token_hint_invalid",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDenied>(
        "?state=xyz&error=access_denied&error_detail=user_terms_agreement_required",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDeniedUserDeleted>(
        "#state=xyz&error=access_denied&error_detail=user_deleted",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusAccessDenied>(
        "#state=xyz&error=access_denied&error_detail=user_terms_agreement_required&foo=bar",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInvalidScope>(
        "?state=xyz&error=invalid_scope",
        "xyz"));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInvalidScope>(
        "?state=xyz&error=invalid_scope&error_detail=scope_token_unknown ",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenUnknown>(
        "?state=xyz&error=invalid_scope&error_detail=scope_token_unknown",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInvalidScopeScopeTokenProhibited>(
        "?state=xyz&error=invalid_scope&error_detail=scope_token_prohibited",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusServerError>(
        "#state=xyz&error=server_error",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusLoginRequired>(
        "?state=xyz&error=login_required",
        "xyz"));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusLoginRequired>(
        "?state=xyz&error=login_required&error_detail=user_not_logged_in0",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusLoginRequiredUserNotLoggedIn>(
        "?state=xyz&error=login_required&error_detail=user_not_logged_in",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusLoginRequiredUserDifferentFromIdTokenHint>(
        "?state=xyz&error=login_required&error_detail=user_different_from_id_token_hint",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusConsentRequired>(
        "#state=xyz&error=consent_required",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequired>(
        "?state=xyz&error=interaction_required",
        "xyz"));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequired>(
        "?state=xyz&error=interaction_required&error_detail=user_banned1",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequiredUserBanned>(
        "#state=xyz&error=interaction_required&error_detail=user_banned",
        "xyz"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequiredUserBanned>(
        "?state=xyz&error=interaction_required&error_detail=user_banned&foo=bar",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequiredUserSuspended>(
        "#state=xyz&error=interaction_required&error_detail=user_suspended",
        "xyz"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequiredUserSuspended>(
        "?state=xyz&error=interaction_required&error_detail=user_suspended&foo=bar",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusInteractionRequiredUserTermsAgreementRequired>(
        "#state=xyz&error=interaction_required&error_detail=user_terms_agreement_required",
        "xyz"));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasAuthorizationStatusUnderMaintenance>(
        "#state=xyz&error=under_maintenance",
        "xyz"));

    // State エラー
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::ResultNasDataBroken>(
        "#foo=bar",
        "xyz"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::ResultNasDataBroken>(
        "?state=xyz",
        "xyz "));

    // 未知の error
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::ResultNasDataBroken>(
        "#state=xyz&error=foo",
        "xyz"));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::ResultNasDataBroken>(
        "?state=xyz&error=foo&error_detail=user_banned",
        "xyz"));

    // 正常
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestSuccessCase(
        "#state=xyz",
        "xyz"));

    return true;
}
} // ~namespace authorization

namespace nasTokenRequest {
class TokenRequestErrorAdaptor final
    : public a::nas::NasTokenRequestAdaptorBase<8, 32>
{
protected:
    virtual nn::Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
};

template <typename ResultType>
bool TestCase(const char *document, int httpCode) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(document, strlen(document));

    TokenRequestErrorAdaptor adaptor;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(a::json::ImportJsonByRapidJson(adaptor, stream, nullptr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(ResultType, adaptor.Adapt(httpCode));
    return true;
}

bool TestSuccessCase(const char *document, int httpCode) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(document, strlen(document));

    TokenRequestErrorAdaptor adaptor;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(a::json::ImportJsonByRapidJson(adaptor, stream, nullptr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Adapt(httpCode));
    return true;
}

bool RunTest() NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidRequest>(
        "{\"error\":\"invalid_request\"}", 400));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidRequest>(
        "{\"error\":\"invalid_request\",\"error_detail\":\"user_banned\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidClient>(
        "{\"error\":\"invalid_client\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrant>(
        "{\"error\":\"invalid_grant\"}", 400));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrant>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_banned \"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserDeleted>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_deleted\"}", 400));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserDeleted>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_deleted\",\"foo\":\"bar\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserBanned>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_banned\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserSuspended>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_suspended\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserWithdrawn>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_withdrawn\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidGrantUserTermsAgreementRequired>(
        "{\"error\":\"invalid_grant\",\"error_detail\":\"user_terms_agreement_required\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidScope>(
        "{\"error\":\"invalid_scope\"}", 400));
    // 未定義の error_detail
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidScope>(
        "{\"error\":\"invalid_scope\",\"error_detail\":\"scope_token_unknown_\"}}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidScopeScopeTokenUnknown>(
        "{\"error\":\"invalid_scope\",\"error_detail\":\"scope_token_unknown\"}}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidScopeScopeTokenProhibited>(
        "{\"error\":\"invalid_scope\",\"error_detail\":\"scope_token_prohibited\"}}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusInvalidScopeScopeTokenNotAuthorized>(
        "{\"error\":\"invalid_scope\",\"error_detail\":\"scope_token_not_authorized\"}}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusUnauthorizedClient>(
        "{\"error\":\"unauthorized_client\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusUnsupportedGrantType>(
        "{\"error\":\"unsupported_grant_type\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusServerError>(
        "{\"error\":\"server_error\"}", 504));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasTokenRequestStatusUnderMaintenance>(
        "{\"error\":\"under_maintenance\"}", 503));

    // 未知の error
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::http::ResultHttpStatus400BadRequest>(
        "{\"error\":\"foo\"}", 400));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::http::ResultHttpStatus400BadRequest>(
        "{}", 400));

    // 正常
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestSuccessCase(
        "{\"error\":\"invalid_request\"}", 200));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestSuccessCase(
        "{}", 200));

    return true;
}
} // ~namespace tokenRequest

namespace nasUserResource {
class ErrorAdaptor final
    : public a::nas::NasCommonAdaptor<8, 64>
{
protected:
    virtual nn::Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
};

template <typename ResultType>
bool TestCase(const char *document, int httpCode) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(document, strlen(document));

    ErrorAdaptor adaptor;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(a::json::ImportJsonByRapidJson(adaptor, stream, nullptr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(ResultType, adaptor.Adapt(httpCode));
    return true;
}

bool TestSuccessCase(const char *document, int httpCode) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(document, strlen(document));

    ErrorAdaptor adaptor;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(a::json::ImportJsonByRapidJson(adaptor, stream, nullptr));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(adaptor.Adapt(httpCode));
    return true;
}

bool RunTest() NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasUserResourceStatusInvalidToken>(
        "{\"error\":\"invalid_token\"}", 400));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasUserResourceStatusInvalidToken>(
        "{\"error\":\"invalid_token\",\"error_detail\":\"user_banned\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasUserResourceStatusInsufficientScope>(
        "{\"error\":\"insufficient_scope\"}", 400));

    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::nas::ResultNasUserResourceStatusUnderMaintenance>(
        "{\"error\":\"under_maintenance\"}", 503));

    // 未知の error
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::http::ResultHttpStatus400BadRequest>(
        "{\"error\":\"foo\"}", 400));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestCase<a::http::ResultHttpStatus400BadRequest>(
        "{}", 400));

    // 正常
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestSuccessCase(
        "{\"error\":\"invalid_token\"}", 200));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(TestSuccessCase(
        "{}", 200));

    return true;
}
} // ~namespace userResource


TEST(AccountAdaptor, NasProblemDetail)
{
    EXPECT_TRUE(nasAuthorization::RunTest());
    EXPECT_TRUE(nasTokenRequest::RunTest());
    EXPECT_TRUE(nasUserResource::RunTest());
}
