﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ndas/account_AuthenticationAdaptor.h"
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/ndas/account_ResultForNdas.h>

#include <nn/account/http/account_ResultForHttp.h>
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>

#include "testAccount_Mounter.h"
#include "testAccount_Util.h"
#include "testAccount_JsonPrintAdaptor.h"
#include "testAccount_RapidJsonInputStream.h"

#include <nnt/nntest.h>

#include <nn/nn_Log.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_APPLICATION_AUTH
#define NNT_ACCOUNT_ENABLE_EXPIRATION_APPLICATION_AUTH
#define NNT_ACCOUNT_ENABLE_ERROR_APPLICATION_AUTH

namespace
{
struct Error
{
    int32_t httpCode;
    const char* json;
};
} // ~namespace <anonymous>

#if defined(NNT_ACCOUNT_ENABLE_APPLICATION_AUTH)
namespace aauth
{

const char ApplicationAuthenticationResponse[] =
    "{"
        "\"expires_in\":86400,"
        "\"application_auth_token\":\"eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1Ni"
            "IsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2Iiwia2lkIjoiOGE0OTA5OTYtYzc3My00MWJkLTkwMDgtYz"
            "k1NjBjOWZlNjBmIn0.eyJzdWIiOiIxMjM0NTY3ODkiLCJpc3MiOiJhYXV0aC10ZDEubmRhcy5zcnYubm"
            "ludGVuZG8ubmV0IiwiZXhwIjoxNDUxOTgwOTA4LCJpYXQiOjE0NTE4OTQ1MDgsImp0aSI6ImFiMmRjZD"
            "M2LTQ4N2UtNGFlMC05Y2E3LWYwMmJiNjJhODQ0OCIsIm5pbnRlbmRvIjp7ImRpIjoiNjEwMDAwMDAwMD"
            "AwMDAyMiIsInNuIjoiWEFWMDAwMDAwMDAzNDUiLCJwYyI6IkhBQy1TLVpaWlpWKFVTWikiLCJkdCI6Ik"
            "5YIFZEZXYgMSJ9fQ.HE38ZZMGv-ZStY427rCY6-Tzx0ADWHA587VCGB43RYl3E_h5247iDGDg6tOLM5o"
            "gVt3NMfGV6zgfLE9n2MC9tTaGrCbt6Nl1GhtYlO8MVq9schh3a527uls9okLP7puWw6yrIvarnVEWjyl"
            "zyYzwbMEKHXybGdWYa_B-77chen4DubfXTajP45yS7RhjE66myy6xioEtZlrGncSikz11fNYhIzzLKJp"
            "07q2StErKyU3f91ZLKW8l6c3c5Z-ItVRVb_vZY23a9oekyr-M-4Q3auaWcmho_p35GjEMVDDlbElTHX8"
            "8ju15f4xiNUhwl1R6DekSn2_lMYALJYU2QOpONQ\""
    "}";
const char ApplicationAuthenticationToken[] = "eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1Ni"
"IsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2Iiwia2lkIjoiOGE0OTA5OTYtYzc3My00MWJkLTkwMDgtYz"
"k1NjBjOWZlNjBmIn0.eyJzdWIiOiIxMjM0NTY3ODkiLCJpc3MiOiJhYXV0aC10ZDEubmRhcy5zcnYubm"
"ludGVuZG8ubmV0IiwiZXhwIjoxNDUxOTgwOTA4LCJpYXQiOjE0NTE4OTQ1MDgsImp0aSI6ImFiMmRjZD"
"M2LTQ4N2UtNGFlMC05Y2E3LWYwMmJiNjJhODQ0OCIsIm5pbnRlbmRvIjp7ImRpIjoiNjEwMDAwMDAwMD"
"AwMDAyMiIsInNuIjoiWEFWMDAwMDAwMDAzNDUiLCJwYyI6IkhBQy1TLVpaWlpWKFVTWikiLCJkdCI6Ik"
"5YIFZEZXYgMSJ9fQ.HE38ZZMGv-ZStY427rCY6-Tzx0ADWHA587VCGB43RYl3E_h5247iDGDg6tOLM5o"
"gVt3NMfGV6zgfLE9n2MC9tTaGrCbt6Nl1GhtYlO8MVq9schh3a527uls9okLP7puWw6yrIvarnVEWjyl"
"zyYzwbMEKHXybGdWYa_B-77chen4DubfXTajP45yS7RhjE66myy6xioEtZlrGncSikz11fNYhIzzLKJp"
"07q2StErKyU3f91ZLKW8l6c3c5Z-ItVRVb_vZY23a9oekyr-M-4Q3auaWcmho_p35GjEMVDDlbElTHX8"
"8ju15f4xiNUhwl1R6DekSn2_lMYALJYU2QOpONQ";

void TestApplicationAuthentication(const char* token, const char* json) NN_NOEXCEPT
{
    const auto info = a::detail::ApplicationInfo::Get(1234567890123456, 1234, a::detail::ApplicationMediaType::GameCard);

    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    a::ndas::ApplicationAuthenticationCache cache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cache.Initialize(s));

    ASSERT_FALSE(cache.IsAvailable(info));

    a::ndas::ApplicationAuthenticationAdaptor adaptor(cache, info);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(-1));

    ASSERT_TRUE(cache.IsAvailable(info));

    t::Buffer buffer(4096);
    size_t sizeActual;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(cache.Load(&sizeActual, buffer.Get<char>(), buffer.GetSize(), info));
    ASSERT_EQ(strlen(token), sizeActual);
    ASSERT_EQ(0, std::memcmp(token, buffer.GetAddress(), sizeActual));
}

} // ~namespace aauth

TEST(AccountAdaptor, ApplicationAuthentication)
{
    aauth::TestApplicationAuthentication(
        aauth::ApplicationAuthenticationToken,
        aauth::ApplicationAuthenticationResponse);
}
#endif // NNT_ACCOUNT_ENABLE_APPLICATION_AUTH

#if defined(NNT_ACCOUNT_ENABLE_EXPIRATION_APPLICATION_AUTH)
namespace aexpr
{

const char* ApplicationAuthenticationExpirations[] =
    {
        "{\"expires_in\":3600,\"application_auth_token\":\"________________\"}",
        "{\"expires_in\":3603,\"application_auth_token\":\"________________\"}",
    };
uint32_t ApplicationAuthenticationExpirationWaits[] = { 0, 3 };

void TestApplicationAuthentication(uint32_t expiresIn, const char* json) NN_NOEXCEPT
{
    const auto info = a::detail::ApplicationInfo::Get(1234567890123456, 1234, a::detail::ApplicationMediaType::GameCard);

    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    a::ndas::ApplicationAuthenticationCache cache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cache.Initialize(s));

    ASSERT_FALSE(cache.IsAvailable(info));

    a::ndas::ApplicationAuthenticationAdaptor adaptor(cache, info);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(-1));

    if (expiresIn > 0)
    {
        // 有効 → 無効
        ASSERT_TRUE(cache.IsAvailable(info));
    }
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(expiresIn));
    ASSERT_FALSE(cache.IsAvailable(info));
}

} // ~namespace aexpr

TEST(AccountAdaptor, ApplicationAuthentication_Expiration)
{
    for (auto i = 0; i < sizeof(aexpr::ApplicationAuthenticationExpirations) / sizeof(aexpr::ApplicationAuthenticationExpirations[0]); ++ i)
    {
        aexpr::TestApplicationAuthentication(
            aexpr::ApplicationAuthenticationExpirationWaits[i],
            aexpr::ApplicationAuthenticationExpirations[i]);
    }
}
#endif // NNT_ACCOUNT_ENABLE_EXPIRATION_APPLICATION_AUTH

#if defined(NNT_ACCOUNT_ENABLE_ERROR_APPLICATION_AUTH)
namespace aerr
{

const Error ApplicationAuthenticationErrors[] =
    {
        {400, "{\"errors\":[{\"code\":\"0100\",\"message\":\"Missing parameter in request.\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0101\",\"message\":\"Invalid parameter in request.\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0102\",\"message\":\"Invalid device token\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0103\",\"message\":\"Device token expired\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0104\",\"message\":\"Invalid gamecard cert\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0105\",\"message\":\"Gamecard has been banned\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0106\",\"message\":\"Unauthorized application\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0107\",\"message\":\"Device has been banned by application\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0108\",\"message\":\"Service is not started\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0109\",\"message\":\"Service closed\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0110\",\"message\":\"Service is not available\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0111\",\"message\":\"Application update is required\"}]}"},
        {500, "{\"errors\":[{\"code\":\"0112\",\"message\":\"Internal server error\"}]}"},
        {503, "{\"errors\":[{\"code\":\"0113\",\"message\":\"Under maintenance\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0114\",\"message\":\"Invalid device token - Invalid signature\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0115\",\"message\":\"Invalid device token - Unkown issuer\"}]}"},
        {400, "{\"errors\":[{\"code\":\"0116\",\"message\":\"Invalid device token - Unsupported algorithm\"}]}"},
        {404, "{\"errors\":[{\"code\":\"0117\",\"message\":\"API not found.\"}]}"},
        {400, "{}"},
        {401, "{}"},
        {500, "{}"},
        {503, "{}"},
        {404, "{}"},
        {200, "{"
            "\"expires_in\":86400,"
            "\"application_auth_token\":\"eyJ0eXAiOiJKV1QiLCJhbGciOiJSUzI1Ni"
                "IsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2Iiwia2lkIjoiOGE0OTA5OTYtYzc3My00MWJkLTkwMDgtYz"
                "k1NjBjOWZlNjBmIn0.eyJzdWIiOiIxMjM0NTY3ODkiLCJpc3MiOiJhYXV0aC10ZDEubmRhcy5zcnYubm"
                "ludGVuZG8ubmV0IiwiZXhwIjoxNDUxOTgwOTA4LCJpYXQiOjE0NTE4OTQ1MDgsImp0aSI6ImFiMmRjZD"
                "M2LTQ4N2UtNGFlMC05Y2E3LWYwMmJiNjJhODQ0OCIsIm5pbnRlbmRvIjp7ImRpIjoiNjEwMDAwMDAwMD"
                "AwMDAyMiIsInNuIjoiWEFWMDAwMDAwMDAzNDUiLCJwYyI6IkhBQy1TLVpaWlpWKFVTWikiLCJkdCI6Ik"
                "5YIFZEZXYgMSJ9fQ.HE38ZZMGv-ZStY427rCY6-Tzx0ADWHA587VCGB43RYl3E_h5247iDGDg6tOLM5o"
                "gVt3NMfGV6zgfLE9n2MC9tTaGrCbt6Nl1GhtYlO8MVq9schh3a527uls9okLP7puWw6yrIvarnVEWjyl"
                "zyYzwbMEKHXybGdWYa_B-77chen4DubfXTajP45yS7RhjE66myy6xioEtZlrGncSikz11fNYhIzzLKJp"
                "07q2StErKyU3f91ZLKW8l6c3c5Z-ItVRVb_vZY23a9oekyr-M-4Q3auaWcmho_p35GjEMVDDlbElTHX8"
                "8ju15f4xiNUhwl1R6DekSn2_lMYALJYU2QOpONQ\","
            "\"errors\":[{\"code\":\"0100\",\"message\":\"Missing parameter in request.\"}]"
        "}"},
        {400, "{\"errors\":[{\"code\":\"0100\"}]}"},
        {400, "{\"errors\":[{\"message\":\"Missing parameter in request.\"}]}"},
    };
const nn::Result ApplicationAuthenticationResults[] = {
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasStatusNo0103(),
    nn::account::ndas::ResultNdasStatusNo0104(),
    nn::account::ndas::ResultNdasStatusNo0105(),
    nn::account::ndas::ResultNdasStatusNo0106(),
    nn::account::ndas::ResultNdasStatusNo0107(),
    nn::account::ndas::ResultNdasStatusNo0108(),
    nn::account::ndas::ResultNdasStatusNo0109(),
    nn::account::ndas::ResultNdasStatusNo0110(),
    nn::account::ResultUnacceptableApplicationVersion(),
    nn::account::ndas::ResultNdasStatusNo0112(),
    nn::account::ndas::ResultNdasStatusNo0113(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasStatusNo0117(),
    nn::account::http::ResultHttpStatus400BadRequest(),
    nn::account::http::ResultHttpStatus401Unauthorized(),
    nn::account::http::ResultHttpStatus500InternalServerError(),
    nn::account::http::ResultHttpStatus503ServiceUnavailable(),
    nn::account::http::ResultHttpStatus404NotFound(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::ndas::ResultNdasRemovedStatus(),
    nn::account::http::ResultHttpStatus400BadRequest(),
};

void TestApplicationAuthentication(nn::Result expectation, const Error& e) NN_NOEXCEPT
{
    const auto info = a::detail::ApplicationInfo::Get(1234567890123456, 1234, a::detail::ApplicationMediaType::GameCard);

    t::MemoryInputStreamForRapidJson stream;
    stream.Set(e.json, strlen(e.json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());
    a::ndas::ApplicationAuthenticationCache cache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(cache.Initialize(s));

    ASSERT_FALSE(cache.IsAvailable(info));

    a::ndas::ApplicationAuthenticationAdaptor adaptor(cache, info);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    ASSERT_EQ(expectation.GetInnerValueForDebug(), adaptor.Adapt(e.httpCode).GetInnerValueForDebug());
}

} // ~namespace aerr

TEST(AccountAdaptor, ApplicationAuthentication_Error)
{
    for (auto i = 0; i < sizeof(aerr::ApplicationAuthenticationErrors) / sizeof(aerr::ApplicationAuthenticationErrors[0]); ++ i)
    {
        aerr::TestApplicationAuthentication(
            aerr::ApplicationAuthenticationResults[i],
            aerr::ApplicationAuthenticationErrors[i]);
    }
}
#endif // NNT_ACCOUNT_ENABLE_ERROR_APPLICATION_AUTH
