﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "testAccount_NasProxy.h"

#include "nas/account_NasResourceResolver.h"
#include "http/account_HttpUtil.h"
#include "http/account_RedirectUriParser.h"
#include "http/account_UrlEncoder.h"

#include <algorithm>
#include <memory>
#include <utility>
#include <curl/curl.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/http/account_CurlInputStream.h>
#include <nn/account/json/account_JsonAdaptor.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/util/util_ScopeExit.h>

#include "testAccount_Util.h"

#define NNT_ACCOUNT_NAS_HELPER_URI "https://hikarie.nntt.mng.nintendo.net/api/v1/nas"

namespace nnt {
namespace account {
namespace {

const int DefaultTryCount = 2;

class NasAuthorizationRequestExtractionAdaptor final
    : public nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1>
{
private:
    typedef nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1> ParserBase;

    nn::account::http::UriLookupEntry m_ClientIdEntry;
    nn::account::http::UriLookupEntry m_RedirectUriEntry;
    nn::account::http::UriLookupEntry m_StateEntry;
    nn::account::http::UriLookupEntry m_CodeChallengeEntry;
    nn::account::http::UriLookupEntry m_CodeChallengeMethodEntry;
    nn::account::http::UriLookupEntry m_ScopeEntry;

    char m_CallbackUri[256];
    nn::account::nas::NasClientInfo m_ClientInfo;
    char m_State[128];
    char m_CodeChallenge[256];
    char m_Scope[256];
    nnt::account::Buffer m_Uri;

    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit NasAuthorizationRequestExtractionAdaptor(const char* baseUrl) NN_NOEXCEPT;
    nnt::account::Buffer Adapt(const NasLoginInfo& credential) NN_NOEXCEPT;
};
class NasApplicationAuthorizationRequestExtractionAdaptor final
    : public nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1>
{
private:
    typedef nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1> ParserBase;

    nn::account::http::UriLookupEntry m_ClientIdEntry;
    nn::account::http::UriLookupEntry m_IdTokenHintEntry;
    nn::account::http::UriLookupEntry m_RedirectUriEntry;
    nn::account::http::UriLookupEntry m_ScopeEntry;
    nn::account::http::UriLookupEntry m_StateEntry;
    nn::account::http::UriLookupEntry m_NonceEntry;

    char m_CallbackUri[256];
    nn::account::nas::NasClientInfo m_ClientInfo;
    char m_IdTokenHint[nn::account::detail::NasIdTokenSizeMax + 1];
    char m_Scope[256];
    char m_State[128];
    char m_Nonce[128];
    nnt::account::Buffer m_Uri;

    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit NasApplicationAuthorizationRequestExtractionAdaptor(const char* baseUrl) NN_NOEXCEPT;
    nnt::account::Buffer Adapt(const NasLoginInfo& credential) NN_NOEXCEPT;
};
class NasAuthorizationRequestExtractionAdaptorForGuest final
    : public nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1>
{
private:
    typedef nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1> ParserBase;

    nn::account::http::UriLookupEntry m_ClientIdEntry;
    nn::account::http::UriLookupEntry m_RedirectUriEntry;
    nn::account::http::UriLookupEntry m_StateEntry;
    nn::account::http::UriLookupEntry m_ScopeEntry;

    char m_CallbackUri[256];
    nn::account::nas::NasClientInfo m_ClientInfo;
    char m_State[128];
    char m_Scope[256];
    nnt::account::Buffer m_Uri;

    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit NasAuthorizationRequestExtractionAdaptorForGuest(const char* baseUrl) NN_NOEXCEPT;
    nnt::account::Buffer Adapt(const NasLoginInfo& credential) NN_NOEXCEPT;
};
class NasAuthorizationRequestExtractionAdaptorForError final
    : public nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1>
{
private:
    typedef nn::account::http::RedirectUriParser<nn::account::detail::NasCallbackUriLengthMax + 1> ParserBase;

    nn::account::http::UriLookupEntry m_RedirectUriEntry;
    nn::account::http::UriLookupEntry m_StateEntry;

    char m_CallbackUri[256];
    const char* const m_Error;
    const char* const m_ErrorDetail;

    char m_RedirectUri[256];
    char m_State[128];
    nnt::account::Buffer m_Uri;

    virtual void UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT NN_OVERRIDE;

public:
    explicit NasAuthorizationRequestExtractionAdaptorForError(const char* baseUrl, const char* error, const char* detail) NN_NOEXCEPT;
    nnt::account::Buffer Adapt() NN_NOEXCEPT;
};
} // ~namespace nnt::account::<anonymous>

nnt::account::Buffer GetAuthorizationViaNasProxy(const NasLoginInfo& credential, const char* url) NN_NOEXCEPT
{
    char baseUrl[256];
    auto l = nn::account::nas::NasResourceResolver::GetUrlForAuthorizationRequest(baseUrl, sizeof(baseUrl));
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < sizeof(baseUrl));
    NasAuthorizationRequestExtractionAdaptor adaptor(baseUrl);
    NN_ABORT_UNLESS(adaptor.Parse(url, strlen(url) + 1));
    return adaptor.Adapt(credential);
}

nnt::account::Buffer GetApplicationAuthorizationViaNasProxy(const NasLoginInfo& credential, const char* url) NN_NOEXCEPT
{
    char baseUrl[256];
    auto l = nn::account::nas::NasResourceResolver::GetUrlForAuthorizationRequest(baseUrl, sizeof(baseUrl));
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < sizeof(baseUrl));
    std::unique_ptr<NasApplicationAuthorizationRequestExtractionAdaptor> adaptor(new NasApplicationAuthorizationRequestExtractionAdaptor(baseUrl));
    NN_ABORT_UNLESS(adaptor->Parse(url, strlen(url) + 1));
    return adaptor->Adapt(credential);
}

nnt::account::Buffer GetAuthorizationViaNasProxyForGuest(const NasLoginInfo& credential, const char* url) NN_NOEXCEPT
{
    char baseUrl[256];
    auto l = nn::account::nas::NasResourceResolver::GetUrlForAuthorizationRequest(baseUrl, sizeof(baseUrl));
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < sizeof(baseUrl));
    std::unique_ptr<NasAuthorizationRequestExtractionAdaptorForGuest> adaptor(new NasAuthorizationRequestExtractionAdaptorForGuest(baseUrl));
    NN_ABORT_UNLESS(adaptor->Parse(url, strlen(url) + 1));
    return adaptor->Adapt(credential);
}

nnt::account::Buffer GetAuthorizationForError(const char* url, const char* error, const char* detail) NN_NOEXCEPT
{
    char baseUrl[256];
    auto l = nn::account::nas::NasResourceResolver::GetUrlForAuthorizationRequest(baseUrl, sizeof(baseUrl));
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < sizeof(baseUrl));
    NasAuthorizationRequestExtractionAdaptorForError adaptor(baseUrl, error, detail);
    NN_ABORT_UNLESS(adaptor.Parse(url, strlen(url) + 1));
    return adaptor.Adapt();
}

void RevokeAuthorization(const NasLoginInfo& credential) NN_NOEXCEPT
{
    nn::nsd::EnvironmentIdentifier envId;
    nn::nsd::GetEnvironmentIdentifier(&envId);
    char postData[256];
    char email[sizeof(credential.email) * 3];
    nn::account::http::UrlEncode(email, sizeof(email), credential.email, sizeof(credential.email));
    char password[sizeof(credential.password) * 3];
    nn::account::http::UrlEncode(password, sizeof(password), credential.password, sizeof(credential.password));
    auto l = nn::util::SNPrintf(
        postData, sizeof(postData),
        "environment=%s&email=%s&password=%s", envId.value, email, password);
    NN_ABORT_UNLESS(l < sizeof(postData));

    CURL* curlHandle = curl_easy_init();
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    nn::Result r = nn::ResultSuccess();
    int tryableCount = DefaultTryCount;
    while ((tryableCount --) > 0)
    {
        class SimpleWebAccessor
            : nn::http::stream::WebApiAccessorBase
        {
        public:
            explicit SimpleWebAccessor(CURL* curlHandle) NN_NOEXCEPT
                : WebApiAccessorBase(curlHandle, nullptr)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetDefault());
                NN_ABORT_UNLESS_RESULT_SUCCESS(SetDebugMode());
            }
            using WebApiAccessorBase::SetHttpPost;
            using WebApiAccessorBase::SetUrl;
            nn::Result Perform() NN_NOEXCEPT
            {
                NN_RESULT_DO(WebApiAccessorBase::Perform());
                NN_RESULT_DO(WebApiAccessorBase::GetResult());
                NN_RESULT_SUCCESS;
            }
        } accessor(curlHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(accessor.SetHttpPost(postData));
        NN_ABORT_UNLESS_RESULT_SUCCESS(accessor.SetUrl(NNT_ACCOUNT_NAS_HELPER_URI "/revoke_all_client_authorizations"));
        r = accessor.Perform();
        if (r.IsSuccess())
        {
            return;
        }

        NN_LOG(
            "[nnt::account] Retry RevokeAuthorization(): error=%03d-%04d, tryableCount=%d\n",
            r.GetModule(), r.GetDescription(), tryableCount);
    }

}
} // ~namespace nnt::account
}

/* --------------------------------------------------------------------------------------------
    実装
*/

#include <nn/account/json/account_RapidJsonApi.h>

namespace nnt {
namespace account {
namespace {
class NasProxyAdaptor
{
private:
    nn::account::json::LookupEntry m_StateEntry;
    nn::account::json::LookupEntry m_CodeEntry;
    nn::account::json::LookupEntry m_IdTokenEntry;

    const char* m_State;
    nnt::account::Buffer m_Code;
    nnt::account::Buffer m_IdToken;

public:
    typedef nn::account::json::JsonPath<8, 32> JsonPathType;

public:
    explicit NasProxyAdaptor(const char* state) NN_NOEXCEPT
        : m_StateEntry("$.state")
        , m_CodeEntry("$.code")
        , m_IdTokenEntry("$.id_token")
        , m_State(state)
    {
    }
    std::pair<nnt::account::Buffer, nnt::account::Buffer> Adapt(int32_t httpCode) NN_NOEXCEPT
    {
        if (!(m_StateEntry && m_CodeEntry && m_IdTokenEntry))
        {
            NN_LOG("[nnt::account] ----------------------------------------------\n");
            NN_LOG("  Error: NAS Proxy Authorization Failed: %04d\n", httpCode);
            NN_LOG("    %s : %s\n", m_CodeEntry ? "+" : "-", m_CodeEntry.path);
            NN_LOG("    %s : %s\n", m_IdTokenEntry ? "+" : "-", m_IdTokenEntry.path);
            NN_LOG("    %s : %s\n", m_StateEntry ? "+" : "-", m_StateEntry.path);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::http::HandleHttpStatusCode(httpCode));
            NN_ABORT("NasProxyFailed\n");
        }
        return std::pair<nnt::account::Buffer, nnt::account::Buffer>(std::move(m_Code), std::move(m_IdToken));
    }
    nnt::account::Buffer AdaptForGuest(int32_t httpCode) NN_NOEXCEPT
    {
        if (!(m_StateEntry && m_IdTokenEntry))
        {
            NN_LOG("[nnt::account] ----------------------------------------------\n");
            NN_LOG("  Error: NAS Proxy Authorization Failed: %04d\n", httpCode);
            NN_LOG("    %s : %s\n", m_IdTokenEntry ? "+" : "-", m_IdTokenEntry.path);
            NN_LOG("    %s : %s\n", m_StateEntry ? "+" : "-", m_StateEntry.path);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::http::HandleHttpStatusCode(httpCode));
            NN_ABORT("NasProxyFailed\n");
        }
        return std::move(m_IdToken);
    }
    void Update(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
    {
        if (m_StateEntry.CanAccept(jsonPath))
        {
            if (std::strncmp(m_State, value, strlen(m_State) + 1) == 0)
            {
                // 意図した state のレスポンスが返ってくることを確認 (assertion)
                m_StateEntry.MarkAccepted();
            }
        }
        else if (m_CodeEntry.CanAccept(jsonPath))
        {
            if (valueLength <= nn::account::detail::NasAuthorizationCodeSizeMax)
            {
                NN_ASSERT(strnlen(value, valueLength + 1) == static_cast<size_t>(valueLength));
                m_Code = nnt::account::Buffer(valueLength);
                std::memcpy(m_Code.GetAddress(), value, valueLength);
                m_CodeEntry.MarkAccepted();
            }
        }
        else if (m_IdTokenEntry.CanAccept(jsonPath))
        {
            if (valueLength <= nn::account::detail::NasIdTokenSizeMax)
            {
                NN_ASSERT(strnlen(value, valueLength + 1) == static_cast<size_t>(valueLength));
                m_IdToken = nnt::account::Buffer(valueLength);
                std::memcpy(m_IdToken.GetAddress(), value, valueLength);
                m_IdTokenEntry.MarkAccepted();
            }
        }
    }
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT{}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT{}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT{}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT{}
    void Update(const JsonPathType&, double) NN_NOEXCEPT{}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}; // NOP
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}; // NOP
};
nn::Result SetupStreamForNasProxy(
    nn::account::http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const NasLoginInfo& credential, const nn::account::nas::NasClientInfo& clientInfo,
    const char* codeChallenge, const char* state, const char* scope) NN_NOEXCEPT
{
    nn::nsd::EnvironmentIdentifier envId;
    nn::nsd::GetEnvironmentIdentifier(&envId);

    // POST データ作成
    static const char PostDataFormat[] =
        "response_type=code+id_token"
        "&email=%.*s"
        "&password=%.*s"
        "&client_id=%016llx"
        "&redirect_uri=%.*s"
        "&environment=%s"
        "&state=%s"
        "&code_challenge=%s"
        "&code_challenge_method=S256"
        "&scope=%s";
    char email[sizeof(credential.email) * 3];
    nn::account::http::UrlEncode(email, sizeof(email), credential.email, sizeof(credential.email));
    char password[sizeof(credential.password) * 3];
    nn::account::http::UrlEncode(password, sizeof(password), credential.password, sizeof(credential.password));
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        strnlen(email, sizeof(email)), email,
        strnlen(password, sizeof(password)), password,
        clientInfo.clientId,
        strnlen(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), clientInfo.redirectUri,
        envId.value,
        state,
        codeChallenge,
        scope);
    NN_ASSERT(static_cast<uint32_t>(l) < postDataBufferSize);
    NN_UNUSED(l);

    // リクエスト生成
    NN_RESULT_DO(stream.SetDebugMode());
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    NN_RESULT_DO(stream.SetUrl(NNT_ACCOUNT_NAS_HELPER_URI "/authorize"));
    NN_RESULT_SUCCESS;
}
nn::Result SetupApplicationAuthroizationStreamForNasProxy(
    nn::account::http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const NasLoginInfo& credential, const nn::account::nas::NasClientInfo& clientInfo,
    const char* idTokenHint,
    const char* scope, const char* state, const char* nonce) NN_NOEXCEPT
{
    nn::nsd::EnvironmentIdentifier envId;
    nn::nsd::GetEnvironmentIdentifier(&envId);

    // POST データ作成
    static const char PostDataFormat[] =
        "response_type=code+id_token"
        "&email=%.*s"
        "&password=%.*s"
        "&client_id=%016llx"
        "&redirect_uri=%.*s"
        "&environment=%s"
        "&id_token_hint=%s"
        "&scope=%s"
        "&state=%s"
        "&nonce=%s"
        "&enforce_consent=true";
    char email[sizeof(credential.email) * 3];
    nn::account::http::UrlEncode(email, sizeof(email), credential.email, sizeof(credential.email));
    char password[sizeof(credential.password) * 3];
    nn::account::http::UrlEncode(password, sizeof(password), credential.password, sizeof(credential.password));
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        strnlen(email, sizeof(email)), email,
        strnlen(password, sizeof(password)), password,
        clientInfo.clientId,
        strnlen(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), clientInfo.redirectUri,
        envId.value,
        idTokenHint,
        scope,
        state,
        nonce);
    NN_ASSERT(static_cast<uint32_t>(l) < postDataBufferSize);
    NN_UNUSED(l);

    // リクエスト生成
    NN_RESULT_DO(stream.SetDebugMode());
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    NN_LOG("PostData(%d): %s\n", strlen(postDataBuffer), postDataBuffer);

    // URL の作成
    NN_RESULT_DO(stream.SetUrl(NNT_ACCOUNT_NAS_HELPER_URI "/authorize"));
    NN_RESULT_SUCCESS;
}
nn::Result SetupStreamForNasProxyForGuest(
    nn::account::http::CurlInputStream& stream,
    char* postDataBuffer, size_t postDataBufferSize,
    const NasLoginInfo& credential, const nn::account::nas::NasClientInfo& clientInfo,
    const char* state, const char* scope) NN_NOEXCEPT
{
    nn::nsd::EnvironmentIdentifier envId;
    nn::nsd::GetEnvironmentIdentifier(&envId);

    // POST データ作成
    static const char PostDataFormat[] =
        "response_type=id_token"
        "&email=%.*s"
        "&password=%.*s"
        "&client_id=%016llx"
        "&redirect_uri=%.*s"
        "&environment=%s"
        "&state=%s"
        "&scope=%s";
    char email[sizeof(credential.email) * 3];
    nn::account::http::UrlEncode(email, sizeof(email), credential.email, sizeof(credential.email));
    char password[sizeof(credential.password) * 3];
    nn::account::http::UrlEncode(password, sizeof(password), credential.password, sizeof(credential.password));
    auto l = nn::util::SNPrintf(
        postDataBuffer, postDataBufferSize,
        PostDataFormat,
        strnlen(email, sizeof(email)), email,
        strnlen(password, sizeof(password)), password,
        clientInfo.clientId,
        strnlen(clientInfo.redirectUri, sizeof(clientInfo.redirectUri)), clientInfo.redirectUri,
        envId.value,
        state,
        scope);
    NN_ASSERT(static_cast<uint32_t>(l) < postDataBufferSize);
    NN_UNUSED(l);

    // リクエスト生成
    NN_RESULT_DO(stream.SetDebugMode());
    NN_RESULT_DO(stream.SetHttpPost(postDataBuffer, false));

    // URL の作成
    NN_RESULT_DO(stream.SetUrl(NNT_ACCOUNT_NAS_HELPER_URI "/authorize"));
    NN_RESULT_SUCCESS;
}

#define NNT_ACCOUNT_MAX(a, b) ((a) > (b) ? (a) : (b))
std::pair<nnt::account::Buffer, nnt::account::Buffer> AcquireAuthorizationCodeByNasProxy(
    const NasLoginInfo& credential,
    const nn::account::nas::NasClientInfo& clientInfo,
    const char* state, const char* codeChallenge, const char* scope) NN_NOEXCEPT
{
    struct WorkBuffer
    {
        char postDataBuffer[1024];
        union
        {
            struct
            {
                char stringBuffer[NNT_ACCOUNT_MAX(nn::account::detail::NasAuthorizationCodeSizeMax, nn::account::detail::NasIdTokenSizeMax) + 1];
                char inputBuffer[nn::account::detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(std::alignment_of<WorkBuffer>::value <= std::alignment_of<std::max_align_t>::value);

    // リクエストの作成
    nnt::account::Buffer rawBuffer(sizeof(WorkBuffer));
    auto* buffer = rawBuffer.Get<WorkBuffer>();

    CURL* curlHandle = curl_easy_init();
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    nn::Result r = nn::ResultSuccess();
    int tryableCount = DefaultTryCount;
    while ((tryableCount --) > 0)
    {
        nn::account::http::CurlInputStream stream(curlHandle, nullptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Initialize());
        stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
        stream.SetInputBuffer(buffer->u.io.inputBuffer, sizeof(buffer->u.io.inputBuffer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(SetupStreamForNasProxy(
            stream, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
            credential, clientInfo, codeChallenge, state, scope));

        // 通信の実行と結果の適用
        NasProxyAdaptor adaptor(state);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Open());
        r = nn::account::json::ImportJsonByRapidJson(adaptor, stream, nullptr);
        stream.Close();

        if (r.IsSuccess())
        {
            return adaptor.Adapt(stream.GetHttpCode());
        }
        NN_LOG(
            "[nnt::account] Retry AcquireAuthorizationCodeByNasProxy(): error=%03d-%04d, tryableCount=%d\n",
            r.GetModule(), r.GetDescription(), tryableCount);
    }

    NN_ASSERT(!r.IsSuccess());
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    NN_ABORT("[nnt::account] Unreachable");
}

std::pair<nnt::account::Buffer, nnt::account::Buffer> AcquireApplicationAuthorizationByNasProxy(
    const NasLoginInfo& credential,
    const nn::account::nas::NasClientInfo& clientInfo,
    const char* idTokenHint,
    const char* scope, const char* state, const char* nonce) NN_NOEXCEPT
{
    struct WorkBuffer
    {
        char postDataBuffer[4096];
        union
        {
            struct
            {
                char stringBuffer[NNT_ACCOUNT_MAX(nn::account::detail::NasAuthorizationCodeSizeMax, nn::account::detail::NasIdTokenSizeMax) + 1];
                char inputBuffer[nn::account::detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(std::alignment_of<WorkBuffer>::value <= std::alignment_of<std::max_align_t>::value);

    // リクエストの作成
    nnt::account::Buffer rawBuffer(sizeof(WorkBuffer));
    auto* buffer = rawBuffer.Get<WorkBuffer>();

    CURL* curlHandle = curl_easy_init();
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    nn::Result r = nn::ResultSuccess();
    int tryableCount = DefaultTryCount;
    while ((tryableCount --) > 0)
    {
        nn::account::http::CurlInputStream stream(curlHandle, nullptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Initialize());
        stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
        stream.SetInputBuffer(buffer->u.io.inputBuffer, sizeof(buffer->u.io.inputBuffer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(SetupApplicationAuthroizationStreamForNasProxy(
            stream, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
            credential, clientInfo, idTokenHint, scope, state, nonce));

        // 通信の実行と結果の適用
        NasProxyAdaptor adaptor(state);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Open());
        r = nn::account::json::ImportJsonByRapidJson(adaptor, stream, nullptr);
        stream.Close();

        if (r.IsSuccess())
        {
            return adaptor.Adapt(stream.GetHttpCode());
        }
        NN_LOG(
            "[nnt::account] Retry AcquireApplicationAuthorizationByNasProxy(): error=%03d-%04d, tryableCount=%d\n",
            r.GetModule(), r.GetDescription(), tryableCount);
    }

    NN_ASSERT(!r.IsSuccess());
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    NN_ABORT("[nnt::account] Unreachable");
}

nnt::account::Buffer AcquireAuthorizationCodeByNasProxyForGuest(
    const NasLoginInfo& credential,
    const nn::account::nas::NasClientInfo& clientInfo,
    const char* state, const char* scope) NN_NOEXCEPT
{
    struct WorkBuffer
    {
        char postDataBuffer[1024];
        union
        {
            struct
            {
                char stringBuffer[nn::account::detail::NasIdTokenSizeMax + 1];
                char inputBuffer[nn::account::detail::IoBufferSizeMin];
            } io;
        } u;
    };
    NN_STATIC_ASSERT(std::alignment_of<WorkBuffer>::value <= std::alignment_of<std::max_align_t>::value);

    // リクエストの作成
    nnt::account::Buffer rawBuffer(sizeof(WorkBuffer));
    auto* buffer = rawBuffer.Get<WorkBuffer>();

    CURL* curlHandle = curl_easy_init();
    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curlHandle);
    };

    nn::Result r = nn::ResultSuccess();
    int tryableCount = DefaultTryCount;
    while ((tryableCount --) > 0)
    {
        nn::account::http::CurlInputStream stream(curlHandle, nullptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Initialize());
        stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
        stream.SetInputBuffer(buffer->u.io.inputBuffer, sizeof(buffer->u.io.inputBuffer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(SetupStreamForNasProxyForGuest(
            stream, buffer->postDataBuffer, sizeof(buffer->postDataBuffer),
            credential, clientInfo, state, scope));

        // 通信の実行と結果の適用
        NasProxyAdaptor adaptor(state);
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Open());
        r = nn::account::json::ImportJsonByRapidJson(adaptor, stream, nullptr);
        stream.Close();

        if (r.IsSuccess())
        {
            return adaptor.AdaptForGuest(stream.GetHttpCode());
        }
        NN_LOG(
            "[nnt::account] Retry AcquireApplicationAuthorizationByNasProxy(): error=%03d-%04d, tryableCount=%d\n",
            r.GetModule(), r.GetDescription(), tryableCount);
    }

    NN_ASSERT(!r.IsSuccess());
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    NN_ABORT("[nnt::account] Unreachable");
}

template <size_t N>
nnt::account::Buffer DecodeRedirectUri(const char (&redirect)[N])
{
    nnt::account::Buffer uri(N);
    int count = 0;
    for (size_t i = 0u; i < N && redirect[i] != '\0'; ++ i)
    {
        auto c = redirect[i];
        if (c == '%')
        {
            auto c1 = redirect[i + 1];
            auto c2 = redirect[i + 2];
            if (c1 == '3' && c2 == 'A')
            {
                uri.Get<char>()[count ++] = ':';
            }
            else if (c1 == '2' && c2 == 'F')
            {
                uri.Get<char>()[count ++] = '/';
            }
            else
            {
                NN_ABORT("Unexpected escaped charactor: %%%c%c\n", c1, c2);
            }
            i += 2;
        }
        else
        {
            uri.Get<char>()[count ++] = c;
        }
    }
    uri.Get<char>()[count ++] = '\0';
    return uri;
}

NasAuthorizationRequestExtractionAdaptor::NasAuthorizationRequestExtractionAdaptor(const char* baseUrl) NN_NOEXCEPT
    : ParserBase(m_CallbackUri)
    , m_ClientIdEntry("client_id")
    , m_RedirectUriEntry("redirect_uri")
    , m_StateEntry("state")
    , m_CodeChallengeEntry("code_challenge")
    , m_CodeChallengeMethodEntry("code_challenge_method")
    , m_ScopeEntry("scope")
{
    strncpy(m_CallbackUri, baseUrl, sizeof(m_CallbackUri));
}
void NasAuthorizationRequestExtractionAdaptor::UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_ClientIdEntry.CanAccept(key, keyLength))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto clientId = nn::account::detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            m_ClientInfo.clientId = clientId;
            m_ClientIdEntry.MarkAccepted();
        }
    }
    else if (m_RedirectUriEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_ClientInfo.redirectUri))
        {
            strncpy(m_ClientInfo.redirectUri, value, valueLength);
            m_ClientInfo.redirectUri[valueLength] = '\0';
            m_RedirectUriEntry.MarkAccepted();
        }
    }
    else if (m_StateEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_State))
        {
            strncpy(m_State, value, valueLength);
            m_State[valueLength] = '\0';
            m_StateEntry.MarkAccepted();
        }
    }
    else if (m_CodeChallengeEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_CodeChallenge))
        {
            strncpy(m_CodeChallenge, value, valueLength);
            m_CodeChallenge[valueLength] = '\0';
            m_CodeChallengeEntry.MarkAccepted();
        }
    }
    else if (m_CodeChallengeMethodEntry.CanAccept(key, keyLength))
    {
        if (true
            && valueLength == strlen("S256")
            && strncmp(value, "S256", valueLength) == 0)
        {
            m_CodeChallengeMethodEntry.MarkAccepted();
        }
    }
    else if (m_ScopeEntry.CanAccept(key, keyLength))
    {
        if (true
            && valueLength < sizeof(m_Scope))
        {
            std::strncpy(m_Scope, value, valueLength);
            m_Scope[valueLength] = '\0';
            m_ScopeEntry.MarkAccepted();
        }
    }
}
nnt::account::Buffer NasAuthorizationRequestExtractionAdaptor::Adapt(const NasLoginInfo& credential) NN_NOEXCEPT
{
    if (!(m_ClientIdEntry && m_RedirectUriEntry && m_StateEntry && m_CodeChallengeEntry && m_CodeChallengeMethodEntry && m_ScopeEntry))
    {
        NN_LOG("[nn::account] -----------------------------------------------\n");
        NN_LOG("  Error: NAS Authorization request extraction failed\n");
        NN_LOG("    %s : %s\n", m_ClientIdEntry? "+": "-", m_ClientIdEntry.path);
        NN_LOG("    %s : %s\n", m_RedirectUriEntry? "+": "-", m_RedirectUriEntry.path);
        NN_LOG("    %s : %s\n", m_StateEntry? "+": "-", m_StateEntry.path);
        NN_LOG("    %s : %s\n", m_CodeChallengeEntry? "+": "-", m_CodeChallengeEntry.path);
        NN_LOG("    %s : %s\n", m_CodeChallengeMethodEntry? "+": "-", m_CodeChallengeMethodEntry.path);
        NN_LOG("    %s : %s\n", m_ScopeEntry? "+": "-", m_ScopeEntry.path);
        NN_ABORT("[nn::account] ABORT: Request parser failure\n");
    }

    // NAS Proxy による Code の取得
    auto response = AcquireAuthorizationCodeByNasProxy(credential, m_ClientInfo, m_State, m_CodeChallenge, m_Scope);

    // 戻り URL の作成
    m_Uri = nnt::account::Buffer(4096);

    auto redirectUri = DecodeRedirectUri(m_ClientInfo.redirectUri);
    auto l = nn::util::SNPrintf(
        m_Uri.Get<char>(), m_Uri.GetSize(),
        "%s?expires_in=0&state=%s&code=%.*s&id_token=%.*s",
        redirectUri.Get<char>(), m_State,
        response.first.GetSize(), response.first.Get<char>(),
        response.second.GetSize(), response.second.Get<char>());
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < m_Uri.GetSize());
    return std::move(m_Uri);
}

NasApplicationAuthorizationRequestExtractionAdaptor::NasApplicationAuthorizationRequestExtractionAdaptor(const char* baseUrl) NN_NOEXCEPT
    : ParserBase(m_CallbackUri)
    , m_ClientIdEntry("client_id")
    , m_IdTokenHintEntry("id_token_hint")
    , m_RedirectUriEntry("redirect_uri")
    , m_ScopeEntry("scope")
    , m_StateEntry("state")
    , m_NonceEntry("nonce")
{
    strncpy(m_CallbackUri, baseUrl, sizeof(m_CallbackUri));
}
void NasApplicationAuthorizationRequestExtractionAdaptor::UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_ClientIdEntry.CanAccept(key, keyLength))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto clientId = nn::account::detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            m_ClientInfo.clientId = clientId;
            m_ClientIdEntry.MarkAccepted();
        }
    }
    else if (m_IdTokenHintEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_IdTokenHint))
        {
            strncpy(m_IdTokenHint, value, valueLength);
            m_IdTokenHint[valueLength] = '\0';
            m_IdTokenHintEntry.MarkAccepted();
        }
    }
    else if (m_RedirectUriEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_ClientInfo.redirectUri))
        {
            strncpy(m_ClientInfo.redirectUri, value, valueLength);
            m_ClientInfo.redirectUri[valueLength] = '\0';
            m_RedirectUriEntry.MarkAccepted();
        }
    }
    else if (m_ScopeEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_Scope))
        {
            std::strncpy(m_Scope, value, valueLength);
            m_Scope[valueLength] = '\0';
            m_ScopeEntry.MarkAccepted();
        }
    }
    else if (m_StateEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_State))
        {
            strncpy(m_State, value, valueLength);
            m_State[valueLength] = '\0';
            m_StateEntry.MarkAccepted();
        }
    }
    else if (m_NonceEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_Nonce))
        {
            std::strncpy(m_Nonce, value, valueLength);
            m_Nonce[valueLength] = '\0';
            m_NonceEntry.MarkAccepted();
        }
    }
}
nnt::account::Buffer NasApplicationAuthorizationRequestExtractionAdaptor::Adapt(const NasLoginInfo& credential) NN_NOEXCEPT
{
    if (!(m_ClientIdEntry && m_IdTokenHintEntry && m_RedirectUriEntry && m_ScopeEntry && m_StateEntry && m_NonceEntry))
    {
        NN_LOG("[nn::account] -----------------------------------------------\n");
        NN_LOG("  Error: NAS Authorization request extraction failed\n");
        NN_LOG("    %s : %s\n", m_ClientIdEntry? "+": "-", m_ClientIdEntry.path);
        NN_LOG("    %s : %s\n", m_IdTokenHintEntry? "+": "-", m_IdTokenHintEntry.path);
        NN_LOG("    %s : %s\n", m_RedirectUriEntry? "+": "-", m_RedirectUriEntry.path);
        NN_LOG("    %s : %s\n", m_ScopeEntry? "+": "-", m_ScopeEntry.path);
        NN_LOG("    %s : %s\n", m_StateEntry? "+": "-", m_StateEntry.path);
        NN_LOG("    %s : %s\n", m_NonceEntry? "+": "-", m_NonceEntry.path);
        NN_ABORT("[nn::account] ABORT: Request parser failure\n");
    }

    // NAS Proxy による Code の取得
    auto response = AcquireApplicationAuthorizationByNasProxy(credential, m_ClientInfo, m_IdTokenHint, m_Scope, m_State, m_Nonce);

    // 戻り URL の作成
    m_Uri = nnt::account::Buffer(4096);

    auto redirectUri = DecodeRedirectUri(m_ClientInfo.redirectUri);
    auto l = nn::util::SNPrintf(
        m_Uri.Get<char>(), m_Uri.GetSize(),
        "%s?expires_in=0&state=%s&code=%.*s&id_token=%*.s",
        redirectUri.Get<char>(), m_State,
        response.first.GetSize(), response.first.GetAddress(),
        response.second.GetSize(), response.second.GetAddress());
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < m_Uri.GetSize());
    return std::move(m_Uri);
}

NasAuthorizationRequestExtractionAdaptorForGuest::NasAuthorizationRequestExtractionAdaptorForGuest(const char* baseUrl) NN_NOEXCEPT
    : ParserBase(m_CallbackUri)
    , m_ClientIdEntry("client_id")
    , m_RedirectUriEntry("redirect_uri")
    , m_StateEntry("state")
    , m_ScopeEntry("scope")
{
    strncpy(m_CallbackUri, baseUrl, sizeof(m_CallbackUri));
}
void NasAuthorizationRequestExtractionAdaptorForGuest::UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_ClientIdEntry.CanAccept(key, keyLength))
    {
        if (valueLength == sizeof(uint64_t) * 2)
        {
            auto clientId = nn::account::detail::ExtractHexadecimal<uint64_t>(value, valueLength);
            m_ClientInfo.clientId = clientId;
            m_ClientIdEntry.MarkAccepted();
        }
    }
    else if (m_RedirectUriEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_ClientInfo.redirectUri))
        {
            strncpy(m_ClientInfo.redirectUri, value, valueLength);
            m_ClientInfo.redirectUri[valueLength] = '\0';
            m_RedirectUriEntry.MarkAccepted();
        }
    }
    else if (m_StateEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_State))
        {
            strncpy(m_State, value, valueLength);
            m_State[valueLength] = '\0';
            m_StateEntry.MarkAccepted();
        }
    }
    else if (m_ScopeEntry.CanAccept(key, keyLength))
    {
        if (true
            && valueLength < sizeof(m_Scope))
        {
            std::strncpy(m_Scope, value, valueLength);
            m_Scope[valueLength] = '\0';
            m_ScopeEntry.MarkAccepted();
        }
    }
}
nnt::account::Buffer NasAuthorizationRequestExtractionAdaptorForGuest::Adapt(const NasLoginInfo& credential) NN_NOEXCEPT
{
    if (!(m_ClientIdEntry && m_RedirectUriEntry && m_StateEntry && m_ScopeEntry))
    {
        NN_LOG("[nn::account] -----------------------------------------------\n");
        NN_LOG("  Error: NAS Guest login request extraction failed\n");
        NN_LOG("    %s : %s\n", m_ClientIdEntry? "+": "-", m_ClientIdEntry.path);
        NN_LOG("    %s : %s\n", m_RedirectUriEntry? "+": "-", m_RedirectUriEntry.path);
        NN_LOG("    %s : %s\n", m_StateEntry? "+": "-", m_StateEntry.path);
        NN_LOG("    %s : %s\n", m_ScopeEntry? "+": "-", m_ScopeEntry.path);
        NN_ABORT("[nn::account] ABORT: Request parser failure\n");
    }

    // NAS Proxy による Code の取得
    auto idToken = AcquireAuthorizationCodeByNasProxyForGuest(credential, m_ClientInfo, m_State, m_Scope);

    // 戻り URL の作成
    m_Uri = nnt::account::Buffer(4096);

    auto redirectUri = DecodeRedirectUri(m_ClientInfo.redirectUri);
    auto l = nn::util::SNPrintf(
        m_Uri.Get<char>(), m_Uri.GetSize(),
        "%s?expires_in=0&state=%s&id_token=%.*s", redirectUri.Get<char>(), m_State, idToken.GetSize(), idToken.Get<char>());
    NN_ABORT_UNLESS(static_cast<uint32_t>(l) < m_Uri.GetSize());
    NN_LOG("NasProxy returned: %s\n", m_Uri.Get<char>());
    return std::move(m_Uri);
}

NasAuthorizationRequestExtractionAdaptorForError::NasAuthorizationRequestExtractionAdaptorForError(const char* baseUrl, const char* error, const char* detail) NN_NOEXCEPT
    : ParserBase(m_CallbackUri)
    , m_RedirectUriEntry("redirect_uri")
    , m_StateEntry("state")
    , m_Error(error)
    , m_ErrorDetail(detail)
{
    NN_SDK_ASSERT_NOT_NULL(error);
    strncpy(m_CallbackUri, baseUrl, sizeof(m_CallbackUri));
}
void NasAuthorizationRequestExtractionAdaptorForError::UpdateImpl(const char* key, size_t keyLength, const char* value, size_t valueLength) NN_NOEXCEPT
{
    if (m_RedirectUriEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_RedirectUri))
        {
            strncpy(m_RedirectUri, value, valueLength);
            m_RedirectUri[valueLength] = '\0';
            m_RedirectUriEntry.MarkAccepted();
        }
    }
    else if (m_StateEntry.CanAccept(key, keyLength))
    {
        if (valueLength < sizeof(m_State))
        {
            strncpy(m_State, value, valueLength);
            m_State[valueLength] = '\0';
            m_StateEntry.MarkAccepted();
        }
    }
}
nnt::account::Buffer NasAuthorizationRequestExtractionAdaptorForError::Adapt() NN_NOEXCEPT
{
    if (!(m_RedirectUriEntry && m_StateEntry))
    {
        NN_LOG("[nn::account] -----------------------------------------------\n");
        NN_LOG("  Error: NAS Authorization request extraction failed\n");
        NN_LOG("    %s : %s\n", m_RedirectUriEntry? "+": "-", m_RedirectUriEntry.path);
        NN_LOG("    %s : %s\n", m_StateEntry? "+": "-", m_StateEntry.path);
        NN_ABORT("[nn::account] ABORT: Request parser failure\n");
    }

    // 戻り URL の作成
    m_Uri = nnt::account::Buffer(4096);

    auto redirectUri = DecodeRedirectUri(m_RedirectUri);
    if (m_ErrorDetail)
    {
        auto l = nn::util::SNPrintf(
            m_Uri.Get<char>(), m_Uri.GetSize(),
            "%s?state=%s&error=%s&error_detail=%s", redirectUri.Get<char>(), m_State, m_Error, m_ErrorDetail);
        NN_ABORT_UNLESS(static_cast<uint32_t>(l) < m_Uri.GetSize());
    }
    else
    {
        auto l = nn::util::SNPrintf(
            m_Uri.Get<char>(), m_Uri.GetSize(),
            "%s?state=%s&error=%s", redirectUri.Get<char>(), m_State, m_Error);
        NN_ABORT_UNLESS(static_cast<uint32_t>(l) < m_Uri.GetSize());
    }
    return std::move(m_Uri);
}

} // ~namespace nnt::account::<anonymous>
}
}
