﻿# -*- coding: utf-8 -*-
from __future__ import division, print_function, absolute_import, unicode_literals
from HACDllDriver import factory, Button, CaptureMode

import sys
import time

"""
HID 入力ライブラリの機能を使用したのサンプルです。
TargetManager にて対象開発機へ接続した状態で、InputDirector を起動してから実行する必要があります。
対象開発機上で、HidSimple サンプル（Samples\Sources\Applications\HidSimple）を起動した状態で実行すると、行われている操作が確認しやすくなります。
"""

class HACDriverExample:
    def __init__(self, target_serial):
        print("Initialize HidInputerSample")
        self.driver = factory(target_serial=target_serial, capture_device_id=None, capture_mode=CaptureMode.CaptureMode_720p)

        # 実行前から接続済みのコントローラーを全て切断
        for i in self.driver.controllers.get_active_cont_id_list():
            self.driver.controllers.delete( i )

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.driver.__exit__(exc_type, exc_val, exc_tb)

    def ex_controller_management(self):
        """Example to connect controllers"""
        print("-----------------------------")
        print("Start controller management")

        controller_id_list = []

        # connect controller
        for i in range(0,3):
            print("Add controller: %d, type=Switch Pro Controller" % ( i ))
            controller_id_list.append( self.driver.controllers.add() )
            time.sleep(1)

        # disconnect controller
        for i in range(len(controller_id_list)):
            print("Reduce controller: %d" % ( i ))
            self.driver.controllers.delete( controller_id_list[i] )
            time.sleep(1)

    def ex_hit_button(self):
        """Example to hit some buttons"""
        print("-----------------------------")

        controller_id_list = []
        for id in range(4):
            controller_id_list.append( self.driver.controllers.add() )

        # hit a button
        print("Hit a button")
        self.driver.controllers[controller_id_list[0]].push("A")
        time.sleep(1)
        self.driver.controllers[controller_id_list[0]].push(Button.A)
        time.sleep(1)
        self.driver.controllers[controller_id_list[1]].push("B")
        time.sleep(1)
        self.driver.controllers[controller_id_list[2]].push("START", ms_press=1000)
        time.sleep(1)
        self.driver.controllers[controller_id_list[3]].push("SELECT", ms_before=1000)
        time.sleep(1)
        self.driver.controllers[controller_id_list[0]].push(Button.HOME, ms_before=3000)
        time.sleep(1)
        self.driver.controllers[controller_id_list[0]].push(Button.CAPTURE, ms_before=3000)
        time.sleep(1)

        # push multi buttons simultaneously
        print("Hit some buttons simultaneously")
        self.driver.controllers[controller_id_list[0]].push_multi(["A", "B", "X", "Y"])
        time.sleep(1)

        # push multi buttons sequentially
        print("Hit some buttons sequentially")
        self.driver.controllers[controller_id_list[1]].push_multi_seq(
            ["A", "B", "X", "Y", "LEFT", "RIGHT", "UP", "DOWN",
            "ZL", "ZR", "L", "R", "START", "SELECT", "STICK_L", "STICK_R"] )
        time.sleep(1)

        # push multi buttons pattern 1
        print("Hit some controllers button by press-release pattern")
        for _con in self.driver.controllers:
            _con.press("A", ms_before=0, ms_after=0)
        time.sleep(3)
        for _con in self.driver.controllers:
            _con.release("A", ms_before=0, ms_after=0)
        time.sleep(1)
        self.driver.controllers.close_all()

    def ex_stick_usage(self):
        """Example for analog stick"""

        print("-----------------------------")
        controller_id_list = []
        for id in range(4):
            controller_id_list.append( self.driver.controllers.add() )

        # tilt analog stick
        print("Tilt an analog stick")
        self.driver.controllers[controller_id_list[0]].stick_L(degree=180)
        time.sleep(1)
        self.driver.controllers[controller_id_list[0]].stick_L(degree=-90, power=0.8, ms_press=2000)
        time.sleep(1)
        self.driver.controllers[controller_id_list[0]].stick_R(90)
        time.sleep(1)

        # push an analog stick all around
        print("Tilt an analog stick all around")
        for deg in range(0, 361, 5):
            self.driver.controllers[controller_id_list[0]].stick_R_hold(degree=deg, power=1.0)
            time.sleep(0.01)
        self.driver.controllers[controller_id_list[0]].stick_R_release()

        for deg in range(0, -361, -5):
            self.driver.controllers[controller_id_list[0]].stick_L_hold(degree=deg, power=0.5)
            time.sleep(0.01)
        self.driver.controllers[controller_id_list[0]].stick_L_release()

        # push multi controllers sticks all around
        print("Tilt multi controllers sticks all around")
        for deg in range(0, 361, 5):
            for _con in self.driver.controllers:
                _con.stick_R_hold(deg, ms_before=0, ms_after=0)
                _con.stick_L_hold(-deg, ms_before=0, ms_after=0)
            time.sleep(0.01)

        self.driver.controllers.close_all()

    def ex_comparison_swipe_flick_drag(self):
        """Example for comparison among swipe flick drag"""
        print("-----------------------------")
        print("Compare swipe function usages")

        start = self.driver.screen.get_center_position()
        angle_deg = 45
        length = 100
        end = 300, 100

        # swipe(touch_id, start_x, start_y, angle_deg, length)
        print("Swipe")
        self.driver.screen.swipe(0, *start, angle_deg=angle_deg, length=length, ms_after=1000)

        # flick(touch_id, start_x, start_y, angle_deg)
        print("Flick")
        self.driver.screen.flick(0, *start, angle_deg=angle_deg, ms_after=1000)

        # drag(touch_id, start_x, start_y, end_x, end_y)
        print("Drag")
        self.driver.screen.drag(0, *(start + end), ms_after=1000)

    def ex_multi_touch(self):
        """Example for multi touch action"""

        print("-----------------------------")
        print("MultiTouch a screen")
        self.driver.screen.touch_multi(touch_ids=[0,1,2],
                                       touches_positions=[(600,400), (700,500), (700, 400)],
                                       ms_press=1000)
        time.sleep(1)

        self.driver.screen.touch_multi(touch_ids=[0, 1, 2, 3, 4],
                                        touches_positions=[(600, 400), (700, 500), (700, 400)],
                                        ms_press=1000)
        time.sleep(1)

    def ex_multi_touch_actions(self):
        """Example for a variety of multi-touch function"""

        print("-----------------------------")
        print("Run a variety of multi-touch function")

        # swipe_multi(touch_ids, center_start_x, center_start_y, center_move_deg, length)
        print("Swipe multi")
        self.driver.screen.swipe_multi([0, 1, 2, 3], 640, 40, 90, 600)
        time.sleep(1)
        self.driver.screen.swipe_multi([0, 1, 2, 3], 640, 40, 90, 600,
                                       #set relative pos in coordination system to the origin of the center point
                                       touches_relative_pos=[(-100, 0), (-50, 50), (50, 50), (100, 0)])
        time.sleep(1)

        # flick_multi(touch_ids, center_start_x, center_start_y, center_move_deg)
        print("Flick multi")
        self.driver.screen.flick_multi([0, 1, 2, 3], 640, 40, 120)
        time.sleep(1)
        self.driver.screen.flick_multi([0, 1, 2, 3], 640, 40, 120,
                                       touches_relative_pos=[(-100, 0), (-50, 50), (50, 50), (100, 0)],
                                       coordsys_deg=30) # set rotate angle of coordination system
        time.sleep(1)

        # drag_multi(touch_ids, center_start_x, center_start_y, center_end_x, center_end_y)
        print("Drag multi")
        self.driver.screen.drag_multi([0, 1, 2, 3], 600, 40, 800, 700)
        time.sleep(1)
        self.driver.screen.drag_multi([0, 1, 2, 3], 600, 40, 800, 700,
                                       touches_relative_pos=[(-100, 0), (-50, 50), (50, 50), (100, 0)],
                                       ms_duration=2000)  # set duration
        time.sleep(1)

        # pinch(touch_ids, center_x, center_y, radius_increase)
        print("Pinch")
        self.driver.screen.pinch([0, 1, 2], 640, 360, 200)
        time.sleep(1)
        self.driver.screen.pinch([0, 1, 2, 3], 640, 360, -200, start_radius=300)
        time.sleep(1)
        self.driver.screen.pinch([0, 1, 2], 640, 360, 200,
                                 touches_relative_deg=[0, 30, 60])
        time.sleep(1)

        # center_touch_pinch(center_toucn_id, center_x, center_y, pinch_touch_ids, radius_increase)
        print("Pinch with center touch")
        self.driver.screen.center_touch_pinch(0, 640, 360, [1, 2], 200)
        time.sleep(1)
        self.driver.screen.center_touch_pinch(0, 640, 360, [1, 2, 3], -200, start_radius=300,
                                              touches_relative_deg=[0, -30, -60]) # set relative degree
        time.sleep(1)

        # rotate(touch_ids, center_x, center_y, rotate_deg)
        print("Rotate")
        self.driver.screen.rotate([0, 1], 640, 360, 180)
        time.sleep(1)
        self.driver.screen.rotate([0, 1, 2], 640, 360, -90, radius=60) # set radius
        time.sleep(1)
        self.driver.screen.rotate([0, 1], 640, 360, 1080, ms_duration=2000,
                                  touches_relative_deg=[0, 30])
        time.sleep(1)

        # center_touch_rotate(center_touch_id, center_x, center_y, rotate_touch_ids, rotate_deg)
        print("Rotate with center touch")
        self.driver.screen.center_touch_rotate(0, 640, 360, [1,2], 180)
        time.sleep(1)
        self.driver.screen.center_touch_rotate(0, 640, 360, [1, 2, 3], 360,
                                               touches_relative_deg=[0, 30, 60], radius=300)
        time.sleep(1)

    def ex_multi_touch_animator(self):
        """Example for multitouch_animator"""
        print("-----------------------------")
        print("Multitouch animator function")

        self.driver.screen.multitouch_animator([0, 1, 2, 3, 6], 200, 360,
                                        [(200, 0), (0, 200), (100, 100), (120, 80), (-110, -110)],
                                        center_move_length=800, coordsys_start_scale=1.0, coordsys_increase_scale=-3.0,
                                        ms_wait=1000,
                                        coordsys_rotate_deg=300, ms_duration=2000)

        self.driver.screen.multitouch_animator([0, 1, 2, 3], 300, 360,
                                        touches_relative_positions=[(0, 0), (-100, 0), (100, 0), (300, 0)],
                                        center_move_length=600,
                                        coordsys_start_deg=0.0, coordsys_rotate_deg=600, coordsys_start_scale=1.0,
                                        coordsys_increase_scale=3.0,
                                        ms_duration=5000, ms_wait=1000)



if __name__ == "__main__":
    # sys.argv[1] = 対象のシリアルナンバー
    serial_number = None
    if len(sys.argv) != 1:
        serial_number = sys.argv[1]

    with HACDriverExample(serial_number) as ex:
        ex.ex_controller_management()
        ex.ex_hit_button()
        ex.ex_stick_usage()

        ex.ex_comparison_swipe_flick_drag()
        ex.ex_multi_touch()
        ex.ex_multi_touch_actions()
        ex.ex_multi_touch_animator()
