﻿# -*- coding: utf-8 -*-
import ctypes
import os
import re
import sys
import time

"""
自動テスト支援ツールを python で利用するためのラッパークラスを定義したサンプルです。
このファイルを対象として実行しても何も行われません。
他の自動テスト支援ツールの python サンプルスクリプトは、このファイルに依存しています。
他のサンプルスクリプトを実行する際は、HACDllDriver.py を同一ディレクトリに配置した状態で実行してください。
"""

class Button:
    (
        HOME,
        CAPTURE,
        A,
        B,
        X,
        Y,
        STICK_L,
        STICK_R,
        L,
        R,
        ZL,
        ZR,
        START,
        SELECT,
        LEFT,
        UP,
        RIGHT,
        DOWN,
        SL,
        SR
    ) = (
        "HOME",
        "CAPTURE",
        "A",
        "B",
        "X",
        "Y",
        "STICK_L",
        "STICK_R",
        "L",
        "R",
        "ZL",
        "ZR",
        "START",
        "SELECT",
        "LEFT",
        "UP",
        "RIGHT",
        "DOWN",
        "SL",
        "SR",
    )

class Stick:
    (
        Stick_L,
        Stick_R,
    ) = range(2)

class SerialNumberString(ctypes.Structure):
    _fields_ = [('value', ctypes.c_char * 15)]

class ImageIdType(ctypes.Structure):
    _fields_ = [('id', ctypes.c_uint)]

class FilterIdType(ctypes.Structure):
    _fields_ = [('id', ctypes.c_uint)]

class LabelingResultHandleType(ctypes.Structure):
    _fields_ = [('Handle', ctypes.c_uint)]

class OcrResultHandleType(ctypes.Structure):
    _fields_ = [('Handle', ctypes.c_uint)]

class ImageRect(ctypes.Structure):
    _fields_ = [('left', ctypes.c_int),
                ('top', ctypes.c_int),
                ('width', ctypes.c_int),
                ('height', ctypes.c_int),]

class ControllerDeviceInfo(ctypes.Structure):
    _fields_ = [('deviceType',   ctypes.c_int),
                ('interfaceType', ctypes.c_int),
                ('mainColor', ctypes.c_int * 3),
                ('subColor', ctypes.c_int * 3)]

class IteratorClassBase(object):
    def __init__(self, list):
        self.__list = list

    def __iter__(self):
        return self.__list.__iter__()

class ControllerDeviceType(IteratorClassBase):
    __list = range(3)
    def __init__(self):
        super(ControllerDeviceType, self).__init__(self.__list)

    (
    ControllerDeviceType_SwitchProController,
    ControllerDeviceType_JoyConLeft,
    ControllerDeviceType_JoyConRight
    ) = __list

class ControllerInterfaceType(IteratorClassBase):
    __list = range(2)
    def __init__(self):
        super(ControllerInterfaceType, self).__init__(self.__list)

    (
    ControllerInterfaceType_Bluetooth,
    ControllerInterfaceType_Rail,
    ) = __list

# Capture
class CaptureMode:
    (
        CaptureMode_720p,
        CaptureMode_1080p,
        CaptureMode_DevKit,
        CaptureMode_None,
    ) = range(4)

class Rgb24Color(ctypes.Structure):
    _fields_ = [('red',   ctypes.c_int),
                ('green', ctypes.c_int),
                ('blue',  ctypes.c_int)]

class MorphologyConversionType:
    (
        MorphologyConversionType_Erosion,
        MorphologyConversionType_Dilation,
        MorphologyConversionType_Opening,
        MorphologyConversionType_Closing,
    ) = range(4)

class OcrLanguage:
    (
        OcrLanguage_Japanese,
        OcrLanguage_English,
    ) = range(2)

class OcrSeparateLevel:
    (
        OcrSeparateLevel_Word,
        OcrSeparateLevel_Line,
    ) = range(2)

class LabelingResult(ctypes.Structure):
    _fields_ = [('rect',   ImageRect),
                ('area',   ctypes.c_int),
                ('centerX',ctypes.c_int),
                ('centerY',ctypes.c_int)]

class PathUtility(object):
    @staticmethod
    def get_sdk_root_path():
        relative_root_path = os.path.dirname(os.path.abspath(__file__)) + r"\..\..\..\..\.."
        if os.path.exists(relative_root_path + r"\NintendoSdkRootMark"):
            return os.path.abspath(relative_root_path)
        if os.path.exists(os.environ["NINTENDO_SDK_ROOT"] + r"\NintendoSdkRootMark"):
            return os.environ["NINTENDO_SDK_ROOT"]
        raise Exception("SDK root was not found")


RESOLUTION_WIDTH_720p   = 1280
RESOLUTION_HEIGHT_720p  = 720
RESOLUTION_WIDTH_1080p  = 1920
RESOLUTION_HEIGHT_1080p = 1080

LIBRARY_PATH = PathUtility.get_sdk_root_path() + r"\Tools\AutoTestAssistTools\Libraries"

CONTROLTARGET_EXE_PATH = PathUtility.get_sdk_root_path() + r"\Tools\CommandLineTools\ControlTarget.exe"

# ===========================================================
#        MultiTouchActionDll
# ===========================================================
class MultiTouchActionDll(object):
    def __init__(self, hid_inputer):
        self.__hid_inputer = hid_inputer

        #=============================================================================
        # デフォルト待機時間
        # INFO: 多数のパターンを用意していますが、今後統一する可能性があります。
        #=============================================================================
        self.default_ms_before  = 0
        self.default_ms_press   = 300
        self.default_ms_after   = 0

        self.default_ms_duration_flick  = 150
        self.default_ms_duration_swipe  = 500
        self.default_ms_duration_drag   = 500

        self.default_ms_span_flick  = 16
        self.default_ms_span_swipe  = 16
        self.default_ms_span_drag   = 16

        self.default_ms_wait_flick  = 0
        self.default_ms_wait_swipe  = 0
        self.default_ms_wait_drag   = 100

        self.default_ms_stay_flick  = 0
        self.default_ms_stay_swipe  = 0
        self.default_ms_stay_drag   = 100

        self.default_ms_duration_straight     = 500
        self.default_ms_duration_swipe_multi  = 500
        self.default_ms_duration_flick_multi  = 150
        self.default_ms_duration_drag_multi   = 500
        self.default_ms_duration_rotate       = 1000
        self.default_ms_duration_pinch        = 500

        self.default_ms_span_straight    = 16
        self.default_ms_span_swipe_multi = 16
        self.default_ms_span_flick_multi = 16
        self.default_ms_span_drag_multi  = 16
        self.default_ms_span_rotate      = 16
        self.default_ms_span_pinch       = 16

        self.default_ms_wait_straight     = 0
        self.default_ms_wait_swipe_multi  = 0
        self.default_ms_wait_flick_multi  = 0
        self.default_ms_wait_drag_multi   = 100
        self.default_ms_wait_rotate       = 100
        self.default_ms_wait_pinch        = 100

        self.default_ms_stay_straight    = 0
        self.default_ms_stay_swipe_multi = 0
        self.default_ms_stay_flick_multi = 0
        self.default_ms_stay_drag_multi  = 100
        self.default_ms_stay_rotate      = 100
        self.default_ms_stay_pinch       = 100


        self.default_point_between_length_straight = 100


        #======================
        # DLL 関数型定義
        #======================
        # タッチ状態設定
        self.__hid_inputer.SetTouchState.argtypes = (ctypes.POINTER( ctypes.c_int ), ctypes.POINTER( ctypes.c_int ), ctypes.POINTER( ctypes.c_float ), ctypes.POINTER( ctypes.c_float ), ctypes.c_int)
        self.__hid_inputer.ResetTouchState.argtypes = ()

        # シングルタッチ
        self.__hid_inputer.TouchPressDown.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.TouchPressDownWithFingerId.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.TouchPressUp.argtypes = (ctypes.c_int,)
        self.__hid_inputer.TouchMove.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.Tap.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)

        # シングルドラッグ
        self.__hid_inputer.DragAngle.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_int, ctypes.c_int, ctypes.c_bool)
        self.__hid_inputer.DragPos.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_bool)

        # マルチタッチ操作
        self.__hid_inputer.MultiTap.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int)

        # マルチドラッグ操作
        self.__hid_inputer.MultiTouchAnimator.argtypes = (ctypes.c_void_p, ctypes.c_int,        # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_float, ctypes.c_float,                                               # center_move_deg, center_move_length
                  ctypes.c_void_p, ctypes.c_void_p,                                             # touch_relative_position_x, touch_relative_position_y
                  ctypes.c_float, ctypes.c_float,                                               # coordsys_start_deg, coordsys_rotate_deg
                  ctypes.c_float, ctypes.c_float,                                               # coordsys_start_scale, coordsys_increase_scale
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration,  ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay

        self.__hid_inputer.MultiDragAngle.argtypes = (ctypes.c_void_p, ctypes.c_int,            # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_float, ctypes.c_float,                                               # center_move_deg, center_move_length
                  ctypes.c_void_p, ctypes.c_void_p,                                             # touch_relative_position_x, touch_relative_position_y
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration,  ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay

        self.__hid_inputer.MultiDragAngle.argtypes = (ctypes.c_void_p, ctypes.c_int,            # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_float, ctypes.c_float,                                               # center_move_deg, center_move_length
                  ctypes.c_void_p, ctypes.c_void_p,                                             # touch_relative_position_x, touch_relative_position_y
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration, ms_before, ms_after, ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay

        self.__hid_inputer.MultiDragPos.argtypes = (ctypes.c_void_p, ctypes.c_int,              # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_int, ctypes.c_int,                                                   # center_end_x, center_end_y
                  ctypes.c_void_p, ctypes.c_void_p,                                             # touch_relative_position_x, touch_relative_position_y
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration, ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay


        self.__hid_inputer.Pinch.argtypes = (ctypes.c_void_p, ctypes.c_int,                     # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_void_p,                                                              # touches_relative_deg
                  ctypes.c_float, ctypes.c_float,                                               # start_radius, coordsys_deg
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration,  ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay

        self.__hid_inputer.CenterTouchPinch.argtypes = (ctypes.c_void_p, ctypes.c_int,          # touchIdList, touchIdNum
                ctypes.c_int, ctypes.c_int,                                                     # center_start_x, center_start_y
                ctypes.c_void_p,                                                                # touches_relative_deg
                ctypes.c_float, ctypes.c_float,                                                 # start_radius, coordsys_deg
                ctypes.c_int,  ctypes.c_int,                                                    # ms_duration, ms_span
                ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                      # ms_wait, ms_stay

        self.__hid_inputer.MultiRotate.argtypes = (ctypes.c_void_p, ctypes.c_int,               # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_void_p, ctypes.c_float,                                              # touches_relative_deg, radius
                  ctypes.c_float, ctypes.c_float,                                               # coordsys_start_deg, coordsys_rotate_deg
                  ctypes.c_int, ctypes.c_int,                                                   # ms_duration, ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay


        self.__hid_inputer.CenterTouchMultiRotate.argtypes = (ctypes.c_void_p, ctypes.c_int,    # touchIdList, touchIdNum
                  ctypes.c_int, ctypes.c_int,                                                   # center_start_x, center_start_y
                  ctypes.c_void_p, ctypes.c_float,                                              # touches_relative_deg, radius
                  ctypes.c_float, ctypes.c_float,                                               # coordsys_start_deg, coordsys_rotate_deg
                  ctypes.c_int,  ctypes.c_int,                                                  # ms_duration,  ms_span
                  ctypes.c_int, ctypes.c_int, ctypes.c_bool)                                    # ms_wait, ms_stay

    # -------------------------------
    #     Single Touch Actions
    # -------------------------------
    def touch_press_down(self, touch_id, x, y, finger_id=None, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after
        time.sleep( ms_before / 1000.0 )
        if finger_id is None:
            self.__hid_inputer.TouchPressDown( int( touch_id ), int( x ), int( y ) )
        else:
            self.__hid_inputer.TouchPressDownWithFingerId( int( touch_id ), int( x ), int( y ), int( finger_id) )

        time.sleep( ms_after / 1000.0 )

    def touch_press_up(self, touch_id, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.TouchPressUp( int( touch_id ) )
        time.sleep( ms_after / 1000.0 )

    def touch_move(self, touch_id, x, y, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.TouchMove( int( touch_id ), int( x ), int( y ) )
        time.sleep( ms_after / 1000.0 )

    def touch(self, touch_id, x, y, ms_press=None, ms_before=None, ms_after=None):

        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.Tap( int( touch_id ), int( x ), int( y ), ms_press )
        time.sleep( ms_after / 1000.0 )
        
    def swipe(self, touch_id, start_x, start_y, angle_deg=0.0, length=100.0, ms_duration=None, ms_before=None, ms_after=None,
              ms_span=None, ms_wait=None, ms_stay=None, releases_after=True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_swipe_multi if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_swipe_multi if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_swipe_multi if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_swipe_multi if ms_wait is None else ms_wait

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.DragAngle( int(touch_id), int(start_x), int(start_y), float(angle_deg), float(length), int(ms_duration), int(ms_span), bool(releases_after) )
        time.sleep( ms_after / 1000.0 )

    def flick(self, touch_id, start_x, start_y, angle_deg=0, ms_duration=None, ms_before=None, ms_after=None, ms_span=None,
              ms_wait=None, ms_stay=None, releases_after=True):
        default_flick_length = 100

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_swipe_multi if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_swipe_multi if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_swipe_multi if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_swipe_multi if ms_wait is None else ms_wait

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.DragAngle( int(touch_id), int(start_x), int(start_y), float(angle_deg), float(default_flick_length), int(ms_duration), int(ms_span), bool(releases_after) )
        time.sleep( ms_after / 1000.0 )

    def drag(self, touch_id, start_x, start_y, end_x, end_y, ms_duration=None, ms_before=None, ms_after=None, ms_span=None,
             ms_wait=None, ms_stay=None, releases_after=True):

        ms_duration = self.default_ms_duration_flick if ms_duration is None else ms_duration
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after
        ms_span = self.default_ms_span_flick if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_flick if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_flick if ms_wait is None else ms_wait

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.DragPos( int(touch_id), int(start_x), int(start_y), int(end_x), int(end_y) , int(ms_duration), int(ms_span), bool(releases_after) )
        time.sleep( ms_after / 1000.0 )

    # -------------------------------
    #     Multi Touch Actions
    # -------------------------------
    def __list_to_ctypes_array(self, list, obj_type=ctypes.c_int):
        for (i, value) in enumerate( list ):
            list[ i ] = obj_type( value )

        ctypes_array = len( list ) * obj_type
        return ctypes_array(*list)

    def __position_to_2Array(self, positions, ob_type=ctypes.c_int ):
        return self.__list_to_ctypes_array( [pos[0] for pos in positions], ob_type), self.__list_to_ctypes_array( [pos[1] for pos in positions], ob_type)

    def set_touch_state( self, touch_ids, finger_ids, touches_positions, ms_before=None, ms_after=None):
        touch_id_list = self.__list_to_ctypes_array( touch_ids )
        finger_id_list = self.__list_to_ctypes_array( finger_ids )
        xpos_list, ypos_list = self.__position_to_2Array( touches_positions, ctypes.c_float )

        touch_num = len( touch_id_list )
        
        if len(xpos_list) < touch_num:
            touch_num = len(xpos_list)

        if len(ypos_list) < touch_num:
            touch_num = len(ypos_list)

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.SetTouchState( touch_id_list, finger_id_list, xpos_list, ypos_list, touch_num )
        time.sleep( ms_after / 1000.0 )

    def reset_touch_state(self):
        self.__hid_inputer.ResetTouchState()

    def touch_multi(self, touch_ids, touches_positions, ms_press=None, ms_before=None, ms_after=None):
        touch_id_list = self.__list_to_ctypes_array( touch_ids )
        xpos_list, ypos_list = self.__position_to_2Array( touches_positions, ctypes.c_float )

        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiTap( touch_id_list, len( touch_ids ), xpos_list, ypos_list, int( ms_press ) )
        time.sleep( ms_after / 1000.0 )

    def multitouch_animator(self, touch_ids, center_start_x, center_start_y, touches_relative_positions,
                            center_move_deg=0.0, center_move_length=100.0, coordsys_start_deg=0.0,
                            coordsys_rotate_deg=0.0, coordsys_start_scale = 1.0, coordsys_increase_scale = 0.0,
                            ms_duration=None, ms_span=None, ms_wait=None,
                            ms_stay=None, ms_before=None, ms_after=None, releases_after=True):

        ms_duration = self.default_ms_duration_straight if ms_duration is None else ms_duration
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after
        ms_span = self.default_ms_span_straight if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_straight if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_straight if ms_wait is None else ms_wait

        xpos_list, ypos_list = self.__position_to_2Array( touches_relative_positions, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiTouchAnimator( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
            int( center_start_x ), int( center_start_y ),
            float( center_move_deg ), float( center_move_length ),
            xpos_list, ypos_list,
            float( coordsys_start_deg ), float( coordsys_rotate_deg ),
            float( coordsys_start_scale ), float( coordsys_increase_scale ),
            int( ms_duration ), int( ms_span ),
            int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )

    def __get_circle_default_degs(self, num):
        between_deg = 360.0 / num
        return [between_deg*index for index in range(num)]

    def __get_straight_default_positions(self, num):
        touches_relative_pos = []
        edge_abs = self.default_point_between_length_straight * (num - 1) / 2
        for index in range(num):
              touches_relative_pos.append( ( -edge_abs + self.default_point_between_length_straight * index, 0 ) )

        return touches_relative_pos

    def swipe_multi(self, touch_ids, center_start_x, center_start_y, center_move_deg, center_move_length = 100.0,
                    touches_relative_pos=None, coordsys_deg=0.0, ms_duration = None, ms_span = None, ms_wait = None,
                    ms_stay = None, ms_before = None, ms_after = None, releases_after = True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_swipe_multi if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_swipe_multi if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_swipe_multi if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_swipe_multi if ms_wait is None else ms_wait

        if touches_relative_pos is None:
            touches_relative_pos = self.__get_straight_default_positions(len(touch_ids))

        xpos_list, ypos_list = self.__position_to_2Array( touches_relative_pos, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiDragAngle( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
                      int( center_start_x ), int( center_start_y ),
                      float( center_move_deg ), float( center_move_length ),
                      xpos_list, ypos_list,
                      int( ms_duration ),  int( ms_span ),
                      int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )
        
    def flick_multi(self, touch_ids, center_start_x, center_start_y, center_move_deg, ms_duration = None,
                    touches_relative_pos=None, coordsys_deg = 0.0, ms_span = None, ms_wait = None,
                    ms_stay = None, ms_before = None, ms_after = None, releases_after = True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_swipe_multi if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_swipe_multi if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_swipe_multi if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_swipe_multi if ms_wait is None else ms_wait

        if touches_relative_pos is None:
            touches_relative_pos = self.__get_straight_default_positions(len(touch_ids))

        xpos_list, ypos_list = self.__position_to_2Array( touches_relative_pos, ctypes.c_float )
        #=======================================================================================
        # TIPS: この関数は、drag の簡易版として、中心点の移動距離を固定値で 600 に指定している。
        #=======================================================================================
        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiDragAngle( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
                      int( center_start_x ), int( center_start_y ),
                      float( center_move_deg ), float( 600 ),
                      xpos_list, ypos_list,
                      int( ms_duration ), int( ms_span ),
                      int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )
        
    def drag_multi(self, touch_ids, center_start_x, center_start_y, center_end_x, center_end_y,
                   touches_relative_pos=None, coordsys_deg=0.0, ms_duration = None, ms_span = None, ms_wait = None,
                   ms_stay = None, ms_before = None, ms_after = None, releases_after = True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_drag_multi if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_drag_multi if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_drag_multi if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_drag_multi if ms_wait is None else ms_wait

        if touches_relative_pos is None:
            touches_relative_pos = self.__get_straight_default_positions(len(touch_ids))

        xpos_list, ypos_list = self.__position_to_2Array( touches_relative_pos, ctypes.c_float )


        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiDragPos( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
                      int( center_start_x ), int( center_start_y ),
                      int( center_end_x ), int( center_end_y ),
                      xpos_list, ypos_list,
                      int( ms_duration ),  int( ms_span ),
                      int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )

    def pinch(self, touch_ids, center_x, center_y, radius_increase,
              start_radius=200, coordsys_deg=0, touches_relative_deg=None,
              ms_duration=None, ms_span=None, ms_wait=None, ms_stay=None, ms_before=None, ms_after=None,
              releases_after=True):


        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_pinch if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_pinch if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_pinch if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_pinch if ms_wait is None else ms_wait

        if touches_relative_deg is None:
            touches_relative_deg = self.__get_circle_default_degs(len(touch_ids))

        rotate_touch_touch_deg = self.__list_to_ctypes_array( touches_relative_deg, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.Pinch( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
            int( center_x ), int( center_y ),
            rotate_touch_touch_deg,
            float( start_radius ), float( radius_increase ),
            int( ms_duration ),  int( ms_span ),
            int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )

    def center_touch_pinch(self, center_touch_id, center_x, center_y,
                           pinch_touch_ids, radius_increase, start_radius=200, coordsys_deg=0, touches_relative_deg=None,
                           ms_duration=None, ms_span=None, ms_wait=None, ms_stay=None, ms_before=None, ms_after=None,
                           releases_after=True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_pinch if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_pinch if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_pinch if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_pinch if ms_wait is None else ms_wait


        if touches_relative_deg is None:
            touches_relative_deg = self.__get_circle_default_degs(len(pinch_touch_ids))

        pinch_touch_ids.insert( 0, center_touch_id )

        rotate_touch_touch_deg = self.__list_to_ctypes_array( touches_relative_deg, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.CenterTouchPinch( self.__list_to_ctypes_array( pinch_touch_ids ), len( pinch_touch_ids ),
            int( center_x ), int( center_y ),
            rotate_touch_touch_deg,
            float( start_radius ), float( radius_increase ),
            int( ms_duration ), int( ms_span ),
            int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )

    def rotate(self, touch_ids, center_x, center_y,
               rotate_deg, start_deg = 0, touches_relative_deg=None, radius=200,
               ms_duration=None, ms_span=None, ms_wait=None, ms_stay=None, ms_before=None, ms_after=None,
               releases_after=True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_rotate if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_rotate if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_rotate if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_rotate if ms_wait is None else ms_wait

        if touches_relative_deg is None:
            touches_relative_deg = self.__get_circle_default_degs(len(touch_ids))

        rotate_touch_touch_deg = self.__list_to_ctypes_array( touches_relative_deg, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.MultiRotate( self.__list_to_ctypes_array( touch_ids ), len( touch_ids ),
            int( center_x ), int( center_y ),
            rotate_touch_touch_deg, float( radius ),
            float( start_deg ), float( rotate_deg ),
            int( ms_duration ),  int( ms_span ),
            int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )

    def center_touch_rotate(self, center_touch_id, center_x, center_y,
                            rotate_touch_ids, rotate_deg, start_deg=0, touches_relative_deg=None, radius=200,
                            ms_duration=None, ms_span=None, ms_wait=None, ms_stay=None, ms_before=None, ms_after=None,
                            releases_after=True):

        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after  = self.default_ms_after  if ms_after  is None else ms_after
        ms_duration = self.default_ms_duration_rotate if ms_duration is None else ms_duration
        ms_span = self.default_ms_span_rotate if ms_span is None else ms_span
        ms_stay = self.default_ms_stay_rotate if ms_stay is None else ms_stay
        ms_wait = self.default_ms_wait_rotate if ms_wait is None else ms_wait


        if touches_relative_deg is None:
            touches_relative_deg = self.__get_circle_default_degs(len(rotate_touch_ids))

        rotate_touch_ids.insert( 0, center_touch_id )

        rotate_touch_touch_deg = self.__list_to_ctypes_array( touches_relative_deg, ctypes.c_float )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.CenterTouchMultiRotate( self.__list_to_ctypes_array( rotate_touch_ids ), len( rotate_touch_ids ),
            int( center_x ), int( center_y ),
            rotate_touch_touch_deg, float( radius ),
            float( start_deg ), float( rotate_deg ),
            int( ms_duration ),  int( ms_span ),
            int( ms_wait ), int( ms_stay ), bool( releases_after ) )
        time.sleep( ms_after / 1000.0 )


# ===========================================================
#        CaptureProcessorDll
# ===========================================================
class CaptureProcessorDll():
    __capture_processor_path = LIBRARY_PATH + "/CaptureProcessor.dll"

    def __init__(self, is_preview=True, capture_device_id=0, target_serial="", capture_mode=CaptureMode.CaptureMode_720p):    
        self.__capture_processor = None
        self.__capture_mode = capture_mode

        # CaptureProcessor ライブラリでのカラーチャンネル数（ColorChannelCount）は 3 で固定
        self.__capture_image_channel = 3

        #===========================
        # DLL ライブラリ読み込み
        #===========================
        self.__capture_processor = ctypes.CDLL(self.__capture_processor_path)

        #===========================
        # DLL 関数型定義
        #===========================
        # 初期化処理
        # void InitializeForCaptureDevice(const char* deviceName, bool isPreview);
        self.__capture_processor.InitializeForCaptureDevice.argtypes = (ctypes.c_int, ctypes.c_bool, ctypes.c_int)

        # CaptureResult InitializeForDevkitCapture(const SerialNumberString serialNumber, bool isPreview);
        self.__capture_processor.InitializeForDevkitCapture.argtypes = (SerialNumberString, ctypes.c_bool)
        self.__capture_processor.InitializeForDevkitCapture.restype = ctypes.c_int

        # キャプチャデバイス数取得
        # CaptureResult GetCaptureDeviceCount(int* pOutDeviceCount);
        self.__capture_processor.GetCaptureDeviceCount.argtypes = (ctypes.POINTER(ctypes.c_int),)
        self.__capture_processor.GetCaptureDeviceCount.restype = ctypes.c_int

        # キャプチャデバイス名取得
        # CaptureResult GetCaptureDeviceName(char* pOutNameBuffer, int nameBufferSize, int deviceId);
        self.__capture_processor.GetCaptureDeviceName.argtypes = (ctypes.c_char_p, ctypes.c_int, ctypes.c_int)
        self.__capture_processor.GetCaptureDeviceName.restype = ctypes.c_int

        # 画像保存
        # CaptureResult SaveImageFile(const char* filepath, ImageIdType id);
        self.__capture_processor.SaveImageFile.argtypes = (ctypes.c_char_p, ImageIdType)
        self.__capture_processor.SaveImageFile.restype = ctypes.c_int

        # 画像取得
        # CaptureResult GetImageFromCaptureDevice( ImageIdType *pOutImageId );
        self.__capture_processor.GetImageFromCaptureDevice.argtypes = (ctypes.POINTER(ImageIdType),)
        self.__capture_processor.GetImageFromCaptureDevice.restype = ctypes.c_int

        # CaptureResult LoadImageFile(ImageIdType *pOutImageId, const char *filepath);
        self.__capture_processor.LoadImageFile.argtypes = (ctypes.POINTER(ImageIdType), ctypes.c_char_p)
        self.__capture_processor.LoadImageFile.restype = ctypes.c_int

        # 画像解放
        # CaptureResult ReleaseImage(ImageIdType id);
        self.__capture_processor.ReleaseImage.argtypes = (ImageIdType,)
        self.__capture_processor.ReleaseImage.restype = ctypes.c_int

        # CaptureResult ReleaseAllImages();
        self.__capture_processor.ReleaseAllImages.restype = ctypes.c_int

        # テンプレートマッチング
        # MatchingResult DetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold );
        self.__capture_processor.DetectObject.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_int), ImageIdType, ctypes.c_double)
        self.__capture_processor.DetectObject.restype = ctypes.c_int

        # MatchingResult WaitDetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold, int timeout);
        self.__capture_processor.WaitDetectObject.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_int), ImageIdType, ctypes.c_double, ctypes.c_int)
        self.__capture_processor.WaitDetectObject.restype = ctypes.c_bool

        # MatchingResult DetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, FilterIdType filterId);
        self.__capture_processor.DetectObjectWithFilter.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_int), ImageIdType, ctypes.c_double, FilterIdType)
        self.__capture_processor.DetectObjectWithFilter.restype = ctypes.c_int

        # MatchingResult WaitDetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, FilterIdType filterId);
        self.__capture_processor.WaitDetectObjectWithFilter.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_int), ImageIdType, ctypes.c_double, ctypes.c_int, FilterIdType)
        self.__capture_processor.WaitDetectObjectWithFilter.restype = ctypes.c_int

        # キャプチャ画像サイズ取得
        # CaptureResult GetImageSize(int* pOutWidth, int* pOutHeight, ImageIdType id);
        self.__capture_processor.GetImageSize.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_int), ImageIdType)
        self.__capture_processor.GetImageSize.restype = ctypes.c_int

        # キャプチャ画像データ取得
        # CaptureResult GetImageData(void* pOutColorBuffer, size_t colorBufferSize, ImageIdType id);
        self.__capture_processor.GetImageData.argtypes = (ctypes.c_void_p, ctypes.c_size_t, ImageIdType)
        self.__capture_processor.GetImageData.restype = ctypes.c_int

        # 色取得
        # CaptureResult GetDotColor( int pOutColor[3],  ImageIdType id, int x, int y );
        self.__capture_processor.GetDotColor.argtypes =  ( ctypes.POINTER(Rgb24Color), ImageIdType, ctypes.c_int, ctypes.c_int)
        self.__capture_processor.GetDotColor.restype = ctypes.c_int

        # CaptureResult DetectDotColor( int x, int y, int lowerColor[3], int upperColoer[3] );
        self.__capture_processor.DetectDotColor.argtypes = (ctypes.c_int, ctypes.c_int, Rgb24Color, Rgb24Color )
        self.__capture_processor.DetectDotColor.restype = ctypes.c_int
        
        # MatchingResult WaitDetectDotColor( int x, int y,  int lowerColor[3], int upperColoer[3], int timeout );
        self.__capture_processor.WaitDetectDotColor.argtypes = (ctypes.c_int, ctypes.c_int, Rgb24Color, Rgb24Color, ctypes.c_int )
        self.__capture_processor.WaitDetectDotColor.restype = ctypes.c_int

        # ラベリング
        # CaptureResult ExecuteLabeling( int *pOutResultAreaCount, LabelingResultHandleType *pOutLabelingResultHandle,  ImageIdType imageId );
        self.__capture_processor.ExecuteLabeling.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(LabelingResultHandleType), ImageIdType )
        self.__capture_processor.ExecuteLabeling.restype = ctypes.c_int
        
        # CaptureResult GetLabelingResults( int *pOutResultsCount, LabelingResult pOutResultArray[], LabelingResultHandleType labelingResultHandle, int arrayCount );
        self.__capture_processor.GetLabelingResults.argtypes = (ctypes.POINTER(ctypes.c_int), ctypes.POINTER(LabelingResult), LabelingResultHandleType, ctypes.c_int)
        self.__capture_processor.GetLabelingResults.restype = ctypes.c_int

        # CaptureResult ReleaseLabelingResult(LabelingResultHandleType labelingResultHandle);
        self.__capture_processor.ReleaseLabelingResult.argtypes = (LabelingResultHandleType,)
        self.__capture_processor.ReleaseLabelingResult.restype = ctypes.c_int

        # OCR
        # MatchingResult DetectTextForImage( ImageRect *pOutRect, ImageIdType id, const char regexString[], OcrLanguage language );
        self.__capture_processor.DetectTextForImage.argtypes = (ctypes.POINTER(ImageRect), ImageIdType, ctypes.c_char_p, ctypes.c_int, ctypes.c_int )
        self.__capture_processor.DetectTextForImage.restype = ctypes.c_int

        # CaptureResult ExecuteOcr( int *pOutResultCount, OcrResultHandleType *pOutOcrResultHandle, ImageIdType id,  OcrSeparateLevel separate, OcrLanguage language );
        self.__capture_processor.ExecuteOcr.argtypes = ( ctypes.POINTER(ctypes.c_int), ctypes.POINTER(OcrResultHandleType), ImageIdType, ctypes.c_int, ctypes.c_int )
        self.__capture_processor.ExecuteOcr.restype = ctypes.c_int

        # CaptureResult GetOcrResultStringSize( int *pOutResultSize, OcrResultHandleType ocrResultHandle, int number );
        self.__capture_processor.GetOcrResultStringSize.argtypes = ( ctypes.POINTER(ctypes.c_int), OcrResultHandleType, ctypes.c_int )
        self.__capture_processor.GetOcrResultStringSize.restype = ctypes.c_int

        # CaptureResult GetOcrResultString( ImageRect *pOutRect, int *pOutResultSize, char pOutResultString[], int bufferSize, OcrResultHandleType ocrResultHandle, int number );
        self.__capture_processor.GetOcrResultString.argtypes = ( ctypes.POINTER(ImageRect), ctypes.POINTER(ctypes.c_int), ctypes.c_char_p, ctypes.c_int, OcrResultHandleType, ctypes.c_int )
        self.__capture_processor.GetOcrResultString.restype = ctypes.c_int

        # CaptureResult ReleaseOcrResult( OcrResultHandleType ocrResultHandle );
        self.__capture_processor.ReleaseOcrResult.argtypes = (OcrResultHandleType,)
        self.__capture_processor.ReleaseOcrResult.restype = ctypes.c_int

        # フィルター
        # CaptureResult CreateFilter( FilterIdType *pOutFilterId );
        self.__capture_processor.CreateFilter.argtypes = ( ctypes.POINTER(FilterIdType), )
        self.__capture_processor.CreateFilter.restype = ctypes.c_int

        # CaptureResult DeleteFilter( FilterIdType filterId );
        self.__capture_processor.DeleteFilter.argtypes = ( FilterIdType, )
        self.__capture_processor.DeleteFilter.restype = ctypes.c_int

        # CaptureResult ApplyFilter( ImageIdType *pOutId, ImageIdType id, FilterIdType filterId );
        self.__capture_processor.ApplyFilter.argtypes = ( ctypes.POINTER(ImageIdType), ImageIdType, FilterIdType )
        self.__capture_processor.ApplyFilter.restype = ctypes.c_int

        # CaptureResult SetBinarizationFilter( FilterIdType filterId, int binarizeThreshold );
        self.__capture_processor.SetBinarizationFilter.argtypes = ( FilterIdType, ctypes.c_int )
        self.__capture_processor.SetBinarizationFilter.restype = ctypes.c_int

        # CaptureResult SetColorRangeFilter( FilterIdType filterId, int lowerColor[3], int upperColor[3] );
        self.__capture_processor.SetColorRangeFilter.argtypes = ( FilterIdType, Rgb24Color, Rgb24Color)
        self.__capture_processor.SetColorRangeFilter.restype = ctypes.c_int

        # CaptureResult SetMorphologyFilter( FilterIdType filterId, MorphologyConverType type, int iterations );
        self.__capture_processor.SetMorphologyFilter.argtypes = ( FilterIdType, ctypes.c_int, ctypes.c_int )
        self.__capture_processor.SetMorphologyFilter.restype = ctypes.c_int

        # その他
        # CaptureResult OpenImageWindow( const char* windowName, ImageIdType id );
        self.__capture_processor.OpenImageWindow.argtypes = (ctypes.c_char_p, ImageIdType)
        self.__capture_processor.OpenImageWindow.restype = ctypes.c_int

        # CaptureResult CloseImageWindow( const char* windowName );
        self.__capture_processor.CloseImageWindow.argtypes = (ctypes.c_char_p,)
        self.__capture_processor.CloseImageWindow.restype = ctypes.c_int

        # CaptureResult ExtractImage( ImageIdType *pOutImageId, ImageIdType id, ImageRect rect );
        self.__capture_processor.ExtractImage.argtypes = (ctypes.POINTER(ImageIdType), ImageIdType, ImageRect )
        self.__capture_processor.ExtractImage.restype = ctypes.c_int

        # 初期化処理
        if capture_mode == CaptureMode.CaptureMode_720p or capture_mode == CaptureMode.CaptureMode_1080p:
            self.__capture_processor.InitializeForCaptureDevice( int(capture_device_id), bool( is_preview ), int(capture_mode) )
        elif capture_mode == CaptureMode.CaptureMode_DevKit and target_serial is not None:
            self.__capture_processor.InitializeForDevkitCapture( SerialNumberString(target_serial.encode()), is_preview)
        elif capture_mode == CaptureMode.CaptureMode_None:
            pass

    def get_image_data( self ):
        c_int_p     = ctypes.POINTER(ctypes.c_int)
        c_uint32_p  = ctypes.POINTER(ctypes.c_uint32)

        # 画像取得
        id = ImageIdType(0)
        self.__capture_processor.GetImageFromCaptureDevice( ctypes.pointer( id ) )

        # 画像サイズ確認
        width = ctypes.c_int(0)
        height = ctypes.c_int(0)
        
        result = self.__capture_processor.GetImageSize( ctypes.pointer( width ), ctypes.pointer( height ), id )
        if( result < 0 ):
            print("get_image_data: Failed to get the size of image.")
            return None

        # 画像バッファ確保
        imageDataSize = width.value * height.value * self.__capture_image_channel
        imageDataBufferType = ctypes.c_uint8 * imageDataSize
        imageDataBuffer = imageDataBufferType(0)

        # データ取得
        result = self.__capture_processor.GetImageData( imageDataBuffer, imageDataSize, id )
        if( result < 0 ):
            print("get_image_data: Failed to get image data.")
            self.__capture_processor.ReleaseImage( id )
            return None

        self.__capture_processor.ReleaseImage( id )
        return height.value, width.value, self.__capture_image_channel, imageDataBuffer

    def get_image( self ):
        image_id = ImageIdType(0)
        if( self.__capture_processor.GetImageFromCaptureDevice( ctypes.pointer( image_id ) ) >= 0 ):
            return image_id
        else:
            print("Failed to get image.")
            return None

    def load_image( self, image_path ):
        image_id = ImageIdType(0)
        if( self.__capture_processor.LoadImageFile( ctypes.pointer( image_id ), image_path.encode() ) >= 0 ):
            return image_id
        else:
            print("Failed to load image file.")
            return None

    def save_image( self, filepath,  image_id ):
        return self.__capture_processor.SaveImageFile( filepath.encode(), image_id ) == 0            

    def release_image( self, image_id ):
        self.__capture_processor.ReleaseImage( image_id )

    def detect_object( self, image_id, threshold=0.95, capture_image_filter=None, template_image_filter=None):
        x = ctypes.c_int( 0 )
        y = ctypes.c_int( 0 )

        result = 0
        template_image_id = image_id
        is_delete_template_image = False
        if template_image_filter is not None:
            filtered_template_image_id = self.apply_filter( template_image_id, template_image_filter )
            if filtered_template_image_id is not None:
                template_image_id = filtered_template_image_id
                is_delete_template_image = True

        if capture_image_filter is None:
            result = self.__capture_processor.DetectObject(ctypes.pointer( x ), ctypes.pointer( y ),  template_image_id, threshold )
        else:
            result = self.__capture_processor.DetectObjectWithFilter(ctypes.pointer( x ), ctypes.pointer( y ),  template_image_id, threshold, capture_image_filter )

        if is_delete_template_image:
            self.release_image(template_image_id)

        if( result == 0 ):
            return x.value, y.value
        else:
            # テンプレートマッチング失敗
            return None

    def detect_object_with_image( self, image_path, threshold=0.95, capture_image_filter=None, template_image_filter=None):
        # 画像読み込み
        image_id = self.load_image( image_path.encode() )
        if( image_id == 0 ):
            return 0,0,0

        # テンプレートマッチング
        result = self.detect_object( image_id, threshold=threshold, capture_image_filter=capture_image_filter, template_image_filter=template_image_filter )
            
        # 画像解放
        self.__capture_processor.ReleaseImage( image_id )

        return result


    def wait_detect_object( self, image_id, timeout=1000, threshold=0.95, capture_image_filter=None, template_image_filter=None):
        x = ctypes.c_int( 0 )
        y = ctypes.c_int( 0 )

        result = 0
        template_image_id = image_id
        is_delete_template_image = False
        if template_image_filter is not None:
            filtered_template_image_id = self.apply_filter( template_image_id, template_image_filter )
            if filtered_template_image_id is not None:
                template_image_id = filtered_template_image_id
                is_delete_template_image = True


        if capture_image_filter is None:
            result = self.__capture_processor.WaitDetectObject(  ctypes.pointer( x ), ctypes.pointer( y ), template_image_id, ctypes.c_double(threshold), int(timeout) )
        else :
            result = self.__capture_processor.WaitDetectObjectWithFilter(  ctypes.pointer( x ), ctypes.pointer( y ), template_image_id, ctypes.c_double(threshold), int(timeout), capture_image_filter )

        if is_delete_template_image:
            self.release_image(template_image_id)

        if( result == 0 ):
            return x.value, y.value
        else:
            # テンプレートマッチ失敗
            return None

    def close(self):
        self.__capture_processor.Finalize()
        del self.__capture_processor
        self.__capture_processor = None

    def get_center_position(self):
        if self.__capture_mode == CaptureMode.CaptureMode_720p:
            return RESOLUTION_WIDTH_720p / 2, RESOLUTION_HEIGHT_720p / 2
        elif self.__capture_mode == CaptureMode.CaptureMode_1080p:
            return RESOLUTION_WIDTH_1080p / 2, RESOLUTION_HEIGHT_1080p / 2
        else:
            return 0, 0

    def __list_to_color_array(self, data, default_value=[0,0,0]):
        if (isinstance( data, tuple )):
            data = list(data)

        if default_value and (not(isinstance(data,list)) or len( data ) is not 3):
            data = default_value
            
        return Rgb24Color(*data)

    def get_dot_color(self, image_id, x, y ):
        color  = Rgb24Color(0)
        result = self.__capture_processor.GetDotColor( ctypes.pointer(color), image_id, x, y )
        
        if result is None:
            return None

        return color

    def detect_dot_color(self, x, y, lower_color=[0,0,0], upper_color=[255,255,255]):
        lower_color = self.__list_to_color_array(lower_color, [0,0,0])
        upper_color = self.__list_to_color_array(upper_color, [255,255,255])

        result = self.__capture_processor.DetectDotColor( ctypes.c_int(x), ctypes.c_int(y), lower_color, upper_color)
        if result is 0:
            return True

        return False

    def wait_detect_dot_color(self, x, y, lower_color=[0,0,0], upper_color=[255,255,255], timeout=1000):
        lower_color = self.__list_to_color_array(lower_color, [0,0,0])
        upper_color = self.__list_to_color_array(upper_color, [255,255,255])

        result = self.__capture_processor.WaitDetectDotColor( ctypes.c_int(x), ctypes.c_int(y), lower_color, upper_color, ctypes.c_int(timeout))
        if result is 0:
            return True

        return False

    def create_filter(self):
        filter_id = FilterIdType(0)
        result = self.__capture_processor.CreateFilter(ctypes.pointer(filter_id))

        if result is not 0:
            return None
        return filter_id

    def delete_filter(self, filter_id ):
        self.__capture_processor.DeleteFilter(filter_id)
    
    def apply_filter(self, image_id, filter_id ):
        result_image_id = ImageIdType(0)
        result = self.__capture_processor.ApplyFilter(ctypes.pointer(result_image_id), image_id, filter_id)

        if result is not 0:
            return None

        return result_image_id

    def set_binarization_filter(self, filter_id, binarize_threshold=128 ):
        self.__capture_processor.SetBinarizationFilter( filter_id, binarize_threshold )

    def set_color_range_filter(self, filter_id, lower_color=[0,0,0], upper_color=[255,255,255] ):
        lower_color = self.__list_to_color_array(lower_color, [0,0,0])
        upper_color = self.__list_to_color_array(upper_color, [255,255,255])

        self.__capture_processor.SetColorRangeFilter( filter_id, lower_color, upper_color )

    def set_morphology_filter(self, filter_id, morphology_type, iterations ):
        self.__capture_processor.SetMorphologyFilter( filter_id, morphology_type, iterations )

    def detect_text(self, image_id, regex_string, separate=OcrSeparateLevel.OcrSeparateLevel_Word, language=OcrLanguage.OcrLanguage_English):
        result_rect = ImageRect(0,0,0,0)
        result = self.__capture_processor.DetectTextForImage( ctypes.pointer(result_rect), image_id, regex_string, separate, language )
        
        if result is not 0:
            return None

        return result_rect

    def execute_ocr(self, image_id, separate=OcrSeparateLevel.OcrSeparateLevel_Word, language=OcrLanguage.OcrLanguage_English):
        result_count = ctypes.c_int(0)
        result_handle = OcrResultHandleType(0)
        result = self.__capture_processor.ExecuteOcr( ctypes.pointer(result_count), ctypes.pointer(result_handle), image_id, separate, language )

        if result is not 0:
            return None

        return (result_count.value, result_handle)
    
    def get_ocr_result(self, result_handle, number):
        text_count = ctypes.c_int(0)

        result = self.__capture_processor.GetOcrResultStringSize( ctypes.pointer( text_count ), result_handle,  number )
        if result is not 0:
            return None

        result_text = ctypes.create_string_buffer(text_count.value)
        result_rect = ImageRect(0,0,0,0)
        text_length = ctypes.c_int(0)
        result = self.__capture_processor.GetOcrResultString( ctypes.pointer(result_rect), ctypes.pointer(text_length), result_text, text_count.value, result_handle, number )
        if result is not 0:
            return None
        return ( result_text.value.decode(errors='ignore'), result_rect )

    def release_ocr_result(self, result_handle):
        self.__capture_processor.ReleaseOcrResult( result_handle )

    def execute_labeling(self, image_id):
        result_count = ctypes.c_int(0)
        result_handle = LabelingResultHandleType(0)
        result = self.__capture_processor.ExecuteLabeling(ctypes.pointer(result_count), ctypes.pointer(result_handle), image_id )
        if result is not 0:
            return None

        return (result_count.value, result_handle)
    
    def get_labeling_results(self, result_handle, results_count ):
        if results_count <= 0:
            return []

        accept_results_count = ctypes.c_int(0)
        results_list = (LabelingResult * results_count)()

        result = self.__capture_processor.GetLabelingResults( ctypes.pointer(accept_results_count), results_list, result_handle, results_count)
        
        if result is not 0:
            return []

        return results_list
    
    def release_labeling_result(self, result_handle):
        self.__capture_processor.ReleaseLabelingResult( result_handle )

    def show_image(self, image_id, window_name=u"HACDllDriver" ):
        self.__capture_processor.OpenImageWindow( window_name.encode(), image_id )

    def close_image(self, window_name="HACDllDriver" ):
        self.__capture_processor.CloseImageWindow( window_name.encode() )

    def extract_image(self, image_id, image_rect):
        if not( isinstance(image_rect, ImageRect )):
            if isinstance(image_rect, list ) and len(image_rect) >= 4:
                image_rect = ImageRect(*image_rect)
            elif isinstance(image_rect, tupple ) and len(image_rect) >= 4:
                image_rect = ImageRect(*image_rect)
            else:
                image_rect = ImageRect(0,0,100,100)

        result_image_id = ImageIdType(0)
        result = self.__capture_processor.ExtractImage(ctypes.pointer(result_image_id), image_id, image_rect)

        if result is not 0:
            return None
        return result_image_id

    @classmethod
    def get_capture_device_list(cls):
        device_list = []

        capture_processor = ctypes.CDLL(cls.__capture_processor_path)
        
        # キャプチャデバイス数取得関数
        capture_processor.GetCaptureDeviceCount.argtypes = (ctypes.POINTER(ctypes.c_int),)
        capture_processor.GetCaptureDeviceCount.restype = ctypes.c_int

        # キャプチャデバイス名取得関数
        capture_processor.GetCaptureDeviceName.argtypes = (ctypes.c_char_p, ctypes.c_int, ctypes.c_int)
        capture_processor.GetCaptureDeviceName.restype = ctypes.c_int

        device_count = ctypes.c_int( 0 )
        device_name = ctypes.create_string_buffer(255)

        # デバイスリスト取得失敗
        if capture_processor.GetCaptureDeviceCount( ctypes.pointer( device_count ) ) == 0:
            for i in range( 0, device_count.value ):
                capture_processor.GetCaptureDeviceName( device_name, 255, i )
                device_list.append( device_name.value.decode() )

        del capture_processor

        return device_list


# ===========================================================
#        Controller
# ===========================================================
class HACControllerDll():
    def __init__(self, hid_inputer, controller_id ):
        self.__controller_id = controller_id
        self.__hid_inputer = hid_inputer

        #======================
        # デフォルト待機時間
        #======================
        self.default_ms_before = 0
        self.default_ms_press = 120
        self.default_ms_after = 90
        self.default_ms_span = 90

        #======================
        # DLL 関数型定義
        #======================
        # デバイス情報取得
        self.__hid_inputer.GetConnectedControllerDeviceInfo.argtypes = (ctypes.POINTER(ControllerDeviceInfo), ctypes.c_int)
        self.__hid_inputer.GetConnectedControllerDeviceInfo.restype  = ctypes.c_int

        # ボタン操作
        self.__hid_inputer.PressButton.argtypes          = (ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.PressMultiButton.argtypes     = (ctypes.c_int, ctypes.c_void_p, ctypes.c_int)
        self.__hid_inputer.ReleaseButton.argtypes        = (ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.ReleaseMultiButton.argtypes   = (ctypes.c_int, ctypes.c_void_p, ctypes.c_int)
        self.__hid_inputer.PushButton.argtypes           = (ctypes.c_int, ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.PushMultiButton.argtypes      = (ctypes.c_int, ctypes.c_void_p, ctypes.c_int, ctypes.c_int)
        # スティック操作
        self.__hid_inputer.HoldAnalogStick.argtypes      = (ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float)
        self.__hid_inputer.ReleaseAnalogStick.argtypes   = (ctypes.c_int, ctypes.c_int)
        self.__hid_inputer.TiltAnalogStick.argtypes      = (ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_int)
        self.__hid_inputer.PushMultiSeq.argtypes         = (ctypes.c_int, ctypes.c_void_p, ctypes.c_int, ctypes.c_int)
        # コントローラー状態設定
        self.__hid_inputer.SetControllerState.argtypes   = (ctypes.c_int, ctypes.POINTER( ctypes.c_int), ctypes.c_int, ctypes.POINTER(ctypes.c_int), ctypes.POINTER(ctypes.c_float), ctypes.POINTER(ctypes.c_float), ctypes.c_int)

    def close(self):
        self.__hid_inputer.ReleaseController.argtypes    = (ctypes.c_int,)
        self.__hid_inputer.ReleaseController( int( self.__controller_id ) )

    def get_device_info(self):
        device_info = ControllerDeviceInfo()
        result = self.__hid_inputer.GetConnectedControllerDeviceInfo(ctypes.pointer(device_info), self.__controller_id)
        if result != 0:
            return None
        return device_info
    
    def set_neutral(self):
        self.__hid_inputer.ResetControllerState.argtypes = (ctypes.c_int,)
        self.__hid_inputer.ResetControllerState( int( self.__controller_id ) )

    def btn_to_num( self, button ):
        self.button_to_flag = {
            "A":            0,
            "B":            1,
            "X":            2,
            "Y":            3,
            "STICK_L":      4,
            "STICK_R":      5,
            "L":            6,
            "R":            7,
            "ZL":           8,
            "ZR":           9,
            "START":        10,
            "SELECT":       11,
            "LEFT":         12,
            "UP":           13,
            "RIGHT":        14,
            "DOWN":         15,
            "SL":           16,
            "SR":           17,
            "HOME":         -2,
            "CAPTURE":      -1,
        }

        return self.button_to_flag[ button ]

    # コントローラー状態設定
    def set_controller_state( self, button_list, l_stick_degree=None, l_stick_power=None, r_stick_degree=None, r_stick_power=None, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after


        _button_list = []
        for i in range( 0,  len( button_list) ):
            _button_list.append( self.btn_to_num( button_list[ i ] ) )

        list_array = ctypes.c_int *  len( button_list)
        button_list_array = list_array( *_button_list )

        stick_type_list = []
        stick_power_list = []
        stick_degree_list = []
        if l_stick_degree is not None and l_stick_power is not None:
            stick_type_list.append( ctypes.c_int(Stick.Stick_L) )
            stick_power_list.append( ctypes.c_float(l_stick_power) )
            stick_degree_list.append( ctypes.c_float(l_stick_degree) )

        if r_stick_degree is not None and r_stick_power is not None:
            stick_type_list.append( ctypes.c_int(Stick.Stick_R) )
            stick_power_list.append( ctypes.c_float(r_stick_power) )
            stick_degree_list.append( ctypes.c_float(r_stick_degree) )


        stick_type_list     = (ctypes.c_int * len(stick_type_list))( *stick_type_list )
        stick_power_list    = (ctypes.c_float * len(stick_type_list))( *stick_power_list )
        stick_degree_list   = (ctypes.c_float * len(stick_type_list))( *stick_degree_list )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.SetControllerState( int( self.__controller_id ), button_list_array, int( len( button_list) ), stick_type_list, stick_degree_list, stick_power_list, len(stick_type_list))
        time.sleep( ms_after / 1000.0 )

    # ボタン押下
    def press(self, button, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after
        
        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.PressButton( int( self.__controller_id ), self.btn_to_num( button ) )
        time.sleep( ms_after / 1000.0 )

    def press_multi(self, button_list, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        button_list_length = len( button_list)

        _button_list = []
        for i in range( 0, button_list_length ):
            _button_list.append( self.btn_to_num( button_list[ i ] ) )

        list_array = ctypes.c_int * button_list_length
        button_list_array = list_array( *_button_list )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.PressMultiButton( int( self.__controller_id ), button_list_array, int( button_list_length ))
        time.sleep( ms_after / 1000.0 )

    # ボタンを離す
    def release(self, button, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.ReleaseButton( int( self.__controller_id ), self.btn_to_num( button ) )
        time.sleep( ms_after / 1000.0 )

    def release_multi(self, button_list, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        button_list_length = len( button_list)

        _button_list = []
        for i in range( 0, button_list_length ):
            _button_list.append( self.btn_to_num( button_list[ i ] ) )

        list_array = ctypes.c_int * button_list_length
        button_list_array = list_array( *_button_list )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.ReleaseMultiButton( int( self.__controller_id ), button_list_array, int( button_list_length ) )
        time.sleep( ms_after / 1000.0 )

    # ボタン押下 -> 離す
    def push(self, button, ms_press=None, ms_before=None, ms_after=None):
        ms_press    = self.default_ms_press if ms_press is None else ms_press
        ms_before   = self.default_ms_before if ms_before is None else ms_before
        ms_after    = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.PushButton( int( self.__controller_id ), self.btn_to_num( button ), int(ms_press))
        time.sleep( ms_after / 1000.0 )

    def push_multi(self, button_list, ms_press=None, ms_before=None, ms_after=None):
        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        ms_press = self.default_ms_press if ms_press is None else ms_press
        button_list_length = len( button_list)

        _button_list = []
        for i in range( 0, button_list_length ):
            _button_list.append( self.btn_to_num( button_list[ i ] ) )

        list_array = ctypes.c_int * button_list_length
        button_list_array = list_array( *_button_list )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.PushMultiButton( int( self.__controller_id ), button_list_array, int( button_list_length ), int(ms_press))
        time.sleep( ms_after / 1000.0 )

    # スティック固定
    def stick_L_hold(self, degree, power=1.0, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.HoldAnalogStick( int( self.__controller_id ), int(Stick.Stick_L), float( degree ), float( power ))
        time.sleep( ms_after / 1000.0 )

    def stick_R_hold(self, degree, power=1.0, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.HoldAnalogStick( int( self.__controller_id ), int(Stick.Stick_R), ( degree ), float( power ) )
        time.sleep( ms_after / 1000.0 )

    # スティック解放
    def stick_L_release(self, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.ReleaseAnalogStick( int( self.__controller_id ), int(Stick.Stick_L) )
        time.sleep( ms_after / 1000.0 )

    def stick_R_release(self, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.ReleaseAnalogStick( int( self.__controller_id ), int(Stick.Stick_R) )
        time.sleep( ms_after / 1000.0 )

    # スティック固定 -> 解放
    def stick_L(self, degree, power=1.0, ms_press=None, ms_before=None, ms_after=None):
        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.TiltAnalogStick( int( self.__controller_id ), int(Stick.Stick_L), float( degree ), float( power ), int(ms_press) )
        time.sleep( ms_after / 1000.0 )

    def stick_R(self, degree, power=1.0, ms_press=None, ms_before=None, ms_after=None):
        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.TiltAnalogStick( int( self.__controller_id ), int(Stick.Stick_R), float( degree ), float( power ), int(ms_press) )
        time.sleep( ms_after / 1000.0 )

    # syntax sugar
    def push_multi_seq(self, button_list, ms_press=None, ms_span=None, ms_before=None, ms_after=None):
        ms_before = self.default_ms_before if ms_before is None else ms_before
        ms_after = self.default_ms_after if ms_after is None else ms_after
        ms_press = self.default_ms_press if ms_press is None else ms_press
        ms_span = self.default_ms_span if ms_span is None else ms_span
        button_list_length = len( button_list)

        _button_list = []
        for i in range( 0, button_list_length ):
            _button_list.append( self.btn_to_num( button_list[ i ] ) )

        list_array = ctypes.c_int * button_list_length
        button_list_array = list_array( *_button_list )

        time.sleep( ms_before / 1000.0 )
        self.__hid_inputer.PushMultiSeq( int( self.__controller_id ), button_list_array, int( button_list_length ), int(ms_press), int(ms_span) )
        time.sleep( ms_after / 1000.0 )


# ===========================================================
#        MultiControllerManagerDll
# ===========================================================
class MultiControllerManagerDll(object):
    def __init__(self,  hid_inputer ):
        # (xpad * 8) + (debugpad * 1)
        self.MAX_CONT_NUM   = 9
        self.__hid_inputer  = hid_inputer
        self.__controllers  = {}


        c_int_p = ctypes.POINTER( ctypes.c_int )
        c_bool_p = ctypes.POINTER( ctypes.c_bool )

        # コントローラー接続
        self.__hid_inputer.AddController.restype = ctypes.c_int
        self.__hid_inputer.AddController.argtypes = (c_int_p,)
        self.__hid_inputer.AddControllerForId.restype = ctypes.c_int
        self.__hid_inputer.AddControllerForId.argtypes = (ctypes.c_int,)
        self.__hid_inputer.AddControllerWithDeviceInfo.restype = ctypes.c_int
        self.__hid_inputer.AddControllerWithDeviceInfo.argtypes = (c_int_p, ControllerDeviceInfo)

        # デバッグコントローラー接続
        self.__hid_inputer.AddDebugController.restype = ctypes.c_int
        self.__hid_inputer.AddDebugController.argtypes = (c_int_p,)

        # コントローラー接続数
        self.__hid_inputer.GetConnectedControllerCount.restype = ctypes.c_int
        self.__hid_inputer.GetConnectedControllerCount.argtypes = (c_int_p,)
        self.__hid_inputer.GetConnectedControllerIds.restype = ctypes.c_int
        self.__hid_inputer.GetConnectedControllerIds.argtypes = (c_int_p, c_int_p, ctypes.c_int)
        self.__hid_inputer.GetConnectedDebugControllerCount.restype = ctypes.c_int
        self.__hid_inputer.GetConnectedDebugControllerCount.argtypes = (c_int_p,)
        self.__hid_inputer.GetConnectedDebugControllerIds.restype = ctypes.c_int
        self.__hid_inputer.GetConnectedDebugControllerIds.argtypes = (c_int_p, c_int_p, ctypes.c_int)

        # コントローラーをセットアップ
        self.__initialize_controller()

    def __initialize_controller(self):
        # 接続済みの仮想コントローラーを読み込み
        controller_count = ctypes.c_int(0)
        self.__hid_inputer.GetConnectedControllerCount(ctypes.pointer(controller_count))

        if( controller_count.value > 0):
            controller_id_list = (ctypes.c_int * controller_count.value)(0)
            controller_id_count = ctypes.c_int(0)

            self.__hid_inputer.GetConnectedControllerIds( ctypes.pointer(controller_id_count) ,controller_id_list, controller_count )
            for i in range(controller_id_count.value):
                self.__controllers[controller_id_list[i]] = HACControllerDll( self.__hid_inputer, controller_id_list[i] )
        
        # 接続済み仮想デバッグコントローラーを読み込み
        controller_count = ctypes.c_int(0)
        self.__hid_inputer.GetConnectedDebugControllerCount(ctypes.pointer(controller_count))
        if( controller_count.value > 0):
            controller_id_list = (ctypes.c_int * controller_count.value)(0)
            controller_id_count = ctypes.c_int(0)

            self.__hid_inputer.GetConnectedDebugControllerIds( ctypes.pointer(controller_id_count) ,controller_id_list, controller_count )
            for i in range(controller_id_count.value):
                self.__controllers[controller_id_list[i]] = HACControllerDll( self.__hid_inputer, controller_id_list[i] )

    def __getitem__(self, key):
        return self.__controllers[key]

    def __iter__(self):
        return list(self.__controllers.values()).__iter__()

    def __len__(self):
        return len(self.__controllers.keys())

    def add( self, cont_id=None ):
        if cont_id is None:
            cont_id = ctypes.c_int(0)
            self.__hid_inputer.AddController( ctypes.pointer( cont_id ) )
            cont_id = cont_id.value
        else:
            self.__hid_inputer.AddControllerForId( cont_id )

        if cont_id >= 0 and  cont_id not in self.__controllers:
            self.__controllers[cont_id] = HACControllerDll( self.__hid_inputer, cont_id )

        return cont_id
    def add_controller( self, device_type=None, interface_type=None, main_color=None, sub_color=None ):
        if device_type is None and interface_type is None and main_color is None and sub_color is None:
            return self.add()


        # デバイス情報の設定
        if device_type is not None or interface_type is not None:
            # 不正な値の場合
            if device_type not in ControllerDeviceType() and device_type is not None:
                return None

            if interface_type not in  ControllerInterfaceType() and interface_type is not None:
                return None

            # デバイスタイプが未指定の場合は、デフォルトのデバイスタイプを設定
            if device_type is None:
                if interface_type == ControllerInterfaceType.ControllerInterfaceType_Bluetooth:
                    device_type = ControllerDeviceType.ControllerDeviceType_SwitchProController
                elif interface_type == ControllerInterfaceType.ControllerInterfaceType_Rail:
                    device_type = ControllerDeviceType.ControllerDeviceType_JoyConLeft
                else:
                    return None

            # インターフェースタイプが未指定の場合は、デフォルトのインターフェースタイプを設定
            if interface_type is None:
                if device_type == ControllerDeviceType.ControllerDeviceType_SwitchProController  \
                    or device_type == ControllerDeviceType.ControllerDeviceType_JoyConLeft \
                    or device_type == ControllerDeviceType.ControllerDeviceType_JoyConRight:
                        interface_type = ControllerInterfaceType.ControllerInterfaceType_Bluetooth
                else:
                    return None

        # デバイス色の指定
        controller_main_color = main_color
        controller_sub_color = sub_color

        if (isinstance( controller_main_color, tuple )):
            controller_main_color = list(controller_main_color)
        if (not isinstance( controller_main_color, list)):
            controller_main_color = [130,130,130]

        if (isinstance( controller_sub_color, tuple )):
            controller_sub_color = list(controller_sub_color)
        if (not isinstance( controller_sub_color, list)):
            controller_sub_color = [15,15,15]

        color_type = ctypes.c_int * 3
        controller_main_color = color_type(*controller_main_color)
        controller_sub_color = color_type(*controller_sub_color)

        device_info = ControllerDeviceInfo(device_type, interface_type, controller_main_color, controller_sub_color)

        cont_id = ctypes.c_int(0)
        self.__hid_inputer.AddControllerWithDeviceInfo( ctypes.pointer(cont_id), device_info)
        
        cont_id = cont_id.value
        self.__controllers[cont_id] = HACControllerDll( self.__hid_inputer, cont_id )
        return cont_id


    def add_debug_pad( self ):
        cont_id = ctypes.c_int(0)
        if self.__hid_inputer.AddDebugController( ctypes.pointer( cont_id ) ) == 0:
            cont_id = cont_id.value

            self.__controllers[cont_id] = HACControllerDll( self.__hid_inputer, cont_id )
            return cont_id
        return -1


    def delete( self, cont_id ):
       if cont_id in self.__controllers:
           self.__controllers[cont_id].close()
           del self.__controllers[cont_id]
    
    def close_all(self):
        for key in list(self.__controllers.keys()):
            self.__controllers[key].close()
            del self.__controllers[key]
            
    def get_free_cont_id_list(self):
        for i in range(0, self.MAX_CONT_NUM):
            if i not in self.__controllers:
                return i

    def get_active_cont_id_list(self):
        return list(self.__controllers.keys())


# ===========================================================
#          HACScreenControl
# ===========================================================
class HACScreenControl(CaptureProcessorDll, MultiTouchActionDll):
    def __init__(self, hid_inputer, capture_device_id=0, target_serial="", capture_mode=CaptureMode.CaptureMode_720p, is_preview=True):
        self.__hid_inputer = hid_inputer

        CaptureProcessorDll.__init__(self, capture_device_id=capture_device_id, target_serial=target_serial, capture_mode=capture_mode, is_preview=is_preview)
        MultiTouchActionDll.__init__(self, self.__hid_inputer)

    def close(self):
        CaptureProcessorDll.close(self)


# =================================
#           LogReader
# =================================
class HACLogReaderDll(object):

    def __init__(self, read_fps=60, hold_log_size=1000):
        self.__log_reader = None
        self.__serialNumber = None
        self.hold_log_size = hold_log_size

    def open(self, *args, **kwargs):
        """ログの溜めこみを開始します。
        これを実行しないと読み込み関数を呼び出すことはできません。

        :param ms_timeout:
        :return:
        """
        self._proper_open_process()

    def close(self, *args, **kwargs):
        """ログの溜めこみを中止し、全てのログキューを削除します。

        :return:
        """
        self._proper_close_process()

    def set_serial(self, serialNumber):
        self.__serialNumber = serialNumber

    def __setup_logreader(self):
        if self.__log_reader is None:
            raise Exception("dll was not loaded")

        self.__log_reader.StartLogStoring.argtypes = (SerialNumberString, ctypes.c_int)
        self.__log_reader.StartLogStoring.restype = ctypes.c_int

        self.__log_reader.GetLogLineSize.argtypes = (ctypes.POINTER(ctypes.c_size_t),)
        self.__log_reader.GetLogLineSize.restype = ctypes.c_int

        self.__log_reader.GetLogLine.argtypes = (ctypes.c_char_p, ctypes.c_size_t)
        self.__log_reader.GetLogLine.restype = ctypes.c_int

        self.__log_reader.WaitForNextLine.argtypes = (ctypes.c_int,)
        self.__log_reader.WaitForNextLine.restype = ctypes.c_int

        self.__log_reader.WaitUntilMatched.argtypes = (ctypes.c_char_p, ctypes.c_int)
        self.__log_reader.WaitUntilMatched.restype = ctypes.c_int

        self.__log_reader.SearchFormerLog.argtypes = (ctypes.c_char_p,)
        self.__log_reader.SearchFormerLog.restype = ctypes.c_int

        self.__log_reader.MoveToNewestLine.argtypes = ()
        self.__log_reader.MoveToNewestLine.restype = None

        self.__log_reader.MoveToFormerLineByTime.argtypes = (ctypes.c_int,)
        self.__log_reader.MoveToFormerLineByTime.restype = None

        self.__log_reader.StopLogStoring.argtypes = ()
        self.__log_reader.StopLogStoring.restype = None

    def _proper_open_process(self):
        if self.__serialNumber is None:
            return

        logreader_path = LIBRARY_PATH + "/LogReader.dll"
        self.__log_reader = ctypes.CDLL(logreader_path)

        self.__setup_logreader()

        serialStr = SerialNumberString(self.__serialNumber.encode())
        if self.__log_reader.StartLogStoring(serialStr, self.hold_log_size) != 0:
            raise Exception("Failed to get telnet port or ip of logs.\n" +
                                     "TargetManager may not have connected to NX.")

    def get_dll(self):
        return self.__log_reader

    def _proper_close_process(self):
        if self.__log_reader is not None:
            self.__log_reader.StopLogStoring()
            self.__log_reader = None

    def create_log_accessor(self):
        return LogAccessor(self)


# =================================
#           LogAccessor
# =================================
class LogAccessor(object):
    def __init__(self, logreader):
        self.__log_reader = logreader
        self.__MS_DEFAULT_TIMEOUT = 5000

    def move_pointer_newest(self):
        self.__log_reader.get_dll().MoveToNewestLine()

    def move_pointer_ms_before(self, ms):
        self.__log_reader.get_dll().MoveToFormerLineByTime(ms)

    def move_pointer_before(self):
        self.__log_reader.get_dll().MoveToPreviousLine()

    def move_pointer_next(self):
        self.__log_reader.get_dll().MoveToNextLine()

    def wait_for_next_line(self, ms_timeout=None):
        timeout = ms_timeout if ms_timeout is not None else self.__MS_DEFAULT_TIMEOUT

        return self.__log_reader.get_dll().WaitForNextLine(int(timeout)) == 0

    def read_line(self, ms_timeout=None):
        timeout = ms_timeout if ms_timeout is not None else self.__MS_DEFAULT_TIMEOUT

        bufSize = ctypes.c_size_t(0)
        if self.__log_reader.get_dll().GetLogLineSize(ctypes.pointer(bufSize)) != 0:
            if self.__log_reader.get_dll().WaitForNextLine(int(timeout)) != 0:
                return None

        buf = ctypes.create_string_buffer(bufSize.value)
        if self.__log_reader.get_dll().GetLogLine(buf, bufSize) != 0:
            return None

        return repr(buf.value)

    def read_line_until(self, key_string, ms_timeout=None, start_log_ms_before=None, is_reverse = False):
        timeout = ms_timeout if ms_timeout is not None else self.__MS_DEFAULT_TIMEOUT 

        if start_log_ms_before is not None:
            self.move_pointer_ms_before(start_log_ms_before)

        start_time = time.time()

        if is_reverse:
            result = self.__log_reader.get_dll().SearchFormerLog(key_string, key_string.__len__())
        else:
            result = self.__log_reader.get_dll().WaitUntilMatched(key_string, key_string.__len__(), int(timeout))

        if result != 0:
            return None

        elapsed_time = time.time() - start_time
        rest_timeout = max(timeout - elapsed_time, 0)

        msg = self.read_line(rest_timeout)
        if msg:
            pattern = re.compile(key_string)
            match = pattern.search(msg)
            if match:
                return match

        return None

    def read_all_after_current(self):
        """現在のログ以降のログを全て行ごとにリストにつめて返します。
        """
        result = []
        while True:
            msg = self.read_line(ms_timeout=0)
            if msg is not None:
                result.append(msg)
                self.__log_reader.get_dll().MoveToNextLine()
            else:
                break

        return result

class TmUtility():
    def __init__(self):
        self.__tm_utility_path = LIBRARY_PATH + "/TmUtility.dll"
        self.__tm_utility = ctypes.CDLL(self.__tm_utility_path)

        self.__tm_utility.GetConnectedTargetCount.argtypes = (ctypes.POINTER(ctypes.c_int),)
        self.__tm_utility.GetConnectedTargetCount.restype = ctypes.c_int
        self.__tm_utility.GetConnectedTargetSerialNumbers.argtypes = (ctypes.c_void_p, ctypes.c_int)
        self.__tm_utility.GetConnectedTargetSerialNumbers.restype = ctypes.c_int

    def get_connected_targets(self):
        count = ctypes.c_int();
        if self.__tm_utility.GetConnectedTargetCount(ctypes.pointer(count)) != 0:
            raise Exception("Failed to access Target Manager")

        if count.value == 0:
            return []

        serialArrayType = SerialNumberString * count.value
        serialArray = serialArrayType()

        if self.__tm_utility.GetConnectedTargetSerialNumbers(serialArray, count) != 0:
            raise Exception("Failed to access Target Manager")

        # str 型 list に変換
        serials = []
        for serial in serialArray:
            serials.append(serial.value.decode())

        return serials

# ===========================================================
#          HACDriverDll
# ===========================================================
class HACDriverDll():
    def __init__(self, target_serial, capture_device_id=0, capture_mode=CaptureMode.CaptureMode_720p, is_preview=True, is_open_logreader=True):
        tm_utility = TmUtility()
        target_list = tm_utility.get_connected_targets()

        if target_serial is not None:
            found_target = False
            for target in target_list:
                if target_serial == target:
                    found_target = True
                    break
            if not found_target:
                raise Exception("Target(" + target_serial + ") was not connected by Target Manager.")

        self.__hid_inputer_path = LIBRARY_PATH + "/HidInputer.dll"
        self.__hid_inputer = ctypes.CDLL( self.__hid_inputer_path )

        # 初期化関数、開始前に必ず呼ぶ必要がある
        self.__hid_inputer.Initialize.restype   = ctypes.c_int
        self.__hid_inputer.Initialize.argtypes  = (SerialNumberString,)
        self.__hid_inputer.InitializeForLocal.restype = ctypes.c_int
        self.__hid_inputer.InitializeForLocal.argtypes = ()

        if target_serial is None:
            if self.__hid_inputer.InitializeForLocal() != 0:
                raise Exception("Failed to initialize HidInputer for local.")
        else:
            if self.__hid_inputer.Initialize( SerialNumberString( target_serial.encode() ) ) != 0:
                raise Exception("Failed to initialize HidInputer.\n" +
                                "InputDirector may not have connected to target.")

        # Hid 記録 API
        self.__hid_inputer.BeginHidInputRecording.argtypes = ()
        self.__hid_inputer.BeginHidInputRecording.restype = ctypes.c_int
        self.__hid_inputer.EndHidInputRecording.argtypes = ()
        self.__hid_inputer.EndHidInputRecording.restype = ctypes.c_int
        self.__hid_inputer.ExportHidInputRecordsWithPythonScript.argtypes = (ctypes.c_char_p,)
        self.__hid_inputer.ExportHidInputRecordsWithPythonScript.restype = ctypes.c_int

        self.controllers = MultiControllerManagerDll(self.__hid_inputer)
        self.screen = HACScreenControl(self.__hid_inputer, capture_device_id, target_serial, capture_mode, is_preview)
        self.log_reader = HACLogReaderDll()
        self.log_reader.set_serial(target_serial)
        if is_open_logreader:
            self.log_reader.open()

    def start_hid_record(self):
        self.__hid_inputer.BeginHidInputRecording()

    def end_hid_record(self):
        self.__hid_inputer.EndHidInputRecording()

    def export_hid_record(self, export_file_name="_DefaultExportScript.py" ):
        self.__hid_inputer.ExportHidInputRecordsWithPythonScript( export_file_name.encode() )

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.screen.close()
        self.__hid_inputer.Finalize()
        self.log_reader.close()


# ===========================================================
#          factory
# ===========================================================
def factory(target_serial=None, capture_device_id=None, capture_mode=CaptureMode.CaptureMode_720p, is_preview=True, is_open_logreader=True):
    # ターゲット未指定の場合、TmUtility.dll より接続済みターゲットのシリアル番号取得を試みる
    if target_serial is None:
        tm_utility = TmUtility()
        target_list = tm_utility.get_connected_targets()

        if target_list == []:
            raise Exception("No targets are connected.")

        print("Please select target.")
        target_list.insert( 0, "Local")
        for (id, target) in enumerate(target_list):
            print( "[%d] : %s" %( id, target ) )

        selected = int( input() )
        if selected not in range( 0, len(target_list)):
            raise Exception("Invalid target is selected.")
        
        if selected is 0:
            target_serial = None
            capture_mode = CaptureMode.CaptureMode_None
        else:
            target_serial = target_list[selected]
        print("\n")


    # 対象キャプチャデバイス未指定の場合、CaptureProcessor.dll よりキャプチャデバイス名取得を試みる
    if target_serial is not None and capture_device_id is None and (capture_mode == CaptureMode.CaptureMode_720p or capture_mode == CaptureMode.CaptureMode_1080p):
        device_list = CaptureProcessorDll.get_capture_device_list()

        if device_list == []:
            raise Exception("Capture devices are not found.\n" + 
                            "Please enable capture device, or use CaptureMode_DevKit (SDEV only).")

        print("Please select capture device.")
        for (id, name) in enumerate( device_list ):
            print( "[%d] : %s" % (id,  name) )

        selected = int( input() )
        if selected not in range( 0, len(device_list)):
            raise Exception("Invalid capture device is selected.")

        capture_device_id = selected
        print("\n")

    return HACDriverDll(target_serial,  capture_device_id, capture_mode, is_preview, is_open_logreader)
