﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/nac/nac_File.h>
#include <cstdio>
#include <memory>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
#include <cafe/fs.h>

// 一時利用しているNWのためにビルド構成定義をリレーする
#ifdef NN_SDK_BUILD_DEBUG
#define NW_DEBUG
#elif NN_SDK_BUILD_DEVELOP
#define NW_DEVELOP
#elif NN_SDK_BUILD_RELEASE
#define NW_RELEASE
#endif
#include <nw/sys.h>

const char* FS_SD01_MOUNT_PATH   = "/vol/disc";
const char* FS_HFIO01_MOUNT_PATH = "/vol/content";
#else
#include <nn/fs.h>
#include <nn/nn_Log.h>
#endif

#ifndef _MAX_PATH
#define _MAX_PATH 260  //NOLINT(preprocessor/const)
#endif

namespace nns {
namespace nac {

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
bool IsFullPath( const char* name )
{
    return
        ((('a' <= name[0] && name[0] <= 'z') ||
        ('A' <= name[0] && name[0] <= 'Z')) &&
        (name[1] == ':'));
}
#endif

bool FileLoad( void**           outBuffer,
               size_t*          outBufferSize,
               const char*      filePath,
               const char*      basePath,
               MemoryAllocator* allocator,
               int              alignment )
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    FSMountSource  mountSrc;
    char           mountPath[FS_MAX_MOUNTPATH_SIZE];
    char           path[FS_MAX_ARGPATH_SIZE];
    FSStatus       status;
    FSStat         stat;
    FSFileHandle   fh;

    memset( path, 0, FS_MAX_ARGPATH_SIZE );
    FSClient* pClient = static_cast<FSClient *>( allocator->Alloc( sizeof( FSClient ) ) );
    if ( pClient == NULL )
    {
        return false;
    }

    FSCmdBlock* pCmd  = static_cast<FSCmdBlock *>( allocator->Alloc( sizeof( FSCmdBlock ) ) );
    if ( pCmd == NULL )
    {
        allocator->Free( pClient );
        return false;
    }

    FSAddClient( pClient, FS_RET_NO_ERROR );
    FSInitCmdBlock( pCmd );

    FSGetMountSource( pClient, pCmd, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR );
    FSMount( pClient, pCmd, &mountSrc, mountPath, sizeof(mountPath), FS_RET_NO_ERROR );

    if ( filePath[0] != '/' )
    {
        const u32 FORMAT_BUFFER_SIZE = 256;
        nw::ut::snprintf( path, FORMAT_BUFFER_SIZE, FORMAT_BUFFER_SIZE - 1, "%s/%s", FS_HFIO01_MOUNT_PATH, filePath );
    }

    if ( filePath[1] == ':' )
    {
        nw::ut::snprintf( path, sizeof(path), "%s/%s", mountPath, filePath );
        for ( int i = 0; i < FS_MAX_ARGPATH_SIZE; i++ )
        {
            if ( path[i] == '\\' || path[i] == ':' )
            {
                path[i] = '/';
            }
        }
    }

    status = FSGetStat( pClient, pCmd, path, &stat, FS_RET_NOT_FOUND );
    if ( status == FS_STATUS_OK )
    {
        *outBuffer = allocator->Alloc( stat.size + 4, alignment );
        if ( outBuffer == NULL )
        {
            allocator->Free( pClient );
            allocator->Free( pCmd );
            return false;
        }

        memset( *outBuffer, 0, stat.size + 4 );

        status = FSOpenFile( pClient, pCmd, path, "r", &fh, FS_RET_NO_ERROR );
        if (status != FS_STATUS_OK)
        {
            NW_LOG("Failed FSOpenFile %s.\n", filePath);
            allocator->Free( pClient );
            allocator->Free( pCmd );
            allocator->Free( outBuffer );
            return false;
        }

        FSReadFile( pClient, pCmd, *outBuffer, 1, stat.size, fh, 0, FS_RET_NO_ERROR );
        *outBufferSize = stat.size;
        FSCloseFile( pClient, pCmd, fh, FS_RET_NO_ERROR );
    }

    FSUnmount(pClient, pCmd, mountPath, FS_RET_NO_ERROR);
    FSDelClient(pClient, FS_RET_NO_ERROR);
    allocator->Free( pClient );
    allocator->Free( pCmd );

    if ( status != FS_STATUS_OK )
    {
        NW_LOG("Failed FSOpenFile %s.\n", filePath);
        return false;
    }
    else
    {
        return true;
    }
#else
    nn::Result ret;

    char filePathName[_MAX_PATH];

    if ( !IsFullPath( filePath ) )
    {
        sprintf( filePathName, "%s\\%s", basePath, filePath );
    }
    else
    {
        memcpy( (void*)&filePathName, filePath, strlen( filePath ) + 1 );
    }

    nn::fs::FileHandle fsHandle;
    ret = nn::fs::OpenFile(&fsHandle, static_cast<char *>( filePathName ), static_cast<nn::fs::OpenMode>( nn::fs::OpenMode_Read ) );
    if ( ret.IsFailure() )
    {
        NN_LOG("failed OpenFile %s.\n", filePath );
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize( &fsSize, fsHandle );
    *outBufferSize = static_cast<size_t>( fsSize );

    *outBuffer = allocator->Alloc( *outBufferSize + 4, alignment );
    if ( !(*outBuffer) )
    {
        nn::fs::CloseFile( fsHandle );
        return false;
    }

    memset(*outBuffer, 0, *outBufferSize + 4);
    nn::fs::ReadFile( fsHandle, 0, *outBuffer, *outBufferSize );
    nn::fs::CloseFile( fsHandle );

    return true;
#endif
} // NOLINT(readability/fn_size)

}
}
