﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @brief
 * NvnVideoRenderer class for rendering of video frames using NVN APIs.
 *
 * @details
 *
 * Purpose:
 *     NvnVideoRenderer class uses NVN APIs for playing back of video frames. Video frames can be submitted either in
 * NVN buffer or NVN texture. If the video frame uses NVN buffer it has to be copied to NVN texture using GPU.
 * If the video frame is in YUV format, it will be converted to RGB format using shaders. There are no threads, client
 * need to drive rendering by calling Draw() API at interval based on video frame rate.
 *
 * Setup:
 *     A client using NvnVideoRenderer need to create an instance of the class by passing nn::vi::NativeWindowHandle
 * and NvnOutputFormat. If nn::vi::NativeWindowHandle is null then NvnVideoRenderer creates one. NvnOutputFormat can
 * be NvnOutputFormat_YuvNv12NvnBuffer or NvnOutputFormat_AbgrNvnTexture. This needs to map with movie video decoder
 * output format. When Initialization() API is called, it will start by reserving memory needed for rendering
 * using NVN. Based on decoder output format shader is compiled using Glsl compiler and loaded to NVN shader memory.
 * Resources needed for rendering using NVN is allocated and initialized. Based on video decoder mode, one NVN buffer
 * or texture is allocated. This will be sent to decoder. NVNcommandBuffer is setup and will be used later in draw call.

 * Main Processing:
 *     Client need to call GetNvnVideoBuffer() or GetNvnVideoTexture() to get either a video buffer or a texture
 * handle. This will be sent to video decoder. Once decoder completes decoding the frame, the client need to submit the
 * frame to NvnVideoRenderer for display using Draw() API. Draw API will acquire a texture from NVN. If the decoder
 * uses CPU buffers, YUV data is copied from NVN buffer to NVN texture. If the decoder output is texture, it is sent
 * to NVN rendering directly, no copy is done. In case of NVN texture mode, client can pass NVN sync point to the decoder.
 * In this case renderer will wait on the sync point before presenting the texture for display. This ensures decoder
 * completing its processing on the texture.
 *
 * Teardown:
 *     When client is done submitting video frames for rendering, Finalize() API need to be called. All resources allocated
 * by NvnVideoRenderer will be released.
 *
 */
#include <nns/mm/mm_NvnVideoRenderer.h>
#include <cstdlib>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nv/nv_MemoryManagement.h>

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>

static const char g_yuv2rgbVertexShader[] =
"#version 450\n"
"layout(location = 0) in vec4 a_position;"
"layout(location = 1) in vec2 a_texCoord;"
"out vec2 v_texCoord;"
"void main() {"
"   gl_Position = a_position;"
"   v_texCoord = a_texCoord;"
"   v_texCoord.y = v_texCoord.y;"
"}";

static const char g_yuv2rgbFragmentShader[] ="#version 450\n"
"in vec2 v_texCoord;"
"layout(binding = 0) uniform sampler2D u_textureY;"
"layout(binding = 1) uniform sampler2D u_textureUV;"
"layout(binding = 2) uniform UBO {"
"    vec3 src_bias;"
"    mat3 src_xform;"
"};"
"out vec4 colorOut;"
"void main() {"
"    float y = texture(u_textureY, v_texCoord).r;"
"    vec2 cbcr = texture(u_textureUV, v_texCoord).rg;"
"    vec3 yuv = vec3(y, cbcr);"
"    yuv -= src_bias;"
"    yuv *= src_xform;"
"    colorOut = vec4(yuv, 1.0);"
"}";

static const char g_rgbFragmentShader[] =
"#version 450\n"
"in vec2 v_texCoord;"
"layout(binding = 0) uniform sampler2D rgb_texture;"
"out vec4 colorOut;"
"void main() {"
"    colorOut = texture(rgb_texture, v_texCoord).rgba;"
"}";

unsigned Align(unsigned value, unsigned alignment)
{
    return value + (alignment - value % alignment) % alignment;
}

char                        g_ShaderHeapBuffer[32 << 20];
nn::mem::StandardAllocator  g_ShaderAllocator(g_ShaderHeapBuffer, sizeof(g_ShaderHeapBuffer));

static void *shaderAllocate(size_t size, size_t alignment, void *userPtr)
{
    return g_ShaderAllocator.Allocate(size, alignment);
}

static void shaderDeallocate(void *block, void *userPtr)
{
    g_ShaderAllocator.Free(block);
}

static void *shaderReallocate(void *block, size_t size, void *userPtr)
{
    return g_ShaderAllocator.Reallocate(block, size);
}

GlslCompiler::GlslCompiler() : m_Init(false)
{
}

GlslCompiler::~GlslCompiler()
{
    Destroy();
}

bool GlslCompiler::Init()
{
    glslcSetAllocator(shaderAllocate, shaderDeallocate, shaderReallocate, NULL);
    uint8_t status = glslcInitialize(&m_Compiler);
    if(status == 0)
        return false;
    m_Init = true;
    return true;
}

void GlslCompiler::Destroy()
{
    if(m_Init)
    {
        glslcFinalize(&m_Compiler);
        m_Init = false;
    }
}

bool GlslCompiler::CompileShader(const char *src, NVNshaderStage stage)
{
    m_Compiler.input.sources = &src;
    m_Compiler.input.stages = &stage;
    m_Compiler.input.count = 1;

    if(!glslcCompile(&m_Compiler))
    {
        NN_SDK_LOG("glslcCompile failed:\n%s\n", m_Compiler.lastCompiledResults->compilationStatus->infoLog);
        return false;
    }
    GLSLCoutput *output = m_Compiler.lastCompiledResults->glslcOutput;
    m_OutputData.data = output;
    m_OutputData.size = output->size;
    return true;
}

DataSection GlslCompiler::GetOutput() const
{
    return m_OutputData;
}

NvnShaderSerialize::NvnShaderSerialize() : m_ShaderData(NULL), m_ScratchSize(0)
{
}

NvnShaderSerialize::~NvnShaderSerialize()
{
    std::free(m_ShaderData);
}

void NvnShaderSerialize::LoadNvnShaderMemory(const uint8_t *shaderBinary)
{
    const GLSLCoutput *output = reinterpret_cast<const GLSLCoutput*>(shaderBinary);
    m_ShaderData = std::malloc(output->size);
    memcpy(m_ShaderData, output, output->size);
    for(int i = 0; i < output->numSections; ++i)
    {
        GLSLCsectionTypeEnum type = output->headers[i].genericHeader.common.type;
        if(type == GLSLC_SECTION_TYPE_GPU_CODE)
        {
            const GLSLCgpuCodeHeader *gpuSection = &output->headers[i].gpuCodeHeader;
            const uint8_t *data = reinterpret_cast<const uint8_t*>(m_ShaderData) + gpuSection->common.dataOffset;
            m_DataSection.size = gpuSection->dataSize;
            m_DataSection.data = data + gpuSection->dataOffset;
            m_ControlSection.size = gpuSection->controlSize;
            m_ControlSection.data = data + gpuSection->controlOffset;
            m_ScratchSize = gpuSection->scratchMemBytesRecommended;
        }
    }
}

DataSection NvnShaderSerialize::GetDataSection()
{
    return m_DataSection;
}

DataSection NvnShaderSerialize::GetControlSection()
{
    return m_ControlSection;
}

unsigned NvnShaderSerialize::GetScratchSize()
{
    return m_ScratchSize;
}

NvnMemoryPool::NvnMemoryPool() : /*m_Allocator(NULL),*/ m_Size(0), m_pMemory(NULL), m_Offset(0)
{
}

NvnMemoryPool::~NvnMemoryPool()
{
    Destroy();
}

bool NvnMemoryPool::Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory)
{
    if(!pMemory)
    {
        NN_SDK_LOG("NvnMemoryPool::Init: pMemory needs to be a valid pointer\n");
        return false;
    }
    if(size % NVN_MEMORY_POOL_STORAGE_GRANULARITY)
    {
        NN_SDK_LOG("NvnMemoryPool::Init: size needs to be aligned to NVN_MEMORY_POOL_STORAGE_GRANULARITY (%i)\n", NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        return false;
    }
    m_Size = size;
    m_pMemory = pMemory;

    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, &device);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder, flags);
    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    return nvnMemoryPoolInitialize(&m_MemoryPool, &poolBuilder) == NVN_TRUE;
}

void NvnMemoryPool::Destroy()
{
    if(m_pMemory)
    {
        nvnMemoryPoolFinalize(&m_MemoryPool);
        m_pMemory = NULL;
        m_Size = 0;
        m_Offset = 0;
    }
}

ptrdiff_t NvnMemoryPool::GetNewMemoryChunkOffset(unsigned size, unsigned alignment)
{
    NN_ASSERT(m_Offset + size <= m_Size, "Memory pool out of memory.");
    m_Offset = Align(m_Offset, alignment);
    ptrdiff_t dataOffset = m_Offset;
    m_Offset += size;
    return dataOffset;
}

NVNmemoryPool &NvnMemoryPool::GetMemoryPool()
{
    return m_MemoryPool;
}

NvnBuffer::NvnBuffer() : m_pDevice(NULL), m_pMappedData(NULL)
{
}

NvnBuffer::~NvnBuffer()
{
    Destroy();
}

bool NvnBuffer::Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size)
{
    m_pDevice = &device;

    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetDevice(&bufferBuilder, &device);
    nvnBufferBuilderSetStorage(&bufferBuilder, &pPool, offset, size);
    if(nvnBufferInitialize(&m_Buffer, &bufferBuilder) == NVN_FALSE)
        return false;
    m_pMappedData = nvnBufferMap(&m_Buffer);
    return true;
}

void NvnBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnBufferFinalize(&m_Buffer);
        m_pDevice = NULL;
    }
}

NVNbufferAddress NvnBuffer::GetAddress() const
{
    return nvnBufferGetAddress(&m_Buffer);
}

NVNbuffer &NvnBuffer::GetBuffer()
{
    return m_Buffer;
}

void *NvnBuffer::Map()
{
    return m_pMappedData;
}

NvnCommandBuffer::NvnCommandBuffer() : m_pDevice(NULL), m_pAllocator(NULL), m_pMemoryPool(NULL), m_CommandOffset(0), m_pControlPool(NULL)
{
}

NvnCommandBuffer::~NvnCommandBuffer()
{
    Destroy();
}

bool NvnCommandBuffer::Init(NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;
    m_pMemoryPool = &memoryPool;

    if(nvnCommandBufferInitialize(&m_CommandBuffer, &device.GetDevice()) == NVN_FALSE)
        return false;

    int commandBufferCommandAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT);
    int commandBufferControlAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT);

    m_CommandOffset = memoryPool.GetNewMemoryChunkOffset(m_CommandSize, commandBufferCommandAlignment);
    m_pControlPool = allocator.Allocate(m_ControlSize, commandBufferControlAlignment);

    Reset();
    return true;
}

void NvnCommandBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnCommandBufferFinalize(&m_CommandBuffer);
        m_pAllocator->Free(m_pControlPool);
        m_pControlPool = NULL;
        m_CommandOffset = 0;
        m_pMemoryPool = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
    }
}

void NvnCommandBuffer::Reset()
{
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, &m_pMemoryPool->GetMemoryPool(), m_CommandOffset, m_CommandSize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, m_ControlSize);
}

size_t NvnCommandBuffer::GetCommandMemoryUsed() const
{
    return nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
}

size_t NvnCommandBuffer::GetControlMemoryUsed() const
{
    return nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
}

NVNcommandBuffer &NvnCommandBuffer::GetCommandBuffer()
{
    return m_CommandBuffer;
}

//in the Windows demos nvnBootstrapLoader needs to be loaded dynamically, but on Horizon it seems to exist in libnvn.a
extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char * name);
}

NvnDevice::NvnDevice() : m_pAllocator(NULL), m_pRenderTargetMemory(NULL), m_FirstResize(true)
{
}

NvnDevice::~NvnDevice()
{
    Destroy();
}

bool NvnDevice::Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator)
{
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(nvnBootstrapLoader("nvnDeviceGetProcAddress"));
    if(!pfnc_nvnDeviceGetProcAddress)
    {
        NN_SDK_LOG("NvnDevice::Init: BootstrapLoader failed to find nvnDeviceGetProcAddress\n");
        return false;
    }

    nvnLoadCProcs(nullptr, pfnc_nvnDeviceGetProcAddress);

    /* If debug is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT | NVN_DEVICE_FLAG_DEBUG_SKIP_CALLS_ON_ERROR_BIT;
#endif

    NVNdeviceBuilder deviceData;
    nvnDeviceBuilderSetDefaults(&deviceData);
    nvnDeviceBuilderSetFlags(&deviceData, deviceFlags);

    if(nvnDeviceInitialize(&m_Device, &deviceData) == NVN_FALSE)
    {
        NN_SDK_LOG("NvnDevice::Init: nvnDeviceInitialize failed\n");
        return false;
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MAJOR_VERSION);
    int MinorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MINOR_VERSION);

    NN_ASSERT(MajorVersion == NVN_API_MAJOR_VERSION && MinorVersion == NVN_API_MINOR_VERSION, "NVN SDK not supported by current driver.");

    /*
     * Debug Layer Callback
     * --------------------
     * Install the debug layer callback if the debug layer was enabled during
     * device initialization. It is possible to pass a pointer to the NVN API
     * to remember and pass back through the debug callback.
     */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

    m_pAllocator = &allocator;

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

    /* Render targets that need to be displayed to the screen need both the render target and display access bits. */
    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    /* Setup the render target memory pool. (See MemoryPool.cpp/.h) */
    unsigned memorySize = Align(m_ColorTargetSize * m_NumColorBuffers, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pRenderTargetMemory = allocator.Allocate(memorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if(!m_RenderTargetMemoryPool.Init(m_Device, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, memorySize, m_pRenderTargetMemory))
    {
        NN_SDK_LOG("NvnDevice::Init: m_RenderTargetMemoryPool.Init() failed\n");
        return false;
    }

   /*
    * NVN Window
    * ----------
    * The NVNwindow is used to present a render target to the screen. It manages
    * an array of render targets and associates them with an NVNnativeWindow. On
    * Horizon NVNnativeWindow defined as ANativeWindow* and on Windows it's defined
    * as HWND. Each frame, the NVNwindow needs to be queried for the active render
    * target for that frame.
    *
    * Window Builder Settings
    * -----------------------
    * SetTextures     - Sets the list of render targets to be used by the window.
    *                   Textures used by the NVNwindow must be initialized with the
    *                   display access bit. A window can be given a max of 6 targets.
    *
    * SetNativeWindow - Sets the handle to the window.
    *
    */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
    return true;
}

void NvnDevice::Destroy()
{
    if(m_pAllocator)
    {
        nvnWindowFinalize(&m_pWindow);

        for( int i = 0; i < sizeof(m_RenderTargets) / sizeof(m_RenderTargets[0]); ++i )
        {
            nvnTextureFinalize(m_RenderTargets + i);
        }

        m_RenderTargetMemoryPool.Destroy();
        m_pAllocator->Free(m_pRenderTargetMemory);
        nvnDeviceFinalize(&m_Device);
        m_pAllocator = NULL;
        m_pRenderTargetMemory = NULL;
        m_FirstResize = true;
    }
}

void NvnDevice::ResizeRenderTargets(unsigned width, unsigned height)
{
    // Check for the window being minimized or having no visible surface.
    if (width == 0 || height == 0)
    {
        return;
    }

    // Set the size of the texture to be created by the texture builder.
    // Implicitly sets depth to one.
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < m_NumColorBuffers; ++i)
    {
        if(!m_FirstResize)
        {
            nvnTextureFinalize(m_RenderTargets + i);
        }

        // Set the memory pool the texture is to be allocated from and offset to where it will be located.
        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, &m_RenderTargetMemoryPool.GetMemoryPool(), m_ColorTargetSize * i);

        // Create the texture using the current state of the texture builder.
        nvnTextureInitialize(m_RenderTargets + i, &m_RenderTargetBuilder);
    }

    // Recreate the NVNwindow with the newly allocated/created render targets
    // for the new resolution.
    if(!m_FirstResize)
    {
        nvnWindowFinalize(&m_pWindow);
    }

    // Sets the number of render targets that the NVNwindow will have available and
    //  gives a pointer to an array of those targets.
    NVNtexture *renderTargets[m_NumColorBuffers];
    for(int i = 0; i < m_NumColorBuffers; ++i)
        renderTargets[i] = m_RenderTargets + i;
    nvnWindowBuilderSetTextures(&m_WindowBuilder, m_NumColorBuffers, renderTargets);
    nvnWindowInitialize(&m_pWindow, &m_WindowBuilder);

    m_FirstResize = false;
}

NVNdevice &NvnDevice::GetDevice()
{
    return m_Device;
}

NVNwindow &NvnDevice::GetWindow()
{
    return m_pWindow;
}

NVNtexture &NvnDevice::GetRenderTarget(int index)
{
    NN_ASSERT(index >= 0 && index < m_NumColorBuffers, "NvnDevice::GetRenderTarget index out of range");
    return m_RenderTargets[index];
}

int NvnDevice::GetDeviceInfo(NVNdeviceInfo name)
{
    int i;
    nvnDeviceGetInteger(&m_Device, name, &i);
    return i;
}

void NvnDevice::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char * message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_SDK_LOG("NVN Debug Layer Callback:\n");
    NN_SDK_LOG("  source:       0x%08x\n", source);
    NN_SDK_LOG("  type:         0x%08x\n", type);
    NN_SDK_LOG("  id:           0x%08x\n", id);
    NN_SDK_LOG("  severity:     0x%08x\n", severity);
    NN_SDK_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "Debug layer callback hit");
}

NvnProgram::NvnProgram() : m_pDevice(NULL)
{
}

NvnProgram::~NvnProgram()
{
    Destroy();
}

bool NvnProgram::Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount)
{
    m_pDevice = &device;
    if(nvnProgramInitialize(&m_Program, &device) == NVN_FALSE)
        return false;
    return nvnProgramSetShaders(&m_Program, stageCount, stageData) == NVN_TRUE;
}

void NvnProgram::Destroy()
{
    if(m_pDevice)
    {
        nvnProgramFinalize(&m_Program);
        m_pDevice = NULL;
    }
}

NVNprogram &NvnProgram::GetProgram()
{
    return m_Program;
}

NvnQueue::NvnQueue() : m_pDevice(NULL)
{
}

NvnQueue::~NvnQueue()
{
    Destroy();
}

bool NvnQueue::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetDevice(&queueBuilder, &device);
    if(nvnQueueInitialize(&m_Queue, &queueBuilder) == NVN_FALSE)
        return false;
    if( nvnSyncInitialize(&m_CommandBufferSync, &device) == NVN_FALSE )
        return false;
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
    nvnQueueFlush(&m_Queue);
    return true;
}

void NvnQueue::Destroy()
{
    if(m_pDevice)
    {
        nvnQueueFinalize(&m_Queue);
        nvnSyncFinalize(&m_CommandBufferSync);
        m_pDevice = NULL;
    }
}

NVNqueue &NvnQueue::GetQueue()
{
    return m_Queue;
}

NVNsync &NvnQueue::GetNvnSync()
{
    return m_CommandBufferSync;
}

NvnSampler::NvnSampler() : m_pDevice(NULL)
{
}

NvnSampler::~NvnSampler()
{
    Destroy();
}

bool NvnSampler::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNsamplerBuilder builder;
    nvnSamplerBuilderSetDefaults(&builder);
    nvnSamplerBuilderSetDevice(&builder, &device);
    return nvnSamplerInitialize(&m_Sampler, &builder) == NVN_TRUE;
}

void NvnSampler::Destroy()
{
    if(m_pDevice)
    {
        nvnSamplerFinalize(&m_Sampler);
        m_pDevice = NULL;
    }
}

NVNsampler &NvnSampler::GetSampler()
{
    return m_Sampler;
}

NvnTexture::NvnTexture() : m_pDevice(NULL)
{
}

NvnTexture::~NvnTexture()
{
    Destroy();
}

bool NvnTexture::Init(NVNdevice &device,
    NvnMemoryPool &memoryPool,
    unsigned width,
    unsigned height,
    NVNformat format,
    int32_t flags,
    unsigned stride,
    bool nvnVideoInteropMode)
{
    m_pDevice = &device;

    NVNtextureBuilder builder;
    nvnTextureBuilderSetDefaults(&builder);
    nvnTextureBuilderSetDevice(&builder, &device);
    nvnTextureBuilderSetSize2D(&builder, width, height);
    nvnTextureBuilderSetFlags(&builder, flags);
    nvnTextureBuilderSetFormat(&builder, format);
    nvnTextureBuilderSetTarget(&builder, NVN_TEXTURE_TARGET_2D);
    if( nvnVideoInteropMode == true )
    {
        nvnTextureBuilderSetFlags(&builder, NVN_TEXTURE_FLAGS_VIDEO_DECODE_BIT);
    }
    nvnTextureBuilderSetStride(&builder, stride);
    size_t size = nvnTextureBuilderGetStorageSize(&builder);
    size_t storageAlignment = nvnTextureBuilderGetStorageAlignment(&builder);
    nvnTextureBuilderSetStorage(&builder, &memoryPool.GetMemoryPool(), memoryPool.GetNewMemoryChunkOffset(size, storageAlignment));
    return nvnTextureInitialize(&m_Texture, &builder) == NVN_TRUE;
}

void NvnTexture::Destroy()
{
    if(m_pDevice)
    {
        nvnTextureFinalize(&m_Texture);
        m_pDevice = NULL;
    }
}

NVNtexture &NvnTexture::GetTexture()
{
    return m_Texture;
}

NvnTextureSamplerPool::NvnTextureSamplerPool() : m_pMemory(NULL), m_pDevice(NULL), m_Id(256)
{
}

NvnTextureSamplerPool::~NvnTextureSamplerPool()
{
    Destroy();
}

bool NvnTextureSamplerPool::Init(NVNdevice &device, nn::mem::StandardAllocator &allocator)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;

    int maxSamplerPoolSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_MAX_SAMPLER_POOL_SIZE, &maxSamplerPoolSize);
    int samplerDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerDescriptorSize);
    int textureDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureDescriptorSize);
    const unsigned poolSize = Align(maxSamplerPoolSize * (samplerDescriptorSize + textureDescriptorSize), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pMemory = allocator.Allocate(poolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_MemoryPool.Init(device, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, poolSize, m_pMemory);
    if(nvnSamplerPoolInitialize(&m_SamplerPool, &m_MemoryPool.GetMemoryPool(), 0, maxSamplerPoolSize) == NVN_FALSE)
        return false;
    int samplerPoolOffset = Align(maxSamplerPoolSize * samplerDescriptorSize, textureDescriptorSize);
    return nvnTexturePoolInitialize(&m_TexturePool, &m_MemoryPool.GetMemoryPool(), samplerPoolOffset, maxSamplerPoolSize) == NVN_TRUE;
}

void NvnTextureSamplerPool::Destroy()
{
    if(m_pDevice)
    {
        nvnTexturePoolFinalize(&m_TexturePool);
        nvnSamplerPoolFinalize(&m_SamplerPool);
        m_MemoryPool.Destroy();
        m_pAllocator->Free(m_pMemory);
        m_pMemory = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
        m_Id = 256;
    }
}

NVNtexturePool &NvnTextureSamplerPool::GetTexturePool()
{
    return m_TexturePool;
}

NVNsamplerPool &NvnTextureSamplerPool::GetSamplerPool()
{
    return m_SamplerPool;
}

unsigned NvnTextureSamplerPool::Register(NVNtexture &pTexture, NVNsampler &pSampler)
{
    NVNtextureView view;
    nvnTextureViewSetDefaults(&view);
    nvnTexturePoolRegisterTexture(&m_TexturePool, m_Id, &pTexture, &view);
    nvnSamplerPoolRegisterSampler(&m_SamplerPool, m_Id, &pSampler);
    return m_Id++;
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

NvnWindow::NvnWindow()
    : m_NativeWindow(NULL),
      m_pDisplay(NULL),
      m_pLayer(NULL),
      m_WindowWidth(1280),
      m_WindowHeight(720),
      m_NativeWindowCreated(false)
{
}

NvnWindow::~NvnWindow()
{
}

bool NvnWindow::Init(nn::vi::NativeWindowHandle nativeWindow)
{
    bool retValue = false;
    InitMemory();
    if( nativeWindow == nullptr )
    {
        if( true == InitWindow() )
        {
            m_NativeWindowCreated = true;
            retValue = true;
        }
    }
    else
    {
        m_NativeWindow = nativeWindow;
        retValue = true;
    }
    return retValue;
}

void NvnWindow::Destroy()
{
    if( m_NativeWindowCreated == true )
    {
        DestroyLayer(m_pLayer);
        CloseDisplay(m_pDisplay);
        nn::vi::Finalize();
        m_NativeWindowCreated = false;
    }
}

nn::vi::NativeWindowHandle NvnWindow::GetNativeWindow()
{
    return m_NativeWindow;
}

void NvnWindow::InitMemory()
{
}

bool NvnWindow::InitWindow()
{
    // Initialize Visual Interface (VI) system to display
    // to the target's screen
    //
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    if( !result.IsSuccess() )
    {
        return false;
    }
    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    if( !result.IsSuccess() )
    {
        return false;
    }
    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
    result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
    if( !result.IsSuccess() )
    {
        return false;
    }
    return true;
}

int NvnWindow::GetWidth() const
{
    return m_WindowWidth;
}

int NvnWindow::GetHeight() const
{
    return m_WindowHeight;
}

NvnVideoRenderer::NvnVideoRenderer(nn::vi::NativeWindowHandle nativeWindow, NvnOutputFormat nvnOutputFormat)
    : m_pAllocatorMemory(nullptr),
    m_pShaderPoolMemory(nullptr),
    m_pVertexPoolMemory(nullptr),
    m_pCommandMemory(nullptr),
    m_pTextureMemory(nullptr),
    m_pBufferMemory(nullptr),
    m_pShaderScratchMemory(nullptr),
    m_videoTextureIdY(0),
    m_videoTextureIdUV(0),
    m_VideoOutputBuffer(nullptr),
    m_YuvBufferSize(0),
    m_NativeWindow(nativeWindow),
    m_NvnOutputFormat(nvnOutputFormat),
    m_RgbTextureHandle(NULL),
    m_RgbTextureId(0)
{
}

void NvnVideoRenderer::GetNvnVideoBuffer(int index, char **bufferMemory, size_t *yuvBufferSize)
{
    if( ( bufferMemory != nullptr ) && ( yuvBufferSize != nullptr ) )
    {
        *bufferMemory = reinterpret_cast< char* >( nvnBufferMap(&m_videoBufferYUV.GetBuffer()) );
        *yuvBufferSize = m_YuvBufferSize;
    }
}

void NvnVideoRenderer::GetNvnVideoTexture(int index, void **textureHandle, size_t *size)
{
    if( ( textureHandle != nullptr ) && ( size != nullptr ) )
    {
        *textureHandle = ( void * ) &m_RgbTexture.GetTexture();
        *size = sizeof(void*);
    }
}

void NvnVideoRenderer::GetNvnVideoSync(void **deviceHandle, void **syncHandle)
{
    if( ( deviceHandle != nullptr ) && ( syncHandle != nullptr ) )
    {
        *deviceHandle = static_cast<void*>(&m_device.GetDevice());
        *syncHandle = static_cast<void*>(&m_queue.GetNvnSync());
    }
}

movie::Status NvnVideoRenderer::Initialize()
{
    const unsigned nvnMemorySize = 96 << 20;
    m_pAllocatorMemory = std::malloc(nvnMemorySize);
    if( m_pAllocatorMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    m_allocator.Initialize(m_pAllocatorMemory, nvnMemorySize);

    GlslCompiler compiler;
    compiler.Init();
    compiler.CompileShader(g_yuv2rgbVertexShader, NVN_SHADER_STAGE_VERTEX);
    m_vertexShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>( compiler.GetOutput().data ));
    if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
    {
        if( !compiler.CompileShader(g_yuv2rgbFragmentShader, NVN_SHADER_STAGE_FRAGMENT) )
        {
            return movie::Status_UnknownError;
        }
    }
    else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
    {
        if( !compiler.CompileShader(g_rgbFragmentShader, NVN_SHADER_STAGE_FRAGMENT) )
        {
            return movie::Status_UnknownError;
        }
    }
    else
    {
        return movie::Status_NotSupported;
    }
    m_fragmentShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>( compiler.GetOutput().data ));
    compiler.Destroy();

    m_vertexDataSection = m_vertexShaderFile.GetDataSection();
    m_vertexControlSection = m_vertexShaderFile.GetControlSection();
    m_fragmentDataSection = m_fragmentShaderFile.GetDataSection();
    m_fragmentControlSection = m_fragmentShaderFile.GetControlSection();

    m_window.Init(m_NativeWindow);
    m_device.Init(m_window.GetNativeWindow(), m_allocator);
    m_queue.Init(m_device.GetDevice());

    m_pCommandMemory = m_allocator.Allocate(NVN_MEMORY_POOL_STORAGE_GRANULARITY, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pCommandMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    m_commandPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_UNCACHED_BIT, NVN_MEMORY_POOL_STORAGE_GRANULARITY, m_pCommandMemory);
    m_drawCommands.Init(m_device, m_allocator, m_commandPool);
    m_renderTargetCommand.Init(m_device, m_allocator, m_commandPool);
    m_texturePool.Init(m_device.GetDevice(), m_allocator);
    m_videoSampler.Init(m_device.GetDevice());

    int shaderScratchGranularity = m_device.GetDeviceInfo(NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY);
    m_pShaderScratchMemory = m_allocator.Allocate(shaderScratchGranularity, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pShaderScratchMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    m_shaderScratchPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, shaderScratchGranularity, m_pShaderScratchMemory);
    unsigned fragmentShaderOffset = Align(m_vertexDataSection.size + 1024, 256);
    unsigned shaderMemorySize = Align(fragmentShaderOffset + m_fragmentDataSection.size + 1024, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pShaderPoolMemory = m_allocator.Allocate(shaderMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pShaderPoolMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    memcpy(m_pShaderPoolMemory, m_vertexDataSection.data, m_vertexDataSection.size);
    memcpy(reinterpret_cast<char*>( m_pShaderPoolMemory ) + fragmentShaderOffset, m_fragmentDataSection.data, m_fragmentDataSection.size);
    m_shaderPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT, shaderMemorySize, m_pShaderPoolMemory);

    m_vertexDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(fragmentShaderOffset, 1), m_vertexDataSection.size);
    m_fragmentDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(m_fragmentDataSection.size + 1024, 256), m_fragmentDataSection.size);

    NVNshaderData stageData[ 2 ];
    stageData[ 0 ].control = m_vertexControlSection.data;
    stageData[ 0 ].data = m_vertexDataBuffer.GetAddress();
    stageData[ 1 ].control = m_fragmentControlSection.data;
    stageData[ 1 ].data = m_fragmentDataBuffer.GetAddress();
    m_program.Init(m_device.GetDevice(), stageData, sizeof(stageData) / sizeof(*stageData));

    m_uniformUsefulSize = sizeof(float) * 4 * 4;
    const unsigned uniformMemorySize = Align(m_uniformUsefulSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);;
    m_pUniformMemory = m_allocator.Allocate(uniformMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pUniformMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    m_uniformPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, uniformMemorySize, m_pUniformMemory);
    if(!m_uniformBuffer.Init(m_device.GetDevice(), m_uniformPool.GetMemoryPool(), m_uniformPool.GetNewMemoryChunkOffset(m_uniformUsefulSize, 1), m_uniformUsefulSize))
        return movie::Status_UnknownError;

    nvnBlendStateSetDefaults(&m_blendState);
    nvnChannelMaskStateSetDefaults(&m_channelMaskState);
    nvnColorStateSetDefaults(&m_colorState);
    nvnMultisampleStateSetDefaults(&m_multisampleState);
    nvnMultisampleStateSetMultisampleEnable(&m_multisampleState, NVN_FALSE);
    nvnPolygonStateSetDefaults(&m_polygonState);
    nvnPolygonStateSetCullFace(&m_polygonState, NVN_FACE_BACK);
    nvnDepthStencilStateSetDefaults(&m_depthStencilState);

    m_device.ResizeRenderTargets(m_window.GetWidth(), m_window.GetHeight());

    return movie::Status_Success;
}//NOLINT(impl/function_size)

void NvnVideoRenderer::Finalize()
{
    m_squareMesh.Destroy();
    m_fragmentDataBuffer.Destroy();
    m_vertexDataBuffer.Destroy();
    if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
    {
        m_videoTextureUV.Destroy();
        m_videoTextureY.Destroy();
        m_videoBufferYUV.Destroy();
    }
    else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
    {
        m_RgbTexture.Destroy();
    }
    m_uniformBuffer.Destroy();
    m_videoSampler.Destroy();
    m_texturePool.Destroy();
    m_drawCommands.Destroy();
    m_program.Destroy();
    m_shaderScratchPool.Destroy();
    m_bufferMemory.Destroy();
    m_textureMemoryPool.Destroy();
    m_uniformPool.Destroy();
    m_renderTargetCommand.Destroy();
    m_commandPool.Destroy();
    m_vertexPool.Destroy();
    m_shaderPool.Destroy();
    m_queue.Destroy();
    m_device.Destroy();
    m_window.Destroy();

    if( m_pShaderPoolMemory != NULL )
    {
        m_allocator.Free(m_pShaderPoolMemory);
        m_pShaderPoolMemory = nullptr;
    }
    if( m_pVertexPoolMemory != NULL )
    {
        m_allocator.Free(m_pVertexPoolMemory);
        m_pVertexPoolMemory = nullptr;
    }
    if( m_pCommandMemory != NULL )
    {
        m_allocator.Free(m_pCommandMemory);
        m_pCommandMemory = nullptr;
    }
    if( m_pTextureMemory != NULL )
    {
        m_allocator.Free(m_pTextureMemory);
        m_pTextureMemory = nullptr;
    }
    if( m_pBufferMemory != NULL )
    {
        m_allocator.Free(m_pBufferMemory);
        m_pBufferMemory = nullptr;
    }
    if( m_pShaderScratchMemory != NULL )
    {
        m_allocator.Free(m_pShaderScratchMemory);
        m_pShaderScratchMemory = nullptr;
    }
    if( m_pUniformMemory != NULL )
    {
        m_allocator.Free(m_pUniformMemory);
    }
    m_allocator.Finalize();
    if( m_pAllocatorMemory != NULL )
    {
        std::free(m_pAllocatorMemory);
        m_pAllocatorMemory = nullptr;
    }
}

void NvnVideoRenderer::Draw(int32_t width, int32_t height, int32_t yOffset, int32_t yStride, int32_t uvOffset, int32_t colorSpace)
{
    NVNsync textureAvailableSync;
    nvnSyncInitialize(&textureAvailableSync, &m_device.GetDevice());

    int textureIndex;
    nvnWindowAcquireTexture(&m_device.GetWindow(), &textureAvailableSync, &textureIndex);
    nvnQueueWaitSync(&m_queue.GetQueue(), &textureAvailableSync);
    nvnQueueFinish(&m_queue.GetQueue());

    m_renderTargetCommand.Reset();

    NVNcommandBuffer *targetCommandBuffer = &m_renderTargetCommand.GetCommandBuffer();
    nvnCommandBufferBeginRecording(targetCommandBuffer);
    {
        NVNtexture *renderTarget = &m_device.GetRenderTarget(textureIndex);
        nvnCommandBufferSetRenderTargets(targetCommandBuffer, 1, &renderTarget, NULL, NULL, NULL);
        if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
        {
            nvnCommandBufferSetCopyRowStride(targetCommandBuffer, yStride);
            NVNcopyRegion copyRegion;
            copyRegion.depth = 1;
            copyRegion.height = height;
            copyRegion.width = width;
            copyRegion.xoffset = 0;
            copyRegion.yoffset = 0;
            copyRegion.zoffset = 0;
            nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, m_videoBufferYUV.GetAddress() + yOffset, &m_videoTextureY.GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
            copyRegion.height = height / 2;
            copyRegion.width = width / 2;
            nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, m_videoBufferYUV.GetAddress() + uvOffset, &m_videoTextureUV.GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
            switch(colorSpace)
            {
                case movie::ColorSpace_YCbCr2020: //just have this case fallthrough to 601 for now
                case movie::ColorSpace_YCbCr601:
                {
                    const float x = 1.164383f;
                    const float y = 1.138393f;
                    const float z = 1.138393f;
                    const float uniforms[] = {
                        16.0f / 255, 128.0f / 255, 128.0f / 255, 0,
                        1.00000000f * x,  0.00000000f * y,  1.40200000f * z, 0,
                        1.00000000f * x, -0.34413629f * y, -0.71413629f * z, 0,
                        1.00000000f * x,  1.77200000f * y,  0.00000000f * z, 0
                    };
                    memcpy(m_pUniformMemory, uniforms, sizeof(uniforms));
                    break;
                }

                case movie::ColorSpace_YCbCr601_ER:
                {
                    const float uniforms[] = {
                        0, 128.0f / 255, 128.0f / 255, 0,
                        1.00000000f,  0.00000000f,  1.40200000f, 0,
                        1.00000000f, -0.34413629f, -0.71413629f, 0,
                        1.00000000f,  1.77200000f,  0.00000000f, 0
                    };
                    memcpy(m_pUniformMemory, uniforms, sizeof(uniforms));
                    break;
                }

                case movie::ColorSpace_YCbCr709:
                {
                    const float x = 1.164383f;
                    const float y = 1.138393f;
                    const float z = 1.138393f;
                    const float uniforms[] = {
                        16.0f / 255, 128.0f / 255, 128.0f / 255, 0,
                        1.00000000f * x,  0.00000000f * y,  1.57480000f * z, 0,
                        1.00000000f * x, -0.18732427f * y, -0.46812427f * z, 0,
                        1.00000000f * x,  1.85560000f * y,  0.00000000f * z, 0
                    };
                    memcpy(m_pUniformMemory, uniforms, sizeof(uniforms));
                    break;
                }

                case movie::ColorSpace_YCbCr709_ER:
                {
                    const float uniforms[] = {
                        0, 128.0f / 255, 128.0f / 255, 0,
                        1.00000000f,  0.00000000f,  1.57480000f, 0,
                        1.00000000f, -0.18732427f, -0.46812427f, 0,
                        1.00000000f,  1.85560000f,  0.00000000f, 0
                    };
                    memcpy(m_pUniformMemory, uniforms, sizeof(uniforms));
                    break;
                }

                default:
                    NN_SDK_LOG("Undefined color space %i returned\n", colorSpace);
            }
        }
        else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
        {
            float aspectRatioDisplay = static_cast<float>(m_window.GetWidth()) / m_window.GetHeight();
            float aspectRatioVideo = static_cast<float>(width) / height;
            float ndcWidth = aspectRatioVideo / aspectRatioDisplay;
            int textureHeight = nvnTextureGetHeight(&m_RgbTexture.GetTexture());
            int textureWidth = nvnTextureGetWidth(&m_RgbTexture.GetTexture());

            const float vertices[] = {
                -ndcWidth, 1, 0, 1,
                0, 0,
                -ndcWidth, -1, 0, 1,
                0, (height - 1.0f) / textureHeight,
                ndcWidth, 1, 0, 1,
                (width - 1.0f) / textureWidth, 0,
                ndcWidth, -1, 0, 1,
                (width - 1.0f) / textureWidth, (height - 1.0f) / textureHeight,
            };
            memcpy(m_pVertexPoolMemory, vertices, sizeof(vertices));
        }
    }
    NVNcommandHandle commandHandleTarget = nvnCommandBufferEndRecording(targetCommandBuffer);
    nvnSyncWait(&m_queue.GetNvnSync(), NVN_WAIT_TIMEOUT_MAXIMUM);
    nvnQueueSubmitCommands(&m_queue.GetQueue(), 1, &commandHandleTarget);
    nvnQueueSubmitCommands(&m_queue.GetQueue(), 1, &m_commandHandleDraw);

    nvnQueueFinish(&m_queue.GetQueue());    // nvnQueueFinish() synchronizes with the completion of the commands
                                            // that we have just submitted; so it is now safe to release our texture
                                            // back to the decoder to be filled again.

    nvnQueuePresentTexture(&m_queue.GetQueue(), &m_device.GetWindow(), textureIndex);
    nvnSyncFinalize(&textureAvailableSync);

} //NOLINT(impl/function_size)

movie::Status NvnVideoRenderer::ResizeTextures(int32_t width, int32_t height)
{
    if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
    {
        m_videoTextureUV.Destroy();
        m_videoTextureY.Destroy();
        m_videoBufferYUV.Destroy();
    }
    else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
    {
        m_RgbTexture.Destroy();
    }
    m_textureMemoryPool.Destroy();
    if( m_pTextureMemory != NULL )
    {
        m_allocator.Free(m_pTextureMemory);
        m_pTextureMemory = nullptr;
    }
    m_bufferMemory.Destroy();
    if( m_pBufferMemory != NULL )
    {
        m_allocator.Free(m_pBufferMemory);
        m_pBufferMemory = nullptr;
    }
    m_squareMesh.Destroy();
    m_vertexPool.Destroy();
    if( m_pVertexPoolMemory != NULL )
    {
        m_allocator.Free(m_pVertexPoolMemory);
        m_pVertexPoolMemory = nullptr;
    }
    //vertices and uv coordinates for a cube
    float aspectRatioDisplay = static_cast<float>( m_window.GetWidth() ) / m_window.GetHeight();
    float aspectRatioVideo = static_cast<float>( width ) / height;
    float ndcWidth = aspectRatioVideo / aspectRatioDisplay;
    const float vertices[] = {
        -ndcWidth, 1, 0, 1,
        0, 0,
        -ndcWidth, -1, 0, 1,
        0, 1,
        ndcWidth, 1, 0, 1,
        1, 0,
        ndcWidth, -1, 0, 1,
        1, 1,
    };
    unsigned vertexPoolSize = Align(sizeof(vertices), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pVertexPoolMemory = m_allocator.Allocate(vertexPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pVertexPoolMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    memcpy(m_pVertexPoolMemory, vertices, sizeof(vertices));
    m_vertexPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, vertexPoolSize, m_pVertexPoolMemory);
    m_squareMesh.Init(m_device.GetDevice(), m_vertexPool.GetMemoryPool(), 0, sizeof(vertices));

    int textureStrideAlignment = m_device.GetDeviceInfo(NVN_DEVICE_INFO_LINEAR_TEXTURE_STRIDE_ALIGNMENT);
    int videoStride = Align(width, textureStrideAlignment);
    unsigned texturePoolSize = Align(videoStride * height * 6, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pTextureMemory = m_allocator.Allocate(texturePoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pTextureMemory == NULL )
    {
        return movie::Status_OutOfMemory;
    }
    m_textureMemoryPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, texturePoolSize, m_pTextureMemory);

    if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
    {
        m_videoTextureY.Init(m_device.GetDevice(), m_textureMemoryPool, width, height, NVN_FORMAT_R8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
        m_videoTextureUV.Init(m_device.GetDevice(), m_textureMemoryPool, width / 2, height / 2, NVN_FORMAT_RG8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
        m_videoTextureIdY = m_texturePool.Register(m_videoTextureY.GetTexture(), m_videoSampler.GetSampler());
        m_videoTextureIdUV = m_texturePool.Register(m_videoTextureUV.GetTexture(), m_videoSampler.GetSampler());
        m_videoTextureHandleY = nvnDeviceGetTextureHandle(&m_device.GetDevice(), m_videoTextureIdY, m_videoTextureIdY);
        m_videoTextureHandleUV = nvnDeviceGetTextureHandle(&m_device.GetDevice(), m_videoTextureIdUV, m_videoTextureIdUV);
        unsigned videoBufferSize = width * height * 3 / 2;
        const unsigned bufferMemorySize = Align(videoBufferSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        m_pBufferMemory = m_allocator.Allocate(bufferMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
        if( m_pBufferMemory == NULL )
        {
            return movie::Status_OutOfMemory;
        }
        m_bufferMemory.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, bufferMemorySize, m_pBufferMemory);
        m_videoBufferYUV.Init(m_device.GetDevice(), m_bufferMemory.GetMemoryPool(), m_bufferMemory.GetNewMemoryChunkOffset(videoBufferSize, 1), videoBufferSize);
        m_VideoOutputBuffer = reinterpret_cast<char*>( nvnBufferMap(&m_videoBufferYUV.GetBuffer()) );
        m_YuvBufferSize = videoBufferSize;
    }
    else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
    {
        bool nvnVideoInteropMode = true;
        m_RgbTexture.Init(m_device.GetDevice(),
            m_textureMemoryPool,
            width,
            height,
            NVN_FORMAT_RGBA8,
            NVN_TEXTURE_FLAGS_LINEAR_BIT, Align(videoStride * 4, textureStrideAlignment),
            nvnVideoInteropMode);
        m_RgbTextureId = m_texturePool.Register(m_RgbTexture.GetTexture(), m_videoSampler.GetSampler());
        m_RgbTextureHandle = nvnDeviceGetTextureHandle(&m_device.GetDevice(), m_RgbTextureId, m_RgbTextureId);
    }

    NVNcommandBuffer *commandBuffer = &m_drawCommands.GetCommandBuffer();
    m_drawCommands.Reset();

    nvnCommandBufferBeginRecording(commandBuffer);
    {
        nvnCommandBufferSetScissor(commandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());
        nvnCommandBufferSetViewport(commandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());

        nvnCommandBufferBindBlendState(commandBuffer, &m_blendState);
        nvnCommandBufferBindChannelMaskState(commandBuffer, &m_channelMaskState);
        nvnCommandBufferBindColorState(commandBuffer, &m_colorState);
        nvnCommandBufferBindMultisampleState(commandBuffer, &m_multisampleState);
        nvnCommandBufferBindPolygonState(commandBuffer, &m_polygonState);
        nvnCommandBufferBindDepthStencilState(commandBuffer, &m_depthStencilState);

        nvnCommandBufferSetShaderScratchMemory(commandBuffer, &m_shaderScratchPool.GetMemoryPool(), 0, m_device.GetDeviceInfo(NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY));

        float grey[ 4 ] = { 0.5f, 0.5f, 0.5f, 1.0f };
        nvnCommandBufferClearColor(commandBuffer, 0, grey, NVN_CLEAR_COLOR_MASK_RGBA);

        nvnCommandBufferSetSampleMask(commandBuffer, static_cast<int>( ~0 ));

        NVNvertexAttribState vertexAttrib[ 2 ];
        for( int i = 0; i < sizeof(vertexAttrib) / sizeof(*vertexAttrib); ++i )
        {
            nvnVertexAttribStateSetDefaults(vertexAttrib + i);
            nvnVertexAttribStateSetStreamIndex(vertexAttrib + i, i);
        }
        nvnVertexAttribStateSetFormat(vertexAttrib + 0, NVN_FORMAT_RGBA32F, 0);
        nvnVertexAttribStateSetFormat(vertexAttrib + 1, NVN_FORMAT_RG32F, sizeof(float) * 4);
        nvnCommandBufferBindVertexAttribState(commandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexAttrib);

        NVNvertexStreamState vertexStream[ 2 ];
        for( int i = 0; i < sizeof(vertexStream) / sizeof(*vertexStream); ++i )
        {
            nvnVertexStreamStateSetDefaults(vertexStream + i);
            nvnVertexStreamStateSetStride(vertexStream + i, sizeof(float) * 6);
        }
        nvnCommandBufferBindVertexStreamState(commandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexStream);

        nvnCommandBufferBindVertexBuffer(commandBuffer, 0, m_squareMesh.GetAddress(), sizeof(vertices));
        nvnCommandBufferBindVertexBuffer(commandBuffer, 1, m_squareMesh.GetAddress(), sizeof(vertices));
        nvnCommandBufferBindUniformBuffer(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 2, m_uniformBuffer.GetAddress(), m_uniformUsefulSize);

        nvnCommandBufferBindProgram(commandBuffer, &m_program.GetProgram(), NVN_SHADER_STAGE_VERTEX_BIT | NVN_SHADER_STAGE_FRAGMENT_BIT);

        nvnCommandBufferSetTexturePool(commandBuffer, &m_texturePool.GetTexturePool());
        nvnCommandBufferSetSamplerPool(commandBuffer, &m_texturePool.GetSamplerPool());
        if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
        {
            nvnCommandBufferBindTexture(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, m_videoTextureHandleY);
            nvnCommandBufferBindTexture(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 1, m_videoTextureHandleUV);
        }
        else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
        {
            nvnCommandBufferBindTexture(commandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, m_RgbTextureHandle);
        }
        nvnCommandBufferDrawArrays(commandBuffer, NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP, 0, 4);
    }
    m_commandHandleDraw = nvnCommandBufferEndRecording(commandBuffer);

    return movie::Status_Success;
}//NOLINT(impl/function_size)
