﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>

#include <nn/hidbus/hidbus.h>

#include <nns/hidbus/hidbus_Ronde.h>

namespace nns { namespace hidbus {

namespace
{
    const uint32_t RondeDeviceId            = 0x20;
    const size_t   RondeHeaderSize          = 4;
    const size_t   RondeFirmwareVersionSize = RondeHeaderSize + 4;
    const size_t   RondePollingDataSize     = RondeHeaderSize + 4;
    const size_t   RondeCalDataSize         = RondeHeaderSize + 4;
    const size_t   RondeUniqueIdDataSize    = RondeHeaderSize + 12;
    const int      RondeResultOffset        = 0;

    enum RondeResult : uint8_t
    {
        RondeResult_Success          = 0x00,
        RondeResult_InvaidCommand    = 0x01,
        RondeResult_InvalidDataName  = 0x02,
        RondeResult_InvalidDataSize  = 0x03,

        RondeResult_UartParityError  = 0x11,
        RondeResult_UartNoiseError   = 0x12,
        RondeResult_UartFrameError   = 0x13,
        RondeResult_UartOverrunError = 0x14,
        RondeResult_UartDmaError     = 0x15,
        RondeResult_UartBusyError    = 0x16,

        RondeResult_ResetAds         = 0x31,
    };

    nns::hidbus::RondeInternalErrorInfo g_InternalErrorInfo = {0};

    void IncrementResultCount(uint32_t* pResultCount)
    {
        if (*pResultCount < UINT32_MAX)
        {
            (*pResultCount)++;
        }
    }

    void RondeResultHandler(RondeResult rondeResult) NN_NOEXCEPT
    {
        switch (rondeResult)
        {
        case RondeResult_InvaidCommand:
            IncrementResultCount(&g_InternalErrorInfo.invalidCommandCount);
            break;

        case RondeResult_InvalidDataName:
            IncrementResultCount(&g_InternalErrorInfo.invalidDataNameCount);
            break;

        case RondeResult_InvalidDataSize:
            IncrementResultCount(&g_InternalErrorInfo.invalidDataSizeCount);
            break;

        case RondeResult_UartParityError:
            IncrementResultCount(&g_InternalErrorInfo.uartParityErrorCount);
            break;

        case RondeResult_UartNoiseError:
            IncrementResultCount(&g_InternalErrorInfo.uartNoiseErrorCount);
            break;

        case RondeResult_UartFrameError:
            IncrementResultCount(&g_InternalErrorInfo.uartFrameErrorCount);
            break;

        case RondeResult_UartOverrunError:
            IncrementResultCount(&g_InternalErrorInfo.uartOverrunErrorCount);
            break;

        case RondeResult_UartDmaError:
            IncrementResultCount(&g_InternalErrorInfo.uartDmaErrorCount);
            break;

        case RondeResult_UartBusyError:
            IncrementResultCount(&g_InternalErrorInfo.uartBusyErrorCount);
            break;

        case RondeResult_ResetAds:
            IncrementResultCount(&g_InternalErrorInfo.resetAdsCount);
            break;

        default:
            break;

        }
    }

    nn::Result GetManufactureCalData(int16_t* pOutData,
                                     const nn::hidbus::BusHandle& handle,
                                     const void* pInBuffer, size_t inBufferSize)
    {
        uint8_t receiveBuffer[RondeCalDataSize];
        size_t  outSize;

        auto result = nn::hidbus::SendAndReceive(&outSize, receiveBuffer, sizeof(receiveBuffer), handle,
                                                pInBuffer, inBufferSize);

        if (result.IsFailure())
        {
            return result;
        }

        if (outSize != RondeCalDataSize || receiveBuffer[RondeResultOffset] != RondeResult_Success)
        {
            if (receiveBuffer[RondeResultOffset] != RondeResult_Success)
            {
                RondeResultHandler(static_cast<RondeResult>(receiveBuffer[RondeResultOffset]));
            }
            // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
            // Timeout 相当として、上位でリトライします。
            return nn::hidbus::ResultExternalDeviceTimeout();
        }
        uint16_t receiveData = receiveBuffer[5] << 8 | receiveBuffer[4];

        *pOutData = *reinterpret_cast<int16_t*>(&receiveData);
        return nn::ResultSuccess();
    }

}  // anonymous

nn::Result EnableRonde(const nn::hidbus::BusHandle& handle, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    const uint8_t RondePollingDataCommand[4] = { 0x01, 0x01, 0x02, 0x00 };

    auto result = nn::hidbus::EnableExternalDevice(true, RondeDeviceId, handle);
    if (result.IsFailure())
    {
        return result;
    }
    return nn::hidbus::EnableJoyPollingReceiveMode(handle, RondePollingDataCommand, sizeof(RondePollingDataCommand),
                                                         workBuffer, workBufferSize, nn::hidbus::JoyPollingMode_SixAxisSensorEnable);
}

nn::Result DisableRonde(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    auto result = nn::hidbus::DisableJoyPollingReceiveMode(handle);
    if (result.IsFailure())
    {
        return result;
    }
    return nn::hidbus::EnableExternalDevice(false, RondeDeviceId, handle);
}

nn::Result GetRondeSensorData(RondeSensorData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    nn::hidbus::JoyPollingReceivedData data;
    auto result = nn::hidbus::GetJoyPollingReceivedData(&data, handle);
    if (result.IsFailure())
    {
        return result;
    }
    if (data.outSize != RondePollingDataSize ||
        data.data[RondeResultOffset] != RondeResult_Success)
    {
        if (data.data[RondeResultOffset] != RondeResult_Success)
        {
            RondeResultHandler(static_cast<RondeResult>(data.data[RondeResultOffset]));
        }
        // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
        // Timeout 相当として、上位でリトライします。
        return nn::hidbus::ResultExternalDeviceTimeout();
    }
    uint16_t sensorData = data.data[5] << 8 | data.data[4];
    pOutData->data = *reinterpret_cast<int16_t*>(&sensorData);
    pOutData->samplingNumber = data.samplingNumber;
    return nn::ResultSuccess();
}

nn::Result GetRondeSensorData(RondeSensorData* pOutData, int count, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(count, 10);
    nn::hidbus::JoyPollingReceivedData data[10];
    auto result = nn::hidbus::GetJoyPollingReceivedData(data, count, handle);
    if (result.IsFailure())
    {
        return result;
    }
    for (int i = 0; i < count; i++)
    {
        if (data[i].outSize != RondePollingDataSize ||
            data[i].data[RondeResultOffset] != RondeResult_Success)
        {
            if (data[i].data[RondeResultOffset] != RondeResult_Success)
            {
                RondeResultHandler(static_cast<RondeResult>(data[i].data[RondeResultOffset]));
            }

            // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
            // Timeout 相当として、上位でリトライします。
            return nn::hidbus::ResultExternalDeviceTimeout();
        }

        uint16_t sensorData = data[i].data[5] << 8 | data[i].data[4];
        pOutData[i].data = *reinterpret_cast<int16_t*>(&sensorData);
        pOutData[i].samplingNumber = data[i].samplingNumber;
    }
    return nn::ResultSuccess();
}

nn::Result GetRondeThermisterData(int16_t* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    nn::hidbus::JoyPollingReceivedData data;
    auto result = nn::hidbus::GetJoyPollingReceivedData(&data, handle);
    if (result.IsFailure())
    {
        return result;
    }

    if (data.outSize != RondePollingDataSize ||
        data.data[RondeResultOffset] != RondeResult_Success)
    {
        if (data.data[RondeResultOffset] != RondeResult_Success)
        {
            RondeResultHandler(static_cast<RondeResult>(data.data[RondeResultOffset]));
        }

        // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
        // Timeout 相当として、上位でリトライします。
        return nn::hidbus::ResultExternalDeviceTimeout();
    }

    uint16_t thermistorData = data.data[7] << 8 | data.data[6];
    *pOutData = *reinterpret_cast<int16_t*>(&thermistorData);
    return nn::ResultSuccess();
}

nn::Result GetRondeManufactureCalibrationData(RondeManufactureCalibrationData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    const uint8_t RondeManufactureCalOsMaxCommand[4] = { 0x04, 0x01, 0x02, 0x00 };
    const uint8_t RondeManufactureCalHkMaxCommand[4] = { 0x04, 0x02, 0x02, 0x00 };
    const uint8_t RondeManufactureCalZeroMaxCommand[4] = { 0x04, 0x03, 0x02, 0x00 };
    const uint8_t RondeManufactureCalZeroMinCommand[4] = { 0x04, 0x04, 0x02, 0x00 };

    auto osMaxResult= GetManufactureCalData(&pOutData->osMax, handle, RondeManufactureCalOsMaxCommand, sizeof(RondeManufactureCalOsMaxCommand));
    if (osMaxResult.IsFailure())
    {
        return osMaxResult;
    }

    auto hkMaxResult = GetManufactureCalData(&pOutData->hkMax, handle, RondeManufactureCalHkMaxCommand, sizeof(RondeManufactureCalHkMaxCommand));
    if (hkMaxResult.IsFailure())
    {
        return hkMaxResult;
    }

    auto zeroMax = GetManufactureCalData(&pOutData->zeroMax, handle, RondeManufactureCalZeroMaxCommand, sizeof(RondeManufactureCalZeroMaxCommand));
    if (zeroMax.IsFailure())
    {
        return zeroMax;
    }
    return GetManufactureCalData(&pOutData->zeroMin, handle, RondeManufactureCalZeroMinCommand, sizeof(RondeManufactureCalZeroMinCommand));
}

nn::Result GetRondeUniqueId(uint8_t* pOutBuffer, size_t bufferSize, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(bufferSize), 12);
    NN_UNUSED(bufferSize);
    const uint8_t RondeGetUnqueIdCommand[4] = { 0x00, 0x01, 0x02, 0x00 };

    uint8_t receiveBuffer[RondeUniqueIdDataSize];
    size_t  outSize;

    auto result = nn::hidbus::SendAndReceive(&outSize, receiveBuffer, sizeof(receiveBuffer), handle,
                                            RondeGetUnqueIdCommand, sizeof(RondeGetUnqueIdCommand));
    if (result.IsFailure())
    {
        return result;
    }

    if (outSize != RondeUniqueIdDataSize || receiveBuffer[RondeResultOffset] != RondeResult_Success)
    {
        if (receiveBuffer[RondeResultOffset] != RondeResult_Success)
        {
            RondeResultHandler(static_cast<RondeResult>(receiveBuffer[RondeResultOffset]));
        }

        // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
        // Timeout 相当として、上位でリトライします。
        return nn::hidbus::ResultExternalDeviceTimeout();
    }

    memcpy(pOutBuffer, &receiveBuffer[4], 12);

    return nn::ResultSuccess();
}

nn::Result GetRondeFirmwareVersion(RondeFwVersion* pOutFwVersion, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT
{
    const uint8_t RondeGetFirmwareVersionCommand[4] = { 0x00, 0x00, 0x02, 0x00 };

    uint8_t receiveBuffer[RondeFirmwareVersionSize];
    size_t  outSize;

    auto result = nn::hidbus::SendAndReceive(&outSize, receiveBuffer, sizeof(receiveBuffer), handle,
        RondeGetFirmwareVersionCommand, sizeof(RondeGetFirmwareVersionCommand));
    if (result.IsFailure())
    {
        return result;
    }

    if (outSize != RondeFirmwareVersionSize || receiveBuffer[RondeResultOffset] != RondeResult_Success)
    {
        if (receiveBuffer[RondeResultOffset] != RondeResult_Success)
        {
            RondeResultHandler(static_cast<RondeResult>(receiveBuffer[RondeResultOffset]));
        }

        // 入っているデータサイズが足りてない場合や、 Ronde からのデータの Result が 0x00 ではない場合は、
        // Timeout 相当として、上位でリトライします。
        return nn::hidbus::ResultExternalDeviceTimeout();
    }

    pOutFwVersion->mainVersion = receiveBuffer[5];
    pOutFwVersion->subVersion = receiveBuffer[4];

    return nn::ResultSuccess();
}

void GetRondeInternalErrorInfo(RondeInternalErrorInfo* pOutInternalErrorInfo) NN_NOEXCEPT
{
    *pOutInternalErrorInfo = g_InternalErrorInfo;
}

}}  // nns::hidbus
