﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_DeviceAssetId.h>
#include <nns/hid/hid_TouchScreen.h>
#include <nns/hid/hid_TouchScreenAsset.h>

namespace nns { namespace hid {

TouchScreen::TouchScreen(ControllerManager* pManager) NN_NOEXCEPT
    : Controller(pManager)
    , m_IsTouched(false)
{
    NN_ASSERT_NOT_NULL(pManager);

    // 対応する TouchScreen のデバイスアセットを取得します。
    m_pTouchScreenAsset = static_cast<TouchScreenAsset*>(
        pManager->GetDeviceAsset(DeviceAssetId_TouchScreen, 0));
}

TouchScreen::~TouchScreen() NN_NOEXCEPT {}

void TouchScreen::Update() NN_NOEXCEPT
{
    // 対応する TouchScreen のデバイスアセットが存在しない場合は何も行いません。
    if (!m_pTouchScreenAsset)
    {
        return;
    }

    // 前回の先頭のタッチ状態を退避します。
    int32_t fingerId = -1;
    if (m_States.size() > 0)
    {
        fingerId = m_States[0].fingerId;
    }

    // タッチ状態を更新します。
    this->TouchScreen::UpdateTouchStates();

    // デジタルボタンと見做した際の押下状態を更新します。
    if (m_IsTouched)
    {
        // タッチが存在しないか先頭のタッチが入れ替わった場合は、
        // デジタルボタンは無効とします。
        if (m_States.size() == 0 || m_States[0].fingerId != fingerId)
        {
            m_IsTouched = false;
        }
    }
    else
    {
        // 新たにタッチが検出された場合はポインティングデバイスは有効とします。
        if (m_States.size() > 0 && m_States[0].IsBegan())
        {
            m_IsTouched = true;
        }
    }

    // ポインティングデバイスと見做した際の座標値を決定します。
    nn::util::Float2 pointer = Controller::GetInvalidPointer();
    if (m_IsTouched)
    {
        // デジタルボタンと見做した場合に押下中であれば座標値を設定します。
        pointer = m_States[0].position;
    }

    // コントローラの基底の状態を更新します。
    const ButtonSet buttons = {};
    const nn::util::Float2 stick = {};
    this->UpdateBase(buttons, stick, stick, m_IsTouched, pointer);
}

void TouchScreen::UpdateTouchState(
    TouchState* pTouchState,
    const TouchScreenAsset::TouchScreenState& touchScreenState) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTouchState);

    TouchState& state = *pTouchState;

    bool isEnded = false;

    bool found = false;

    for (int i = 0; i < touchScreenState.count; ++i)
    {
        const nn::hid::TouchState& touch = touchScreenState.touches[i];

        // タッチを検索して状態を更新します。
        if (state.fingerId == touch.fingerId)
        {
            state.deltaTime += touch.deltaTime;
            state.position.x = static_cast<float>(touch.x);
            state.position.y = static_cast<float>(touch.y);
            isEnded = touch.attributes.Test<nn::hid::TouchAttribute::End>();
            found = true;
            break;
        }
    }

    if (isEnded)
    {
        state.phase |= PointerPhase::Ended::Mask;
    }
    else if (!found)
    {
        // タッチが見つからなかった場合はキャンセルされたものとして処理します。
        state.phase |= PointerPhase::Canceled::Mask;
    }
}

void TouchScreen::UpdateTouchStates() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pTouchScreenAsset);

    // TouchScreen の入力状態を取得します。
    const std::vector<TouchScreenAsset::TouchScreenState>& screenStates =
        m_pTouchScreenAsset->GetTouchScreenStates();

    std::vector<TouchState> states;

    // 前回のタッチ状態を処理します。
    for (std::vector<TouchState>::const_iterator it = m_States.begin();
         it != m_States.end(); ++it)
    {
        // 前回に離れたタッチはタッチ状態から削除します。
        if (it->IsEnded() || it->IsCanceled())
        {
            continue;
        }

        TouchState state = {};
        state.fingerId = it->fingerId;

        // 前回のタッチ座標を初期値として流用します。
        state.position = it->position;

        // TouchScreen の入力状態を古いものから順にタッチ状態に反映します。
        for (std::vector<TouchScreenAsset::TouchScreenState
                         >::const_reverse_iterator rit = screenStates.rbegin();
             rit != screenStates.rend(); ++rit)
        {
            UpdateTouchState(&state, *rit);

            if (state.IsEnded() || state.IsCanceled())
            {
                break;
            }
        }

        // 移動距離を設定します。
        state.deltaPosition.x = state.position.x - it->position.x;
        state.deltaPosition.y = state.position.y - it->position.y;

        states.push_back(state);
    }

    // 新たに検出されたタッチ状態を処理します。
    if (screenStates.size() > 0)
    {
        // 現在の末尾位置を記録します。
        const size_t count = states.size();

        // 新たに検出されたタッチを検索します。
        const TouchScreenAsset::TouchScreenState& screenState = screenStates[0];

        for (int i = 0; i < screenState.count; ++i)
        {
            bool found = false;

            for (size_t j = 0; j < count; ++j)
            {
                if (states[j].fingerId == screenState.touches[i].fingerId)
                {
                    found = true;
                    break;
                }
            }

            if (!found)
            {
                // 新たに検出されたタッチであればタッチ状態に反映します。
                TouchState state = {};
                state.fingerId = screenState.touches[i].fingerId;
                state.phase |= PointerPhase::Began::Mask;
                state.position.x = static_cast<float>(screenState.touches[i].x);
                state.position.y = static_cast<float>(screenState.touches[i].y);
                states.push_back(state);
            }
        }
    }

    // タッチ状態を更新します。
    m_States = states;
}

}} // namespace nns::hid
