﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nns/hid/hid_GamePadNxAsset.h>

namespace nns { namespace hid {

//!< サポートする NpadIdType と コントローラ番号 の対応づけを定義する構造体です
struct NpadPlayerId
{
    nn::hid::NpadIdType npadId;           //!< NpadId の Id
    int controllerNumber;                 //!< コントローラの番号
};

//!< サポートする Npad の Id です。
const NpadPlayerId GamePadNxIds[] =
{
    { nn::hid::NpadId::No1, 0},
    { nn::hid::NpadId::No2, 1},
    { nn::hid::NpadId::No3, 2},
    { nn::hid::NpadId::No4, 3},
    { nn::hid::NpadId::Handheld, 4}
};

namespace {

nn::hid::NpadButtonSet ConvertHorizontalButtonLayout(const nn::hid::NpadStyleSet& style, const nn::hid::NpadButtonSet& button)
{
    nn::hid::NpadButtonSet convButton = button;
    if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
    {
        if (style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            convButton.Set<nn::hid::NpadButton::A>(button.Test<nn::hid::NpadButton::Down>());
            convButton.Set<nn::hid::NpadButton::B>(button.Test<nn::hid::NpadButton::Left>());
            convButton.Set<nn::hid::NpadButton::X>(button.Test<nn::hid::NpadButton::Right>());
            convButton.Set<nn::hid::NpadButton::Y>(button.Test<nn::hid::NpadButton::Up>());

            convButton.Set<nn::hid::NpadButton::Right>(button.Test<nn::hid::NpadButton::StickLDown>());
            convButton.Set<nn::hid::NpadButton::Down>(button.Test<nn::hid::NpadButton::StickLLeft>());
            convButton.Set<nn::hid::NpadButton::Up>(button.Test<nn::hid::NpadButton::StickLRight>());
            convButton.Set<nn::hid::NpadButton::Left>(button.Test<nn::hid::NpadButton::StickLUp>());

            convButton.Set<nn::hid::NpadButton::L>(button.Test<nn::hid::NpadJoyButton::LeftSL>());
            convButton.Set<nn::hid::NpadButton::R>(button.Test<nn::hid::NpadJoyButton::LeftSR>());

            convButton.Reset<nn::hid::NpadButton::ZL>();

            convButton.Set<nn::hid::NpadButton::StickLRight>(button.Test<nn::hid::NpadButton::StickLDown>());
            convButton.Set<nn::hid::NpadButton::StickLDown>(button.Test<nn::hid::NpadButton::StickLLeft>());
            convButton.Set<nn::hid::NpadButton::StickLUp>(button.Test<nn::hid::NpadButton::StickLRight>());
            convButton.Set<nn::hid::NpadButton::StickLLeft>(button.Test<nn::hid::NpadButton::StickLUp>());

            convButton.Reset<nn::hid::NpadJoyButton::LeftSL>();
            convButton.Reset<nn::hid::NpadJoyButton::LeftSR>();
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            convButton.Set<nn::hid::NpadButton::A>(button.Test<nn::hid::NpadButton::X>());
            convButton.Set<nn::hid::NpadButton::B>(button.Test<nn::hid::NpadButton::A>());
            convButton.Set<nn::hid::NpadButton::X>(button.Test<nn::hid::NpadButton::Y>());
            convButton.Set<nn::hid::NpadButton::Y>(button.Test<nn::hid::NpadButton::B>());

            convButton.Set<nn::hid::NpadButton::Right>(button.Test<nn::hid::NpadButton::StickRUp>());
            convButton.Set<nn::hid::NpadButton::Down>(button.Test<nn::hid::NpadButton::StickRRight>());
            convButton.Set<nn::hid::NpadButton::Up>(button.Test<nn::hid::NpadButton::StickRLeft>());
            convButton.Set<nn::hid::NpadButton::Left>(button.Test<nn::hid::NpadButton::StickRDown>());

            convButton.Set<nn::hid::NpadButton::L>(button.Test<nn::hid::NpadJoyButton::RightSL>());
            convButton.Set<nn::hid::NpadButton::R>(button.Test<nn::hid::NpadJoyButton::R>() | button.Test<nn::hid::NpadJoyButton::RightSR>());

            convButton.Reset<nn::hid::NpadButton::ZR>();

            convButton.Set<nn::hid::NpadButton::StickRRight>(button.Test<nn::hid::NpadButton::StickRUp>());
            convButton.Set<nn::hid::NpadButton::StickRDown>(button.Test<nn::hid::NpadButton::StickRRight>());
            convButton.Set<nn::hid::NpadButton::StickRUp>(button.Test<nn::hid::NpadButton::StickRLeft>());
            convButton.Set<nn::hid::NpadButton::StickRLeft>(button.Test<nn::hid::NpadButton::StickRDown>());

            convButton.Reset<nn::hid::NpadJoyButton::RightSL>();
            convButton.Reset<nn::hid::NpadJoyButton::RightSR>();
        }
    }
    return convButton;
}

nn::hid::AnalogStickState ConvertHorizontalAnalogStick(const nn::hid::NpadStyleSet& style, const nn::hid::AnalogStickState& analogStick)
{
    nn::hid::AnalogStickState convStickAxis = analogStick;
    if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
    {
        if (style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            convStickAxis.y = analogStick.x;
            convStickAxis.x = -analogStick.y;
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            convStickAxis.y = -analogStick.x;
            convStickAxis.x = analogStick.y;
        }
    }
    return convStickAxis;
}

} // namespace

GamePadNxAsset::GamePadNxAsset(nns::hid::ControllerManager* pManager) NN_NOEXCEPT
    : DeviceAsset(pManager)
    , m_States(new GamePadNxState[NN_ARRAY_SIZE(GamePadNxIds)])
    , m_StatesTable(new int[ControllerCountMax])
{
    NN_ASSERT_NOT_NULL(pManager);
    NN_ASSERT_NOT_NULL(m_States);
    NN_ASSERT_NOT_NULL(m_StatesTable);

    // controllerNumber に対応するバッファの割り当てを初期化
    for (int controllerNumberIndex = 0; controllerNumberIndex < ControllerCountMax; ++controllerNumberIndex)
    {
        for (size_t GamePadNxIndex = 0; GamePadNxIndex < NN_ARRAY_SIZE(GamePadNxIds); ++GamePadNxIndex)
        {
            if (GamePadNxIds[GamePadNxIndex].controllerNumber == controllerNumberIndex)
            {
                m_StatesTable[controllerNumberIndex] = static_cast<int>(GamePadNxIndex);
                break;
            }
        }
    }
}

GamePadNxAsset::~GamePadNxAsset() NN_NOEXCEPT
{
    delete[] m_States;
}

void GamePadNxAsset::Initialize() NN_NOEXCEPT
{
    // Npad を初期化します。
    nn::hid::InitializeNpad();

    // Npad の Id を初期化します
    nn::hid::NpadIdType supportedNpadIds[NN_ARRAY_SIZE(GamePadNxIds)];
    for (size_t i = 0; i < NN_ARRAY_SIZE(supportedNpadIds); ++i)
    {
        supportedNpadIds[i] = GamePadNxIds[i].npadId;
    }
    nn::hid::SetSupportedNpadIdType(supportedNpadIds, NN_ARRAY_SIZE(supportedNpadIds));

    // Npad の Style を初期化します
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
}

void GamePadNxAsset::Update() NN_NOEXCEPT
{
    // コントローラの入力状態を取得します。
    for (size_t i = 0; i < NN_ARRAY_SIZE(GamePadNxIds); ++i)
    {
        auto style = nn::hid::GetNpadStyleSet(GamePadNxIds[i].npadId);
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            m_States[i].buttons = ConvertHorizontalButtonLayout(style, m_States[i].buttons);
            nn::hid::GetNpadState(&m_States[i], GamePadNxIds[i].npadId);
        }
        else if (style.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState handheldState;
            nn::hid::GetNpadState(&handheldState, GamePadNxIds[i].npadId);

            // NpadHandheldState を NpadFullKeyState に変換
            m_States[i].buttons = ConvertHorizontalButtonLayout(style, handheldState.buttons);
            m_States[i].analogStickL = handheldState.analogStickL;
            m_States[i].analogStickR = handheldState.analogStickR;
            m_States[i].attributes = handheldState.attributes;
            m_States[i].samplingNumber = handheldState.samplingNumber;
        }
        else if (style.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState joyDualState;
            nn::hid::GetNpadState(&joyDualState, GamePadNxIds[i].npadId);

            // NpadJoyDualState を NpadFullKeyState に変換
            m_States[i].buttons = ConvertHorizontalButtonLayout(style, joyDualState.buttons);
            m_States[i].analogStickL = joyDualState.analogStickL;
            m_States[i].analogStickR = joyDualState.analogStickR;
            m_States[i].attributes = joyDualState.attributes;
            m_States[i].attributes.Set<GamePadNxAttribute::IsConnected>(
                joyDualState.attributes.Test<GamePadNxAttribute::IsLeftConnected>() &&
                joyDualState.attributes.Test<GamePadNxAttribute::IsRightConnected>()
                );
            m_States[i].samplingNumber = joyDualState.samplingNumber;
        }
        else if (style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            nn::hid::NpadJoyLeftState joyLeftState;
            nn::hid::GetNpadState(&joyLeftState, GamePadNxIds[i].npadId);

            // NpadJoyLeftState を NpadFullKeyState に変換
            m_States[i].buttons = ConvertHorizontalButtonLayout(style, joyLeftState.buttons);
            m_States[i].analogStickL = ConvertHorizontalAnalogStick(style, joyLeftState.analogStickL);
            m_States[i].analogStickR = m_States[i].analogStickL;
            m_States[i].attributes = joyLeftState.attributes;
            m_States[i].samplingNumber = joyLeftState.samplingNumber;
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::NpadJoyRightState joyRightState;
            nn::hid::GetNpadState(&joyRightState, GamePadNxIds[i].npadId);

            // NpadJoyRightState を NpadFullKeyState に変換
            m_States[i].buttons = ConvertHorizontalButtonLayout(style, joyRightState.buttons);
            m_States[i].analogStickR = ConvertHorizontalAnalogStick(style, joyRightState.analogStickR);
            m_States[i].analogStickL = m_States[i].analogStickR;
            m_States[i].attributes = joyRightState.attributes;
            m_States[i].samplingNumber = joyRightState.samplingNumber;
        }
        else
        {
            m_States[i] = GamePadNxState();
        }
    }
}

bool GamePadNxAsset::GetGamePadNxState(GamePadNxState* pOutValue, int controllerNumber
                                   ) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    // コントローラの管理番号が範囲外にある場合は入力状態の取得に失敗します。
    if (controllerNumber < 0 || ControllerCountMax < controllerNumber)
    {
        return false;
    }

    // 入力状態のバッファと controllerNumber の対応関係を更新
    if (m_States[m_StatesTable[controllerNumber]].buttons.IsAnyOn() == false)
    {
        // 今対応関係にあるものが操作状態にある場合は他に操作状態にあるバッファがないか探索する
        for (size_t i = 0; i < NN_ARRAY_SIZE(GamePadNxIds); ++i)
        {
            // controllerNumber が一致して入力状態にある場合は、バッファと controllerNumber の対応関係を更新
            if (GamePadNxIds[i].controllerNumber == controllerNumber &&
                m_States[i].buttons.IsAnyOn())
            {
                m_StatesTable[controllerNumber] = i;
                break;
            }
        }
    }

    *pOutValue = m_States[m_StatesTable[controllerNumber]];

    return true;
}

}} // namespace nns::hid
