﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_DeviceAssetId.h>
#include <nns/hid/hid_GamePadNx.h>

namespace nns { namespace hid {

namespace {

/**
 * @brief       スティックの座標値を正規化します。
 *
 * @param[in]   x                           スティックの座標値です。
 *
 * @return      正規化されたスティックの座標値です。
 */
float GetNormalizedPosition(int x)
{
    NN_ASSERT_MINMAX(x, -::nn::hid::AnalogStickMax, ::nn::hid::AnalogStickMax);

    // 境界値については丸め込み誤差が発生しないよう即値を返します。
    if (x == 0)
    {
        return 0.0f;
    }

    if (x == ::nn::hid::AnalogStickMax)
    {
        return 1.0f;
    }

    if (x == -::nn::hid::AnalogStickMax)
    {
        return -1.0f;
    }

    // 正規化した位置を返します。
    return static_cast<float>(x) / ::nn::hid::AnalogStickMax;
}

} // namespace

GamePadNx::GamePadNx(nns::hid::ControllerManager* pManager) NN_NOEXCEPT
    : Controller(pManager)
    , m_ControllerNumber(0)
    , m_IsConnected(false)
{
    NN_ASSERT_NOT_NULL(pManager);

    // 対応するコントローラのデバイスアセットを取得します。
    m_pGamePadNxAsset = static_cast<GamePadNxAsset*>(
        pManager->GetDeviceAsset(0));

    m_Attributes.Reset();
}

GamePadNx::~GamePadNx() NN_NOEXCEPT {}

void GamePadNx::Update() NN_NOEXCEPT
{
    // 対応するコントローラのデバイスアセットが存在しない場合は何も行いません。
    if (!m_pGamePadNxAsset)
    {
        return;
    }

    // 一旦コントローラを切断状態にします。
    m_IsConnected = false;
    m_Attributes.Reset();

    nns::hid::ButtonSet buttons = {};
    nn::util::Float2 leftStick = {};
    nn::util::Float2 rightStick = {};

    // コントローラの入力状態を取得します。
    GamePadNxAsset::GamePadNxState state = {};
    if (m_pGamePadNxAsset->GetGamePadNxState(&state, m_ControllerNumber))
    {
        // 接続状態を設定します。
        m_IsConnected = state.attributes.Test<
            GamePadNxAsset::GamePadNxAttribute::IsConnected>();

        m_Attributes.Set<Attribute::IsConnected>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsConnected>());
        m_Attributes.Set<Attribute::IsWired>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsWired>());
        m_Attributes.Set<Attribute::IsLeftConnected>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsLeftConnected>());
        m_Attributes.Set<Attribute::IsLeftWired>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsLeftWired>());
        m_Attributes.Set<Attribute::IsRightConnected>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsRightConnected>());
        m_Attributes.Set<Attribute::IsRightWired>(
            state.attributes.Test<GamePadNxAsset::GamePadNxAttribute::IsRightWired>());

        // デジタルボタンの押下状態を対応付けます。
        buttons.Set<nns::hid::Button::A>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::A>());
        buttons.Set<nns::hid::Button::B>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::B>());
        buttons.Set<nns::hid::Button::X>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::X>());
        buttons.Set<nns::hid::Button::Y>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Y>());
        buttons.Set<nns::hid::Button::StickL>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::StickL>());
        buttons.Set<nns::hid::Button::StickR>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::StickR>());
        buttons.Set<nns::hid::Button::Minus>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Minus>());
        buttons.Set<nns::hid::Button::Plus>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Plus>());
        buttons.Set<nns::hid::Button::Start>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Plus>());
        buttons.Set<nns::hid::Button::Select>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Minus>());
        buttons.Set<nns::hid::Button::ZL>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::ZL>());
        buttons.Set<nns::hid::Button::ZR>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::ZR>());
        buttons.Set<nns::hid::Button::L>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::L>());
        buttons.Set<nns::hid::Button::R>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::R>());
        buttons.Set<nns::hid::Button::Up>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Up>());
        buttons.Set<nns::hid::Button::Down>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Down>());
        buttons.Set<nns::hid::Button::Left>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Left>());
        buttons.Set<nns::hid::Button::Right>(
            state.buttons.Test<GamePadNxAsset::GamePadNxButton::Right>());

        // 左スティックの入力座標を対応付けます。
        leftStick.x = GetNormalizedPosition(state.analogStickL.x);
        leftStick.y = GetNormalizedPosition(state.analogStickL.y);

        // 右スティックの入力座標を対応付けます。
        rightStick.x = GetNormalizedPosition(state.analogStickR.x);
        rightStick.y = GetNormalizedPosition(state.analogStickR.y);
    }

    // コントローラの基底の状態を更新します。
    this->UpdateBase(buttons,
                     leftStick,
                     rightStick,
                     false,
                     Controller::GetInvalidPointer());
}

}} // namespace nns::hid
