﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nns/hid/hid_GamePadAsset.h>

namespace nns { namespace hid {

namespace {

//!< サポートする NpadIdType と プレイヤー番号の対応づけを定義する構造体です
struct GamePadId
{
    nn::hid::NpadIdType npadId;           //!< NpadId の Id
    int playerNumber;                     //!< プレイヤー番号
};

//!< サポートする Npad の Id です。
const GamePadId GamePadIds[] =
{
    { nn::hid::NpadId::No1, 0},
    { nn::hid::NpadId::No2, 1},
    { nn::hid::NpadId::No3, 2},
    { nn::hid::NpadId::No4, 3},
    { nn::hid::NpadId::Handheld, 0}
};

} // namespace

GamePadAsset::GamePadAsset(ControllerManager* pManager) NN_NOEXCEPT
    : DeviceAsset(pManager)
    , m_States(new GamePadState[NN_ARRAY_SIZE(GamePadIds)])
    , m_StatesTable(new int[PlayerNumberCountMax])
{
    NN_ASSERT_NOT_NULL(pManager);
    NN_ASSERT_NOT_NULL(m_States);
    NN_ASSERT_NOT_NULL(m_StatesTable);

    // PlayerNumber に対応するバッファの割り当てを初期化
    for (int playerNumberIndex = 0; playerNumberIndex < PlayerNumberCountMax; ++playerNumberIndex)
    {
        for (int gamePadIndex = 0; gamePadIndex < NN_ARRAY_SIZE(GamePadIds); ++gamePadIndex)
        {
            if (GamePadIds[gamePadIndex].playerNumber == playerNumberIndex)
            {
                m_StatesTable[playerNumberIndex] = gamePadIndex;
                break;
            }
        }
    }
}

GamePadAsset::~GamePadAsset() NN_NOEXCEPT
{
    delete[] m_States;
}

void GamePadAsset::Initialize() NN_NOEXCEPT
{
    // Npad を初期化します。
    nn::hid::InitializeNpad();

    // Npad の Id を初期化します
    nn::hid::NpadIdType supportedNpadIds[NN_ARRAY_SIZE(GamePadIds)];
    for (int i = 0; i < NN_ARRAY_SIZE(supportedNpadIds); ++i)
    {
        supportedNpadIds[i] = GamePadIds[i].npadId;
    }
    nn::hid::SetSupportedNpadIdType(supportedNpadIds, NN_ARRAY_SIZE(supportedNpadIds));

    // Npad の Style を初期化します
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
}

void GamePadAsset::Update() NN_NOEXCEPT
{
    // ゲームパッドの入力状態を取得します。
    for (size_t i = 0; i < NN_ARRAY_SIZE(GamePadIds); ++i)
    {
        auto style = nn::hid::GetNpadStyleSet(GamePadIds[i].npadId);
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::GetNpadState(&m_States[i], GamePadIds[i].npadId);
        }
        else if (style.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState handheldState;
            nn::hid::GetNpadState(&handheldState, GamePadIds[i].npadId);

            // NpadHandheldState を NpadFullKeyState に変換
            m_States[i].buttons = handheldState.buttons;
            m_States[i].analogStickL = handheldState.analogStickL;
            m_States[i].analogStickR = handheldState.analogStickR;
            m_States[i].attributes = handheldState.attributes;
            m_States[i].samplingNumber = handheldState.samplingNumber;
        }
        else
        {
            m_States[i] = GamePadState();
        }
    }
}

bool GamePadAsset::GetGamePadState(GamePadState* pOutValue, int playerNumber
                                   ) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    // プレイヤー番号が範囲外にある場合は入力状態の取得に失敗します。
    if (playerNumber < 0 || PlayerNumberCountMax < playerNumber)
    {
        return false;
    }

    // 入力状態のバッファと playerNumber の対応関係を更新
    if (m_States[m_StatesTable[playerNumber]].buttons.IsAnyOn() == false)
    {
        // 今対応関係にあるものが操作状態にある場合は他に操作状態にあるバッファがないか探索する
        for (int i = 0; i < NN_ARRAY_SIZE(GamePadIds); ++i)
        {
            // playerNumber が一致して入力状態にある場合は、バッファと playerNumber の対応関係を更新
            if (GamePadIds[i].playerNumber == playerNumber &&
                m_States[i].buttons.IsAnyOn())
            {
                m_StatesTable[playerNumber] = i;
                break;
            }
        }
    }

    *pOutValue = m_States[m_StatesTable[playerNumber]];

    return true;
}

}} // namespace nns::hid
