﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_DeviceAssetId.h>
#include <nns/hid/hid_GamePad.h>

namespace nns { namespace hid {

namespace {

/**
 * @brief       コントローラ用にゲームパッドのスティックの座標値を正規化します。
 *
 * @param[in]   x                           ゲームパッドのスティックの座標値です。
 *
 * @return      正規化されたゲームパッドのスティックの座標値です。
 */
float GetNormalizedPosition(int x)
{
    NN_ASSERT_MINMAX(x, -::nn::hid::AnalogStickMax, ::nn::hid::AnalogStickMax);

    // 境界値については丸め込み誤差が発生しないよう即値を返します。
    if (x == 0)
    {
        return 0.0f;
    }

    if (x == ::nn::hid::AnalogStickMax)
    {
        return 1.0f;
    }

    if (x == -::nn::hid::AnalogStickMax)
    {
        return -1.0f;
    }

    // 正規化した位置を返します。
    return static_cast<float>(x) / ::nn::hid::AnalogStickMax;
}

} // namespace

GamePad::GamePad(ControllerManager* pManager) NN_NOEXCEPT
    : Controller(pManager)
    , m_PlayerNumber(0)
    , m_IsConnected(false)
{
    NN_ASSERT_NOT_NULL(pManager);

    // 対応するゲームパッドのデバイスアセットを取得します。
    m_pGamePadAsset = static_cast<GamePadAsset*>(
        pManager->GetDeviceAsset(DeviceAssetId_GamePad, 0));
}

GamePad::~GamePad() NN_NOEXCEPT {}

void GamePad::Update() NN_NOEXCEPT
{
    // 対応するゲームパッドのデバイスアセットが存在しない場合は何も行いません。
    if (!m_pGamePadAsset)
    {
        return;
    }

    // 一旦ゲームパッドを切断状態にします。
    m_IsConnected = false;

    ButtonSet buttons = {};
    nn::util::Float2 leftStick = {};
    nn::util::Float2 rightStick = {};

    // ゲームパッドの入力状態を取得します。
    GamePadAsset::GamePadState state = {};
    if (m_pGamePadAsset->GetGamePadState(&state, m_PlayerNumber))
    {
        // 接続状態を設定します。
        m_IsConnected = state.attributes.Test<
            GamePadAsset::GamePadAttribute::IsConnected>();

        // デジタルボタンの押下状態を対応付けます。
        buttons.Set<Button::A>(
            state.buttons.Test<GamePadAsset::GamePadButton::A>());
        buttons.Set<Button::B>(
            state.buttons.Test<GamePadAsset::GamePadButton::B>());
        buttons.Set<Button::X>(
            state.buttons.Test<GamePadAsset::GamePadButton::X>());
        buttons.Set<Button::Y>(
            state.buttons.Test<GamePadAsset::GamePadButton::Y>());
        buttons.Set<Button::StickL>(
            state.buttons.Test<GamePadAsset::GamePadButton::StickL>());
        buttons.Set<Button::StickR>(
            state.buttons.Test<GamePadAsset::GamePadButton::StickR>());
        buttons.Set<Button::Minus>(
            state.buttons.Test<GamePadAsset::GamePadButton::Minus>());
        buttons.Set<Button::Plus>(
            state.buttons.Test<GamePadAsset::GamePadButton::Plus>());
        buttons.Set<Button::Start>(
            state.buttons.Test<GamePadAsset::GamePadButton::Plus>());
        buttons.Set<Button::Select>(
            state.buttons.Test<GamePadAsset::GamePadButton::Minus>());
        buttons.Set<Button::ZL>(
            state.buttons.Test<GamePadAsset::GamePadButton::ZL>());
        buttons.Set<Button::ZR>(
            state.buttons.Test<GamePadAsset::GamePadButton::ZR>());
        buttons.Set<Button::L>(
            state.buttons.Test<GamePadAsset::GamePadButton::L>());
        buttons.Set<Button::R>(
            state.buttons.Test<GamePadAsset::GamePadButton::R>());
        buttons.Set<Button::Up>(
            state.buttons.Test<GamePadAsset::GamePadButton::Up>());
        buttons.Set<Button::Down>(
            state.buttons.Test<GamePadAsset::GamePadButton::Down>());
        buttons.Set<Button::Left>(
            state.buttons.Test<GamePadAsset::GamePadButton::Left>());
        buttons.Set<Button::Right>(
            state.buttons.Test<GamePadAsset::GamePadButton::Right>());

        // 左スティックの入力座標を対応付けます。
        leftStick.x = GetNormalizedPosition(state.analogStickL.x);
        leftStick.y = GetNormalizedPosition(state.analogStickL.y);

        // 右スティックの入力座標を対応付けます。
        rightStick.x = GetNormalizedPosition(state.analogStickR.x);
        rightStick.y = GetNormalizedPosition(state.analogStickR.y);
    }

    // コントローラの基底の状態を更新します。
    this->UpdateBase(buttons,
                     leftStick,
                     rightStick,
                     false,
                     Controller::GetInvalidPointer());
}

}} // namespace nns::hid
