﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>


#include "gfxLogGpuBuffer.h"
#include "gfxLogPrimitiveRendererShaderRes.h"


namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//---------------------------------------------------------------------------
//! @brief 前方宣言です。
//---------------------------------------------------------------------------
class MemoryInfo;

//! @brief バッファの種類です。
enum BufferType
{
    BufferType_VertexBuffer = 0,
    BufferType_IndexBuffer,
    BufferType_ConstantBuffer,

    BufferType_End,
};

//! @brief プリミティブの形状です。
enum ShapeType
{
    ShapeType_Quad              = 0,
    ShapeType_CountMax  // 種類の数です。
};

//! @brief サーフェイスの種類です。
//! @details 関数で使うため、例外として短い名前にしています。
enum Surface
{
    Surface_Wired = 0,
    Surface_Solid,

    Surface_CountMax,

    Wired = Surface_Wired,
    Solid = Surface_Solid,
};

//! @brief サーフェイスの細分化レベルです。
//! @details 関数で使うため、例外として短い名前にしています。
enum Subdiv
{
    Subdiv_Coarse = 0,
    Subdiv_Normal,
    Subdiv_Beautiful,

    Subdiv_CountMax,

    Coarse = Subdiv_Coarse,
    Normal = Subdiv_Normal,
    Beautiful = Subdiv_Beautiful
};

//---------------------------------------------------------------------------
//! @brief メッシュです。
//! @details 1つだけでなく、複数の形状を設定することが出来ます。
//---------------------------------------------------------------------------
class PrimitiveMesh
{
    NN_DISALLOW_COPY(PrimitiveMesh);
    NN_DISALLOW_MOVE(PrimitiveMesh);
public:
    //! @brief コンストラクタです。
    PrimitiveMesh() NN_NOEXCEPT;

    //----------------------------------------
    //! @name 初期化／後処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief initializeを行います。
    //!
    //! @param[in] pGpuBuffer   利用するGpuBufferのポインタ
    //! @param[in] numVertices  頂点数
    //! @param[in] numIndices   インデックス数
    //! @param[in] vertexFormat 頂点バッファのフォーマット
    //! @return initializeに成功した場合に、trueを返します。
    //---------------------------------------------------------------------------
    bool Initialize( GpuBuffer* pGpuBuffer,
                     int numVertices,
                     int numIndices,
                     VertexFormat vertexFormat = VertexFormat_Default ) NN_NOEXCEPT;

    //@}


    //----------------------------------------
    //! @name 設定／取得（パラメータ）
    //! @{

    //! @brief インデック数を取得します。
    //! @return インデック数です。
    int GetIndexCount() const NN_NOEXCEPT;

    //! @brief 頂点属性のストライドを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return ストライドです。
    size_t GetVertexStride( VertexAttribute attrib ) const NN_NOEXCEPT;

    //! @brief 指定された頂点属性のメモリサイズを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return メモリサイズを返します。
    size_t GetVertexMemorySize( VertexAttribute attrib ) const NN_NOEXCEPT;

    //! @brief 頂点バッファのポインタを取得します。
    //! @param[in] attribute   頂点属性です。
    //! @return ポインタを返します。
    void* GetVertexBufferCpuAddress( VertexAttribute attribute ) const NN_NOEXCEPT;

    //! @brief インデックスバッファのポインタを取得します。
    //! @return ポインタを返します。
    uint32_t* GetIndexBufferCpuAddress() const NN_NOEXCEPT;

    //! @brief 指定された頂点属性のメモリサイズを取得します。
    //! @param[in] attrib   頂点属性です。
    //! @param[out] pGpuAddress
    void GetVertexBufferGpuAddress(
        VertexAttribute attrib,
        nn::gfx::GpuAddress* pGpuAddress) const NN_NOEXCEPT;

    //! @brief 指定された頂点属性が初期化済みか取得します。
    //! @param[in] attrib   頂点属性です。
    //! @return trueなら初期化済。falseなら未初期化
    bool IsValidVertexBuffer( VertexAttribute attrib ) const NN_NOEXCEPT;

    //! @brief インデックスバッファが初期化済みか取得します。
    //! @return trueなら初期化済。falseなら未初期化
    bool IsValidIndexBuffer() const NN_NOEXCEPT;

    //! @brief インデックスバッファのGpuAddressを取得します。
    //! @param[out] pGpuAddress インデックスバッファのGpuAddress
    void GetIndexBufferGpuAddress(
        nn::gfx::GpuAddress* pGpuAddress ) const NN_NOEXCEPT;

    //! @}

    //----------------------------------------
    //! @name コピー
    //! @{

    template<typename T>
    //! @brief 頂点属性の頂点バッファに、指定のデータ型で、指定のデータでコピーします。
    //! @param[in] attribute    頂点属性です。
    //! @param[in] pPodData     PODデータです。
    //! @param[in] size         データサイズです。
    //! @return コピー結果です。trueならコピー成功です。
    bool CopyVertexBuffer(
        VertexAttribute attribute,
        const T* pPodData,
        size_t size ) NN_NOEXCEPT;

    //! @brief インデックスバッファに、指定データをコピーします。
    //! @param[in] pPodData     PODデータです。
    //! @param[in] size         データサイズです。
    //! @return コピー結果です。trueならコピー成功です。
    bool CopyIndexBuffer( const uint32_t* pPodData, size_t size ) NN_NOEXCEPT;

    //! @}
private:

    // インタリーブではなく、頂点属性ごとにバッファを持ちます。
    // 未使用の頂点属性は、アロケーションしません。
    GpuBuffer* m_pGpuBuffer;
    void* m_pVertexBufferPointer[ VertexAttribute_CountMax ];
    void* m_pIndexBufferPointer;
    int m_VertexCount;
    int m_IndexCount;
};

//---------------------------------------------------------------------------
//! @brief メッシュの集合です。
//---------------------------------------------------------------------------
class MeshSet
{
    NN_DISALLOW_COPY(MeshSet);
    NN_DISALLOW_MOVE(MeshSet);
public:
    //! @brief コンストラクタです。
    MeshSet() NN_NOEXCEPT;

    //! @brief メッシュを取得します。
    //! @param[in] shapeType    メッシュ形状の種類です。
    //! @return メッシュです。
    PrimitiveMesh* Get( ShapeType shapeType ) NN_NOEXCEPT;

    //! @brief MeshSetに必要なMemoryPoolのサイズを計算します。
    //! @param[in] alignment    アライメントのサイズ
    //! @return 必要なMemoryPoolのサイズ
    static size_t CalculateMemoryPoolSize( const size_t alignment ) NN_NOEXCEPT;

private:
    PrimitiveMesh m_Mesh[ ShapeType_CountMax ];
};



} // namespace PrimitiveRenderer
} // namespace gfxLog
} // namespace nns
