﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/gfx.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Color.h>

#include "gfxLogGpuBuffer.h"
#include "gfxLogPrimitiveRendererGfxRes.h"

namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! @brief 前方宣言
//------------------------------------------------------------------------------
struct RendererInfo;
class Renderer;


//! @brief レンダラクラスです。
//! @details １つのインスタンスが、１つのスレッドで処理します。
class Renderer
{
    NN_DISALLOW_COPY(Renderer);
    NN_DISALLOW_MOVE(Renderer);

public:

    // TODO: ユーザが直接指定できる描画インターフェイスを用意します。

    //! @brief モデルごとの情報を与える UniformBlock
    struct Model
    {
        nn::util::FloatT4x4 u_userMatrix;
        nn::util::Float4 u_color;
        nn::util::Float2 u_uv_src;
        nn::util::Float2 u_uv_size;
        // xは、テクスチャではLODです。テクスチャ配列ではレイヤーです。
        nn::util::Float4 u_layer;
        float rate;
    };

    //! @brief mvp行列を与える UniformBlock
    struct View
    {
        nn::util::FloatT4x4         u_mvp;
    };

    //! @brief シェーダで利用するパラメータを管理する構造体
    struct Param
    {
        nn::util::Matrix4x4fType    projMatrix;
        nn::util::Matrix4x4fType    viewMatrix;
        nn::util::Matrix4x4fType    modelMatrix;

        float lineWidth;

        void SetDefault()
        {
            nn::util::MatrixIdentity( &projMatrix );
            nn::util::MatrixIdentity( &viewMatrix );
            nn::util::MatrixIdentity( &modelMatrix );
            lineWidth = 1.0f;
        }
    };

public:
    //! @brief デフォルトシェーダのパラメータを初期値にします。
    //! @param[in]    pModel モデルデータです。
    static void SetDefault( Model* pModel ) NN_NOEXCEPT;

    //! @brief 必要なMemoryPoolを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        レンダラ初期化に使うパラメータ情報です。
    //! @return                   必要なMemoryPool
    static size_t GetRequiredMemoryPoolSize( nn::gfx::Device* pGfxDevice,
                                             const RendererInfo& info ) NN_NOEXCEPT;

    //! @brief MemoryPoolのアライメントを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        レンダラ初期化に使うパラメータ情報です。
    //! @return                   MemoryPoolのアライメント
    static size_t GetMemoryPoolAlignment( nn::gfx::Device* pGfxDevice,
                                           const RendererInfo& info ) NN_NOEXCEPT;

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    Renderer() NN_NOEXCEPT;

    //@}


    //----------------------------------------
    //! @name 初期化／後処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief 初期化します
    //!
    //! @param[in]    pGfxDevice         gfxデバイスです。
    //! @param[in]    info               レンダラ初期化に使うパラメータ情報です。
    //! @param[in]    pGraphicsResource  レンダラが利用するリソースです。
    //! @param[in]    pMemoryPool        Renderer内で利用するメモリプールのポインタです。
    //! @param[in]    memoryPoolOffset   pMemoryPoolのオフセット値です。
    //! @param[in]    memoryPoolSize     pMmoeryPoolのサイズです。
    //! @return                          初期化に成功したらtrueを返します。
    //---------------------------------------------------------------------------
    bool Initialize( nn::gfx::Device* pGfxDevice,
                     const RendererInfo& info,
                     GraphicsResource* pGraphicsResource,
                     nn::gfx::MemoryPool* pMemoryPool,
                     ptrdiff_t memoryPoolOffset,
                     size_t memoryPoolSize ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief 終了処理です。
    //! @param[in]    pGfxDevice              Initialize時に使用したgfxデバイスです。
    //---------------------------------------------------------------------------
    void Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 設定／取得（パラメータ）
    //! @{

    //---------------------------------------------------------------------------
    //! @brief カラーを設定します。
    //! @details プリミティブを描画するときのカラーを設定します。カラー0,1をまとめて設定します。
    //! @param[in]    color カラー
    //---------------------------------------------------------------------------
    void SetColor( const nn::util::Color4u8& color ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief スクリーンの幅を設定します。
    //! @param[in]    screenWidth スクリーンの幅
    //---------------------------------------------------------------------------
    void SetScreenWidth(const int screenWidth) NN_NOEXCEPT
    {
        m_ScreenWidth = screenWidth;
    }

    //---------------------------------------------------------------------------
    //! @brief スクリーンの高さを設定します。
    //! @param[in]    screenHeight スクリーンの高さ
    //---------------------------------------------------------------------------
    void SetScreenHeight(const int screenHeight) NN_NOEXCEPT
    {
        m_ScreenHeight = screenHeight;
    }

    //! @}

    //----------------------------------------
    //! @name 更新
    //! @{

    //---------------------------------------------------------------------------
    //! @brief 更新します。描画コール数が0になり、定数バッファがクリアされます。
    //! @param[in]    bufferIndex   利用する定数バッファのインデックス
    //---------------------------------------------------------------------------
    void Update( int bufferIndex = 0 ) NN_NOEXCEPT;

    //! @}

    //----------------------------------------
    //! @name 描画
    //! @{

     //---------------------------------------------------------------------------
    //! @brief        スクリーンに四角形を描画します。
    //!
    //! @param[in]    pCommandBuffer コマンドバッファです。
    //! @param[in]    x              四角形の左上頂点のx座標です。
    //! @param[in]    y              四角形の左上頂点のy座標です。
    //! @param[in]    width          四角形の幅です。
    //! @param[in]    height         四角形の高さです。
    //---------------------------------------------------------------------------
    void Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                     const float x,
                     const float y,
                     const float width,
                     const float height ) NN_NOEXCEPT;

    //----------------------------------------
    //! @name ユーティリティー
    //! @{

    void SetGraphicsResource(GraphicsResource* pGraphicResource) NN_NOEXCEPT
    {
        m_pGraphicsResource = pGraphicResource;
    }
    GraphicsResource* GetGraphicsResource() const NN_NOEXCEPT
    {
        return m_pGraphicsResource;
    }

    void SetMemoryPool(nn::gfx::MemoryPool* pMemoryPool) NN_NOEXCEPT
    {
        m_pMemoryPool = pMemoryPool;
    }
    nn::gfx::MemoryPool* GetMemoryPool() const NN_NOEXCEPT
    {
        return m_pMemoryPool;
    }

    //! @}


private:
    //---------------------------------------------------------------------------
    //! @brief デフォルトシェーダの定数バッファを更新します。
    //! @param[in]    pPodData 定数バッファに設定するPODデータです。
    //---------------------------------------------------------------------------
    void SetViewConstantBuffer( View* pPodData ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief デフォルトシェーダの定数バッファを更新します。
    //! @details TODO：ユーザが設定できるようにします。現行版は呼び出しても反映されません。
    //! @param[in]    pPodData 定数バッファに設定するPODデータです。
    //---------------------------------------------------------------------------
    void SetModelConstantBuffer( Model* pPodData ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief ビュー依存の定数バッファを更新します。
    //---------------------------------------------------------------------------
    void UpdateGpuView() NN_NOEXCEPT;

    //! @brief プリミティブを描画します。
    void DrawPrimitiveInternal( ShapeType shapeType,
                                nn::gfx::CommandBuffer* pCommandBuffer,
                                Model* pModel,
                                const SamplerType samplerType = SamplerType_CountMax,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot = NULL,
                                const nn::gfx::DescriptorSlot* pSamplerSlot = NULL
                                ) NN_NOEXCEPT;

    //! @brief プリミティブを描画します。
    void DrawPrimitiveInternal( nn::gfx::PrimitiveTopology topology,
                                ShaderVariation shaderVariation,
                                nn::gfx::CommandBuffer* pCommandBuffer,
                                const PrimitiveMesh* pMesh,
                                Model* pModel,
                                const int indexBase,
                                const int numUserIndices,
                                const int userBaseVertex,
                                const SamplerType samplerType,
                                const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                const nn::gfx::DescriptorSlot* pSamplerSlot
                                ) NN_NOEXCEPT;

    //! @brief 定数バッファの状態をリセットします。
    void ResetConstantBufferStatus() NN_NOEXCEPT;

    //! @brief オフセットから定数バッファのデータを取得します。
    void* GetUniformBlock(ptrdiff_t offset, size_t size) NN_NOEXCEPT;

    template<typename T>
    inline T* GetUniformBlock(ptrdiff_t offset) NN_NOEXCEPT
    {
        return static_cast<T*>(GetUniformBlock(offset, sizeof(T)));
    }

    //! @brief UniformBlock のためのメモリを確保します。
    void* CreateUniformBlock(size_t size) NN_NOEXCEPT;

    //! @brief UniformBlock のためのメモリを確保します。
    template <typename T>
    inline T* CreateUniformBlock() NN_NOEXCEPT
    {
        return static_cast<T*>(CreateUniformBlock(sizeof(T)));
    }

    void UpdateConstantBufferPointer(ConstantBufferType type, void* pointer) NN_NOEXCEPT;

    template<typename T>
    T* UpdateConstantBuffer( ConstantBufferType type, T* pPodData ) NN_NOEXCEPT
    {
        NN_ASSERT(m_ConstantUpdateCallCount[type] < m_ConstantUpdateCallCountMax[type],
                      "[type %d] UpdateCall(%d) >= UpdateCallCountMax(%d)",
                      m_ConstantUpdateCallCount[type], m_ConstantUpdateCallCountMax[type]);

        T* pData = CreateUniformBlock<T>();
        if (pData != NULL)
        {
            std::memcpy(pData, pPodData, sizeof(T));
            UpdateConstantBufferPointer(type, pData);
            m_ConstantUpdateCallCount[type]++;
            return pData;
        }
        return NULL;
    }

    void SetConstantBuffer( nn::gfx::CommandBuffer* pCommandBuffer,
                            int slot,
                            nn::gfx::ShaderStage shaderStage,
                            void* pointer,
                            size_t memorySize ) NN_NOEXCEPT;

    //! @brief モデル構築
    void MakeModelConstantBuffer( Model* pOutModel,
                                  const nn::util::Vector3fType& scale,
                                  const nn::util::Vector3fType& trans,
                                  const nn::util::Float4& color ) NN_NOEXCEPT;
    //! @brief 描画共通処理
    void SetupShader( nn::gfx::CommandBuffer* pCommandBuffer,
                      Shader*                 pDefaultShader,
                      const nn::gfx::Shader*  pUserPixelShader ) NN_NOEXCEPT;

private:

    GraphicsResource*       m_pGraphicsResource;
    nn::gfx::MemoryPool*    m_pMemoryPool;
    Param                   m_Param;
    View*                   m_pView;
    Model*                  m_pModel;
    nn::util::Float4        m_Color;
    nn::gfx::Shader*        m_pUserPixelShader;
    size_t                  m_UserConstatnElementSize;
    MeshSet*                m_pMeshSet;
    GpuBuffer               m_GpuBuffer;
    int                     m_MultiBufferQuantity;
    int                     m_DrawCallCount;
    int                     m_DrawCallCountMax;
    void*                   m_pConstantMemoryPointer[ConstantBufferType_CountMax];
    int                     m_ConstantUpdateCallCount[ConstantBufferType_CountMax];
    int                     m_ConstantUpdateCallCountMax[ConstantBufferType_CountMax];
    int                     m_ScreenWidth;
    int                     m_ScreenHeight;
};


//! @brief レンダラ初期化情報です。
struct RendererInfo
{
public:
    enum Constants
    {
        Constants_MultiBufferingQuantityMax     = 1,    //!< 1スレッド、1インスタンスです。
        Constants_DrawCallCountMax              = 1024, //!< 1フレームのドローコール最大数です。
        Constants_ViewFunctionCallCountMax      = 1024, //!< 1フレームのビュー設定コール最大数です。
        Constants_UserFunctionCallCountMax      = 64,   //!< 1フレームのユーザ設定コール最大数です。
        Constants_UserConstantBufferDataSizeMax = 64,   //!< ユーザ定数バッファの上限サイズ(Bytes)です。
    };

    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    RendererInfo()  NN_NOEXCEPT {}

    //@}

    //---------------------------------------------------------------------------
    //! @brief 初期値を設定します。
    //---------------------------------------------------------------------------
    void SetDefault() NN_NOEXCEPT
    {
        m_DrawCallCountMax          = Constants_DrawCallCountMax;
        m_ViewFunctionCallCountMax  = Constants_ViewFunctionCallCountMax;
        m_UserFunctionCallCountMax  = Constants_UserFunctionCallCountMax;
        m_MultiBufferQuantity       = Constants_MultiBufferingQuantityMax;
        m_AdditionalBufferSize      = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief 定数バッファのバッファリング数を設定します。
    //! @param[in]    quantity バッファリング数です。
    //---------------------------------------------------------------------------
    void SetMultiBufferQuantity( int quantity ) NN_NOEXCEPT
    {
        m_MultiBufferQuantity = quantity;
    }

    //---------------------------------------------------------------------------
    //! @brief  初期化時に使用されるメモリアロケータを設定します。
    //! @param[in]  pAllocateFunction   アロケータ関数へのポインタです。
    //! @param[in]  pAllocateFunctionUserData   アロケータ関数へ渡すユーザーデータポインタです。
    //---------------------------------------------------------------------------
    void SetAllocator(
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        void* pAllocateFunctionUserData) NN_NOEXCEPT
    {
        m_pAllocateFunction = pAllocateFunction;
        m_pAllocateFunctionUserData = pAllocateFunctionUserData;
    }

    //---------------------------------------------------------------------------
    //! @brief 実際に初期化されるコンスタントバッファサイズを計算します。
    //! @param[in]    pGfxDevice         gfxデバイスです。
    //! @return       実際に初期化されるコンスタントバッファサイズ
    //---------------------------------------------------------------------------
    size_t CalculateConstantBufferSize( nn::gfx::Device* pGfxDevice ) const NN_NOEXCEPT
    {
        nn::gfx::Buffer::InfoType constantBufferInfo;
        constantBufferInfo.SetDefault();
        constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment( pGfxDevice, constantBufferInfo );

        size_t uniformViewSize  =
            nn::util::align_up( sizeof( Renderer::View ), alignment );
        size_t uniformModelSize =
            nn::util::align_up( sizeof( Renderer::Model ), alignment );
        size_t uniformUserSize  =
            nn::util::align_up( sizeof( Constants_UserConstantBufferDataSizeMax), alignment );

        size_t uniformModelTotalSize = uniformModelSize * m_DrawCallCountMax;
        size_t uniformViewTotalSize  = uniformViewSize * m_ViewFunctionCallCountMax;
        size_t uniformUserTotalSize  = uniformUserSize * m_UserFunctionCallCountMax;

        // １つのコンスタントバッファのサイズ
        size_t constantBufferMemorySize =
            uniformModelTotalSize + uniformViewTotalSize + uniformUserTotalSize;

        return constantBufferMemorySize;
    }

    nn::AlignedAllocateFunctionWithUserData GetAllocateFunction() const NN_NOEXCEPT
    {
        return m_pAllocateFunction;
    }

    void* GetAllocateFunctionUserData() const NN_NOEXCEPT
    {
        return m_pAllocateFunctionUserData;
    }

private:
    friend class Renderer;
    friend class GraphicsResource;

    int                                     m_DrawCallCountMax;
    int                                     m_ViewFunctionCallCountMax;
    int                                     m_UserFunctionCallCountMax;
    int                                     m_MultiBufferQuantity;
    nn::AlignedAllocateFunctionWithUserData m_pAllocateFunction;
    void*                                   m_pAllocateFunctionUserData;
    size_t                                  m_AdditionalBufferSize;
};

//---------------------------------------------------------------------------
//! @brief PrimitiveRendererのインスタンスを生成します。
//! @param[in]    pGfxDevice   gfxデバイスです。
//! @param[in]    info         レンダラ初期化に使うパラメータ情報です。
//! @return                    PrimitiveRendererのインスタンスを返します。
//---------------------------------------------------------------------------
Renderer* CreateRenderer( nn::gfx::Device*                        pGfxDevice,
                          RendererInfo&                           info ) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief PrimitiveRendererを破棄します。
//! @param[in]    pRenderer               破棄するPrimitiveRendererです。
//! @param[in]    pGfxDevice              初期化時に利用したgfxデバイスです。
//! @param[in]    pFreeFunction           CreateRenderer で確保した内部データ構造の
//!                                       メモリを開放するための関数へのポインタです。
//! @param[in]    pFreeFunctionUserData   pFreeFunction へ渡すユーザーデータへのポインタです。
//---------------------------------------------------------------------------
void DestroyRenderer( Renderer* pRenderer,
                      nn::gfx::Device* pGfxDevice,
                      nn::FreeFunctionWithUserData pFreeFunction,
                      void* pFreeFunctionUserData ) NN_NOEXCEPT;
} // namespace PrimitiveRenderer
} // namespace gfxLog
} // namespace nns
