﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>
#include "gfxLogPrimitiveRenderer.h"

namespace nns
{
namespace gfxLog
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 定数です。
//------------------------------------------------------------------------------

namespace
{

//! @brief 初期値です。
const nn::util::Float4 Float4One  = NN_UTIL_FLOAT_4_INITIALIZER( 1.f, 1.f, 1.f, 1.f );
const nn::util::Float4 Float4Zero = NN_UTIL_FLOAT_4_INITIALIZER( 0.f, 0.f, 0.f, 0.f );

const nn::util::Float2 Float2One  = NN_UTIL_FLOAT_2_INITIALIZER( 1.f, 1.f );
const nn::util::Float2 Float2Zero = NN_UTIL_FLOAT_2_INITIALIZER( 0.f, 0.f );

const nn::util::MatrixT4x4fType MatrixT4x4Unit =
    NN_UTIL_MATRIX_T4X4F_INITIALIZER(
    1.f, 0.f, 0.f, 0.f,
    0.f, 1.f, 0.f, 0.f,
    0.f, 0.f, 1.f, 0.f,
    0.f, 0.f, 0.f, 1.f );

const ShaderVariation ShaderVariationTable[] =
{
    ShaderVariation_VertexColor, // Quad
    ShaderVariation_SolidColor,  // QuadWired
    ShaderVariation_SolidColor,  // Cube
    ShaderVariation_SolidColor,  // CubeWired
    ShaderVariation_SolidColor,  // Sphere
    ShaderVariation_SolidColor,  // SphereWired
    ShaderVariation_SolidColor,  // SphereCoarse
    ShaderVariation_SolidColor,  // SphereWiredCoarse
    ShaderVariation_SolidColor,  // Line
    ShaderVariation_SolidColor,  // Point
    ShaderVariation_VertexColor, // Triangle
    ShaderVariation_SolidColor,  // TriangleWired
    ShaderVariation_SolidColor,  // Circle
    ShaderVariation_SolidColor,  // CircleWired
    ShaderVariation_SolidColor,  // CircleCoarse
    ShaderVariation_SolidColor,  // CircleWiredCoarse
    ShaderVariation_Texture,     // ScreenQuad
    ShaderVariation_Texture,     // ScreenYFlipQuad
    ShaderVariation_SolidColor,  // UpperHalfSphere
    ShaderVariation_VertexColor, // UpperHalfSphereWired
    ShaderVariation_SolidColor,  // LowerHalfSphere
    ShaderVariation_VertexColor, // LowerHalfSphereWired
    ShaderVariation_SolidColor,  // Pipe
    ShaderVariation_VertexColor, // PipeWired
    ShaderVariation_SolidColor,  // Cone
    ShaderVariation_VertexColor, // ConeWired
    ShaderVariation_Texture,     // User
};

const ShaderVariation SamplerShaderVariationTable[] =
{
    ShaderVariation_Texture,
    ShaderVariation_TextureArray,
    ShaderVariation_Cubemap
};

const nn::gfx::PrimitiveTopology TopologyTable[] =
{
    nn::gfx::PrimitiveTopology_TriangleList,  // Quad
    nn::gfx::PrimitiveTopology_LineList,      // QuadWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Cube
    nn::gfx::PrimitiveTopology_LineList,      // CubeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Sphere
    nn::gfx::PrimitiveTopology_LineList,      // SphereWired
    nn::gfx::PrimitiveTopology_TriangleList,  // SphereCoarse
    nn::gfx::PrimitiveTopology_LineList,      // SphereWiredCoarse
    nn::gfx::PrimitiveTopology_LineList,      // Line
    nn::gfx::PrimitiveTopology_PointList,     // Point
    nn::gfx::PrimitiveTopology_TriangleList,  // Triangle
    nn::gfx::PrimitiveTopology_LineStrip,     // TriangleWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Circle
    nn::gfx::PrimitiveTopology_LineStrip,     // CircleWired
    nn::gfx::PrimitiveTopology_TriangleList,  // CircleCoarse
    nn::gfx::PrimitiveTopology_LineStrip,     // CircleWiredCoarse
    nn::gfx::PrimitiveTopology_TriangleList,  // ScreenQuad
    nn::gfx::PrimitiveTopology_TriangleList,  // ScreenYFlipQuad
    nn::gfx::PrimitiveTopology_TriangleList,  // UpperHalfSphere
    nn::gfx::PrimitiveTopology_LineList,      // UpperHalfSphereWired
    nn::gfx::PrimitiveTopology_TriangleList,  // LowerHalfSphere
    nn::gfx::PrimitiveTopology_LineList,      // LowerHalfSphereWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Pipe
    nn::gfx::PrimitiveTopology_LineList,      // PipeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // Cone
    nn::gfx::PrimitiveTopology_LineList,      // ConeWired
    nn::gfx::PrimitiveTopology_TriangleList,  // User
};

// グラフィックスリソース
class GraphicResourceManager
{
    NN_DISALLOW_COPY(GraphicResourceManager);
    NN_DISALLOW_MOVE(GraphicResourceManager);
public:
    static GraphicResourceManager& GetInstance() NN_NOEXCEPT;

    void SetMemoryPoolBuffer(void* pMemoryPoolBuffer) NN_NOEXCEPT
    {
        m_pMemoryPoolBuffer = pMemoryPoolBuffer;
    }

    void* GetMemoryPoolBuffer() const NN_NOEXCEPT
    {
        return m_pMemoryPoolBuffer;
    }

    void SetMemoryBuffer(void* pMemoryBuffer) NN_NOEXCEPT
    {
        m_pMemoryBuffer = pMemoryBuffer;
    }

    void* GetMemoryBuffer() const NN_NOEXCEPT
    {
        return m_pMemoryBuffer;
    }

    void SetIsPrimitiveRendererInitialized(bool IsPrimitiveRendererInitialized) NN_NOEXCEPT
    {
        m_IsPrimitiveRendererInitialized = IsPrimitiveRendererInitialized;
    }

    bool IsPrimitiveRendererInitialized() const NN_NOEXCEPT
    {
        return m_IsPrimitiveRendererInitialized;
    }

private:
    GraphicResourceManager() NN_NOEXCEPT
        :   m_pMemoryPoolBuffer(NULL),
            m_pMemoryBuffer(NULL),
            m_IsPrimitiveRendererInitialized(false)
    {
    }

private:
    void* m_pMemoryPoolBuffer;
    void* m_pMemoryBuffer;
    bool m_IsPrimitiveRendererInitialized;
};

GraphicResourceManager& GraphicResourceManager::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(GraphicResourceManager, s_GraphicResourceManager);
    return s_GraphicResourceManager;
}
}; // namespace

//  プリミティブレンダラを初期化します。
Renderer* CreateRenderer(
    nn::gfx::Device*                        pGfxDevice,
    RendererInfo&                           info ) NN_NOEXCEPT
{
    if (GraphicResourceManager::GetInstance().IsPrimitiveRendererInitialized())
    {
        return NULL;
    }

    GraphicResourceManager::GetInstance().SetIsPrimitiveRendererInitialized(false);

    size_t memoryPoolSize = 0;
    ptrdiff_t memoryPoolOffset = 0;
    size_t memoryBufferSize = 0;

    // メモリプールを初期化
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    nn::gfx::MemoryPool* pMemoryPool = new nn::gfx::MemoryPool();
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached );

    // Renderer1つ分のMemoryPoolのサイズを計算
    memoryPoolSize =  Renderer::GetRequiredMemoryPoolSize(
        pGfxDevice,
        info
    );
    memoryPoolSize += Renderer::GetMemoryPoolAlignment(
        pGfxDevice,
        info
    );
    // GraphicsResourceに必要なMemoryPoolのサイズを計算して追加
    memoryPoolSize += GraphicsResource::GetRequiredMemoryPoolSize(
        pGfxDevice
    );
    memoryPoolSize += GraphicsResource::GetMemoryPoolAlignment(
        pGfxDevice
    );
    memoryPoolSize = nn::util::align_up( memoryPoolSize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(
            pGfxDevice,
            memoryPoolInfo
        )
    );
    GraphicResourceManager::GetInstance().SetMemoryPoolBuffer(
        info.GetAllocateFunction()(
            memoryPoolSize,
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(
                pGfxDevice,
                memoryPoolInfo
            ),
            info.GetAllocateFunctionUserData() )
        );
    memoryPoolInfo.SetPoolMemory(GraphicResourceManager::GetInstance().GetMemoryPoolBuffer(), memoryPoolSize );
    pMemoryPool->Initialize( pGfxDevice, memoryPoolInfo );

    // グラフィックスリソースに必要なメモ領域を確保
    memoryBufferSize = GraphicsResource::GetRequiredMemorySize(
        pGfxDevice
    );
    GraphicResourceManager::GetInstance().SetMemoryBuffer(
        info.GetAllocateFunction()(
            memoryBufferSize,
            GraphicsResource::GetRequiredMemoryAlignment(
                pGfxDevice
            ),
            info.GetAllocateFunctionUserData()
        )
    );

    GraphicsResource* pGraphicsResource = new GraphicsResource();
    // グラフィックスリソースを初期化
    if (!pGraphicsResource->Initialize(
        pGfxDevice,
        GraphicResourceManager::GetInstance().GetMemoryBuffer(),
        memoryBufferSize,
        pMemoryPool,
        memoryPoolOffset,
        memoryPoolSize ))
    {
        return NULL;
    }

    // メモリプールのオフセットを、GraphicsResource分ずらす
    memoryPoolOffset += GraphicsResource::GetRequiredMemoryPoolSize(
        pGfxDevice
    );
    memoryPoolOffset = nn::util::align_up( memoryPoolOffset,
        Renderer::GetMemoryPoolAlignment(
            pGfxDevice,
            info
        )
    );

    // Rendererを1つ生成
    Renderer* pRenderer = NULL;
    const size_t size = sizeof(Renderer);
    void* ptr = info.GetAllocateFunction()(size, 16, info.GetAllocateFunctionUserData());
    pRenderer = new (ptr) Renderer;
    NN_ASSERT_NOT_NULL(pRenderer);
    if (pRenderer == NULL)
    {
        return NULL;
    }
    pRenderer->SetGraphicsResource(pGraphicsResource);
    pRenderer->SetMemoryPool(pMemoryPool);

    // Rendererを初期化する
    if (!pRenderer->Initialize( pGfxDevice,
                                info,
                                pGraphicsResource,
                                pMemoryPool,
                                memoryPoolOffset,
                                memoryPoolSize ))
    {
        return NULL;
    }

    GraphicResourceManager::GetInstance().SetIsPrimitiveRendererInitialized(true);

    return pRenderer;
}


//  プリミティブレンダラを破棄します。
void DestroyRenderer(
    Renderer* pRenderer,
    nn::gfx::Device* pGfxDevice,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pFreeFunctionUserData ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( pRenderer );

    if (!GraphicResourceManager::GetInstance().IsPrimitiveRendererInitialized())
    {
        return;
    }

    if (pRenderer)
    {
        pRenderer->Finalize( pGfxDevice );
        pFreeFunction( pRenderer, pFreeFunctionUserData );
    }

    // グラフィックスリソースを初期化
    GraphicsResource* pGraphicsResource = pRenderer->GetGraphicsResource();
    pGraphicsResource->Finalize( pGfxDevice );
    pFreeFunction(GraphicResourceManager::GetInstance().GetMemoryBuffer(), pFreeFunctionUserData );
    GraphicResourceManager::GetInstance().SetMemoryBuffer(NULL);

    nn::gfx::MemoryPool* pMemoryPool = pRenderer->GetMemoryPool();
    pMemoryPool->Finalize( pGfxDevice );
    pFreeFunction(GraphicResourceManager::GetInstance().GetMemoryPoolBuffer(), pFreeFunctionUserData );

    GraphicResourceManager::GetInstance().SetIsPrimitiveRendererInitialized(true);
}

void Renderer::SetDefault( Model* pModel ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pModel);
    pModel->rate = 1.f;
    pModel->u_color  = Float4One;
    pModel->u_layer  = Float4Zero;
    pModel->u_uv_src = Float2Zero;
    pModel->u_uv_size = Float2One;
    nn::util::MatrixStore( &pModel->u_userMatrix, MatrixT4x4Unit );
}

size_t Renderer::GetRequiredMemoryPoolSize(
    nn::gfx::Device* pGfxDevice,
    const RendererInfo& info ) NN_NOEXCEPT
{
    // １つのコンスタントバッファのサイズ
    size_t constantBufferSize = info.CalculateConstantBufferSize( pGfxDevice );

    GpuBuffer::InitializeArg gpuBufferArg;
    gpuBufferArg.SetBufferCount(info.m_MultiBufferQuantity);
    gpuBufferArg.SetBufferSize(constantBufferSize + info.m_AdditionalBufferSize);
    gpuBufferArg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
        nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    return GpuBuffer::CalculateGpuBufferPoolSize( pGfxDevice, gpuBufferArg );

}

size_t Renderer::GetMemoryPoolAlignment(
    nn::gfx::Device* pGfxDevice,
    const RendererInfo& info ) NN_NOEXCEPT
{
    // １つのコンスタントバッファのサイズ
    size_t constantBufferSize = info.CalculateConstantBufferSize( pGfxDevice );

    GpuBuffer::InitializeArg gpuBufferArg;
    gpuBufferArg.SetBufferCount(info.m_MultiBufferQuantity);
    gpuBufferArg.SetBufferSize(constantBufferSize);
    gpuBufferArg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
        nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

    return GpuBuffer::GetGpuBufferAlignement( pGfxDevice, gpuBufferArg );
}

Renderer::Renderer() NN_NOEXCEPT
    : m_pGraphicsResource(NULL)
    , m_pMemoryPool(NULL)
    , m_Param()
    , m_pView(NULL)
    , m_pModel(NULL)
    , m_Color()
    , m_pUserPixelShader()
    , m_UserConstatnElementSize(0)
    , m_pMeshSet(NULL)
    , m_GpuBuffer()
    , m_MultiBufferQuantity(0)
    , m_DrawCallCount(0)
    , m_DrawCallCountMax(0)
    , m_ScreenWidth(1280)
    , m_ScreenHeight(720)
{
    for (int i = 0; i < ConstantBufferType_CountMax; i++)
    {
        m_pConstantMemoryPointer[i] = NULL;
    }
    for (int i = 0; i < ConstantBufferType_CountMax; i++)
    {
        m_ConstantUpdateCallCount[i] = 0;
    }
    for (int i = 0; i < ConstantBufferType_CountMax; i++)
    {
        m_ConstantUpdateCallCountMax[i] = 0;
    }

    m_Param.SetDefault();
}

bool Renderer::Initialize( nn::gfx::Device* pGfxDevice,
                           const RendererInfo& info,
                           GraphicsResource* pGraphicsResource,
                           nn::gfx::MemoryPool* pMemoryPool,
                           ptrdiff_t memoryPoolOffset,
                           size_t memoryPoolSize ) NN_NOEXCEPT
{
    m_pGraphicsResource = pGraphicsResource;

    // デフォルトメッシュ
    m_pMeshSet = m_pGraphicsResource->GetDefaultMeshSet();

    // 内部の定数バッファの状態をリセットします。
    ResetConstantBufferStatus();

    // デフォルトの定数バッファ
    m_MultiBufferQuantity = info.m_MultiBufferQuantity;
    m_DrawCallCountMax = info.m_DrawCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_Model] = info.m_DrawCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_View]  = info.m_ViewFunctionCallCountMax;
    m_ConstantUpdateCallCountMax[ConstantBufferType_User]  = info.m_UserFunctionCallCountMax;

    // 定数バッファを初期化します。
    // 描画コールごとに内容を書き換えるので、最初に最大描画コール分だけ確保します。
    // また、このサイズに追加バッファのサイズも足しています。
    {
        GpuBuffer::InitializeArg arg;
        arg.SetBufferCount(m_MultiBufferQuantity);
        arg.SetBufferSize(
            info.CalculateConstantBufferSize( pGfxDevice ) + info.m_AdditionalBufferSize
        );
        arg.SetGpuAccessFlag(nn::gfx::GpuAccess::GpuAccess_ConstantBuffer |
            nn::gfx::GpuAccess::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_IndexBuffer);

        ptrdiff_t alignedMemoryPoolOffset = nn::util::align_up(
            memoryPoolOffset,
            GetMemoryPoolAlignment( pGfxDevice, info )
        );

        // メモリプールのサイズが足りない場合は、何もせずにreturnする。
        if ( memoryPoolSize < GetRequiredMemoryPoolSize( pGfxDevice, info ) )
        {
            return false;
        }

        m_GpuBuffer.Initialize(pGfxDevice, arg, pMemoryPool, alignedMemoryPoolOffset);
    }

    return true;
}

void Renderer::Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{

    m_DrawCallCount = 0;
    m_pMeshSet = NULL;
    ResetConstantBufferStatus();

    m_GpuBuffer.Finalize( pGfxDevice );
}

void Renderer::SetColor( const nn::util::Color4u8& color ) NN_NOEXCEPT
{
    m_Color.v[ 0 ] = static_cast<float>( color.GetR() ) / 255.f;
    m_Color.v[ 1 ] = static_cast<float>( color.GetG() ) / 255.f;
    m_Color.v[ 2 ] = static_cast<float>( color.GetB() ) / 255.f;
    m_Color.v[ 3 ] = static_cast<float>( color.GetA() ) / 255.f;
}

void Renderer::SetViewConstantBuffer( View* pPodData ) NN_NOEXCEPT
{
    ConstantBufferType type = ConstantBufferType_View;
    m_pView = UpdateConstantBuffer<View>( type, pPodData );
}

void Renderer::SetModelConstantBuffer( Model* pPodData ) NN_NOEXCEPT
{
    ConstantBufferType type = ConstantBufferType_Model;
    m_pModel = UpdateConstantBuffer<Model>( type, pPodData );
}

void Renderer::ResetConstantBufferStatus() NN_NOEXCEPT
{
    for( int idxBuffer = 0; idxBuffer < ConstantBufferType_CountMax; ++idxBuffer )
    {
        m_pConstantMemoryPointer[ idxBuffer ] = NULL;
        m_ConstantUpdateCallCount[ idxBuffer ] = 0;
    }
    m_pView = NULL;
    m_pModel = NULL;
}

void* Renderer::CreateUniformBlock(size_t size) NN_NOEXCEPT
{
    if ( m_GpuBuffer.GetAllocatedSize() + size > m_GpuBuffer.GetBufferSize() )
    {
        return NULL;
    }

    void* pointer = m_GpuBuffer.Allocate(size);
    return pointer;
}

void Renderer::UpdateConstantBufferPointer(ConstantBufferType type, void* pointer) NN_NOEXCEPT
{
    m_pConstantMemoryPointer[type] = pointer;
}

void Renderer::SetConstantBuffer( nn::gfx::CommandBuffer* pCommandBuffer,
                                  int slot,
                                  nn::gfx::ShaderStage shaderStage,
                                  void* pointer,
                                  size_t memorySize ) NN_NOEXCEPT
{
    NN_ASSERT(pointer != NULL);
    if( slot == ShaderSlotNone )
    {
        return;
    }

    nn::gfx::GpuAddress gpuAddress;
    m_GpuBuffer.GetGpuAddress( &gpuAddress, pointer );

    // データサイズは、実際のPODを設定します。
    pCommandBuffer->SetConstantBuffer( slot, shaderStage, gpuAddress, memorySize );
}

void Renderer::MakeModelConstantBuffer(
                    Model* pOutModel,
                    const nn::util::Vector3fType& scale,
                    const nn::util::Vector3fType& trans,
                    const nn::util::Float4& color ) NN_NOEXCEPT
{
    nn::util::Matrix4x3fType modelMatrix = {};
    nn::util::MatrixSetScale( &modelMatrix, scale );
    nn::util::MatrixSetTranslate( &modelMatrix, trans );

    nn::util::Matrix4x4fType modelMtx4x4f = {};
    nn::util::MatrixConvert(&modelMtx4x4f, modelMatrix);
    nn::util::MatrixStore(&pOutModel->u_userMatrix, modelMtx4x4f);

    pOutModel->u_color = color;
}

void Renderer::SetupShader( nn::gfx::CommandBuffer* pCommandBuffer,
                            Shader*                 pDefaultShader,
                            const nn::gfx::Shader*  pUserPixelShader ) NN_NOEXCEPT
{
    // 頂点シェーダ
    pCommandBuffer->SetShader( pDefaultShader->GetVertexShader(), nn::gfx::ShaderStageBit_Vertex );

    // フラグメントシェーダ
    if( pUserPixelShader != NULL  )
    {
        pCommandBuffer->SetShader( pUserPixelShader, nn::gfx::ShaderStageBit_Pixel );
    }
    else
    {
        pCommandBuffer->SetShader(
            pDefaultShader->GetPixelShader(),
            nn::gfx::ShaderStageBit_Pixel
        );
    }

    // 頂点ステート
    pCommandBuffer->SetVertexState( pDefaultShader->GetVertexState() );
}

void Renderer::Update(int bufferIndex) NN_NOEXCEPT
{
    // 描画コール数を初期化します。
    m_DrawCallCount = 0;

    // 定数バッファの状態をリセットします。
    ResetConstantBufferStatus();

    m_GpuBuffer.Unmap();
    m_GpuBuffer.Map(bufferIndex);
}

void Renderer::UpdateGpuView() NN_NOEXCEPT
{
    View view = {};
    nn::util::Matrix4x4fType mvp;
    nn::util::MatrixMultiply( &mvp, m_Param.modelMatrix, m_Param.viewMatrix);
    nn::util::MatrixMultiply( &mvp, mvp,  m_Param.projMatrix );
    nn::util::MatrixStore( &view.u_mvp, mvp );

    SetViewConstantBuffer( &view );
}

void Renderer::DrawPrimitiveInternal( nn::gfx::PrimitiveTopology topology,
                                      ShaderVariation shaderVariation,
                                      nn::gfx::CommandBuffer* pCommandBuffer,
                                      const PrimitiveMesh* pMesh,
                                      Model* pModel,
                                      const int indexBase,
                                      const int numUserIndices,
                                      const int userBaseVertex,
                                      const SamplerType samplerType,
                                      const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                      const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pMesh);

    // pMeshがnullだったら、何もせずにreturnする。
    if ( pMesh == NULL )
    {
        return;
    }

    // インデックスバッファ
    NN_ASSERT( pMesh->IsValidIndexBuffer() );

    // インデックスバッファが不正だったら、何もせずにreturnする。
    if (!pMesh->IsValidIndexBuffer())
    {
        return;
    }

    // ドローコール数をチェックします。
    if( m_DrawCallCount >= m_DrawCallCountMax )
    {
        NN_LOG( "Draw Call(%d) >= limit(%d), Please Call Update()",
            m_DrawCallCount, m_DrawCallCountMax );
        return;
    }
    UpdateGpuView();

    // 定数バッファ
    SetModelConstantBuffer( pModel );

    // シェーダバリエーションからデフォルトシェーダを取得します。
    Shader* pDefaultShader = m_pGraphicsResource->GetDefaultShader( shaderVariation );

    // シェーダ
    SetupShader( pCommandBuffer, pDefaultShader, m_pUserPixelShader );

    // 頂点バッファ
    nn::gfx::GpuAddress vertexGpuAddress;
    for( int idx = 0; idx < VertexAttribute_CountMax; ++idx )
    {
        VertexAttribute attrib = static_cast<VertexAttribute>(idx);
        if( !pMesh->IsValidVertexBuffer( attrib ) )
        {
            continue;
        }
        const size_t stride = pMesh->GetVertexStride( attrib );
        const size_t size = pMesh->GetVertexMemorySize( attrib );
        pMesh->GetVertexBufferGpuAddress( attrib, &vertexGpuAddress );
        pCommandBuffer->SetVertexBuffer( idx, vertexGpuAddress, stride, size );
    }

    // 定数バッファのロード
    int slotView = pDefaultShader->GetSlotView();
    int slotModelVs = pDefaultShader->GetSlotModel(0);
    int slotModelPs = pDefaultShader->GetSlotModel(1);

    // SetViewConstantBuffer() / UpdateGpuView()で、更新済みとします。
    SetConstantBuffer( pCommandBuffer,
        slotView,
        nn::gfx::ShaderStage_Vertex,
        m_pConstantMemoryPointer[ConstantBufferType_View],
        sizeof(View));

    // SetModelConstantBuffer() で、更新済みとします。
    SetConstantBuffer( pCommandBuffer,
        slotModelVs,
        nn::gfx::ShaderStage_Vertex,
        m_pConstantMemoryPointer[ConstantBufferType_Model],
        sizeof(Model));

    SetConstantBuffer( pCommandBuffer,
        slotModelPs,
        nn::gfx::ShaderStage_Pixel,
        m_pConstantMemoryPointer[ConstantBufferType_Model],
        sizeof(Model));

    // サンプラ
    int idxSampler = ShaderSlotNone;
    if( SamplerType_CountMax != samplerType &&
        pTextureViewSlot != NULL && pSamplerSlot != NULL )
    {
        idxSampler = pDefaultShader->GetSlotSamplerPs(samplerType);
    }

    // テクスチャとサンプラ
    if( ShaderSlotNone != idxSampler )
    {
        pCommandBuffer->SetTextureAndSampler( idxSampler,
            nn::gfx::ShaderStage_Pixel,
            *pTextureViewSlot, *pSamplerSlot );
    }

    // ドローコール
    const int numIndices = (0 < numUserIndices) ? numUserIndices : pMesh->GetIndexCount();
    NN_ASSERT(0 < numIndices);

    nn::gfx::GpuAddress indexGpuAddress;
    pMesh->GetIndexBufferGpuAddress( &indexGpuAddress );
    ptrdiff_t indexOffset = indexBase * sizeof( uint32_t );
    indexGpuAddress.Offset( indexOffset );
    pCommandBuffer->DrawIndexed( topology,
        nn::gfx::IndexFormat_Uint32, indexGpuAddress, numIndices, userBaseVertex );

    // 描画コール数をカウントします。
    m_DrawCallCount++;
}

void Renderer::DrawPrimitiveInternal( ShapeType shapeType,
                                      nn::gfx::CommandBuffer* pCommandBuffer,
                                      Model* pModel,
                                      const SamplerType samplerType,
                                      const nn::gfx::DescriptorSlot* pTextureViewSlot,
                                      const nn::gfx::DescriptorSlot* pSamplerSlot ) NN_NOEXCEPT
{
    const PrimitiveMesh* pMesh = m_pMeshSet->Get( shapeType );
    NN_ASSERT_NOT_NULL(pMesh);

    // pMeshがnullだったら、何もせずにreturnする。
    if ( pMesh == NULL )
    {
        return;
    }

    // インデックスバッファ
    NN_ASSERT( pMesh->IsValidIndexBuffer() );

    // サンプラが指定されている場合は、シェーダバリエーションを変更します。
    ShaderVariation shaderVariation = ShaderVariationTable[ shapeType ];
    if( SamplerType_CountMax != samplerType &&
        pTextureViewSlot != NULL && pSamplerSlot != NULL )
    {
        shaderVariation = SamplerShaderVariationTable[ samplerType ];
    }

    nn::gfx::PrimitiveTopology topology = TopologyTable[ shapeType ];

    DrawPrimitiveInternal( topology,
                           shaderVariation,
                           pCommandBuffer,
                           pMesh,
                           pModel,
                           0,
                           0,
                           0,
                           samplerType,
                           pTextureViewSlot,
                           pSamplerSlot );
}

void Renderer::Draw2DRect( nn::gfx::CommandBuffer* pCommandBuffer,
                           const float x,
                           const float y,
                           const float width,
                           const float height ) NN_NOEXCEPT
{
    // 定数バッファ
    Model model = {};
    SetDefault( &model );
    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER(
            2.f * width / m_ScreenWidth,
            2.f * height / m_ScreenHeight, 2.f
        ); // [-1,1]x[-1,1]
    const nn::util::Vector3fType center
        = NN_UTIL_VECTOR_3F_INITIALIZER(
            -1.f + 2.f * x / m_ScreenWidth,
            1.f - 2.f * (y + height) / m_ScreenHeight, 0.f
        );
    MakeModelConstantBuffer( &model, size, center, m_Color );

    DrawPrimitiveInternal( ShapeType_Quad,
                           pCommandBuffer,
                           &model,
                           SamplerType_CountMax,
                           NULL,
                           NULL );
}

} // namespace PrimitiveRenderer
} // namespace gfxLog
} // namespace nns
