﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Color.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include "gfxLogPrimitiveRenderer.h"

namespace {

/**
* @brief エスケープシーケンスを解析した結果を表す enum 値です。
*
* @details エスケープシーケンスを解析した結果を表す enum 値です。無効値は ColorIndex_Invalid です。
*/
enum ColorIndex
{
    ColorIndex_Invalid = 0,

    ColorIndex_TextBlack,
    ColorIndex_TextRed,
    ColorIndex_TextGreen,
    ColorIndex_TextYellow,
    ColorIndex_TextBlue,
    ColorIndex_TextMagenta,
    ColorIndex_TextCyan,
    ColorIndex_TextWhite,
    ColorIndex_TextDefault,

    ColorIndex_BackBlack,
    ColorIndex_BackRed,
    ColorIndex_BackGreen,
    ColorIndex_BackYellow,
    ColorIndex_BackBlue,
    ColorIndex_BackMagenta,
    ColorIndex_BackCyan,
    ColorIndex_BackWhite,
    ColorIndex_BackDefault,
};
};


namespace nns {
namespace gfxLog {

class GraphicsSystem;

/**
* @brief ログ出力処理を行うクラスです。
*
* @details ログ出力処理を行うクラスです。このクラスはシングルトンです。
*/
class LogOut
{
    NN_DISALLOW_COPY(LogOut);
    NN_DISALLOW_MOVE(LogOut);
public:
    static void Log(const std::string& String) NN_NOEXCEPT;
    static void SetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;
    static void SetBackgroudColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;
    static void SetDrawRect(
        const float Left,
        const float Top,
        const float Width,
        const float Height
    ) NN_NOEXCEPT;
    static void WriteCommand() NN_NOEXCEPT;

    static void SetThreadCoreNumber(int idealCore) NN_NOEXCEPT;
    static void SetConfiguration(
        nn::gfx::Device* pDevice,
        nn::gfx::CommandBuffer* pCommandBuffer,
        nn::gfx::util::DebugFontTextWriter* pDebugFontWriter) NN_NOEXCEPT;
private:
    /**
    * @brief 改行コードを表すenum
    */
    enum LineEnding
    {
        LineEnding_NO,              //!< 改行なし
        LineEnding_CarriageReturn,  //!< CR(\r) 同じ行で先頭に移動する
        LineEnding_LineFeed,        //!< LF(\n) 次の行で先頭に移動する
        LineEnding_AutoLineFeed,    //!< 自動整形の改行
    };

    /**
    * @brief 文字列データと色の設定と改行の有無を管理するクラスです。
    *
    * @details 文字列データと色の設定と改行の有無を管理するクラスです。
    */
    class StringData
    {
    public:
        StringData() NN_NOEXCEPT
            : m_String()
            , m_LineEnding(LineEnding_NO)
        {

        }
        void SetString(const std::string& string) NN_NOEXCEPT
        {
            m_String = string;
        }
        const std::string GetString() const NN_NOEXCEPT
        {
            return m_String;
        }
        void SetLineEnding(LineEnding lineEnding) NN_NOEXCEPT
        {
            m_LineEnding = lineEnding;
        }
        const LineEnding GetLineEnding() const NN_NOEXCEPT
        {
            return m_LineEnding;
        }
        void SetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
        {
            m_TextColor = Color;
        }
        const nn::util::Color4u8 GetTextColor() const NN_NOEXCEPT
        {
            return m_TextColor;
        }
        void SetBackColor(const nn::util::Color4u8& Color) NN_NOEXCEPT
        {
            m_BackColor = Color;
        }
        const nn::util::Color4u8 GetBackColor() const NN_NOEXCEPT
        {
            return m_BackColor;
        }
    private:
        std::string m_String; // 文字列
        LineEnding m_LineEnding; // 改行の有無
        nn::util::Color4u8 m_TextColor; // テキスト色
        nn::util::Color4u8 m_BackColor; // 背景色
    };

    /**
    * @brief WrapString の戻り値を返すために定義されたクラスです。
    *
    * @details WrapString の戻り値を返すために定義されたクラスです。
    *          指定した幅に収まる文字列と収まらなかった文字列を管理します。
    */
    class StringPair
    {
    public:
        StringPair() NN_NOEXCEPT
            : m_String()
            , m_Remain()
        {
        }
        void SetString(const std::string& string) NN_NOEXCEPT
        {
            m_String = string;
        }
        const std::string GetString() const NN_NOEXCEPT
        {
            return m_String;
        }
        void SetRemain(const std::string& string) NN_NOEXCEPT
        {
            m_Remain = string;
        }
        const std::string GetRemain() const NN_NOEXCEPT
        {
            return m_Remain;
        }
        void SetTotalFontWidth(float& totalFontWidth) NN_NOEXCEPT
        {
            m_TotalFontWidth = totalFontWidth;
        }
        const float GetTotalFontWidth() const NN_NOEXCEPT
        {
            return m_TotalFontWidth;
        }
    private:
        std::string m_String;
        std::string m_Remain;
        float m_TotalFontWidth;
    };

private:
    static const size_t ThreadStackSize = 128 * 1024;

private:
    static LogOut& GetInstance() NN_NOEXCEPT;
    LogOut() NN_NOEXCEPT;
    ~LogOut() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void AddToQueue(const std::string& String) NN_NOEXCEPT;
    void InternalDraw() NN_NOEXCEPT;
    std::vector<LogOut::StringData> Split(const LogOut::StringData& Data) const NN_NOEXCEPT;
    void LimitLines(std::vector<StringData>& stringData) NN_NOEXCEPT;
    StringPair WrapString(const std::string& String, const float width) NN_NOEXCEPT;
    void ProcessCR(
        std::vector<StringData>& destination,
        const std::vector<StringData>& data
    ) NN_NOEXCEPT;
    void Concat(std::vector<StringData>& stringData) NN_NOEXCEPT;
    void WrapStrings(std::vector<LogOut::StringData>& Data) NN_NOEXCEPT;
    void InternalSetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;
    void InternalSetBackgroudColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;
    void InternalSetDrawRect(
        const float Left,
        const float Top,
        const float Width,
        const float Height
    ) NN_NOEXCEPT;
    void InternalSetThreadCoreNumber(int idealCore) NN_NOEXCEPT;
    std::string ClipString(const std::string& String, const float Width) NN_NOEXCEPT;
    std::vector<LogOut::StringData> SplitEscapeSequences(const std::string& String) NN_NOEXCEPT;
    void SetTextAndBackColor(const ColorIndex ColorIndex) NN_NOEXCEPT;
    void ThreadFuncImpl() NN_NOEXCEPT;
    static void ThreadFunc(void* pArg) NN_NOEXCEPT;

    nn::gfx::CommandBuffer& GetCommandBuffer() NN_NOEXCEPT;
    nn::gfx::util::DebugFontTextWriter& GetDebugFont() NN_NOEXCEPT;
    PrimitiveRenderer::Renderer& GetPrimitiveRenderer() const NN_NOEXCEPT;

    void InitializePrimitiveRenderer() NN_NOEXCEPT;
    void FinalizePrimitiveRenderer() NN_NOEXCEPT;

    static void* AllocateFunction(size_t size,
        size_t alignment,
        void* pUserData) NN_NOEXCEPT;
    static void FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT;

private:
    bool m_IsCustomDraw;
    bool m_IsRunning;
    nn::gfx::Device* m_pDevice;
    nn::gfx::CommandBuffer* m_pCommandBuffer;
    PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* m_pDebugFontWriter;

    GraphicsSystem* m_pGraphicsSystem;
    std::vector<StringData> m_TempDisplayText;
    std::vector<StringData> m_DisplayText;

    nn::util::Color4u8 m_TextColorToSet;
    nn::util::Color4u8 m_BackColorToSet;

    nn::util::Color4u8 m_TextColor;
    nn::util::Color4u8 m_BackColor;

    nn::os::Mutex m_Mutex;
    char*  m_pThreadStack;
    nn::os::ThreadType m_Thread; // スレッド
    nn::os::EventType m_EventType;
    int m_IdealCore; // ログ描画スレッドが動作するコア番号

    size_t m_Current;
    std::vector<StringData> m_TempOneLine;

    float m_LeftToSet;
    float m_TopToSet;
    float m_WidthToSet;
    float m_HeightToSet;

    float m_Left;
    float m_Top;
    float m_Width;
    float m_Height;

    float m_PosX;
};


} // namespace gfxLog
} // namespace nns
