﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_PrimitiveRendererGfxRes.h>
#include <nns/gfx/gfx_PrimitiveRendererShaderRes.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//------------------------------------------------------------------------------
//! 実装です。
//------------------------------------------------------------------------------
uint32_t Shader::GetDefaultVertexFormat( ShaderVariation shaderVariation )
{
    uint32_t vertexFormat = 0;
    switch( shaderVariation )
    {
    case ShaderVariation_P:
        vertexFormat = VertexFormat_Pos;
        break;
    case ShaderVariation_PN:
        vertexFormat = VertexFormat_Pos | VertexFormat_Normal;
        break;
    case ShaderVariation_PUv:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv;
        break;
    case ShaderVariation_PNUv:
        vertexFormat = VertexFormat_Pos | VertexFormat_Normal| VertexFormat_Uv;
        break;
    case ShaderVariation_PC:
        vertexFormat = VertexFormat_Pos | VertexFormat_Color;
        break;
    case ShaderVariation_PUvC:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_TextureArray:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_Cubemap:
        vertexFormat = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color;
        break;
    case ShaderVariation_CountMax:
    default: NN_UNEXPECTED_DEFAULT;
    }
    return vertexFormat;
}

bool Shader::Create( Shader*                 pShader,
                     nn::gfx::ResShaderFile* pResShaderFile,
                     ShaderVariation         shaderVariation,
                     nn::gfx::Device*        pGfxDevice,
                     nn::util::BytePtr&      pMemory )
{
#if NN_GFX_IS_TARGET_GL
    nn::gfx::ShaderCodeType shaderCodeType = nn::gfx::ShaderCodeType_Source;
#elif NN_GFX_IS_TARGET_NVN
    nn::gfx::ShaderCodeType shaderCodeType = nn::gfx::ShaderCodeType_Binary;
#else
#error Unsupported Platform
#endif

    nn::gfx::ResShaderContainer* pContainer = pResShaderFile->GetShaderContainer();
    nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( shaderVariation );
    if( nullptr != pVariation )
    {
        nn::gfx::Shader* pGfxShader = pVariation->GetResShaderProgram( shaderCodeType )->GetShader();
        NN_SDK_ASSERT_NOT_NULL(pGfxShader);

        pShader->SetVertexShader( pGfxShader );
        pShader->SetPixelShader( pGfxShader );

        // リソースから頂点ステートを生成します。
        if ( CreateVertexState( pShader,
                                shaderVariation,
                                pGfxShader,
                                pGfxDevice,
                                pMemory ) == false )
        {
            return false;
        }

        pShader->UpdateSystemSlots();
        return true;
    }
    return false;
}

// 配列版頂点ステートを初期化する
int Shader::InitializeVertexStateInfo( nn::gfx::VertexState::InfoType*      info,
                                       nn::gfx::VertexAttributeStateInfo*   attribs,
                                       nn::gfx::VertexBufferStateInfo*      buffers,
                                       ShaderVariation                      shaderVariation,
                                       nn::gfx::Shader*                     pVertexShader )
{
    uint32_t vertexFormat = Shader::GetDefaultVertexFormat( shaderVariation );

    info->SetDefault();

    int attribCount = 0;
    int slot = -1;

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_Position" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Pos && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Position" );
        attrib.SetBufferIndex( attribCount );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attrib.SetOffset( 0 );
        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof( nn::util::Float3 ) );
        attribCount++;
    }

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_Color" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Color && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Color" );
        attrib.SetBufferIndex( attribCount );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );
        attrib.SetOffset( 0 );
        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof( nn::util::Float4 ) );
        attribCount++;
    }


    if (pVertexShader != nullptr)
    {
        slot = pVertexShader->GetInterfaceSlot(nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_TexCoord");
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Uv && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_TexCoord" );
        attrib.SetBufferIndex( attribCount );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attrib.SetOffset( 0 );
        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof( nn::util::Float2 ) );
        attribCount++;
    }

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_Normal" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Normal && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Normal" );
        attrib.SetBufferIndex( attribCount );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attrib.SetOffset( 0 );
        nn::gfx::VertexBufferStateInfo& buffer = buffers[ attribCount ];
        buffer.SetDefault();
        buffer.SetStride( sizeof( nn::util::Float3 ) );
        attribCount++;
    }

    if (0 == attribCount)
    {
        return 0;
    }

    info->SetVertexAttributeStateInfoArray(attribs, attribCount);
    info->SetVertexBufferStateInfoArray(buffers, attribCount);

    return attribCount;
}

int Shader::InitializeInterleavedVertexStateInfo( nn::gfx::VertexState::InfoType*      info,
                                                  nn::gfx::VertexAttributeStateInfo*   attribs,
                                                  nn::gfx::VertexBufferStateInfo*      buffer,
                                                  ShaderVariation                      shaderVariation,
                                                  nn::gfx::Shader*                     pVertexShader )
{
    NN_SDK_ASSERT_NOT_NULL( buffer );

    uint32_t vertexFormat = Shader::GetDefaultVertexFormat( shaderVariation );

    int attribCount = 0;

    info->SetDefault();

    int slot = -1;
    size_t  stride = 0;

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_Position" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Pos && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Position" );
        attrib.SetBufferIndex( 0 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attrib.SetOffset( 0 );
        stride += sizeof( nn::util::Float3 );
        attribCount++;
    }

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_Normal" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Normal && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_Normal" );
        attrib.SetBufferIndex( 0 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attrib.SetOffset( stride );
        stride += sizeof( nn::util::Float3 );
        attribCount++;
    }

    if ( pVertexShader != nullptr )
    {
        slot = pVertexShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "i_TexCoord" );
    }
    else
    {
        slot = 0;
    }

    if ( vertexFormat & VertexFormat_Uv && slot != -1 )
    {
        nn::gfx::VertexAttributeStateInfo& attrib = attribs[ attribCount ];
        attrib.SetDefault();
        attrib.SetNamePtr( "i_TexCoord" );
        attrib.SetBufferIndex( 0 );
        attrib.SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attrib.SetOffset( stride );
        stride += sizeof( nn::util::Float2 );
        attribCount++;
    }

    if ( 0 == attribCount )
    {
        return 0;
    }

    info->SetVertexAttributeStateInfoArray( attribs, attribCount );
    buffer->SetDefault();
    buffer->SetStride( stride );
    info->SetVertexBufferStateInfoArray( buffer, 1 );

    return attribCount;
}

bool Shader::CreateVertexState( Shader*            pShader,
                                ShaderVariation    shaderVariation,
                                nn::gfx::Shader*   pVertexShader,
                                nn::gfx::Device*   pGfxDevice,
                                nn::util::BytePtr& pMemory )
{
    // インターリブ版と非インターリブ版の２つを作成する。
    nn::gfx::VertexState::InfoType info;
    nn::gfx::VertexAttributeStateInfo attribs[ VertexAttribute_CountMax ];
    nn::gfx::VertexBufferStateInfo buffers[VertexAttribute_CountMax];

    int attribCount = InitializeVertexStateInfo(&info, attribs, buffers, shaderVariation, pVertexShader);
    if ( attribCount == 0 )
    {
        return false;
    }

    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );

    pShader->m_VertexState.SetMemory( pMemory.Get(), size );
    pMemory.Advance( size );
    pShader->m_VertexState.Initialize( pGfxDevice, info, pVertexShader );

    attribCount = InitializeInterleavedVertexStateInfo(&info, attribs, &buffers[0], shaderVariation, pVertexShader);
    if ( attribCount == 0 )
    {
        return false;
    }

    size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );

    pShader->m_VertexStateInterleaved.SetMemory( pMemory.Get(), size );
    pMemory.Advance( size );
    pShader->m_VertexStateInterleaved.Initialize( pGfxDevice, info, pVertexShader );

    return true;
}

void Shader::Cleanup( nn::gfx::Device* pGfxDevice )
{
    m_VertexState.Finalize( pGfxDevice );
    m_VertexStateInterleaved.Finalize( pGfxDevice );
    SetDefault();
}

Shader::Shader():
vertexFormat(VertexFormat_Default),
shaderVariationIndex(0),
shaderSlotCount(0)
{
    SetDefault();
}

Shader::~Shader()
{
    SetDefault();
}

void Shader::SetDefault()
{
    for( int idx = 0; idx < nn::gfx::ShaderStage_End; ++idx )
    {
        m_pGfxShader[ idx ] = nullptr;
    }

    slotView = ShaderSlotNone;
    for( int idx = 0; idx < Constants_ModelSlotCountMax; ++idx )
    {
        slotModel[ idx ] = ShaderSlotNone;
    }
    for( int idx = 0; idx < Constants_UserSlotCountMax; ++idx )
    {
        slotUser[ idx ] = ShaderSlotNone;
    }
}

void Shader::SetVertexShader( nn::gfx::Shader* pGfxShader )
{
    m_pGfxShader[ nn::gfx::ShaderStage_Vertex ] = pGfxShader;
}

void Shader::SetPixelShader( nn::gfx::Shader* pGfxShader )
{
    m_pGfxShader[ nn::gfx::ShaderStage_Pixel ] = pGfxShader;
}

nn::gfx::Shader* Shader::GetVertexShader()
{
    return m_pGfxShader[ nn::gfx::ShaderStage_Vertex ];
}

nn::gfx::Shader* Shader::GetPixelShader()
{
    return m_pGfxShader[ nn::gfx::ShaderStage_Pixel ];
}

const nn::gfx::VertexState* Shader::GetVertexState() const
{
    return &m_VertexState;
}

const nn::gfx::VertexState* Shader::GetInterleavedVertexState() const
{
    return &m_VertexStateInterleaved;
}

bool Shader::HasVertexAttribute( VertexAttribute attribute ) const
{
    return ( vertexFormat & attribute ) ? true : false;
}

//! @brief サンプラスロットを取得します。
int Shader::GetSamplerSlot( const char* name, const nn::gfx::ShaderStage shaderStage )
{
    nn::gfx::Shader* pShader = m_pGfxShader[ shaderStage ];
    return pShader->GetInterfaceSlot( shaderStage,
                       nn::gfx::ShaderInterfaceType_Sampler, name );
}

void Shader::UpdateSystemSlots()
{
    // スロットを取得します。
    nn::gfx::Shader* pVertexShader = m_pGfxShader[ nn::gfx::ShaderStage_Vertex ];
    nn::gfx::Shader* pPixelShader = m_pGfxShader[ nn::gfx::ShaderStage_Pixel ];

    if( pVertexShader != nullptr )
    {
        slotView = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "View" );

        slotModel[0] = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "Model" );

        slotUser[0] = pVertexShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Vertex,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "User" );
    }

    if( pPixelShader != nullptr )
    {
        slotModel[1] = pPixelShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "Model" );

        slotUser[1] = pPixelShader->GetInterfaceSlot(
                            nn::gfx::ShaderStage::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_ConstantBuffer,
                            "User" );

        slotSamplerPs[0] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "texture0" );

        slotSamplerPs[1] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "textureArray" );

        slotSamplerPs[2] = pPixelShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
                            nn::gfx::ShaderInterfaceType_Sampler, "textureCubemap" );
    }
}

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
