﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_GpuBuffer.h>
#include <nn/nn_SdkLog.h>

namespace nns
{
namespace gfx
{

//----------------------------------------
bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    const InitializeArg&  arg,
    nn::gfx::MemoryPool* pMemoryPool,
    size_t memoryPoolOffset
    )
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT(arg.m_BufferSize > 0);
    NN_SDK_ASSERT(arg.m_BufferCount >= 1);

    // バッファのアライメント値を保存
    m_Alignment = GetGpuBufferAlignement(pDevice, arg);
    // 渡されたメモリプールオフセットのアライメントチェック
    NN_SDK_ASSERT( (memoryPoolOffset & ( m_Alignment - 1 ))  == 0 );

    // アライメントしてバッファのサイズを決定
    m_Size = nn::util::align_up(arg.m_BufferSize, m_Alignment);

    // バッファの個数
    m_Count = arg.m_BufferCount;

    // nn::gfx::Bufferを確保する
    m_AllBufferSize = CalculateGpuBufferPoolSize(pDevice, arg);
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);
    info.SetSize(m_AllBufferSize);
    m_Buffer.Initialize(pDevice, info, pMemoryPool, memoryPoolOffset, m_AllBufferSize);

    m_isInitialized = true;

    return true;
}

//----------------------------------------
bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    nn::gfx::GpuAccess gpuAccess,
    size_t bufferSize,
    int bufferCount,
    nn::gfx::MemoryPool* pMemoryPool,
    size_t memoryPoolOffset
    )
{
    InitializeArg   arg;

    arg.SetGpuAccessFlag(gpuAccess);
    arg.SetBufferSize(bufferSize);
    arg.SetBufferCount(bufferCount);

    return Initialize(pDevice, arg, pMemoryPool, memoryPoolOffset);
}

//----------------------------------------
void GpuBuffer::Finalize(
    nn::gfx::Device* pDevice)
{
    NN_UNUSED(pDevice);

    m_Buffer.Finalize(pDevice);

    // 再利用に備えてすべてのパラメータをクリアする。
    m_Size = 0;
    m_Alignment = 0;
    m_Count = 0;
    m_AllBufferSize = 0;
    m_MappedBufferIndex = -1;
    m_pMappedPointer = NULL;
    m_isInitialized = false;
}

//----------------------------------------
void GpuBuffer::Map(int bufferIndex)
{
    if(m_MappedBufferIndex >= 0)
    {
        return ;
    }

    NN_SDK_ASSERT(m_isInitialized);

    m_MappedBufferIndex = bufferIndex;

    m_pMappedPointer = m_Buffer.Map();

    m_BufferAllocatedSize.m_Raw = 0;
}

//----------------------------------------
void GpuBuffer::Unmap()
{
    NN_SDK_ASSERT(m_isInitialized);

    if(m_MappedBufferIndex < 0)
    {
        return ;
    }

#if defined(NN_BUILD_APISET_CAFE)
    size_t  bufferAllocatedSize = m_BufferAllocatedSize.m_Raw;
    GX2EndianSwap(m_pMappedPointer, bufferAllocatedSize);
#endif

    m_Buffer.Unmap();

    m_MappedBufferIndex = -1;

    m_BufferAllocatedSize.m_Raw = 0;
}

//----------------------------------------
size_t GpuBuffer::GetGpuBufferAlignement(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg)
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);
    return nn::gfx::Buffer::GetBufferAlignment(pDevice, info);
}

//----------------------------------------
size_t GpuBuffer::CalculateGpuBufferPoolSize(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg)
{
    size_t alignmentSize = GetGpuBufferAlignement(pDevice, arg);
    return nn::util::align_up(arg.m_BufferSize, alignmentSize) * arg.m_BufferCount;
}

} // namespace gfx
} // namespace nns
