﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_FrameBuffer.h>
#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_DepthStencilBuffer.h>
#include <nn/nn_SdkLog.h>

namespace nns
{
namespace gfx
{
//--------------------------------------------------------------------------------------------------
FrameBuffer::FrameBuffer() NN_NOEXCEPT :
    m_Width(0),
    m_Height(0),
    m_pColorBuffer(NULL),
    m_pDepthBuffer(NULL)
{
    m_ClearColor[ClearColorComponent_Red] = 0.0f;
    m_ClearColor[ClearColorComponent_Green] = 0.0f;
    m_ClearColor[ClearColorComponent_Blue] = 0.0f;
    m_ClearColor[ClearColorComponent_Alpha] = 1.0f;
}

//--------------------------------------------------------------------------------------------------
size_t FrameBuffer::GetRequiredMemoryPoolSize(nn::gfx::Device* pGfxDevice, const InfoType& info)
{
    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetWidth(info.GetWidth());
    textureInfo.SetHeight(info.GetHeight());
    textureInfo.SetMipCount(1);

    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetImageFormat(info.GetColorBufferFormat());
    size_t colorBufferRequiredSize = nn::gfx::Texture::CalculateMipDataSize(pGfxDevice, textureInfo);
    size_t depthStencilBufferRequiredSize = 0;

    if (info.IsDepthBufferUsed())
    {
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture);
        textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        textureInfo.SetImageFormat(nn::gfx::ImageFormat_D32_Float);
        depthStencilBufferRequiredSize = nn::gfx::Texture::CalculateMipDataSize(pGfxDevice, textureInfo);
    }

    return colorBufferRequiredSize + depthStencilBufferRequiredSize;
}

//--------------------------------------------------------------------------------------------------
size_t FrameBuffer::GetMemoryPoolAlignment(nn::gfx::Device* pGfxDevice, const InfoType& info)
{
    NN_UNUSED(info);
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();
    bufferInfo.SetGpuAccessFlags(nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);
    return nn::gfx::Buffer::GetBufferAlignment(pGfxDevice, bufferInfo);
}

//--------------------------------------------------------------------------------------------------
void FrameBuffer::Initialize(
    nn::gfx::Device* pDevice,
    const InfoType* pInfoType,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pAllocateFunctionUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pInfoType);
    NN_ASSERT_NOT_NULL(pMemoryPool);
    NN_ASSERT_NOT_NULL(pAllocateFunction);
    NN_ASSERT_NOT_NULL(pFreeFunction);

    NN_ASSERT(pInfoType->GetWidth() < Max_Width);
    NN_ASSERT(pInfoType->GetHeight() < Max_Height);

    m_Width = pInfoType->GetWidth();
    m_Height = pInfoType->GetHeight();

    // カラーバッファを初期化
    void* pData = (*pAllocateFunction)(sizeof(nns::gfx::ColorBuffer), NN_ALIGNOF(nns::gfx::ColorBuffer), pAllocateFunctionUserData);
    NN_ASSERT_NOT_NULL(pData);

    size_t  offset = memoryPoolOffset;

    m_pColorBuffer = new(pData) nns::gfx::ColorBuffer();
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(pInfoType->GetWidth());
        info.SetHeight(pInfoType->GetHeight());
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(pInfoType->GetColorBufferFormat());
        info.SetMipCount(1);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);

        m_pColorBuffer->Initialize(pDevice, info, pMemoryPool, offset, size);
        offset += size;
    }

    // デプスバッファを初期化
    if (pInfoType->IsDepthBufferUsed())
    {
        pData = (*pAllocateFunction)(sizeof(nns::gfx::DepthStencilBuffer), NN_ALIGNOF(nns::gfx::DepthStencilBuffer), pAllocateFunctionUserData);
        NN_ASSERT_NOT_NULL(pData);
        m_pDepthBuffer = new(pData) nns::gfx::DepthStencilBuffer();
        {
            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetWidth(pInfoType->GetWidth());
            info.SetHeight(pInfoType->GetHeight());
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture);
            info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            info.SetImageFormat(pInfoType->GetDepthBufferFormat());
            info.SetMipCount(1);
            size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
            size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
            offset = nn::util::align_up(offset, align);
            m_pDepthBuffer->Initialize(pDevice, info, pMemoryPool, offset, size);
        }
    }

    // ビューポート&シザーを初期化
    SetupViewport(pDevice, 0, 0, m_Width, m_Height);
}

//--------------------------------------------------------------------------------------------------
void FrameBuffer::Finalize(
            nn::gfx::Device* pDevice,
            nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
            nn::FreeFunctionWithUserData pFreeFunction,
            void* pAllocateFunctionUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pAllocateFunction);
    NN_ASSERT_NOT_NULL(pFreeFunction);

    if (nn::gfx::IsInitialized(m_ViewportScissor))
    {
        m_ViewportScissor.Finalize(pDevice);
    }

    if (m_pColorBuffer)
    {
        m_pColorBuffer->Finalize(pDevice);
        (*pFreeFunction)(m_pColorBuffer, pAllocateFunctionUserData);
    }
    m_pColorBuffer = NULL;

    if (m_pDepthBuffer)
    {
        m_pDepthBuffer->Finalize(pDevice);
        (*pFreeFunction)(m_pDepthBuffer, pAllocateFunctionUserData);
    }
    m_pDepthBuffer = NULL;
}

//--------------------------------------------------------------------------------------------------
ptrdiff_t FrameBuffer::GetMemoryPoolOffset() const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pColorBuffer);

    return m_pColorBuffer->GetMemoryPoolOffset();
}

//--------------------------------------------------------------------------------------------------
void FrameBuffer::SetupViewport(nn::gfx::Device* pDevice, int x, int y, int width, int height) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    if (nn::gfx::IsInitialized(m_ViewportScissor))
    {
        m_ViewportScissor.Finalize(pDevice);
    }

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetOriginX(static_cast<float>(x));
        viewportInfo.SetOriginY(static_cast<float>(y));
        viewportInfo.SetWidth(static_cast<float>(width));
        viewportInfo.SetHeight(static_cast<float>(height));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetOriginX(x);
        scissorInfo.SetOriginY(y);
        scissorInfo.SetWidth(width);
        scissorInfo.SetHeight(height);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    m_ViewportScissor.Initialize(pDevice, info);
}

} // namespace gfx
} // namespace nns
