﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/nn_SdkAssert.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>
#include <nns/gfx/gfx_DebugGraphicsFramework.h>

namespace nns {
namespace gfx {

void DebugGraphicsFramework::Initialize(
    const FrameworkInfo& info,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pAllocateFunctionUserData
) NN_NOEXCEPT
{
    // GraphicsFramework の初期化
    GraphicsFramework::Initialize(info, pAllocateFunction, pFreeFunction, pAllocateFunctionUserData);

    // プリミティブレンダラーの初期化
    {
        nns::gfx::PrimitiveRenderer::RendererInfo rendererInfo;
        rendererInfo.SetDefault();
        rendererInfo.SetMultiBufferQuantity(m_BufferCount);
        rendererInfo.SetAllocator(m_AllocateFunction, m_pAllocateFunctionUserData);
        m_pRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(&m_Device, rendererInfo);
    }

    // デバッグフォントの初期化
    {
        void* ptr = AllocateMemory(sizeof(nns::gfx::GraphicsFramework::DebugFontTextWriter), 16);
        pFontWriter = new (ptr)nns::gfx::GraphicsFramework::DebugFontTextWriter();
        GraphicsFramework::InitializeDebugFontTextWriter(pFontWriter, 1024);
    }

    m_IsDebugInitialized = true;
}

void DebugGraphicsFramework::Initialize(const FrameworkInfo& info) NN_NOEXCEPT
{
    DebugGraphicsFramework::Initialize(
        info,
        DefaultAllocateFunction,
        DefaultFreeFunction,
        nullptr
    );
}

void DebugGraphicsFramework::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // デバッグフォントの解放
    FinalizeDebugFontTextWriter(pFontWriter);
    FreeMemory(pFontWriter);

    // プリミティブレンダラーの解放
    nns::gfx::PrimitiveRenderer::DestroyRenderer(m_pRenderer, &m_Device, m_FreeFunction, m_pAllocateFunctionUserData);

    // GraphicsFramework の終了処理
    GraphicsFramework::Finalize();

    m_IsDebugInitialized = false;
}

void DebugGraphicsFramework::InitializePrimitiveRenderer(DebugGraphicsFramework::PrimitiveRenderer* pPrimitiveRenderer, const nns::gfx::PrimitiveRenderer::RendererInfo& rendererInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    size_t memoryPoolSize = nns::gfx::PrimitiveRenderer::Renderer::GetRequiredMemoryPoolSize(&m_Device, rendererInfo);
    pPrimitiveRenderer->memoryPoolOffset = AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
        memoryPoolSize,
        nns::gfx::PrimitiveRenderer::Renderer::GetMemoryPoolAlignment(&m_Device, rendererInfo));

    pPrimitiveRenderer->object.Initialize(&m_Device, rendererInfo, nns::gfx::PrimitiveRenderer::GetGraphicsResource(), GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer), pPrimitiveRenderer->memoryPoolOffset, memoryPoolSize);
    pPrimitiveRenderer->object.SetScreenWidth(GetDisplayWidth());
    pPrimitiveRenderer->object.SetScreenHeight(GetDisplayHeight());
}

void DebugGraphicsFramework::FinalizePrimitiveRenderer(DebugGraphicsFramework::PrimitiveRenderer* pPrimitiveRenderer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, pPrimitiveRenderer->memoryPoolOffset);
    pPrimitiveRenderer->object.Finalize(&m_Device);
}

void DebugGraphicsFramework::InitializePerf(const nn::perf::LoadMeterCenterInfo& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(!IsPerfInitialized());

    // info をコピーしてフレームワーク用のバッファー数を設定する
    // FrameworkMode が変わった時に、フレームワーク内で perf を再構築する際に使用する
    {
        m_DebugPerf.info = info;
        switch (m_FrameworkMode)
        {
        case FrameworkMode_DeferredExecution:
        case FrameworkMode_DeferredSubmission:
            {
                m_DebugPerf.info.SetCpuBufferCount(2);
                m_DebugPerf.info.SetGpuBufferCount(3);
            }
            break;
        case FrameworkMode_Immediate:
            {
                m_DebugPerf.info.SetCpuBufferCount(2);
                m_DebugPerf.info.SetGpuBufferCount(2);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // perf の初期化
    {
        size_t memorySize = NN_PERF_GET_BUFFER_SIZE(m_DebugPerf.info);
        size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
        m_DebugPerf.pMemory = AllocateMemory(memorySize, memoryAlignment);

        size_t size = NN_PERF_GET_MEMORY_POOL_SIZE(&m_Device, m_DebugPerf.info);
        size_t alignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT(&m_Device, m_DebugPerf.info);
        m_DebugPerf.memoryPoolOffset = AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others, size, alignment);

        NN_PERF_INITIALIZE_METER(&m_Device, m_DebugPerf.info, m_DebugPerf.pMemory, memorySize, GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Others), m_DebugPerf.memoryPoolOffset, size);
        NN_PERF_SET_ENABLED(true);
    }

    // 計測用コマンドバッファーの初期化
    {
        nn::gfx::CommandBuffer::InfoType commandBufferInfo;
        commandBufferInfo.SetDefault();
        commandBufferInfo.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
        commandBufferInfo.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
        size_t commandMemorySize = 1024 * 1024 * 1;
        size_t controlMemorySize = 256;

         m_DebugPerf.pCommandBufferBegin = reinterpret_cast<GraphicsFramework::CommandBuffer*>(m_AllocateFunction(sizeof(GraphicsFramework::CommandBuffer) * m_BufferCount, NN_ALIGNOF(GraphicsFramework::CommandBuffer), m_pAllocateFunctionUserData));
         m_DebugPerf.pCommandBufferEnd = reinterpret_cast<GraphicsFramework::CommandBuffer*>(m_AllocateFunction(sizeof(GraphicsFramework::CommandBuffer) * m_BufferCount, NN_ALIGNOF(GraphicsFramework::CommandBuffer), m_pAllocateFunctionUserData));

        for (int bufferIndex = 0; bufferIndex < m_BufferCount; bufferIndex++)
        {
            new(&m_DebugPerf.pCommandBufferBegin[bufferIndex])GraphicsFramework::CommandBuffer;
            new(&m_DebugPerf.pCommandBufferEnd[bufferIndex])GraphicsFramework::CommandBuffer;
            InitializeCommandBuffer(&m_DebugPerf.pCommandBufferBegin[bufferIndex], commandBufferInfo, commandMemorySize, controlMemorySize);
            InitializeCommandBuffer(&m_DebugPerf.pCommandBufferEnd[bufferIndex], commandBufferInfo, commandMemorySize, controlMemorySize);
        }
    }

    // パフォーマンスメーターの初期化
    {
        void* ptr = AllocateMemory(sizeof(nns::gfx::PrimitiveRenderer::MeterDrawer), 16);
        m_DebugPerf.pMeterDrawer = new (ptr)nns::gfx::PrimitiveRenderer::MeterDrawer();
        m_DebugPerf.pMeterDrawer->SetDefault();
        m_DebugPerf.pMeterDrawer->SetDebugFontTextWriter(&pFontWriter->object);
        m_DebugPerf.pMeterDrawer->SetWidth(GetDisplayWidth() * 0.9f);
    }

    m_DebugPerf.isInitialized = true;
}

void DebugGraphicsFramework::InitializePerf() NN_NOEXCEPT
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(1);
    info.SetCpuSectionCountMax(128);
    info.SetGpuSectionCountMax(128);
    InitializePerf(info);
}

void DebugGraphicsFramework::FinalizePerf() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsPerfInitialized());

    // パフォーマンスメーターの解放
    FreeMemory(m_DebugPerf.pMeterDrawer);

    // 計測用コマンドバッファーの破棄
    for (int bufferIndex = 0; bufferIndex < m_BufferCount; bufferIndex++)
    {
        FinalizeCommandBuffer(&m_DebugPerf.pCommandBufferBegin[bufferIndex]);
        FinalizeCommandBuffer(&m_DebugPerf.pCommandBufferEnd[bufferIndex]);
    }

    // perf の終了処理
    {
        NN_PERF_SET_ENABLED(false);
        NN_PERF_FINALIZE_METER(&m_Device);
        FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others, m_DebugPerf.memoryPoolOffset);
        FreeMemory(m_DebugPerf.pMemory);
    }

    m_DebugPerf.isInitialized = false;
}

void DebugGraphicsFramework::BeginFrame(int bufferIndex) NN_NOEXCEPT
{
    GraphicsFramework::BeginFrame(bufferIndex);

    // プリミティブレンダラーの更新処理
    m_pRenderer->Update(bufferIndex);
}

void DebugGraphicsFramework::EndFrame(int bufferIndex) NN_NOEXCEPT
{
    EndFrame(bufferIndex, true);
}

void DebugGraphicsFramework::EndFrame(int bufferIndex, bool enabledCopyImage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    nn::gfx::CommandBuffer* pCommandBuffer = GetRootCommandBuffer(bufferIndex);

    // カラーバッファーからスキャンバッファーへコピー
    if (enabledCopyImage) {
        nn::gfx::Texture* pScanBuffer = GetScanBuffer(GetNextScanBufferIndex());
        nn::gfx::Texture* pColorBuffer = GetColorBuffer();
        NN_SDK_ASSERT_NOT_NULL(pColorBuffer);

        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(GetDisplayWidth());
        region.SetHeight(GetDisplayHeight());
        pCommandBuffer->BlitImage(pScanBuffer, region, pColorBuffer, region, 0);
    }

    if (IsPerfInitialized())
    {
        m_pRenderer->SetDefaultParameters();
        if (NN_STATIC_CONDITION(NN_PERF_IS_ENABLED()))
        {
            // パフォーマンスメーターをスキャンバッファーに描画
            nn::gfx::ColorTargetView* pTarget = GetScanBufferView(GetNextScanBufferIndex());
            pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

            // 描画位置を自動調整
            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(GetDisplayWidth() * 0.05f,
                GetDisplayHeight() - m_DebugPerf.pMeterDrawer->GetHeight(NN_PERF_GET_FRAME_METER()) - m_DebugPerf.pMeterDrawer->GetBarHeight());
            m_DebugPerf.pMeterDrawer->SetPosition(pos);
            m_DebugPerf.pMeterDrawer->Draw(pCommandBuffer, m_pRenderer, NN_PERF_GET_FRAME_METER());
        }
    }

    // デバッグフォントの描画
    pFontWriter->object.Draw(pCommandBuffer);

    EndRootCommandBuffer(bufferIndex);
}

void DebugGraphicsFramework::ProcessFrame() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());

    // InitializePerf() で初期化されてない場合は通常の ProcessFrame() を呼ぶ
    if (!IsPerfInitialized())
    {
        GraphicsFramework::ProcessFrame();
        return;
    }

    if (m_FrameworkMode != m_PrevFrameworkMode)
    {
        if (m_CurrentFrameIndex > 0)
        {
            if (m_PrevFrameworkMode == FrameworkMode_DeferredSubmission || m_PrevFrameworkMode == FrameworkMode_Immediate)
            {
                QueuePresentTexture(m_PresentInterval);
            }
            QueueFinish();
            bool isPrevPerfEnabled = NN_PERF_IS_ENABLED();
            FinalizePerf();
            InitializePerf(m_DebugPerf.info);
            NN_PERF_SET_ENABLED(isPrevPerfEnabled);
        }
        m_CurrentFrameIndex = 0;
        m_PrevFrameworkMode = m_FrameworkMode;
    }

    NN_PERF_BEGIN_FRAME();

    int currentBufferIndex = m_CurrentFrameIndex % m_BufferCount;
    int previousBufferIndex = (currentBufferIndex + m_BufferCount - 1) % m_BufferCount;
    int nextBufferIndex = (currentBufferIndex + 1) % m_BufferCount;

    switch (m_FrameworkMode)
    {
    case FrameworkMode_DeferredExecution:
        {
            NN_PERF_SET_COLOR(nn::util::Color4u8::Gray());
            NN_PERF_BEGIN_MEASURE_NAME("CPU");

            Calculate();
            AcquireTexture(currentBufferIndex);
            QueueWaitSync(currentBufferIndex);
            MakeDebugCommand(currentBufferIndex);
            ExecuteDebugCommand(currentBufferIndex);

            NN_PERF_END_MEASURE();

            WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);

            if (m_CurrentFrameIndex > 0)
            {
                WaitGpuSync(previousBufferIndex, m_WaitSyncTimeout);
            }
        }
        break;
    case FrameworkMode_DeferredSubmission:
        {
            NN_PERF_SET_COLOR(nn::util::Color4u8::Gray());
            NN_PERF_BEGIN_MEASURE_NAME("CPU");

            if (m_CurrentFrameIndex > 0)
            {
                ExecuteDebugCommand(previousBufferIndex);
            }

            Calculate();
            AcquireTexture(currentBufferIndex);
            MakeDebugCommand(currentBufferIndex);

            NN_PERF_END_MEASURE();

            WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);

            if (m_CurrentFrameIndex > 0)
            {
                WaitGpuSync(previousBufferIndex, m_WaitSyncTimeout);
            }
        }
        break;
    case FrameworkMode_Immediate:
        {
            if (m_CurrentFrameIndex == 0)
            {
                AcquireTexture(currentBufferIndex);
                WaitDisplaySync(currentBufferIndex, m_WaitSyncTimeout);
            }

            NN_PERF_SET_COLOR(nn::util::Color4u8::Gray());
            NN_PERF_BEGIN_MEASURE_NAME("CPU");

            Calculate();
            MakeDebugCommand(currentBufferIndex);
            ExecuteDebugCommand(currentBufferIndex);
            AcquireTexture(nextBufferIndex);

            NN_PERF_END_MEASURE();

            WaitDisplaySync(nextBufferIndex, m_WaitSyncTimeout);
            WaitGpuSync(currentBufferIndex, m_WaitSyncTimeout);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_CurrentFrameIndex++;
    if (m_CurrentFrameIndex > 10000000)
    {
        m_CurrentFrameIndex = (m_CurrentFrameIndex % m_BufferCount) + m_BufferCount;
    }

    NN_PERF_END_FRAME();

} // NOLINT(impl/function_size)

void DebugGraphicsFramework::MakeDebugCommand(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    ResetCommandBuffer(&m_DebugPerf.pCommandBufferBegin[bufferIndex]);
    m_DebugPerf.pCommandBufferBegin[bufferIndex].object.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Gray());
    NN_PERF_BEGIN_MEASURE_NAME_GPU(&m_DebugPerf.pCommandBufferBegin[bufferIndex].object, "GPU");
    m_DebugPerf.pCommandBufferBegin[bufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    m_DebugPerf.pCommandBufferBegin[bufferIndex].object.End();

    GraphicsFramework::MakeCommand(bufferIndex);

    ResetCommandBuffer(&m_DebugPerf.pCommandBufferEnd[bufferIndex]);
    m_DebugPerf.pCommandBufferEnd[bufferIndex].object.Begin();
    NN_PERF_END_MEASURE_GPU(&m_DebugPerf.pCommandBufferEnd[bufferIndex].object);
    m_DebugPerf.pCommandBufferEnd[bufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    m_DebugPerf.pCommandBufferEnd[bufferIndex].object.End();
}

void DebugGraphicsFramework::ExecuteDebugCommand(int bufferIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    GetQueue()->ExecuteCommand(&m_DebugPerf.pCommandBufferBegin[bufferIndex].object, NULL);
    GetQueue()->ExecuteCommand(GetRootCommandBuffer(bufferIndex), NULL);
    GraphicsFramework::QueuePresentTexture(m_PresentInterval);
    GetQueue()->ExecuteCommand(&m_DebugPerf.pCommandBufferEnd[bufferIndex].object, GetGpuFence(bufferIndex));
    GetQueue()->Flush();
}

} // namespace gfx
} // namespace nns
