﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/g3d/g3d_RenderModelObj.h>
#include <nns/g3d/g3d_Utility.h>

namespace nns { namespace g3d {

bool RenderUnitObj::Initialize(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel, int shapeIndex) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());
    NN_ASSERT_NOT_NULL(pModelObj);
    NN_ASSERT_NOT_NULL(pRenderModel);
    NN_ASSERT(pRenderModel->IsInitialized());

    m_PassCount = pRenderModel->GetPassCount();
    m_pRenderShapePtr = Allocate<RenderShape*>(sizeof(RenderShape*) * m_PassCount);

    bool isInitialized = true;
    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        m_pRenderShapePtr[passIndex] = pRenderModel->GetRenderShape(passIndex, shapeIndex);
        isInitialized &= m_pRenderShapePtr[passIndex]->IsInitialized();
    }

    nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
    if (!isInitialized)
    {
        // 確保したメモリを解放します。
        Free(m_pRenderShapePtr);
        m_pRenderShapePtr = NULL;
        m_PassCount = 0;

        NNS_G3D_WARNING("Failed to initialize RenderUnitObj for shape \"%s\". Because specified RenderShape was not initialized.", pShapeObj->GetName());
        return false;
    }

    // 同一リソースを用いているか確認
    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        NN_UNUSED(passIndex);
        NN_ASSERT(pShapeObj->GetResource() == m_pRenderShapePtr[passIndex]->GetResShape(),
            "ResShape is not the same.\npass index: %d\nshape name: \"%s\", \"%s\"\naddress: 0x%016x, 0x%016x",
            passIndex,
            pShapeObj->GetName(), m_pRenderShapePtr[passIndex]->GetResShape()->GetName(),
            pShapeObj->GetResource(), m_pRenderShapePtr[passIndex]->GetResShape());
    }

    m_pModelObj = pModelObj;
    m_pShapeObj = pShapeObj;
    m_ShapeIndex = shapeIndex;

    // ユニフォームブロックの管理領域を作成
    {
        const int bufferingCount = m_pModelObj->GetSkeleton()->GetBufferingCount();
        m_pUniformBlockPtr = Allocate<UniformBlock*>(sizeof(UniformBlock*) * m_PassCount);
        for (int passIndex = 0; passIndex < m_PassCount; passIndex++)
        {
            const nn::g3d::ShadingModelObj* pShadingModelObj = m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetShadingModelObj();
            const int uniformBlockCount = pShadingModelObj->GetResource()->GetUniformBlockCount();
            if (uniformBlockCount > 0)
            {
                // 必要なメモリーサイズを計算
                nn::util::MemorySplitter::MemoryBlock uniformBlockListMemoryBlock;
                nn::util::MemorySplitter::MemoryBlock uniformBlockBufferListMemoryBlock;
                uniformBlockListMemoryBlock.Initialize();
                uniformBlockBufferListMemoryBlock.Initialize();
                uniformBlockListMemoryBlock.SetSizeBy<UniformBlock>(NN_ALIGNOF(UniformBlock), uniformBlockCount);
                uniformBlockBufferListMemoryBlock.SetSizeBy<nn::gfx::Buffer*>(NN_ALIGNOF(nn::gfx::Buffer*), uniformBlockCount * bufferingCount);

                nn::util::MemorySplitter memorySplitter;
                memorySplitter.Initialize();
                memorySplitter.Append(&uniformBlockListMemoryBlock);
                memorySplitter.Append(&uniformBlockBufferListMemoryBlock);

                // メモリーを確保
                void* buffer = Allocate(memorySplitter.GetSize(), memorySplitter.GetAlignment());
                NN_ASSERT_NOT_NULL(buffer);
                memset(buffer, 0, sizeof(memorySplitter.GetSize()));

                m_pUniformBlockPtr[passIndex] = uniformBlockListMemoryBlock.Get<UniformBlock>(buffer);
                nn::util::BytePtr bytePtr(uniformBlockBufferListMemoryBlock.Get(buffer));
                for (int uniformBlockIndex = 0; uniformBlockIndex < uniformBlockCount; ++uniformBlockIndex)
                {
                    new(&m_pUniformBlockPtr[passIndex][uniformBlockIndex]) UniformBlock(bytePtr.Get(), bufferingCount);
                    bytePtr.Advance(sizeof(nn::gfx::Buffer*) * bufferingCount);
                }

                // RenderShape に ユニフォームブロックを設定
                m_pRenderShapePtr[passIndex]->RegisterUniformBlockList(m_pUniformBlockPtr[passIndex], uniformBlockCount);
            }
            else
            {
                m_pUniformBlockPtr[passIndex] = NULL;
            }
        }
    }

    // シェーダーストレージブロックの管理領域を作成
    {
        const int bufferingCount = m_pModelObj->GetSkeleton()->GetBufferingCount();
        m_pShaderStorageBlockPtr = Allocate<ShaderStorageBlock*>(sizeof(ShaderStorageBlock*) * m_PassCount);
        for (int passIndex = 0; passIndex < m_PassCount; passIndex++)
        {
            const nn::g3d::ShadingModelObj* pShadingModelObj = m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetShadingModelObj();
            const int shaderStorageBlockCount = pShadingModelObj->GetResource()->GetShaderStorageBlockCount();
            if (shaderStorageBlockCount > 0)
            {
                // 必要なメモリーサイズを計算
                nn::util::MemorySplitter::MemoryBlock shaderStorageBlockListMemoryBlock;
                nn::util::MemorySplitter::MemoryBlock shaderStorageBlockBufferListMemoryBlock;
                shaderStorageBlockListMemoryBlock.Initialize();
                shaderStorageBlockBufferListMemoryBlock.Initialize();
                shaderStorageBlockListMemoryBlock.SetSizeBy<ShaderStorageBlock>(NN_ALIGNOF(ShaderStorageBlock), shaderStorageBlockCount);
                shaderStorageBlockBufferListMemoryBlock.SetSizeBy<nn::gfx::Buffer*>(NN_ALIGNOF(nn::gfx::Buffer*), shaderStorageBlockCount * bufferingCount);

                nn::util::MemorySplitter memorySplitter;
                memorySplitter.Initialize();
                memorySplitter.Append(&shaderStorageBlockListMemoryBlock);
                memorySplitter.Append(&shaderStorageBlockBufferListMemoryBlock);

                // メモリーを確保
                void* buffer = Allocate(memorySplitter.GetSize(), memorySplitter.GetAlignment());
                NN_ASSERT_NOT_NULL(buffer);
                memset(buffer, 0, sizeof(memorySplitter.GetSize()));

                m_pShaderStorageBlockPtr[passIndex] = shaderStorageBlockListMemoryBlock.Get<ShaderStorageBlock>(buffer);
                nn::util::BytePtr bytePtr(shaderStorageBlockBufferListMemoryBlock.Get(buffer));
                for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < shaderStorageBlockCount; ++shaderStorageBlockIndex)
                {
                    new(&m_pShaderStorageBlockPtr[passIndex][shaderStorageBlockIndex]) ShaderStorageBlock(bytePtr.Get(), bufferingCount);
                    bytePtr.Advance(sizeof(nn::gfx::Buffer*) * bufferingCount);
                }

                // RenderShape に シェーダーストレージブロックを設定
                m_pRenderShapePtr[passIndex]->RegisterShaderStorageBlockList(m_pShaderStorageBlockPtr[passIndex], shaderStorageBlockCount);
            }
            else
            {
                m_pShaderStorageBlockPtr[passIndex] = NULL;
            }
        }
    }

    // ViewVolumeポインター用メモリー確保
    {
        size_t size = sizeof(nn::g3d::ViewVolume*) * pShapeObj->GetViewCount();
        if (size > 0)
        {
            void* buffer = Allocate(size);
            NN_ASSERT_NOT_NULL(buffer);

            m_pViewVolumePtr = reinterpret_cast<const nn::g3d::ViewVolume**>(buffer);
            memset(m_pViewVolumePtr, 0, size);
        }
    }

    // ICalculateLodLevelFunctorポインター用メモリー確保
    {
        size_t size = sizeof(nn::g3d::ICalculateLodLevelFunctor*) * pShapeObj->GetViewCount();
        if (size > 0)
        {
            void* buffer = Allocate(size);
            NN_ASSERT_NOT_NULL(buffer);

            m_pCalculateLodLevelFunctorPtr = reinterpret_cast<nn::g3d::ICalculateLodLevelFunctor**>(buffer);
            memset(m_pCalculateLodLevelFunctorPtr, 0, size);
        }
    }

    m_IsInitialized = true;
    return true;
} // NOLINT

void RenderUnitObj::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    Free(m_pCalculateLodLevelFunctorPtr);
    m_pCalculateLodLevelFunctorPtr = NULL;
    Free(m_pViewVolumePtr);
    m_pViewVolumePtr = NULL;

    // ユニフォームブロック管理領域を破棄
    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        if (m_pUniformBlockPtr[passIndex])
        {
            Free(m_pUniformBlockPtr[passIndex]);
            m_pUniformBlockPtr[passIndex] = NULL;
        }
    }
    Free(m_pUniformBlockPtr);
    m_pUniformBlockPtr = NULL;
    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        if (m_pShaderStorageBlockPtr[passIndex])
        {
            Free(m_pShaderStorageBlockPtr[passIndex]);
            m_pShaderStorageBlockPtr[passIndex] = NULL;
        }
    }
    Free(m_pShaderStorageBlockPtr);
    m_pShaderStorageBlockPtr = NULL;
    Free(m_pRenderShapePtr);
    m_pRenderShapePtr = NULL;

    m_IsInitialized = false;
}

void RenderUnitObj::BindUniformBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::UniformBlock::MaxIdSize - 1);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        BindUniformBlock(passIndex, id, pBufferPtr, bufferCount);
    }
}

void RenderUnitObj::BindUniformBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::UniformBlock::MaxIdSize - 1);

    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int uniformBlockIndex = pResShadingModel->FindUniformBlockIndex(id);
    if (uniformBlockIndex == nn::util::ResDic::Npos)
    {
        NNS_G3D_WARNING("Failed to bind uniform block to material \"%s\". Because uniform block \"%s\" is not found in shading model \"%s\".",
        m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetResMaterial()->GetName(), id, pResShadingModel->GetName());
        return;
    }

    const nn::g3d::ResUniformBlockVar* pResUniformBlockVar = pResShadingModel->GetUniformBlock(uniformBlockIndex);
    m_pUniformBlockPtr[passIndex][uniformBlockIndex].Initialize(id, pBufferPtr, pResUniformBlockVar->GetSize(), bufferCount);
}

void RenderUnitObj::UnbindUniformBlock(const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        UnbindUniformBlock(passIndex,id);
    }
}

void RenderUnitObj::UnbindUniformBlock(int passIndex, const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);

    UniformBlock* pUniformBlock = FindUniformBlock(passIndex, id);
    if (!pUniformBlock)
    {
        return;
    }
    pUniformBlock->Finalize();
}

nns::g3d::UniformBlock* RenderUnitObj::FindUniformBlock(int passIndex, const char* id) NN_NOEXCEPT
{
    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int blockCount = pResShadingModel->GetUniformBlockCount();
    for (int blockIndex = 0; blockIndex < blockCount; ++blockIndex)
    {
        UniformBlock& uniformBlock = m_pUniformBlockPtr[passIndex][blockIndex];
        if (strncmp(uniformBlock.GetId(), id, nns::g3d::UniformBlock::MaxIdSize) != 0)
        {
            continue;
        }

        return &uniformBlock;
    }
    return NULL;
}

void RenderUnitObj::BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::ShaderStorageBlock::MaxIdSize - 1);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        BindShaderStorageBlock(passIndex, id, pBufferPtr, bufferCount);
    }
}

void RenderUnitObj::BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::ShaderStorageBlock::MaxIdSize - 1);

    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int shaderStorageBlockIndex = pResShadingModel->FindShaderStorageBlockIndex(id);
    if (shaderStorageBlockIndex == nn::util::ResDic::Npos)
    {
        NNS_G3D_WARNING("Failed to bind shader storage block to material \"%s\". Because shader storage block \"%s\" is not found in shading model \"%s\".",
            m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetResMaterial()->GetName(), id, pResShadingModel->GetName());
        return;
    }

    const nn::g3d::ResShaderStorageBlockVar* pResShaderStorageBlockVar = pResShadingModel->GetShaderStorageBlock(shaderStorageBlockIndex);
    m_pShaderStorageBlockPtr[passIndex][shaderStorageBlockIndex].Initialize(id, pBufferPtr, pResShaderStorageBlockVar->GetSize(), bufferCount);
}

void RenderUnitObj::BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::ShaderStorageBlock::MaxIdSize - 1);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        BindShaderStorageBlock(passIndex, id, pBufferPtr, size, bufferCount);
    }
}

void RenderUnitObj::BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_NOT_NULL(pBufferPtr);
    NN_ASSERT_GREATER(bufferCount, 0);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::ShaderStorageBlock::MaxIdSize - 1);

    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int shaderStorageBlockIndex = pResShadingModel->FindShaderStorageBlockIndex(id);
    if (shaderStorageBlockIndex == nn::util::ResDic::Npos)
    {
        NNS_G3D_WARNING("Failed to bind shader storage block to material \"%s\". Because shader storage block \"%s\" is not found in shading model \"%s\".",
            m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetResMaterial()->GetName(), id, pResShadingModel->GetName());
        return;
    }

    m_pShaderStorageBlockPtr[passIndex][shaderStorageBlockIndex].Initialize(id, pBufferPtr, size, bufferCount);
}

void RenderUnitObj::UnbindShaderStorageBlock(const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        UnbindShaderStorageBlock(passIndex, id);
    }
}

void RenderUnitObj::UnbindShaderStorageBlock(int passIndex, const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);

    ShaderStorageBlock* pShaderStorageBlock = FindShaderStorageBlock(passIndex, id);
    if (!pShaderStorageBlock)
    {
        return;
    }
    pShaderStorageBlock->Finalize();
}

nns::g3d::ShaderStorageBlock* RenderUnitObj::FindShaderStorageBlock(int passIndex, const char* id) NN_NOEXCEPT
{
    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int blockCount = pResShadingModel->GetShaderStorageBlockCount();
    for (int blockIndex = 0; blockIndex < blockCount; ++blockIndex)
    {
        ShaderStorageBlock& shaderStorageBlock = m_pShaderStorageBlockPtr[passIndex][blockIndex];
        if (strncmp(shaderStorageBlock.GetId(), id, nns::g3d::ShaderStorageBlock::MaxIdSize) != 0)
        {
            continue;
        }

        return &shaderStorageBlock;
    }
    return NULL;
}

void RenderUnitObj::BindSampler(const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);
    NN_ASSERT(textureDescriptorSlot.IsValid());
    NN_ASSERT(samplerDescriptorSlot.IsValid());

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        BindSampler(passIndex, id, textureDescriptorSlot, samplerDescriptorSlot);
    }
}

void RenderUnitObj::BindSampler(int passIndex, const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);
    NN_ASSERT(textureDescriptorSlot.IsValid());
    NN_ASSERT(samplerDescriptorSlot.IsValid());

    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderShapePtr[passIndex]->GetShaderSelector()->GetShadingModel()->GetResource();
    int samplerIndex = pResShadingModel->FindSamplerIndex(id);
    if (samplerIndex == nn::util::ResDic::Npos)
    {
        NNS_G3D_WARNING("Failed to bind sampler to material \"%s\". Because sampler \"%s\" is not found in shading model \"%s\".",
                m_pRenderShapePtr[passIndex]->GetRenderMaterial()->GetResMaterial()->GetName(), id, pResShadingModel->GetName());
        return;
    }
    m_pRenderShapePtr[passIndex]->InitializeSampler(id, samplerIndex, textureDescriptorSlot, samplerDescriptorSlot);
}

void RenderUnitObj::UnbindSampler(const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        UnbindSampler(passIndex, id);
    }
}

void RenderUnitObj::UnbindSampler(int passIndex, const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);

    m_pRenderShapePtr[passIndex]->FinalizeSampler(id);
}

nns::g3d::Sampler* RenderUnitObj::FindSampler(int passIndex, const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);

    return m_pRenderShapePtr[passIndex]->FindSampler(id);
}

void RenderUnitObj::UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        UpdateShader(passIndex, pDevice);
    }
}

void RenderUnitObj::UpdateShader(int passIndex, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    m_pRenderShapePtr[passIndex]->UpdateShader(pDevice);
}

void RenderUnitObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingFunction) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT(nn::gfx::IsInitialized(*pCommandBuffer));
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_RANGE(bufferIndex, 0, m_pModelObj->GetSkeleton()->GetBufferingCount());
    NN_ASSERT_RANGE(viewIndex, 0, m_pShapeObj->GetViewCount());

    // ビジビリティー判定
    if (!m_pModelObj->IsShapeVisible(m_ShapeIndex))
    {
        return;
    }

    // フラスタムカリング
    if (m_pViewVolumePtr[viewIndex])
    {
        const nn::g3d::Sphere* pSphere = m_pShapeObj->GetBounding();
        if (pSphere && !m_pViewVolumePtr[viewIndex]->TestIntersection(*pSphere))
        {
            return;
        }
    }

    // ユーザー定義のコールバック関数によるカリング
    if (pDrawCullingFunction)
    {
        if (!pDrawCullingFunction(this, passIndex))
        {
            return;
        }
    }

    // 頂点バッファーを設定
    nn::gfx::GpuAddress gpuAddress;
    const nn::g3d::ResVertex* pResVertex = m_pShapeObj->GetResVertex();
    for (int vertexBufferIndex = 0, vertexBufferCount = m_pShapeObj->GetVertexBufferCount(); vertexBufferIndex < vertexBufferCount; ++vertexBufferIndex)
    {
        const nn::gfx::Buffer* pBuffer = m_pShapeObj->GetDynamicVertexBuffer(vertexBufferIndex, bufferIndex);
        if (!pBuffer || !m_pShapeObj->HasValidBlendWeight())
        {
            pBuffer = m_pShapeObj->GetVertexBuffer(vertexBufferIndex);
        }
        const nn::gfx::BufferInfo* pBufferInfo = m_pShapeObj->GetVertexBufferInfo(vertexBufferIndex);
        pBuffer->GetGpuAddress(&gpuAddress);

        pCommandBuffer->SetVertexBuffer(
            vertexBufferIndex,
            gpuAddress,
            pResVertex->GetVertexBufferStride(vertexBufferIndex),
            pBufferInfo->GetSize()
        );
    }


    // コマンドバッファーに設定の書き込み
    m_pRenderShapePtr[passIndex]->SetRenderStateTo(pCommandBuffer, this, viewIndex, bufferIndex);

    int maxLodLevel = m_pShapeObj->GetMeshCount() - 1;
    int meshIndex = (m_DrawLodLevel <= InvalidIndex) ? 0 : m_DrawLodLevel;
    meshIndex = (meshIndex > maxLodLevel) ? 0 : meshIndex;
    const nn::g3d::ResMesh* pResMesh = m_pShapeObj->GetResMesh(meshIndex);

     // サブメッシュをビューフラスタムカリングを行い描画
    if (m_pViewVolumePtr[viewIndex] && m_pShapeObj->GetResource()->GetSubMeshCount() > 1)
    {
        nn::g3d::ICalculateLodLevelFunctor* pCalculateLodLevelFunctor = m_pCalculateLodLevelFunctorPtr[viewIndex];
        nn::g3d::CullingContext ctx;

        bool canCalcLodSubmeshCulling = pCalculateLodLevelFunctor && m_DrawLodLevel == InvalidLod;
        if (canCalcLodSubmeshCulling)
        {
            while (m_pShapeObj->TestSubMeshLodIntersection(&ctx, *m_pViewVolumePtr[viewIndex], *pCalculateLodLevelFunctor))
            {
                int selectLodLevel = (ctx.submeshLodLevel < maxLodLevel) ? ctx.submeshLodLevel : maxLodLevel;
                if (m_DrawSingleSubmeshIndex == InvalidIndex)
                {
                    m_pShapeObj->GetResMesh(selectLodLevel)->DrawSubMesh(pCommandBuffer, ctx.submeshIndex, ctx.submeshCount);
                }
                // 特定のサブメッシュのみ描画
                else if (ctx.submeshIndex <= m_DrawSingleSubmeshIndex && m_DrawSingleSubmeshIndex < ctx.submeshIndex + ctx.submeshCount)
                {
                    m_pShapeObj->GetResMesh(selectLodLevel)->DrawSubMesh(pCommandBuffer, m_DrawSingleSubmeshIndex, 1);
                }
            }

            return;
        }

        while (m_pShapeObj->TestSubMeshIntersection(&ctx, *m_pViewVolumePtr[viewIndex]))
        {
            if (m_DrawSingleSubmeshIndex == InvalidIndex)
            {
                pResMesh->DrawSubMesh(pCommandBuffer, ctx.submeshIndex, ctx.submeshCount);
            }

            // 特定のサブメッシュのみ描画
            else if (ctx.submeshIndex <= m_DrawSingleSubmeshIndex && m_DrawSingleSubmeshIndex < ctx.submeshIndex + ctx.submeshCount)
            {
                pResMesh->DrawSubMesh(pCommandBuffer, m_DrawSingleSubmeshIndex, 1);
            }
        }

        return;
    }

    // 特定のサブメッシュのみ描画
    if (m_DrawSingleSubmeshIndex != InvalidIndex)
    {
        pResMesh->DrawSubMesh(pCommandBuffer, m_DrawSingleSubmeshIndex, 1);
        return;
    }

    pResMesh->Draw(pCommandBuffer);
}

void RenderUnitObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex) const NN_NOEXCEPT
{
    Draw(pCommandBuffer, passIndex, viewIndex, bufferIndex, NULL);
}

//-------------------------------------------------------------------------------------------------------------------------

void RenderModelObj::Initialize(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());
    NN_ASSERT_NOT_NULL(pModelObj);
    NN_ASSERT_NOT_NULL(pRenderModel);

    m_pRenderModel = pRenderModel;
    m_pModelObj = pModelObj;

    int shapeCount = m_pModelObj->GetShapeCount();
    if (shapeCount > 0)
    {
        m_pRenderUnitObj = Allocate<RenderUnitObj>(sizeof(RenderUnitObj) * shapeCount);
        NN_ASSERT_NOT_NULL(m_pRenderUnitObj);

        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            RenderUnitObj* pRenderUnitObj = new(&m_pRenderUnitObj[shapeIndex]) RenderUnitObj();
            pRenderUnitObj->Initialize(pModelObj, pRenderModel, shapeIndex);
        }
    }

    // マテリアルのシェーダーパラメータに初期値を設定
    {
        for (int passIndex = 0, passCount = pRenderModel->GetPassCount(); passIndex < passCount; ++passIndex)
        {
            for (int materialIndex = 0, materialCount = m_pModelObj->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
            {
                const nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(passIndex, materialIndex);
                if (!pRenderMaterial->IsMaterialShaderAssigned())
                {
                    continue;
                }

                nn::g3d::MaterialObj* pMaterialObj = m_pModelObj->GetMaterial(materialIndex);
                if (pMaterialObj->IsBlockBufferValid())
                {
                    nn::g3d::ShaderUtility::InitializeShaderParam(pMaterialObj, pRenderMaterial->GetResShadingModel());
                }

                // マテリアルブロックがシェーディングモデルのデフォルト値に設定されるため、ダーティフラグを設定します。
                pMaterialObj->ResetDirtyFlags();
            }
        }
    }

    m_IsInitialized = true;
}

void RenderModelObj::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    int shapeCount =  m_pModelObj->GetShapeCount();
    if (shapeCount > 0)
    {
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            if (m_pRenderUnitObj[shapeIndex].IsInitialized())
            {
                m_pRenderUnitObj[shapeIndex].Finalize();
            }
            m_pRenderUnitObj[shapeIndex].~RenderUnitObj();
        }
        Free(m_pRenderUnitObj);
        m_pRenderUnitObj = NULL;
    }

    m_IsInitialized = false;
}

void RenderModelObj::UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        m_pRenderUnitObj[shapeIndex].UpdateShader(pDevice);
    }
}

void RenderModelObj::UpdateShader(int passIndex, nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));

    for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        m_pRenderUnitObj[shapeIndex].UpdateShader(passIndex, pDevice);
    }
}

void RenderModelObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingCallback) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT(nn::gfx::IsInitialized(*pCommandBuffer));
    NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
    NN_ASSERT_RANGE(viewIndex, 0, m_pModelObj->GetViewCount());
    NN_ASSERT_RANGE(bufferIndex, 0, m_pModelObj->GetSkeleton()->GetBufferingCount());

    if (m_DrawSingleShapeIndex != InvalidIndex)
    {
        m_pRenderUnitObj[m_DrawSingleShapeIndex].Draw(pCommandBuffer, passIndex, viewIndex, bufferIndex, pDrawCullingCallback);
        return;
    }

    for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        m_pRenderUnitObj[shapeIndex].Draw(pCommandBuffer, passIndex, viewIndex, bufferIndex, pDrawCullingCallback);
    }
}

void RenderModelObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex) const NN_NOEXCEPT
{
    Draw(pCommandBuffer, passIndex, viewIndex, bufferIndex, NULL);
}

void RenderModelObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int viewIndex, int bufferIndex) const NN_NOEXCEPT
{
    Draw(pCommandBuffer, 0, viewIndex, bufferIndex, NULL);
}

void RenderModelObj::Draw(nn::gfx::CommandBuffer* pCommandBuffer, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingCallback) const NN_NOEXCEPT
{
    Draw(pCommandBuffer,0,viewIndex,bufferIndex,pDrawCullingCallback);
}

}}
