﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "dbgui_UserInputs.h"

#include "dbgui_Allocator.h"
#include "dbgui_ImguiIO.h"

#include <nn/nn_Assert.h>

#if defined(NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD)
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/os/os_MemoryHeapCommon.h>

#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_CharacterEncodingResult.h>
#include <nn/util/util_StringUtil.h>
#endif

namespace nns { namespace dbgui {

namespace
{

const float InputReferenceWidth = 1280.0;
const float InputReferenceHeight = 720.0;

struct InputCharacterConfig
{
    int     hidKeyIndex;
    char    ch;
    char    shiftCh;
};

const InputCharacterConfig g_InputCharacterConfigTable[] =
{
    { nn::hid::KeyboardKey::A::Index, 'a', 'A' },
    { nn::hid::KeyboardKey::B::Index, 'b', 'B' },
    { nn::hid::KeyboardKey::C::Index, 'c', 'C' },
    { nn::hid::KeyboardKey::D::Index, 'd', 'D' },
    { nn::hid::KeyboardKey::E::Index, 'e', 'E' },
    { nn::hid::KeyboardKey::F::Index, 'f', 'F' },
    { nn::hid::KeyboardKey::G::Index, 'g', 'G' },
    { nn::hid::KeyboardKey::H::Index, 'h', 'H' },
    { nn::hid::KeyboardKey::I::Index, 'i', 'I' },
    { nn::hid::KeyboardKey::J::Index, 'j', 'J' },
    { nn::hid::KeyboardKey::K::Index, 'k', 'K' },
    { nn::hid::KeyboardKey::L::Index, 'l', 'L' },
    { nn::hid::KeyboardKey::M::Index, 'm', 'M' },
    { nn::hid::KeyboardKey::N::Index, 'n', 'N' },
    { nn::hid::KeyboardKey::O::Index, 'o', 'O' },
    { nn::hid::KeyboardKey::P::Index, 'p', 'P' },
    { nn::hid::KeyboardKey::Q::Index, 'q', 'Q' },
    { nn::hid::KeyboardKey::R::Index, 'r', 'R' },
    { nn::hid::KeyboardKey::S::Index, 's', 'S' },
    { nn::hid::KeyboardKey::T::Index, 't', 'T' },
    { nn::hid::KeyboardKey::U::Index, 'u', 'U' },
    { nn::hid::KeyboardKey::V::Index, 'v', 'V' },
    { nn::hid::KeyboardKey::W::Index, 'w', 'W' },
    { nn::hid::KeyboardKey::X::Index, 'x', 'X' },
    { nn::hid::KeyboardKey::Y::Index, 'y', 'Y' },
    { nn::hid::KeyboardKey::Z::Index, 'z', 'Z' },

    { nn::hid::KeyboardKey::D1::Index, '1', '!' },
    { nn::hid::KeyboardKey::D2::Index, '2', '"' },
    { nn::hid::KeyboardKey::D3::Index, '3', '#' },
    { nn::hid::KeyboardKey::D4::Index, '4', '$' },
    { nn::hid::KeyboardKey::D5::Index, '5', '%' },
    { nn::hid::KeyboardKey::D6::Index, '6', '&' },
    { nn::hid::KeyboardKey::D7::Index, '7', '\'' },
    { nn::hid::KeyboardKey::D9::Index, '8', '(' },
    { nn::hid::KeyboardKey::D8::Index, '9', ')' },
    { nn::hid::KeyboardKey::D0::Index, '0', '\0' },

    { nn::hid::KeyboardKey::Space::Index, ' ', ' ' },
    { nn::hid::KeyboardKey::Minus::Index, '-', '_' },
    { nn::hid::KeyboardKey::Plus::Index, '=', '+' },
    { nn::hid::KeyboardKey::OpenBracket::Index, '[', '{' },
    { nn::hid::KeyboardKey::CloseBracket::Index, ']', '}' },
    { nn::hid::KeyboardKey::Pipe::Index, '\\', '|' },
    { nn::hid::KeyboardKey::Tilde::Index, '^', '~' },
    { nn::hid::KeyboardKey::Semicolon::Index, ';', ':' },
    { nn::hid::KeyboardKey::Quote::Index, '\'', '\"' },
    { nn::hid::KeyboardKey::Backquote::Index, '`', '~' },
    { nn::hid::KeyboardKey::Comma::Index, ',', '<' },
    { nn::hid::KeyboardKey::Period::Index, '.', '>' },
    { nn::hid::KeyboardKey::Slash::Index, '/', '?' },
    { nn::hid::KeyboardKey::NumPadDivide::Index, '/', '/' },
    { nn::hid::KeyboardKey::NumPadMultiply::Index, '*', '*' },
    { nn::hid::KeyboardKey::NumPadSubtract::Index, '-', '-' },
    { nn::hid::KeyboardKey::NumPadAdd::Index, '+', '+' },
    { nn::hid::KeyboardKey::NumPad1::Index, '1', '1' },
    { nn::hid::KeyboardKey::NumPad2::Index, '2', '2' },
    { nn::hid::KeyboardKey::NumPad3::Index, '3', '3' },
    { nn::hid::KeyboardKey::NumPad4::Index, '4', '4' },
    { nn::hid::KeyboardKey::NumPad5::Index, '5', '5' },
    { nn::hid::KeyboardKey::NumPad6::Index, '6', '6' },
    { nn::hid::KeyboardKey::NumPad7::Index, '7', '7' },
    { nn::hid::KeyboardKey::NumPad8::Index, '8', '8' },
    { nn::hid::KeyboardKey::NumPad9::Index, '9', '9' },
    { nn::hid::KeyboardKey::NumPad0::Index, '0', '0' },
    { nn::hid::KeyboardKey::NumPadDot::Index, '.', '.' },
    { nn::hid::KeyboardKey::Backslash::Index, '\\', '|' },
    { nn::hid::KeyboardKey::NumPadEquals::Index, '=', '=' },
    { nn::hid::KeyboardKey::NumPadComma::Index, ',', ',' },
    { nn::hid::KeyboardKey::Yen::Index, '\\', '|' },
    { nn::hid::KeyboardKey::Ro::Index, '\\', '_' },
};

} // anonymous namespace

UserInputs::UserInputs()
: m_PreviousKeyboardState()
, m_pAllocator(nullptr)
, m_ScalableViewport()
{
}

UserInputs::~UserInputs()
{
}

void UserInputs::Initialize(
    Allocator* pAllocator,
    int imguiDiplayWidth, int imguiDiplayHeight)
{
    m_pAllocator = pAllocator;

    io::SetupUserInputConfiguration();

    ResetPadNavigationInputs();
    ResetMouseAndTouchScreenInputs();

    nn::gfx::util::ScalableViewport::WindowCoordinate virtualWindowCoordinate;
    virtualWindowCoordinate.SetOriginMode(nn::gfx::util::ScalableViewport::OriginMode_UpperLeft);
    virtualWindowCoordinate.SetSize(
        static_cast<float>(imguiDiplayWidth),
        static_cast<float>(imguiDiplayHeight));

    nn::gfx::util::ScalableViewport::WindowCoordinate physicalWindowCoordinate;
    physicalWindowCoordinate.SetOriginMode(nn::gfx::util::ScalableViewport::OriginMode_UpperLeft);
    physicalWindowCoordinate.SetSize(InputReferenceWidth, InputReferenceHeight);

    m_ScalableViewport.Initialize(virtualWindowCoordinate, physicalWindowCoordinate);
}

void UserInputs::Finalize()
{
    m_pAllocator = nullptr;
}


void UserInputs::AddInputCharactersFromKeyboardState(const nn::hid::KeyboardState& keyboardState)
{
    bool shiftPressed = keyboardState.modifiers.Test<nn::hid::KeyboardModifier::Shift>();
    int inputCharacterTableCount = sizeof(g_InputCharacterConfigTable) / sizeof(g_InputCharacterConfigTable[0]);
    for (int inputCharacterIndex = 0; inputCharacterIndex < inputCharacterTableCount; ++inputCharacterIndex)
    {
        const InputCharacterConfig* pInputCharacterConfig = &g_InputCharacterConfigTable[inputCharacterIndex];
        int keyIndex = pInputCharacterConfig->hidKeyIndex;
        bool isPressed = keyboardState.keys.Test(keyIndex);
        bool wasPressed = m_PreviousKeyboardState.keys.Test(keyIndex);

        if (!wasPressed && isPressed)
        {
            char charToAdd = shiftPressed ? pInputCharacterConfig->shiftCh : pInputCharacterConfig->ch;
            io::AddInputCharacter(charToAdd);
        }
    }

    m_PreviousKeyboardState = keyboardState;
}

void UserInputs::AddInputCharacter(char c)
{
    io::AddInputCharacter(c);
}

void UserInputs::AddInputCharactersUtf8(const char* c)
{
    io::AddInputCharactersUtf8(c);
}

void UserInputs::SetKeyboardState(const nn::hid::KeyboardState& keyboardState)
{
    io::UpdateKeyboardState(keyboardState);
}

void UserInputs::ResetPadNavigationInputs()
{
    io::ResetPadNavigationInputs();
}

void UserInputs::SetDebugPadState(
    const nn::hid::DebugPadButtonSet& debugPadButtonSet,
    const nn::hid::AnalogStickState& analogStickState)
{
    int32_t stickActivationThreshold = 4096;
    io::UpdateNavigationFromDebugPadState(
        debugPadButtonSet, analogStickState, stickActivationThreshold);
}

void UserInputs::ResetMouseAndTouchScreenInputs()
{
    io::ResetMouseInputs();
}

void UserInputs::SetMouseState(const nn::hid::MouseState& mouseState)
{
    nn::hid::MouseState scaledMouseState = mouseState;

    ConvertCoordinatesInputToImgui(&scaledMouseState.x, &scaledMouseState.y);
    ConvertLengthInputToImgui(&scaledMouseState.deltaX, &scaledMouseState.deltaY);

    io::UpdateMouseState(scaledMouseState);
}

void UserInputs::SetTouchState(const nn::hid::TouchState& touchState)
{
    nn::hid::TouchState scaledTouchState = touchState;

    ConvertCoordinatesInputToImgui(&scaledTouchState.x, &scaledTouchState.y);
    ConvertLengthInputToImgui(&scaledTouchState.diameterX, &scaledTouchState.diameterY);

    io::UpdateMouseStateFromTouchState(scaledTouchState);
}

void UserInputs::SetNpadState(
    const nn::hid::NpadButtonSet& buttons,
    const nn::hid::AnalogStickState& analogStickState)
{
    int32_t stickActivationThreshold = 4096;
    io::UpdateNavigationFromNpadState(buttons, analogStickState, stickActivationThreshold);
}

#if defined(NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD)
void UserInputs::UpdateSoftwareKeyboard()
{
    if (!io::FinishTextInput())
    {
        return;
    }

    if (io::WantTextInput())
    {
        io::TextInputProperties textInputProperties = {};
        io::BeginTextInput(&textInputProperties);
        ShowSoftwareKeyboard(&textInputProperties);
    }
}

void UserInputs::ShowSoftwareKeyboard(const io::TextInputProperties* pTextInputProperties)
{
    // Exit parameter settings
    size_t stringBufferSize = nn::swkbd::GetRequiredStringBufferSize();
    void* pStringBuffer = m_pAllocator->AllocateMemory(stringBufferSize, nn::os::MemoryPageSize);

    // keyboard settings
    nn::swkbd::ShowKeyboardArg showKeyboardArg = {};
    nn::swkbd::MakePreset(&showKeyboardArg.keyboardConfig, nn::swkbd::Preset_Default);
    showKeyboardArg.keyboardConfig.keyboardMode = nn::swkbd::KeyboardMode_Full;
    showKeyboardArg.keyboardConfig.inputFormMode =
        pTextInputProperties->isMultiline ? nn::swkbd::InputFormMode_MultiLine : nn::swkbd::InputFormMode_OneLine;
    showKeyboardArg.keyboardConfig.isPredictionEnabled = true;

    // Guide string settings
    const char* guide_string = nullptr;
    nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig, guide_string);

    // Allocate buffers for shared memory
    size_t inHeapSize = nn::swkbd::GetRequiredWorkBufferSize(false);
    void* swkbdWorkBuffer = m_pAllocator->AllocateMemory(inHeapSize, nn::os::MemoryPageSize);
    showKeyboardArg.workBuf = swkbdWorkBuffer;
    showKeyboardArg.workBufSize = inHeapSize;

    nn::util::ConvertStringUtf8ToUtf16Native(
        reinterpret_cast<uint16_t*>(pStringBuffer), stringBufferSize, pTextInputProperties->pInitialText);
    nn::swkbd::SetInitialText(
        &showKeyboardArg, reinterpret_cast<const char16_t*>(pStringBuffer));

    // キーボードの表示
    nn::swkbd::String outputString;
    outputString.ptr = pStringBuffer;
    outputString.bufSize = stringBufferSize;

    nn::Result result = nn::swkbd::ShowKeyboard(&outputString, showKeyboardArg);
    if (result.IsSuccess())
    {
        uint16_t* pOutputStringUtf16 = reinterpret_cast<uint16_t*>(outputString.ptr);
        int outputStringIndex = 0;
        char convertUtf8Buffer[4];

        while (pOutputStringUtf16[outputStringIndex] != 0)
        {
            memset(convertUtf8Buffer, 0, sizeof(convertUtf8Buffer));
            nn::util::CharacterEncodingResult encodingResult =
                nn::util::ConvertCharacterUtf16NativeToUtf8(
                    convertUtf8Buffer, pOutputStringUtf16 + outputStringIndex);
            NN_ASSERT(encodingResult == nn::util::CharacterEncodingResult_Success);
            io::AddInputCharactersUtf8(convertUtf8Buffer);
            outputStringIndex++;
        }
    }

    m_pAllocator->FreeMemory(outputString.ptr);
    m_pAllocator->FreeMemory(swkbdWorkBuffer);
}
#endif


void UserInputs::ConvertCoordinatesImguiToInput(int* pX, int* pY)
{
    float posX = static_cast<float>(*pX);
    float posY = static_cast<float>(*pY);

    m_ScalableViewport.ConvertPointVirtualToPhysical(
        &posX, &posY,
        posX, posY);

    *pX = static_cast<int>(posX);
    *pY = static_cast<int>(posY);
}

void UserInputs::ConvertCoordinatesInputToImgui(int* pX, int* pY)
{
    float posX = static_cast<float>(*pX);
    float posY = static_cast<float>(*pY);

    m_ScalableViewport.ConvertLengthPhysicalToVirtual(
        &posX, &posY,
        posX, posY);

    *pX = static_cast<int>(posX);
    *pY = static_cast<int>(posY);
}

void UserInputs::ConvertLengthImguiToInput(int* pOffsetX, int* pOffsetY)
{
    float offsetX = static_cast<float>(*pOffsetX);
    float offsetY = static_cast<float>(*pOffsetY);

    m_ScalableViewport.ConvertLengthVirtualToPhysical(
        &offsetX, &offsetY,
        offsetX, offsetY);

    *pOffsetX = static_cast<int>(offsetX);
    *pOffsetY = static_cast<int>(offsetY);
}

void UserInputs::ConvertLengthInputToImgui(int* pOffsetX, int* pOffsetY)
{
    float offsetX = static_cast<float>(*pOffsetX);
    float offsetY = static_cast<float>(*pOffsetY);

    m_ScalableViewport.ConvertLengthPhysicalToVirtual(
        &offsetX, &offsetY,
        offsetX, offsetY);

    *pOffsetX = static_cast<int>(offsetX);
    *pOffsetY = static_cast<int>(offsetY);
}


} } // namespace nns { namespace dbgui {

